#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2006 Jd, Hap Hazard & Yves Perrenoud
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU General Public License (GPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/gpl.txt
#
import types
import gtk, gtk.glade, gobject
from xen.xend.XendProtocol import XendError
import xmlrpclib

import xen.xm.create
import vte, re, os, sys, rpm, platform, glob

import socket
import datetime 

from phelper import PHelper
from NodeProxy import Node
import dialogs
from dialogs import CreateDialog, DomSettings, InitParamsDialog, AddNodeDialog
from dialogs import CredentialsHelper
from dialogs import wtree, paused_pb, resident_pb, not_resident_pb, node_pb, pool_pb, unknown_pb
from dialogs import cleanupQCDomain, file_selection, checkAndSetDefaultPaths
from dialogs import main_context

import utils
from utils import is_host_remote
from constants import *

import traceback
import xml.parsers.expat

import subprocess

#
# defaults
#

default_tcp_port = "8005"
default_path = "/RPC2"
default_user = "root"
default_host = 'localhost'

## Add from xenauto or auto_start flag in the domconfig/dom.
# So this is not required
XENAUTO = '/etc/xen/auto'   # pick it up from managed node


#
# class definitions
#

from ManagedNode import ManagedNode,NodeException
from XenNode import XenNode,XenDomain
# _RESTRUCTURING


class LeftNav:
    """
    Keeps track of the left side navigation
    List of Managed Nodes as well as virtual machines running on them.
    """
    # Two types of nodes in the tree. MangedNode and Doms
    SERVER_POOL = "SERVER_POOL"
    MANAGED_NODE = "NODE"
    DOMAIN  = "DOMAIN"

    # NodeList Constructor
    def __init__(self, wtree):
        """
        constructor, initialize the widgets
        Tree populated with nodes from grid manager
        Assumes manager as global variable.
        """
        self.wtree = wtree
        self.left_nav_view = wtree.get_widget("DomView")
        self.left_nav_view.get_selection().connect("changed",
                                               self.handle_selection)
        self.left_nav_view.connect("row-activated",
                                   self.handle_row_activated)

        self.left_nav_view.connect("row-expanded",
                                self.handle_row_expanded)


        # initialize columns
        pbrenderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn("State", pbrenderer)
        column.set_cell_data_func(pbrenderer, self.get_state_pixbuf)
        self.left_nav_view.append_column(column)

        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("Name", textrenderer, text=0)

        self.left_nav_view.append_column(column)

        self.left_nav_tree_model = gtk.TreeStore(gobject.TYPE_STRING,
                                                 gobject.TYPE_STRING,
                                                 gobject.TYPE_PYOBJECT)

        # Add a sorter inbetween
        #sorted_model = gtk.TreeModelSort(self.left_nav_tree_model)
        #sorted_model.set_sort_column_id(0, gtk.SORT_ASCENDING)

        sorted_model = self.left_nav_tree_model

        # This is also behaving strange, when timer is enabled.
        # some multi-threaded problem!!
        #sorted_model.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.left_nav_view.set_model(sorted_model)
        
        
        ## Does not work..! property not available.!
        #self.left_nav_view.set_property("show-expanders", True)
        #self.left_nav_view.set_show_expanders(True)

        self.refresh_nodes()


    # Cell renderers for the colums in the tree view.
    def get_state_pixbuf(self, column, renderer, model, iter):
        """ Call back to display the state column image"""
        dom_name = model.get_value(iter,0)
        node_type = model.get_value(iter,1)
        managed_node =  model.get_value(iter,2)
        
        if node_type is None :
            return

        pb = unknown_pb
        
        if node_type == self.MANAGED_NODE:
            pb = node_pb
        elif managed_node is not None and node_type == self.DOMAIN: # dom
            dom = managed_node.get_dom(dom_name)
            
            if dom and dom.is_resident:
                try:
                    state = dom["state"]
                except xmlrpclib.Fault :
                    state = None

                if state is not None and state[2] == 'p':
                    pb = paused_pb
                else:
                    pb = resident_pb
            else:
                pb = not_resident_pb
        elif node_type == self.SERVER_POOL:
            pb = pool_pb

            
        renderer.set_property('pixbuf', pb)



    # signal handlers
    def handle_row_activated(self, treeview, path, column):
        """
        handle double-click on the managed node.
        """
        model = treeview.get_model()
        iter = model.get_iter(path)
        if iter:
            node_type = model.get_value(iter,1)
            tree_node = model.get_value(iter,2)
            if node_type == self.MANAGED_NODE:
                if self._refresh_doms(model, tree_node, iter, path, path):
                    path = model.get_path(iter)
                    treeview.expand_row(path, True)
            elif node_type == self.DOMAIN:
                Coordinator.handle_row_activated()

        return True


    # A node in the tree is expanded/collapsed
    def handle_row_expanded(self, treeview, iter, path, userdata=None):
        model = treeview.get_model()
        iter = model.get_iter(path)
        if iter:
            node_type = model.get_value(iter,1)
            tree_node = model.get_value(iter,2)
            
            if node_type == self.MANAGED_NODE:
                self._refresh_doms(model, tree_node, iter, path, path)

                return False

        return True

        
    def handle_selection(self, widget=None):
        """Handles the selection of a domain in the domain tree view"""

        Coordinator.handle_nav_selection(widget)


    def refresh_doms(self, auto=False, dom_name = None):
        """ take current selection and refresh dom list."""
        tree_selection = self.left_nav_view.get_selection()
        model, iter = tree_selection.get_selected()
        if not iter :
            # no selection
            return True

        node_type = model.get_value(iter,1)
        selected_node = model.get_value(iter,2)

        if node_type == self.DOMAIN:
            # current selection on dom, get the managed node
            parent_iter = model.iter_parent(iter)
            managed_node = model.get_value(parent_iter,2)
            path = model.get_path(parent_iter)
            current_path = model.get_path(iter)
            return self._refresh_doms(model, managed_node, parent_iter, path,
                                      current_path, auto)
        elif node_type == self.MANAGED_NODE:
            # current selection is managed node itself
            managed_node = selected_node
            path = model.get_path(iter)
            return self._refresh_doms(model, managed_node, iter, path, path, auto)



    # utility methods
    def _append_dummy_row(self, iter):
        self.left_nav_tree_model.append(iter, ["Domains not found",
                                               "DUMMY",None])
    
    def _refresh_doms(self, model, managed_node,
                      iter, path, current_path,
                      auto = False):
        """
        refresh dom list for the managed node, refered by iter
        """
        
        # This is done in the reverse way, as the normal way
        # does not work. We add the new nodes and remove the old
        # ones later.Need to be careful, at some point there
        # would be more than one instance of the same dom here!!
        # Also, the order matters. (careful, when we add sorting!)
        
        #print managed_node.hostname, managed_node.is_authenticated()
        if isinstance(managed_node, ManagedNode) and iter:
            if auto and (not managed_node.is_authenticated() or \
                                            managed_node.is_in_error()):
                return True

            num_old_doms = model.iter_n_children(iter)            
            # get new ones and insert them
            # catch the expact error. Intermittently we get this because
            # server returns an empty string. (dont know why, yet)
            try:
                managed_node.refresh()
                doms = managed_node.get_dom_names()
            except xml.parsers.expat.ExpatError, e:
                #print "_refresh_doms() :", e
                return                
            except Exception, e:
                showmsg('Exception '+str(e))
                return False
            

            doms.sort()
            # add the dummy row back if no current doms
            if len(doms) == 0:
                self._append_dummy_row(iter)

            for dom_name in doms:
                #if not dom.is_resident:
                #    Coordinator.handle_kill_dom(dom)
                node = model.iter_children(iter)
                exists = False
                while node:
                    d_name = model.get_value(node,0)
                    node_type = model.get_value(node,1)
                    m_node = model.get_value(node,2)

                    #update the managed node instance if required.
                    if m_node is not None and m_node is not managed_node:
                        model.set_value(node,2, managed_node)
                        
                    if node_type == self.DOMAIN and d_name == dom_name:
                        exists = True
                        break

                    if d_name > dom_name:  # keep things in sorted order
                        break

                    node= model.iter_next(node)
                    
                if not exists:
                    #print "adding new node", dom_name
                    model.insert_before(iter,node,
                                       [dom_name,self.DOMAIN,managed_node,])

            # clear old nodes that do not exist
            next_node = model.iter_children(iter)
            while next_node: 
                this_node = next_node
            
                dom_name = model.get_value(this_node,0)
                node_type = model.get_value(this_node,1)

                if dom_name is not None and dom_name != "" and \
                       node_type == self.DOMAIN:

                    if dom_name in doms:
                        d = managed_node.get_dom(dom_name)
                        if not managed_node.isResident(dom_name):
                            Coordinator.handle_kill_dom(d)
                        else:
                            Coordinator.handle_existing_dom(d)

                        next_node = model.iter_next(this_node)
                            
                    else:
                        #print "removing " , dom_name
                        Coordinator.handle_missing_dom(managed_node,
                                                       dom_name)
                        next_node = model.iter_next(this_node)
                        model.remove(this_node)
                        
                else:
                    #print "removing ", dom_name
                    next_node = model.iter_next(this_node)
                    model.remove(this_node)



            # set  selection to current selection
            try:
                selection = self.left_nav_view.get_selection()
                if auto: # suppress callbacks
                    Coordinator.fromUserAction = False
                selection.select_path(current_path)
            finally:
                Coordinator.fromUserAction = True
        else :
            print "refresh dom called without Managed Node"
            print managed_node.__class__, iter

        return True
    
    def refresh_nodes(self):
        """
        refresh the nodes.
        Not expected to be called other than constructor.
        
        """
        nodes = manager.getNodeList()
        self.left_nav_tree_model.clear()
        # Append a Global Server Pool
        g_iter = self.left_nav_tree_model.append(None,
                                                 ["Global Server Pool",
                                                  self.SERVER_POOL,
                                                  None])
                                        
        # Kludge, append a blank node so the expanders show up
        for node in nodes.itervalues():
            iter = \
                 self.left_nav_tree_model.append(g_iter,
                                                [node.hostname,
                                                 self.MANAGED_NODE,
                                                 node,])
            self._append_dummy_row(iter)

        self.left_nav_view.expand_row((0,),False)
        

    def add_node(self, node):
        if node.hostname not in manager.getNodeNames():
            manager.addNode(node)
            self.refresh_nodes()


    def remove_node(self, node):
        manager.removeNode(node.hostname)
        client_config.removeHost(node.hostname)
        self.refresh_nodes()

    # returns tuple with name, node_type and managed node
    def get_selection(self):
        """
        return current selection 
        returns either ManagedNode,  Dom or None
        """
        widget = self.left_nav_view.get_selection()
        treemodel, treeiter = widget.get_selected()
        if treeiter:
            return (treemodel.get_value(treeiter, 0),
                    treemodel.get_value(treeiter, 1),
                    treemodel.get_value(treeiter, 2))

        return (None,None,None)

    def get_selection_type(self):
        (name, node_type, managed_node) = self.get_selection()
        return node_type

    def get_managed_node(self):
        """
        return managed node for the currently selected dom
        """
        (name,node_type, managed_node) = self.get_selection()
        return managed_node
    
    def set_selection(self, model_node):
        """
        set the current selection to either a Dom or a ManagedNode
        """
        # this may be tricky.
        
    def addDomFile(self, filename):
        """Adds a dom specified by it's file name to the index"""
        try:
            managed_node = self.get_managed_node()
            dom = managed_node.add_dom_config(filename)
            self.refresh_doms()
        except (Exception, StandardError), e:
            showmsg('FAILED. Invalid File: '+str(e))


    def remDomFile(self, filename):
        """Remove a dom specified by it's file name to the index"""
        managed_node = self.get_managed_node()
        if managed_node.remove_dom_config(filename):
            self.refresh_doms()
        else:
            return False
        


class Coordinator:
    """ Class that takes care of cordination """
    # currently the toolbar, menu and popup are assumed to be part of
    # cordinator, later can be moved out in to a separate class



    # going with static methods (can make it singleton later)
    # Also, this is not a pure cordinator (yet) but central place
    # for all interactiona.

    nav = None
    notebook = None
    fromUserAction = True

    # timer vars
    skipcount = 0
    stat_refresh_rate = 5
    stat_refresh_count = stat_refresh_rate

##  Singleton bit later.
##     @classmethod
##     def instance():
##         return self._instance

##     @classmethod
##     def initialize(nav, notebook):
##         self._instance = Coordinator(nav, notebook)

##     def __init__(self,nav, notebook):
##         self.nav = nav
##         self.notebook = notebook
    
    
    @classmethod
    def set_nav(cls,nav):
        cls.nav = nav

    @classmethod
    def set_notebook(cls, notebook):
        cls.notebook =  notebook
    

    @classmethod
    def handle_row_activated(cls):
        if cls.nav == None:
            return
        
        (name, node_type, managed_node) = cls.nav.get_selection()
        if managed_node is None:
            return

        if node_type == LeftNav.DOMAIN:
            if managed_node.isDomU(name):
                if managed_node.isResident(name):
                    dom = managed_node.get_dom(name)
                    cls.notebook.showTab(dom)

       
    @classmethod
    def handle_nav_selection(cls, widget):
        cls.stateRefresh()
        if cls.nav == None or cls.fromUserAction == False:
            return


        (name, node_type, managed_node) = cls.nav.get_selection()
        if managed_node is None:
            return
        if not cls.fromUserAction:
            return
        
        #Handle if the selection was a Dom on a Node

        if node_type == LeftNav.DOMAIN:
            cls.notebook.showDomTab(managed_node,name)

        # handle node selection
        elif node_type==LeftNav.MANAGED_NODE:
            if managed_node.is_authenticated() \
                   and not managed_node.is_in_error():
                cls.notebook.showSummaryTab()
            



    @classmethod
    def handle_timer_callback(cls):
        if cls.nav is None : return
        try:
            try:
                cls.fromUserAction = False
                cls.nav.refresh_doms(True)

                cls.stat_refresh_count -= 1
                if cls.stat_refresh_count <= 0:
                   cls.stat_refresh_count = cls.stat_refresh_rate
                   # fetch stats only if home tab is shown
                   if (cls.notebook.getTabPos() == 0):
                       cls.notebook.showSummaryTab()

                if cls.skipcount:
                    cls.skipcount -= 1
                    return True

                Coordinator.stateRefresh()
            except:
                print "Exception in callback...ignoring ", \
                      traceback.print_exc()
                pass
        finally:
            cls.fromUserAction = True
            
        return True


    @classmethod
    def handle_save_dom(cls, dom):
        cls.notebook.removeTab(dom)
        
    @classmethod
    def handle_missing_dom(cls, managed_node, dom_name):
        cls.notebook.removeTabUsingName(managed_node,dom_name)

    @classmethod
    def handle_remove_dom(cls, dom):
        cls.notebook.removeTab(dom)

    @classmethod
    def handle_shutdown_dom(cls, dom):
        cls.notebook.removeTab(dom)
        
    @classmethod
    def handle_existing_dom(cls, dom):
        """ Handle refresh of existing dom"""
        # allow notebook to potentially clean up
        cls.notebook.refresh_tab(dom)

    @classmethod
    def handle_kill_dom(cls, dom):
        cls.notebook.removeTab(dom)
        

    @classmethod
    def stateRefresh(cls):
        """Refreshes UI elements based on current dom selection"""
        if cls.nav == None: return
        
        (name,node_type,managed_node) = cls.nav.get_selection()
        #if node_type != LeftNav.DOMAIN:
        #    return 


        pausebutton = wtree.get_widget("pausebutton")
        pausemenu  = wtree.get_widget("menu_pause")
        pausepopup = wtree.get_widget("Popup_pause")

        remove = ("MenuRemove", "PopupRemove", "delete", "delete_domain")
        
        for widget in ("MenuSettings", "PopupSettings"):
            wtree.get_widget(widget).set_sensitive(True)

        
        restore = node_type == LeftNav.MANAGED_NODE
        for action in ("restore",):
            wtree.get_widget(action + "button").set_sensitive(restore)
            wtree.get_widget("menu_" + action).set_sensitive(restore)
            wtree.get_widget("Popup_" + action).set_sensitive(restore)

        if node_type != LeftNav.DOMAIN or \
           (node_type == LeftNav.DOMAIN and  managed_node.isDom0(name)):

            for action in ("start", "pause", "reboot", "shutdown", "kill",
                           "snapshot", "show_console"):
                wtree.get_widget(action + "button").set_sensitive(False)
                wtree.get_widget("menu_" + action).set_sensitive(False)
                wtree.get_widget("Popup_" + action).set_sensitive(False)

            
            cls.fromUserAction = False
            pausebutton.set_active(False)
            cls.fromUserAction = True

            pausemenu.set_sensitive(False)
            pausepopup.set_sensitive(False)

            for widget in remove:
                wtree.get_widget(widget).set_sensitive(False)

            return
        else:
            dom_is_resident = managed_node.isResident(name)

            for widget in ("startbutton", "menu_start", "Popup_start"):
                wtree.get_widget(widget)\
                                          .set_sensitive(not dom_is_resident)

            for action in ("pause", "reboot", "shutdown", "kill", "snapshot",
                           "show_console"):
                wtree.get_widget(action + "button")\
                                        .set_sensitive(dom_is_resident)
                wtree.get_widget("menu_" + action)\
                                         .set_sensitive(dom_is_resident)
                wtree.get_widget("Popup_" + action)\
                                          .set_sensitive(dom_is_resident)

            isauto = False
            #_RESTRUCT
            dom = managed_node.get_dom(name)
            if dom.get_config() is not None and \
                   dom.get_config().filename.startswith(XENAUTO):
                isauto = True
            

            for widget in remove:
                if isauto or dom_is_resident:
                    wtree.get_widget(widget).set_sensitive(False)
                else:
                    wtree.get_widget(widget).set_sensitive(True)
            

            if dom_is_resident:
                state = dom["state"]
            else:
                state = None

            cls.fromUserAction = False
            if state:
                pausemenu.set_sensitive(True)
                pausepopup.set_sensitive(True)

                if (state[2] == 'p'):
                    pausebutton.set_active(True)
                    cls.notebook.pausedTab(dom, True)
                    
                    for widget in (pausebutton, pausemenu.child,
                                   pausepopup.child):
                        widget.set_label("Resume")
                else:
                    pausebutton.set_active(False)
                    cls.notebook.pausedTab(dom, False)
                    
                    for widget in (pausebutton, pausemenu.child,
                                   pausepopup.child):
                        widget.set_label("Pause")
            else:
                pausebutton.set_active(False)
                for widget in (pausemenu.child, pausepopup.child):
                    widget.set_sensitive(False)
            cls.fromUserAction = True


        

class NoteBook:
    """This class manages the gtk.NoteBook"""

    
    def __init__(self, left_nav):
        self.notebook = left_nav.wtree.get_widget("MainNotebook")
        self.left_nav = left_nav
        self.summary_tab = SummaryTab(wtree, left_nav)
        self.info_tab = InfoTab(wtree, left_nav)
        
        self.nbooklist = [0,0]
        self.terms = {}
        
        # add handlers : connecting through glade not working.
        self.notebook.connect("switch_page", \
                              self.on_notebook_switch_page)

    def _getDomToken(self, managed_node,dom_name):
        if managed_node is None or dom_name is None:
            return None
        else:
            return managed_node.hostname + ","  + dom_name


    def __getDomToken(self, dom):
        """ std way of generating unique id for a given dom"""
        if dom is None:
            #print "ERROR : getDomToken :dom is None"
            return None
        if dom.node is None :
            #print "ERROR : getDomToken :dom.node is None"
            return None
        return self._getDomToken(dom.node,dom.name)


    def showDomTab(self,managed_node, dom_name):
        if self.getTabPos()== 1: 
            self.showInfoTab()


    def showInfoTab(self,widget=None,dom=None):
        if self.notebook.get_current_page() == 1:
            self.info_tab.refresh()
        else:
            self.notebook.set_current_page(1)
        

    def showSummaryTab(self, widget=None, manged_node=None):
        if self.notebook.get_current_page() == 0:
            self.summary_tab.refresh()
        else:
            self.notebook.set_current_page(0)
            
    def showTab(self, dom):
        """Displays the tab for the requested Dom or Dom ID"""
        dom_token = self.__getDomToken(dom)

        if dom_token in self.terms.keys():
            self.notebook.set_current_page(self.nbooklist.index(dom_token))
        else:
            self.__addTab(dom)
            self.pausedTab(dom, False)

    def removeTabUsingName(self, managed_node, dom_name):
        dom_token = self._getDomToken(managed_node,dom_name)
        if dom_token:
            self.__removeTab(dom_token)

    def removeTab(self, dom):
        dom_token = self.__getDomToken(dom)
        if dom_token is not None:
            self.__removeTab(dom_token)

    def __removeTab(self, dom_token):
        """Removes the tab for the requested Dom ID"""
        if dom_token in self.terms.keys():
            self.notebook.remove_page(self.nbooklist.index(dom_token))
            del self.terms[dom_token]
            self.nbooklist.remove(dom_token)


    def getTabPos(self):
        return self.notebook.get_current_page()

    def pausedTab(self, dom, state):
        """Set the sensitive state of the contents of a tab specified by
        the requested Dom or Dom ID"""

        dom_token = self.__getDomToken(dom)
        
        if dom_token in self.terms.keys():
            term_tab = self.terms[dom_token]
            term_tab.showPaused(state)

    def __addTab(self, dom):
        """Adds a console tab to the notebook for the specified Dom"""

        # assume that dom is only passed.
        #if not isinstance(dom, Dom):
        #    return

        dom_token = self.__getDomToken(dom)
        if dom_token:
            term_tab = TerminalTab(self, dom, dom_token)
            self.terms[dom_token] = term_tab
            self.nbooklist.append(dom_token)

        
    def closeTab(self, widget, dom_token):
        """Handles the request to close a notebook tab"""
        self.__removeTab(dom_token)


    # TODO : add the tab at the same position as itwas.
    def refresh_tab(self, dom):
        """ If a tab for the same dom with old id exists, clean it up"""
        
        if dom:
            dom_token = self.__getDomToken(dom)
            if dom_token in self.terms.keys():
                tab = self.terms[dom_token]
                if tab.dom_id != dom.id:
                    self.__removeTab(dom_token)
                    self.__addTab(dom)
                

    #### NOTE THIS IS NOT CALLED FROM ANYWHERE.
    #### MAY BE RETIRED (NOT CUTOVER FOR MULTINODE)
    def cleanup(self):
        """Goes through the list of notebook tabs and checks if there is a
        corresponding Dom and what it's state is, in order to cleanup tabs
        that no longer should be displayed"""

        changes = 0

        
        for dom_token in self.terms.keys():
            try:
                dom = self.terms[dom_token]
                dom.refresh()
            except NodeException:
                self.__removeTab(dom_token)
                changes += 1
                continue
            #except Exception, e:
            #    print 'domid: ',domid
            #    continue

        if changes:
            # create new tab if required (after reboot)
            self.left_nav.domselected()

            
    def on_notebook_switch_page(self, widget, page, page_no):
        """if the user switched to info tab, populate with the
        dom information """
        if page_no == 1:
            self.info_tab.refresh()
        else:
            self.info_tab.set_default_title()
        if page_no == 0:
            self.summary_tab.refresh()
        else:
            self.summary_tab.set_default_title()

        return True



class TerminalTab:
    """ Class represents terminal tab for a dom """

    def __init__(self, notebook, dom, dom_token):
        
        self.notebook = notebook
        
        
        self.term = vte.Terminal()
        self.savedTerm = None

        #self.dom = dom
        self.host_name = dom.node.hostname
        self.dom_id = dom.id 
        self.dom_name = dom.name

        self.dom_token = dom_token
        
        managed_node = dom.node

        # determine the architecture for the node        
        if re.search('64', managed_node.environ[prop_env_MACHINE_TYPE]):
            arch_libdir = 'lib64'
        else:
            arch_libdir = 'lib'
            
        if not managed_node.is_remote():            
            self.term.fork_command("/usr/"+arch_libdir+"/xen/bin/xenconsole",
                                   ("xenconsole", str(self.dom_id)))
        else:
            hostname = managed_node.hostname
            if managed_node.username is not None:
                hostname = managed_node.username + "@" + hostname
            self.term.fork_command("ssh",
                                   ("xenconsole","-t",
                                    managed_node.username+ "@" +
                                    managed_node.hostname,
                                    "/usr/"+arch_libdir+"/xen/bin/xenconsole",
                                    str(self.dom_id)))

        
        self.tbox = gtk.VBox()
        self.tbox.add(self.term)
        self.tbox.show_all()

        self.nbox = gtk.HBox()
        self.event_box = gtk.EventBox()
        self.tab_name = gtk.Label(self.dom_name)
        self.event_box.add(self.tab_name)
        self.nbox.pack_start(self.event_box)
        self.closebutton = gtk.Button()
        self.closebutton.set_relief(gtk.RELIEF_NONE)
        self.closebutton.set_size_request(20, 17)
        self.closeimage = gtk.Image()
        self.closeimage.set_from_stock(gtk.STOCK_CLOSE,
                                       gtk.ICON_SIZE_MENU)
        self.closebutton.add(self.closeimage)
        self.closebutton.connect("clicked", self.__closeTab, self.dom_token)

        """ Some other time...
        infobutton = gtk.Button()
        infobutton.set_relief(gtk.RELIEF_NONE)
        infobutton.set_size_request(28, 28)
        info_image = gtk.Image()
        info_image.set_from_stock(gtk.STOCK_INFO,
                                  gtk.ICON_SIZE_MENU)
        infobutton.add(info_image)
        infobutton.connect("clicked", self.notebook.showInfoTab, dom)
        nbox.pack_start(infobutton)
        """
        self.nbox.pack_start(self.closebutton)
        self.tool_tip = gtk.Tooltips()

        self.tool_tip.set_tip(self.event_box, managed_node.hostname)
        self.nbox.show_all()

        # add the tab to the notebook
        # Kludge : done like this because append page requires
        # tab content (tbox etc)
        page_id = self.notebook.notebook.append_page(self.tbox, self.nbox)
        self.notebook.notebook.set_current_page(page_id)


    def __closeTab(self, widget, dom_token):
        self.notebook.closeTab(widget, dom_token)

    def showPaused(self,paused):
        """
        show paused or normal console for a running dom
        """
        
        if paused:
            if not self.savedTerm:
                self.savedTerm = self.term
                self.tbox.remove(self.term)
                
                label = gtk.Label()
                label.set_markup('<span size="xx-large">Paused</span>')
                self.tbox.add(label)
                self.tbox.show_all()
                
        elif self.savedTerm:
            self.savedTerm = None
            self.tbox.remove(self.tbox.get_children()[0])
            self.tbox.add(self.term)


class InfoTab:
    """ Class representing information tab for current selection """
    
    def __init__(self, wtree, left_nav):
        self.left_nav = left_nav
        self.info_tree_model = gtk.TreeStore(*[gobject.TYPE_STRING]*2)
        self.info_tree_view = wtree.get_widget("info_tree")
        self.tab_name = wtree.get_widget("info_tree_tab_label")
        textrenderer = gtk.CellRendererText()

        column = gtk.TreeViewColumn("Configuration", textrenderer, text=0)
        self.info_tree_view.append_column(column)

        column = gtk.TreeViewColumn("", textrenderer, text=1)
        self.info_tree_view.append_column(column)

        """
        column = gtk.TreeViewColumn("Value", textrenderer, text=2)
        self.info_tree_view.append_column(column)
        """
        self.info_tree_view.set_model(self.info_tree_model)
        self.info_tree_view.show()


    def refresh(self):
        (name, node_type, managed_node) = self.left_nav.get_selection()
        if managed_node is None:
            #clean up info tab with "No selection"
            return
        
        self.info_tree_model.clear()

        if  node_type != LeftNav.DOMAIN:
            env_iter  = self.info_tree_model.append(None,["Environment",""])
            self.populateEnvInfo(env_iter, managed_node)
            self.info_tree_view.expand_all()
            return

        # self.tab_name.set_text("Information" + "\n" + name)
        gen_iter  = self.info_tree_model.append(None,["General",""])
        boot_iter = self.info_tree_model.append(None,["Boot",""])
        res_iter  = self.info_tree_model.append(None,["Resource",""])

        dom = managed_node.get_dom(name)
        self.populateGeneralInfo(gen_iter, dom)
        self.populateBootInfo(boot_iter, dom)
        self.populateResourceInfo(res_iter,dom)
        self.info_tree_view.expand_all()

    def set_default_title(self):
        pass
        # self.tab_name.set_text("Information")

    def populateEnvInfo(self, iter, managed_node):
        for name in managed_node.environ:
            value = managed_node.environ[name]
            if type(value) == int:
                value = str(value)
            self.info_tree_model.append(iter, [name,value])
            

    def append_info_node(self, iter, dom,  name, param):
        """ get the value from dom index and append Name / Value in the
        tree at 'iter' position."""
        
        value = dom[param]

        if value:
            if type(value) == int:
                value = str(value)
            self.info_tree_model.append(iter, [name,value])
        else:
            config = dom.get_config()
            if config:
                value = config[param]
                if value:
                    if type(value) == int:
                        value = str(value)
                    self.info_tree_model.append(iter, [name,value])
                    return

            self.info_tree_model.append(iter, [name,"N/A"])

    
            
    def populateGeneralInfo(self, iter, dom):
        for name, param in ( ("Name", "name"),
                             ("FileName", "filename"),
                             ):
            self.append_info_node(iter, dom, name, param)
        

    def populateBootInfo(self, iter, dom):
        for name, param in ( 
                             ("Kernel", "kernel"),
                             ("Ramdisk", "ramdisk"),
                             ("Bootloader", "bootloader"),

                             ("On Crash", "on_crash"),
                             ("On Reboot", "on_reboot"),):
            self.append_info_node(iter, dom,name, param)


    def populateResourceInfo(self, iter, dom):
        for name, param in ( 
                             ("Memory", "memory"),
                             ("CPU", "vcpus"),
                             ("Network", "vif"),
                             ("Disks", "disk"),
                             ):
            self.append_info_node(iter, dom, name, param)




class SummaryTab:
    """ Summary information of all running doms"""
    # Can add some more information about the node.
    def __init__(self, wtree, left_nav):
        self.left_nav = left_nav

        self.statmodel = gtk.ListStore(*[gobject.TYPE_STRING]*6)

        statview = wtree.get_widget("statTable")
        self.tab_name = wtree.get_widget("summaryTabLabel")
        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("Id", textrenderer, text=0)
        statview.append_column(column)
        column = gtk.TreeViewColumn("Name", textrenderer, text=1)
        statview.append_column(column)
        column = gtk.TreeViewColumn("State", textrenderer, text=2)
        statview.append_column(column)
        column = gtk.TreeViewColumn("Memory", textrenderer, text=3)
        statview.append_column(column)
        rt_textrenderer = gtk.CellRendererText()
        rt_textrenderer.set_property("xalign", 1.0)
        column = gtk.TreeViewColumn("CPU Time(sec)", rt_textrenderer, text=4)
        statview.append_column(column)
        column = gtk.TreeViewColumn("", textrenderer, text=5)
        statview.append_column(column)
        statview.set_model(self.statmodel)
        statview.show()


    def get_state_string(self,dom):
        if not dom:
            return "Unknown"

        state = dom._state()
        if state == XenDomain.RUNNING: return "Running"
        elif state ==  XenDomain.BLOCKED : return "Blocked"
        elif state ==  XenDomain.PAUSED : return "Paused"
        elif state == XenDomain.SHUTDOWN: return "Shutdown"
        elif state == XenDomain.CRASHED: return "Crashed"
        elif state == XenDomain.UNKNOWN: return "Unknown"
        
        return "Unknown"
        

    # For refresh_stats, clearing the model isn't ideal as the selection
    # highlight gets cleared at each refresh, which looks very odd to the
    # user... the append()/remove() method used in Left_nav.refresh() for
    # instance, doesn't suffer that problem - should be fixed

    def refresh(self):
        """Refreshes the statistics on the home tab"""

        (name, node_type,managed_node)  = self.left_nav.get_selection()
        if managed_node is None : return
        if not managed_node.is_authenticated(): return
        if managed_node.is_in_error():
            return
        
        #self.tab_name.set_text("Summary" + "\n" + managed_node.hostname)
        self.statmodel.clear()

        doms = managed_node.get_dom_names()
        doms.sort()
            
        for dom_name in doms:
            dom = managed_node.get_dom(dom_name)
            if dom.is_resident:
                iter = self.statmodel.insert_before(None, None)
                self.statmodel.set(iter,
                                   0, dom.id,
                                   1, dom.name,
                                   2, self.get_state_string(dom),
                                   3, dom["memory"],
                                   4, "%12.3f" % float(dom["cpu_time"]),
                                   5,""
                                   )

    def set_default_title(self):
        pass
        #self.tab_name.set_text("Summary")
        

class Timer:
    """This class initializes a GTK call back loop that runs one of it's
    methods on a regular basis to perform refreshes and such operations"""



    def __init__(self):
        self.timeoutid = gobject.timeout_add(2000, self.__callback)

    def __callback(self):
        """This is the callback that handles the refresh"""
        return Coordinator.handle_timer_callback()



#
# function definitions
#

def validate_node_selected():
    if left_nav.get_selection_type() == LeftNav.MANAGED_NODE:
        return True
    else:
        showmsg("Select a host for this operation")
        return False


def quitProgram(*args):
    """Quit the execution of the program"""

    gtk.main_quit()

def show_about(*args):
    """handler that shows the about dialog"""

    wtree.get_widget('AboutDialog').show()

def hide_about(widget, *args):
    """handler that hides the about dialog"""

    widget.hide()
    return True


def savedom(widget):
    """Take Snapshot of a running dom. (a.k.a save)"""

    (name,node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        checkAndSetDefaultPaths(wtree, managed_node)
        dom = managed_node.get_dom(name)
        snap_dir = managed_node.config.get(utils.XMConfig.PATHS,prop_snapshots_dir)
        default_ext = managed_node.config.get(utils.XMConfig.PATHS,prop_snapshot_file_ext);
        snap_file = dom.name
        if default_ext is not None:
            snap_file += default_ext
        mainwin = wtree.get_widget('MainWindow')
        (res, fname) = file_selection(managed_node,
                                      "Save Snapshot", "save", snap_dir,
                                      snap_file,
                                      parentwin = mainwin)
        if res and fname:
            try:
                dom._save(fname)
            except socket.error, e:
                print "error ", e
            
        Coordinator.handle_save_dom(dom)
        left_nav.refresh_doms()

def restoredom(widget):
    """Restore a dom from a file."""

    if not validate_node_selected():
        return
    managed_node = left_nav.get_managed_node()
    checkAndSetDefaultPaths(wtree, managed_node)
    snap_dir = managed_node.config.get(utils.XMConfig.PATHS,prop_snapshots_dir)
    mainwin = wtree.get_widget('MainWindow')
    (res, fname) = file_selection(managed_node,
                                  "Restore from Snapshot","open",
                                  snap_dir,
                                  parentwin = mainwin)
    if res and fname:
        if managed_node is not None:
            managed_node.restore_dom(fname)
            managed_node.refresh()
            left_nav.refresh_doms()

def rebootdom(widget):
    """Reboot the currently selected domain"""
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        if confirmation("Reboot " + dom.name + " on " + managed_node.hostname + "?"):
            dom._reboot()
            managed_node = dom.node
            managed_node.refresh()
            left_nav.refresh_doms()

def shutdowndom(widget):
    """shutdown the currently selected domain"""

    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)

        if confirmation("Shutdown " + dom.name+ " on " + managed_node.hostname + "?"):
            dom._shutdown()
    
            managed_node = dom.node
            left_nav.refresh_doms()

            

def startdom(widget):
    """Start the selected domain"""
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
    else:
        return
    

    if not managed_node.isResident(dom.name):
        #_RESTRUCT
        #errmsgs = left_nav.cursel.validate()

        #if errmsgs:
        #    errmsg = "Can't start the domain because of the following " + \
        #             "problems:\n"

        #    for msg in errmsgs:
        #        errmsg += " - " + msg + "\n"

        #    showmsg(errmsg)
        #    return

        try:
            #_RESTRUCT
            #left_nav.cursel = left_nav.cursel.start()
            dom._start()
        except Exception, e:
            showmsg("Error starting " + dom.name +". "+ str(e))
            return

        # _RESTRUCT
        managed_node = dom.node
        left_nav.refresh_doms()


def killdom(widget):
    """Kills the currently selected domain"""
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)

        if (confirmation("Kill " + dom.name + " on " + managed_node.hostname+ "?")):
            # sometime the xen destroy gives socket error after destorying vm
            # lets ignore it for now.
            try:
                dom._destroy()                
                #left_nav.refresh()
            except (socket.error, xmlrpclib.Fault):
                pass

            
            # _RESTRUCT
            managed_node = dom.node
            left_nav.refresh_doms()
            Coordinator.handle_kill_dom(dom)

def pausedom(widget):
    """Pause the currently selected domain"""

    # We use set_active on the pause button, which may fire
    # unnecessary pasuedom calls, to prevent this the callback from the
    # UI actually sets fromUserAction to True.
    # Note : There must be a better way of doing this. 
    if not Coordinator.fromUserAction: return

    need_refresh = False
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        
        state = dom._state()
        if state == XenDomain.PAUSED:
            if confirmation("Resume " + dom.name + " on " + managed_node.hostname + "?"):
                dom._resume()
                need_refresh = True


        else:
            if confirmation("Pause " + dom.name + " on " + managed_node.hostname+ "?"):
                dom._pause()
                need_refresh = True

        if need_refresh:
            managed_node = dom.node
            left_nav.refresh_doms()

            Coordinator.stateRefresh()
                

def popupmenu(widget, event):
    """Checks whether the right mouse button was clicked within the domain
    TreeView and if so, displays a popup menu"""

    if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
        (name, node_type, managed_node) = left_nav.get_selection()
        if node_type == LeftNav.DOMAIN:
            wtree.get_widget("PopupMenu").popup(None, None, None, event.button,
                                                event.time)
        elif node_type == LeftNav.MANAGED_NODE:
            wtree.get_widget("NodePopup").popup(None, None, None, event.button,
                                            event.time)
        elif node_type == LeftNav.SERVER_POOL:
            wtree.get_widget("PoolPopup").popup(None, None, None, event.button,
                                                event.time)
            

def opendom(widget):
    """Handles the open domain menu item which adds a dom by file name to
    the list of doms in the UI"""
    if not validate_node_selected():
        return
    managed_node = left_nav.get_managed_node()
    
    mainwin = wtree.get_widget('MainWindow')
    result, filename = \
            file_selection(managed_node,
                           "Please select a Dom configuration file", "open",
                           managed_node.config.get(utils.XMConfig.PATHS,prop_xenconf_dir),
                           parentwin = mainwin)

    if result and filename:
        left_nav.addDomFile(filename)

def deletedom(widget):
    """Handles the menu items requesting to delete of a dom from the
    left_nav"""
    
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        msg = "Delete " + dom.name + " on " + managed_node.hostname+ "?"
        msg += "\n" + \
               "WARNING : Related VBDs/Volumes would also be deleted."
        if confirmation(msg):
            host = dom.node.hostname
            cleanupQCDomain(managed_node, dom.name)
            left_nav.remDomFile(dom.get_config().filename)
            Coordinator.handle_remove_dom(dom)

def removedom(widget):
    """Handles the menu items requesting to removal of a dom from the
    left_nav"""
    
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        msg = "Remove " + dom.name + " from list under " + \
              managed_node.hostname+ "?"

        if confirmation(msg):
            host = dom.node.hostname
            left_nav.remDomFile(dom.get_config().filename)
            Coordinator.handle_remove_dom(dom)


def show_create_dialog(widget):
    if not validate_node_selected():
        return
    managed_node = left_nav.get_managed_node()
    checkAndSetDefaultPaths(wtree, managed_node)
    dlg = CreateDialog(wtree, image_store)
    dlg.show(managed_node)


def show_settings(widget, blank = False):
    """ get the current context and show the settings dialog"""
    if blank and not validate_node_selected():
        return
    (name, node_type, managed_node) = left_nav.get_selection()
    if managed_node is not None:
        dom = managed_node.get_dom(name)
        if dom == None:
            blank = True
        settings.show(widget, managed_node, dom, blank)


def kill_all_domains(widget):
    """ kill all running domains on a node """
    if not validate_node_selected():
        return

    err_doms = []
    (name, node_type, managed_node) = left_nav.get_selection()
    if managed_node is not None:
        if confirmation("Kill all running domains on " +
                        managed_node.hostname + "?"):
            domains = managed_node.get_dom_names()
            for dom in domains:
                try:
                    if managed_node.isResident(dom) and \
                           managed_node.isDomU(dom):
                        managed_node.destroy_dom(dom)
                except Exception ,ex:
                    traceback.print_exc()
                    print "kill all : Error killing ",dom,str(ex) 
                    # do best effort start, do not bother user.
                    err_doms.append(dom)
                    
            if len(err_doms) > 0:
                showmsg(str(err_doms) + " could not be killed.\n" +
                        "Please try killing individually.")




def shutdown_all_domains(widget):
    """ shutdown all running domains on a node """
    if not validate_node_selected():
        return

    err_doms = []
    (name, node_type, managed_node) = left_nav.get_selection()
    if managed_node is not None:
        if confirmation("Shutdown all running domains on " +
                        managed_node.hostname + "?"):
            domains = managed_node.get_dom_names()
            for dom in domains:
                try:
                    if managed_node.isResident(dom) and \
                           managed_node.isDomU(dom):
                        managed_node.shutdown_dom(dom)
                except Exception ,ex:
                    traceback.print_exc()
                    print "shutdown  all : Error shutting down ",dom,str(ex) 
                    # do best effort start, do not bother user.
                    err_doms.append(dom)
                    
            if len(err_doms) > 0:
                showmsg(str(err_doms) + " did not shutdown.\n" +
                        "Please try shutting them down individually.")


def start_all_domains(widget):
    """ start all domains on a node"""
    if not validate_node_selected():
        return

    err_doms = []
    (name, node_type, managed_node) = left_nav.get_selection()
    if managed_node is not None:
        if confirmation("Start  all domains on " +
                        managed_node.hostname + "?"):
            domains = managed_node.get_dom_names()
            for dom in domains:
                try:
                    if not managed_node.isResident(dom): 
                        managed_node.start_dom(dom)
                except Exception ,ex:
                    print "Start all : Error starting " , dom, str(ex) 
                    # do best effort start, do not bother user.
                    err_doms.append(dom)

            if len(err_doms) > 0:
                showmsg(str(err_doms) + " could not be started.\n" +
                        "Please try starting them individually.")
        

def add_node(widget):
    """ add a new managed node to xenman """
    dlg = AddNodeDialog(wtree,client_config, left_nav)
    dlg.show()


def remove_node(widget):
    """ add a new managed node to xenman """
    if not validate_node_selected():
        return

    (name, node_type, managed_node) = left_nav.get_selection()
    if confirmation("Remove " + managed_node.hostname + "?"):
        left_nav.remove_node(managed_node)

def show_console(widget):
    """ show cpnsole """
    (name, node_type, managed_node) = left_nav.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        nbook.showTab(dom)


def wrapper(widget, funct, *args, **cargs):
    try:
        funct(widget, *args, **cargs)
    except Exception , ex:
        showmsg("Wrapper : Exception : " + str(ex))

def display_help(widget):
    """ display help """
    
   ##     import gnome
   ##     gnome.help_display_uri("./errata.html")
   ##     return
    browser = client_config.get(utils.XMConfig.CLIENT_CONFIG,
                                prop_browser)
    if browser is None:
        browser ="/usr/bin/yelp"
        
    
    if os.path.exists('doc/xenman_help.html'):
        help_file = 'doc/xenman_help.html'
    elif len(glob.glob('/usr/share/doc/xenman*/xenman_help.html'))>0:
        help_file = glob.glob('/usr/share/doc/xenman*/xenman_help.html')[0]
    else:
        help_file = None
        
    cmd = browser  + " " +  help_file
    p1 = subprocess.Popen(cmd,
                          stdout=subprocess.PIPE,stderr=subprocess.STDOUT,
                          universal_newlines=True,
                          shell=True, close_fds=True)
    #out = p1.communicate()[0]
    #exit_code   = p1.returncode
    #return out, exit_code



def checkLocalEnvironment():
    """Checks a number of required pre-conditions for the tool to be usable,
    if those aren't met, displays an appropriate dialog box and exits"""

    isXen = False
    isRoot = False
    
    # check whether the local kernel is recognisable as xen
    if platform.release().find('xen') != -1:
        isXen = True
    # check if running as root
    if os.getuid() == 0:
        isRoot = True

    return isXen, isRoot


    #ts = rpm.TransactionSet()
    #fedora = ts.dbMatch(rpm.RPMTAG_NAME, "fedora-release").count()
    # Check that we're running a Dom0 Xen kernel locally        

    #if platform.release().find('xen') == -1:
    #    msg = "You must be running a Xen Dom0 kernel to manage Xen domains."
    #
    #    if fedora:
    #        if ts.dbMatch(rpm.RPMTAG_NAME, "kernel-xen0").count():
    #            msg += "\nYou seem to have an appropriate Xen Dom0 kernel " + \
    #                   "installed, so restart your system and boot into it."
    #        else:
    #            msg += "\nYou don't seem to have an appropriate kernel " + \
    #                   'installed, try running "yum install "'
    #
    #            if not ts.dbMatch(rpm.RPMTAG_NAME, "xen").count():
    #                msg += "xen "
    #
    #            msg += 'kernel-xen0" to deploy the appropriate kernel, ' + \
    #                   "then restart your system and boot into that newly " + \
    #                   "installed kernel."
    #
    #    showmsg(msg)
    #    sys.exit(1)

    # Check that we're running as root
    #if os.getuid() != 0:
    #    showmsg("You must be running as root to interact with the hypervisor")
    #    sys.exit(1)
        
    #try:
        #_RESTRUCT
        #server.xend.domains()
        ### With remoting .. this is not a must.
        ### Tbis check would be made when adding local host.
        
        ### manager.getNode(default_host).get_doms()
    #    pass
    #except NodeException:
    #    if not ts.dbMatch(rpm.RPMTAG_NAME, "xen").count():
    #        msg = "You must install the Xen daemon in order manage Xen " + \
    #              "domains."
    #        if fedora:
    #            msg += 'Try running "yum install xen", followed by ' + \
    #                   '"service xend start" as root.'
    #    else:
    #        msg = "The Xen daemon isn't responding, "

    #        if fedora:
    #            msg += 'try starting it using "service xend start" as root.'
    #        else:
    #            msg += "please ensure it is installed and started."
    #
    #    showmsg(msg)
    #    sys.exit(1)


                
#############################################
# Main block                                #
#############################################

# set default timeout to 5 sec. This applies to setting connections.
# Fixes the problem when I tried to connect to my machine booted in
# Windows!!!.
socket.setdefaulttimeout(5)

showmsg = dialogs.showmsg
confirmation = dialogs.confirmation
file_selection = dialogs.file_selection


## initialize the phelper credentials helper
creds_helper = CredentialsHelper(wtree)
#PHelper.set_credentials_helper(creds_helper)

# Look at the command line arguments  and set up the server.
# for now simple checks for arguments

timer = None

# _RESTRUCTURING
from GridManager import GridManager
manager = GridManager()
remote_host = None


# Create a local node
# TODO: change this to a ManagedNode object
nd = XenNode(hostname = LOCALHOST, isRemote = False, helper = creds_helper)


# Sanity check the environment before performing any Xen interactions
isXen,isSuperUser = checkLocalEnvironment()
if isXen and not isSuperUser:
    if not nd.config.get(utils.XMConfig.CLIENT_CONFIG,prop_init_confirmation):
        if not confirmation('WARNING: XenMan must be run from a superuser account '+
                            'in order to manage the local Xen environment.\n\n'+
                            'Would you like to manage remote hosts only? \n\n'+
                            'Click YES to proceed \n'+
                            'Click CANCEL to exit and launch XenMan from a '+
                            'superuser account'):        
            sys.exit(0)
        else:
            nd.config.set(utils.XMConfig.CLIENT_CONFIG,prop_init_confirmation,'False')


if isXen and isSuperUser:
    manager.addNode(nd)

if len(sys.argv) > 1:
    remote_host = sys.argv[1]


# Initialise the configuration objects to gain access to global defaults
client_config = nd.config
image_store = utils.ImageStore(nd.config)

# testing only #
## image_store.addImage('Fedora Installer',
##                      '/var/cache/xenman/vmlinuz.default',
##                      '/var/cache/xenman/initrd.img.default',
##                      '' #'/usr/bin/pygrub'
##                      )
## image_store.setDefault('Fedora Installer')


#KLUDGE :populate the main context so dialogs and utils can share.
main_context["client_config"] = client_config

host_names = client_config.getHosts()

for host in host_names:
    
    _remote = client_config.getHostProperty(prop_isRemote, host)
    if not _remote:
        remote = is_host_remote(host)
        client_config.setHostProperty(prop_isRemote,str(remote),host)
    else:
        remote = eval(_remote)

    username = client_config.getHostProperty(prop_login, host)
    xen_port = client_config.getHostProperty(prop_xen_port, host)
    print host, remote
    node = XenNode(hostname=host,
                   username = username,
                   isRemote = remote,
                   tcp_port = xen_port,
                   helper = creds_helper)
    
    manager.addNode(node)
    
# Add the host on the command line
if remote_host:
    if remote_host not in manager.getNodeNames():
        remote = is_host_remote(remote_host)
        r_nd = XenNode(hostname = remote_host, isRemote = remote)
        manager.addNode(r_nd)
        
# _RESTRUCTURING



# Let's instantiate the dom management object

#left_nav = Left_nav(wtree)
left_nav = LeftNav(wtree)
nbook = NoteBook(left_nav)
Coordinator.set_nav(left_nav)
Coordinator.set_notebook(nbook)



settings = DomSettings(wtree, image_store)

signals = {"on_MainWindow_delete_event": quitProgram,
           "on_startbutton_clicked": (wrapper,startdom),
           "on_killbutton_clicked": (wrapper,killdom),
           "on_pausebutton_toggled": (wrapper,pausedom),
           "on_new_domain_activate": (wrapper,show_settings, True),
           "on_add_node_activate": (wrapper,add_node),
           "on_start_all_domains_activate": (wrapper,start_all_domains),
           "on_shutdown_all_domains_activate" : (wrapper,shutdown_all_domains),
           "on_kill_all_domains_activate" : (wrapper,kill_all_domains),
           "on_show_console_activate": (wrapper,show_console),
           "on_remove_node_activate": (wrapper,remove_node),
           "on_open_domain_activate": (wrapper,opendom),
           "on_Remove_activate": (wrapper,removedom),
           "on_delete_dom_activate": (wrapper,deletedom),
           "on_create_domain_wizard_activate": (wrapper,show_create_dialog),
           "on_DomView_button_press_event": popupmenu,
           "on_Settings_activate": (wrapper,show_settings),
           "on_quit_activate": quitProgram,
           "on_rebootbutton_clicked": (wrapper,rebootdom),
           "on_shutdownbutton_clicked": (wrapper,shutdowndom),
           "on_snapshotbutton_clicked": (wrapper,savedom),
           "on_restorebutton_clicked": (wrapper,restoredom),
           'on_about_activate': show_about,
           'on_AboutDialog_response': hide_about,
           'on_AboutDialog_delete_event': hide_about,
           'on_help_activate': (wrapper,display_help),
           }

wtree.signal_autoconnect(signals)

# Let's define the parent for the dialog windows (glade should do it but
# doesn't)

mainwin = wtree.get_widget('MainWindow')

for dname in ("SettingsDialog", "StatusDialog", "ConfirmationDialog",
              "AboutDialog", "CreateDialog", "initParamsDialog",
              "AddNode", "Credentials"):
    dialog = wtree.get_widget(dname)
    dialog.set_transient_for(mainwin)

# Setup the timer

timer = Timer()

# Run the main GUI loop

gtk.main()
