/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2x00
	Abstract: rt2x00 global information.
	Supported chipsets: rt2400, rt2500, rt2561, rt2561s & rt2661.
 */

#ifndef RT2X00PCI_H
#define RT2X00PCI_H

/*
 * Register defines.
 * When register access attempts should be repeated
 * only REGISTER_BUSY_COUNT attempts with a delay
 * of REGISTER_BUSY_DELAY us should be taken.
 */
#define REGISTER_BUSY_COUNT		5
#define REGISTER_BUSY_DELAY		100

/*
 * EEPROM operation defines.
 */
#define EEPROM_WIDTH_93C46		6
#define EEPROM_WIDTH_93C66		8
#define EEPROM_WRITE_OPCODE		0x05
#define EEPROM_READ_OPCODE		0x06

/*
 * Device specific structure.
 */
struct rt2x00_pci{
	/*
	 * Device flags.
	 */
	unsigned int				flags;

	/*
	 * PCI device structure.
	 */
	struct pci_dev				*pci_dev;

#ifdef CONFIG_RT2X00_BUTTON
	struct radio_button			button;
#endif /* CONFIG_RT2X00_BUTTON */

	/*
	 * IEEE80211 Configuration block.
	 */
	struct ieee80211_hw			hw;

	/*
	 * Chipset identification.
	 */
	struct rt2x00_chip			chip;

	/*
	 * RF values for channel switching.
	 */
	u32					rf1;
	u32					rf2;
	u32					rf3;
	u32					rf4;

	u32					tx_power;

	/*
	 * csr_addr
	 * Base address of device registers.
	 */
	void __iomem				*csr_addr;

	/*
	 * Queue for deferred interrupt work.
	 */
	struct workqueue_struct		*workqueue;

	/*
	 * Interface identication required for requesting beacons
	 * from dscape stack.
	 */
	int					interface_id;

	/*
	 * Current working mode.
	 */
	int					interface_type;

	/*
	 * Monitor mode count, the number of interfaces
	 * in monitor mode that that have been added.
	 */
	int					monitor_count;

	/*
	 * EEPROM bus width.
	 */
	u8					eeprom_width;

	/*
	 * Led status
	 */
	u8					led_mode;

	/*
	 * False CCA counter. (for rt2400pci).
	 */
	u8					false_cca;

	/*
	 * Frequency offset (for rt61pci).
	 */
	u8					freq_offset;

	/*
	 * LED register (for rt61pci).
	 */
	u16					led_reg;

	/*
	 * Sequence number.
	 */
	u16					seq_ctrl;

	/*
	 * EEPROM BBP data.
	 */
	u16					*eeprom;

	/*
	 * Low level statistics which will have
	 * to be kept up to date while device is running.
	 */
	struct ieee80211_low_level_stats	low_level_stats;

	/*
	 * RX configuration information.
	 */
	struct ieee80211_rx_status		rx_params;

	/*
	 * work structure for configuration changes.
	 */
	struct work_struct			config_work;

	/*
	 * Scanning structure and workqueue
	 * required for scheduling scanning requests.
	 */
	struct work_struct			scan_work;
	struct scanning			*scan;

	/*
	 * Data rings for both RX and TX.
	 * The first entries must be the normal TX
	 * rings, followed by a possible ATIM ring
	 * (when atim is used atim_available must be set)
	 * after that the beacon and RX ring follow.
	 */
	struct data_ring			*ring;
};

static inline struct data_ring* rt2x00pci_get_ring(
	struct rt2x00_pci *rt2x00pci, const int ring)
{
	int atim;

	atim = GET_FLAG(rt2x00pci, DEVICE_SUPPORT_ATIM);

	/*
	 * Check if the rings have been allocated.
	 */
	if (!rt2x00pci->ring)
		return NULL;

	/*
	 * Check for beacon ring, the beacon ring
	 * is located behing the normal TX and, when available
	 * the atim ring.
	 */
	if (ring == IEEE80211_TX_QUEUE_BEACON)
		return &rt2x00pci->ring[rt2x00pci->hw.queues + atim];

	if (ring == IEEE80211_TX_QUEUE_AFTER_BEACON && atim)
		return &rt2x00pci->ring[rt2x00pci->hw.queues];

	/*
	 * Make sure the requested ring does not exceed
	 * the maximum number of rings.
	 */
	if (ring < rt2x00pci->hw.queues)
		return &rt2x00pci->ring[ring];

	return NULL;
}

/*
 * Small handlers to determine the descriptor and
 * data address within each entry.
 */
static inline void* rt2x00pci_data_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline void* rt2x00pci_desc_addr(struct data_entry *entry)
{
	return entry->priv;
}

#endif /* RT2X00PCI_H */
