/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2500usb
	Abstract: rt2500usb device specific routines.
	Supported chipsets: RT2570.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/usb.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/ethtool.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME			"rt2500usb"

#ifdef CONFIG_RT2500USB_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT2500USB_DEBUG */

#include "rt2x00.h"
#include "rt2x00usb.h"
#include "rt2500usb.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 * The caller to these register access functions, should take precautions
 * for the correct byte ordering of the values.
 */
static void rt2x00_vendor_request(
	const struct rt2x00_usb *rt2x00usb,
	const u8 request,
	const u8 type,
	const u16 offset,
	u32 value,
	void *buffer,
	const u16 buffer_length)
{
	int status = rt2x00usb_vendor_request(
		rt2x00usb, request, type, offset,
		value, buffer, buffer_length, REGISTER_TIMEOUT);

	if (status < 0)
		ERROR("vendor request error. Request 0x%02x failed "
			"for offset 0x%04x with error %d.\n",
			request, offset, status);
}

static inline void rt2x00_register_read(
	const struct rt2x00_usb *rt2x00usb, const u16 offset, u16 *value)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_SINGLE_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, 2);
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_usb *rt2x00usb,
	const u16 offset, u16 *value, const u16 length)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, length);
}

static inline void rt2x00_register_write(
	const struct rt2x00_usb *rt2x00usb, const u16 offset, u16 value)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_SINGLE_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, value, NULL, 0);
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_usb *rt2x00usb,
	const u16 offset, u16 *value, const u16 length)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, value, length);
}

static void rt2x00_bbp_write(
	const struct rt2x00_usb *rt2x00usb, const u8 reg_id, const u8 value)
{
	u16 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, PHY_CSR8, &reg);
		if (!rt2x00_get_field16_nb(reg, PHY_CSR8_BBP_BUSY))
			goto bbp_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("BBPCSR register busy. Write failed.\n");
	return;

bbp_write:
	reg = 0;
	rt2x00_set_field16_nb(&reg, PHY_CSR7_BBP_DATA, value);
	rt2x00_set_field16_nb(&reg, PHY_CSR7_BBP_REG_ID, reg_id);
	rt2x00_set_field16_nb(&reg, PHY_CSR7_BBP_READ_CONTROL, 0);

	rt2x00_register_write(rt2x00usb, PHY_CSR7, reg);
}

static void rt2x00_bbp_read(
	const struct rt2x00_usb *rt2x00usb, const u8 reg_id, u8 *value)
{
	u16 reg;
	unsigned int i;

	/*
	 * First request the register we wish to read from.
	 */
	rt2x00_set_field16_nb(&reg, PHY_CSR7_BBP_REG_ID, reg_id);
	rt2x00_set_field16_nb(&reg, PHY_CSR7_BBP_READ_CONTROL, 1);

	rt2x00_register_write(rt2x00usb, PHY_CSR7, reg);

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, PHY_CSR8, &reg);
		if (!rt2x00_get_field16_nb(reg, PHY_CSR8_BBP_BUSY)) {
			rt2x00_register_read(rt2x00usb, PHY_CSR7, &reg);
			*value = rt2x00_get_field16_nb(reg, PHY_CSR7_BBP_DATA);
			return;
		}
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("BBPCSR register busy. Read failed.\n");
	*value = 0xff;
}

static void rt2x00_rf_write(
	const struct rt2x00_usb *rt2x00usb, const u32 value)
{
	u16 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, PHY_CSR10, &reg);
		if (!rt2x00_get_field16_nb(reg, PHY_CSR10_RF_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("RFCSR register busy. Write failed.\n");
	return;

rf_write:
	reg = value & 0x0000ffff;
	rt2x00_register_write(rt2x00usb, PHY_CSR9, reg);

	reg = (value >> 16) & 0x0000ffff;
	rt2x00_set_field16_nb(&reg, PHY_CSR10_RF_NUMBER_OF_BITS, 20);
	rt2x00_set_field16_nb(&reg, PHY_CSR10_RF_IF_SELECT, 0);
	rt2x00_set_field16_nb(&reg, PHY_CSR10_RF_BUSY, 1);

	rt2x00_register_write(rt2x00usb, PHY_CSR10, reg);
}

static inline void rt2x00_eeprom_read(
	const struct rt2x00_usb *rt2x00usb, const u16 word, u16 *data)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		word, 0x00, data, 2);
}

static void rt2x00_eeprom_multiread(
	const struct rt2x00_usb *rt2x00usb,
	const u8 word, u16 *data, const u16 length)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		word, 0x00, data, length);
}

static void rt2500usb_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->bus_info, rt2x00usb->usb_intf->dev.bus_id);
}

static int rt2500usb_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt2500usb_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	regs->version = rt2x00_get_chip_id(&rt2x00usb->chip);

	regs->len = CSR_REG_SIZE;

	rt2x00_register_multiread(rt2x00usb, CSR_REG_BASE, data, CSR_REG_SIZE);
}

#ifdef CONFIG_RT2500USB_DEBUG
static u32 rt2500usb_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt2500usb_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT2500USB_DEBUG */

static int rt2500usb_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt2500usb_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	eeprom->magic = rt2x00_get_chip_id(&rt2x00usb->chip);

	rt2x00_eeprom_multiread(rt2x00usb,
		eeprom->offset / sizeof(u16), (void*)data, eeprom->len);

	return 0;
}

static struct ethtool_ops rt2500usb_ethtool_ops = {
	.get_drvinfo	= rt2500usb_get_drvinfo,
	.get_regs_len	= rt2500usb_get_regs_len,
	.get_regs	= rt2500usb_get_regs,
#ifdef CONFIG_RT2500USB_DEBUG
	.get_msglevel	= rt2500usb_get_msglevel,
	.set_msglevel	= rt2500usb_set_msglevel,
#endif /* CONFIG_RT2500USB_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt2500usb_get_eeprom_len,
	.get_eeprom	= rt2500usb_get_eeprom,
};

/*
 * Configuration handlers.
 */
static void rt2500usb_config_bssid(struct rt2x00_usb *rt2x00usb, u8 *bssid)
{
	u16 reg[3] = {0, 0, 0};

	rt2x00_set_field16_nb(&reg[0], MAC_CSR5_BYTE0, *(bssid));
	rt2x00_set_field16_nb(&reg[0], MAC_CSR5_BYTE1, *(bssid + 1));
	rt2x00_set_field16_nb(&reg[1], MAC_CSR6_BYTE2, *(bssid + 2));
	rt2x00_set_field16_nb(&reg[1], MAC_CSR6_BYTE3, *(bssid + 3));
	rt2x00_set_field16_nb(&reg[2], MAC_CSR7_BYTE4, *(bssid + 4));
	rt2x00_set_field16_nb(&reg[2], MAC_CSR7_BYTE5, *(bssid + 5));

	rt2x00_register_multiwrite(rt2x00usb, MAC_CSR5, &reg[0], sizeof(reg));
}

static void rt2500usb_config_promisc(struct rt2x00_usb *rt2x00usb, int promisc)
{
	u16 reg;

	rt2x00_register_read(rt2x00usb, TXRX_CSR2, &reg);

	if (promisc) {
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00usb, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00usb, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00usb, TXRX_CSR2, reg);
}

static void rt2500usb_config_type(struct rt2x00_usb *rt2x00usb, int type)
{
	u16 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (type == rt2x00usb->interface_type
	|| (rt2x00usb->monitor_count
	^ GET_FLAG(rt2x00usb, INTERFACE_ENABLED_MONITOR)))
		return;

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR2, &reg);

	if (type == IEEE80211_IF_TYPE_IBSS
	|| type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_TODS, 1);
	else
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_TODS, 0);

	rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_CRC, 1);
	if (rt2x00usb->monitor_count) {
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_PHYSICAL, 0);
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_CONTROL, 0);
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_PHYSICAL, 1);
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_CONTROL, 1);
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DROP_VERSION_ERROR, 1);
	}

	rt2x00_register_write(rt2x00usb, TXRX_CSR2, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (rt2x00usb->monitor_count)
		rt2500usb_config_promisc(rt2x00usb, 1);

	/*
	 * Enable TSF counter
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR19, &reg);
	rt2x00_set_field16_nb(&reg, TXRX_CSR19_TSF_COUNT, 1);
	if (type == IEEE80211_IF_TYPE_IBSS)
		rt2x00_set_field16_nb(&reg, TXRX_CSR19_TSF_SYNC, 2);
	else if (type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field16_nb(&reg, TXRX_CSR19_TSF_SYNC, 1);
	else
		rt2x00_set_field16_nb(&reg, TXRX_CSR19_TSF_SYNC, 0);
	rt2x00_register_write(rt2x00usb, TXRX_CSR19, reg);

	/*
	 * Update working mode.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR)
		rt2x00usb->interface_type = type;

	if (rt2x00usb->monitor_count)
		SET_FLAG(rt2x00usb, INTERFACE_ENABLED_MONITOR);
	else
		CLEAR_FLAG(rt2x00usb, INTERFACE_ENABLED_MONITOR);
}

static void rt2500usb_config_channel(struct rt2x00_usb *rt2x00usb,
	int rf2, int channel, int freq, int txpower)
{
	u32 rf1 = rt2x00usb->rf1;
	u32 rf3 = rt2x00usb->rf3;
	u32 rf4 = rt2x00usb->rf4;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00usb->rx_params.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00usb->tx_power;
	txpower = TXPOWER_TO_DEV(txpower);

	if ((rt2x00_rf(&rt2x00usb->chip, RF2523)
	|| rt2x00_rf(&rt2x00usb->chip, RF2524)
	|| rt2x00_rf(&rt2x00usb->chip, RF2525))
	&& channel == 14)
		rf4 &= ~0x00000018;

	if (rt2x00_rf(&rt2x00usb->chip, RF2525E)) {
		if (channel & 0x01)
			rf4 = 0x00000e1b;
		else
			rf4 = 0x00000e07;
		if (channel == 14)
			rf4 = 0x00000e23;
	}

	if (rt2x00_rf(&rt2x00usb->chip, RF5222)) {
		if (channel < 14) {
			rf1 = 0x00022020;
			rf4 = 0x00000a0b;
		} else if (channel == 14) {
			rf1 = 0x00022010;
			rf4 = 0x00000a1b;
		} else if (channel < 64) {
			rf1 = 0x00022010;
			rf4 = 0x00000a1f;
		} else if (channel < 140) {
			rf1 = 0x00022010;
			rf4 = 0x00000a0f;
		} else if (channel < 161) {
			rf1 = 0x00022020;
			rf4 = 0x00000a07;
		}
	}

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32_nb(&rf3, RF3_TXPOWER, txpower);

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	/*
	 * For RT2525E we should first set the channel to half band higher.
	 */
	if (rt2x00_rf(&rt2x00usb->chip, RF2525E)) {
		static const u32 vals[] = {
			0x000008aa, 0x000008ae, 0x000008ae, 0x000008b2,
			0x000008b2, 0x000008b6, 0x000008b6, 0x000008ba,
			0x000008ba, 0x000008be, 0x000008b7, 0x00000902,
			0x00000902, 0x00000906
		};

		rt2x00_rf_write(rt2x00usb, vals[channel - 1]);
		if (rf4)
			rt2x00_rf_write(rt2x00usb, rf4);
	}

	rt2x00_rf_write(rt2x00usb, rf1);
	rt2x00_rf_write(rt2x00usb, rf2);
	rt2x00_rf_write(rt2x00usb, rf3);
	if (rf4)
		rt2x00_rf_write(rt2x00usb, rf4);

	/*
	 * Update active info for RX.
	 */
	rt2x00usb->rx_params.freq = freq;
	rt2x00usb->rx_params.channel = channel;

	/*
	 * Update rf fields
	 */
	rt2x00usb->rf1 = rf1;
	rt2x00usb->rf2 = rf2;
	rt2x00usb->rf3 = rf3;
	rt2x00usb->rf4 = rf4;

	rt2x00usb->tx_power = txpower;
}

static void rt2500usb_config_txpower(struct rt2x00_usb *rt2x00usb, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00usb->tx_power)
		return;

	rt2x00_set_field32_nb(&rt2x00usb->rf3, RF3_TXPOWER, txpower);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf3);

	rt2x00usb->tx_power = txpower;
}

static void rt2500usb_config_antenna(struct rt2x00_usb *rt2x00usb, int antenna)
{
	u8 reg_rx;
	u8 reg_tx;
	u16 csr5_reg;
	u16 csr6_reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00usb->rx_params.antenna == antenna)
		return;

	rt2x00_bbp_read(rt2x00usb, 2, &reg_tx);
	rt2x00_bbp_read(rt2x00usb, 14, &reg_rx);
	rt2x00_register_read(rt2x00usb, PHY_CSR5, &csr5_reg);
	rt2x00_register_read(rt2x00usb, PHY_CSR6, &csr6_reg);

	/*
	 * Clear current config antenna bits.
	 */
	reg_tx &= ~0x03;
	reg_rx &= ~0x03;

	/*
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	if (antenna == 0) {
		/* Diversity. */
		reg_rx |= 0x01;
		reg_tx |= 0x01;
		rt2x00_set_field16_nb(&csr5_reg, PHY_CSR5_CCK, 1);
		rt2x00_set_field16_nb(&csr6_reg, PHY_CSR6_OFDM, 1);
	} else if (antenna == 1) {
		/* RX: Antenna B */
		reg_rx |= 0x02;
		/* TX: Antenna A */
		reg_tx |= 0x00;
		rt2x00_set_field16_nb(&csr5_reg, PHY_CSR5_CCK, 0);
		rt2x00_set_field16_nb(&csr6_reg, PHY_CSR6_OFDM, 0);
	} else if (antenna == 2) {
		/* RX: Antenna A */
		reg_rx |= 0x00;
		/* TX: Antenna B */
		reg_tx |= 0x02;
		rt2x00_set_field16_nb(&csr5_reg, PHY_CSR5_CCK, 2);
		rt2x00_set_field16_nb(&csr6_reg, PHY_CSR6_OFDM, 2);
	}

	/*
	 * RT2525E and RT5222 need to flip TX I/Q
	 */
	if (rt2x00_rf(&rt2x00usb->chip, RF2525E)
	|| rt2x00_rf(&rt2x00usb->chip, RF5222)) {
		reg_tx |= 0x04;
		rt2x00_set_field16_nb(&csr5_reg, PHY_CSR5_CCK_FLIP, 1);
		rt2x00_set_field16_nb(&csr6_reg, PHY_CSR6_OFDM_FLIP, 1);

		/*
		 * RT2525E does not need RX I/Q Flip.
		 */
		if (rt2x00_rf(&rt2x00usb->chip, RF2525E))
			reg_rx &= ~0x04;
	} else {
		rt2x00_set_field16_nb(&csr5_reg, PHY_CSR5_CCK_FLIP, 0);
		rt2x00_set_field16_nb(&csr6_reg, PHY_CSR6_OFDM_FLIP, 0);
	}

	rt2x00_bbp_write(rt2x00usb, 2, reg_tx);
	rt2x00_bbp_write(rt2x00usb, 14, reg_rx);
	rt2x00_register_write(rt2x00usb, PHY_CSR5, csr5_reg);
	rt2x00_register_write(rt2x00usb, PHY_CSR6, csr6_reg);

	/*
	 * Update active info for RX.
	 */
	rt2x00usb->rx_params.antenna = antenna;
}

static void rt2500usb_config_duration(
	struct rt2x00_usb *rt2x00usb, int short_slot_time)
{
	u16 reg;

	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_write(rt2x00usb, MAC_CSR10, (u16)short_slot_time);

	rt2x00_register_read(rt2x00usb, TXRX_CSR1, &reg);
	rt2x00_set_field16_nb(&reg, TXRX_CSR1_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_register_write(rt2x00usb, TXRX_CSR1, reg);
}

static void rt2500usb_config_rate(struct rt2x00_usb *rt2x00usb, const int rate)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
		usb_get_intfdata(rt2x00usb->usb_intf));
	u16 reg;
	u16 value;
	u16 preamble;

	preamble = DEVICE_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg = DEVICE_RATE_FIELD(rate, RATEMASK);

	rt2x00_register_write(rt2x00usb, TXRX_CSR11, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR1, &reg);
	value = SIFS + PLCP
		+ (2 * (conf->short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME))
		+ preamble
		+ get_duration(ACK_SIZE, 10);
	rt2x00_set_field16_nb(&reg, TXRX_CSR1_ACK_TIMEOUT, value);
	rt2x00_register_write(rt2x00usb, TXRX_CSR1, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR10, &reg);
	if (DEVICE_RATE_FIELD(rate, PREAMBLE))
		reg |= 0x0004;
	else
		reg &= ~0x0004;
	rt2x00_register_write(rt2x00usb, TXRX_CSR10, reg);
}

static void rt2500usb_config_phymode(
	struct rt2x00_usb *rt2x00usb, const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00usb->rx_params.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A
	&& rt2x00_rf(&rt2x00usb->chip, RF5222))
		rate = &rt2x00usb->hw.modes[2].rates[
			rt2x00usb->hw.modes[2].num_rates - 1];
	else if (phymode == MODE_IEEE80211B)
		rate = &rt2x00usb->hw.modes[1].rates[
			rt2x00usb->hw.modes[1].num_rates - 1];
	else
		rate = &rt2x00usb->hw.modes[0].rates[
			rt2x00usb->hw.modes[0].num_rates - 1];

	rt2500usb_config_rate(rt2x00usb, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00usb->rx_params.phymode = phymode;
}

static void rt2500usb_config_mac_address(
	struct rt2x00_usb *rt2x00usb, void *addr)
{
	u16 reg[3] = {0, 0, 0};

	rt2x00_set_field16_nb(&reg[0], MAC_CSR2_BYTE0, ((u8*)addr)[0]);
	rt2x00_set_field16_nb(&reg[0], MAC_CSR2_BYTE1, ((u8*)addr)[1]);
	rt2x00_set_field16_nb(&reg[1], MAC_CSR3_BYTE2, ((u8*)addr)[2]);
	rt2x00_set_field16_nb(&reg[1], MAC_CSR3_BYTE3, ((u8*)addr)[3]);
	rt2x00_set_field16_nb(&reg[2], MAC_CSR4_BYTE4, ((u8*)addr)[4]);
	rt2x00_set_field16_nb(&reg[2], MAC_CSR4_BYTE5, ((u8*)addr)[5]);

	rt2x00_register_multiwrite(rt2x00usb, MAC_CSR2, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt2500usb_link_tuner(struct rt2x00_usb *rt2x00usb, char rssi)
{
	u16 reg;
	u8 reg_r17;
	u8 up_bound;
	u8 low_bound;

	/*
	 * Don't perform any tuning when it is disabled
	 * in the EEPROM.
	 */
	if (GET_FLAG(rt2x00usb, CONFIG_DISABLE_BBP_TUNING))
		return;

	low_bound = 0x32;
	if (rssi >= 43)
		up_bound = 0x40;
	else
		up_bound = 0x40 - (43 - rssi);
	if (up_bound < low_bound)
		up_bound = low_bound;

	if (rssi > 75) {
		rt2x00_bbp_write(rt2x00usb, 24, 0x70);
		rt2x00_bbp_write(rt2x00usb, 25, 0x40);
		rt2x00_bbp_write(rt2x00usb, 61, 0x6d);
	} else {
		rt2x00_bbp_write(rt2x00usb, 24, 0x80);
		rt2x00_bbp_write(rt2x00usb, 25, 0x50);
		rt2x00_bbp_write(rt2x00usb, 61, 0x60);
	}

	rt2x00_bbp_read(rt2x00usb, 17, &reg_r17);

	if (rssi > 80) {
		if (reg_r17 != 0x60)
			rt2x00_bbp_write(rt2x00usb, 17, 0x60);
		return;
	} else if (rssi >= 62) {
		if (reg_r17 != 0x48)
			rt2x00_bbp_write(rt2x00usb, 17, 0x48);
		return;
	} else if (rssi >= 46) {
		if (reg_r17 != 0x41)
			rt2x00_bbp_write(rt2x00usb, 17, 0x41);
		return;
	} else if (reg_r17 > up_bound) {
		rt2x00_bbp_write(rt2x00usb, 17, up_bound);
		return;
	}

	rt2x00_register_read(rt2x00usb, STA_CSR3, &reg);

	if (reg > 512 && reg_r17 < up_bound)
		rt2x00_bbp_write(rt2x00usb, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > low_bound)
		rt2x00_bbp_write(rt2x00usb, 17, --reg_r17);
}

/*
 * LED functions.
 */
static void rt2500usb_enable_led(struct rt2x00_usb *rt2x00usb)
{
	u16 reg;

	rt2x00_register_read(rt2x00usb, MAC_CSR21, &reg);
	rt2x00_set_field16_nb(&reg, MAC_CSR21_ON_PERIOD, 30);
	rt2x00_set_field16_nb(&reg, MAC_CSR21_OFF_PERIOD, 70);
	rt2x00_register_write(rt2x00usb, MAC_CSR21, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR20, &reg);

	if (rt2x00usb->led_mode == LED_MODE_TXRX_ACTIVITY) {
		rt2x00_set_field16_nb(&reg, MAC_CSR20_LINK, 1);
		rt2x00_set_field16_nb(&reg, MAC_CSR20_ACTIVITY, 0);
	} else if (rt2x00usb->led_mode == LED_MODE_ASUS) {
		rt2x00_set_field16_nb(&reg, MAC_CSR20_LINK, 0);
		rt2x00_set_field16_nb(&reg, MAC_CSR20_ACTIVITY, 1);
	} else {
		rt2x00_set_field16_nb(&reg, MAC_CSR20_LINK, 1);
		rt2x00_set_field16_nb(&reg, MAC_CSR20_ACTIVITY, 1);
	}

	rt2x00_register_write(rt2x00usb, MAC_CSR20, reg);
}

static void rt2500usb_disable_led(struct rt2x00_usb *rt2x00usb)
{
	u16 reg;

	rt2x00_register_read(rt2x00usb, MAC_CSR20, &reg);
	rt2x00_set_field16_nb(&reg, MAC_CSR20_LINK, 0);
	rt2x00_set_field16_nb(&reg, MAC_CSR20_ACTIVITY, 0);
	rt2x00_register_write(rt2x00usb, MAC_CSR20, reg);
}

static void rt2500usb_activity_led(struct rt2x00_usb *rt2x00usb, char activity)
{
	u16 reg;

	if (rt2x00usb->led_mode != LED_MODE_TXRX_ACTIVITY)
		return;

	rt2x00_register_read(rt2x00usb, MAC_CSR20, &reg);
	rt2x00_set_field16_nb(&reg, MAC_CSR20_ACTIVITY, activity);
	rt2x00_register_write(rt2x00usb, MAC_CSR20, reg);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt2500usb_set_state(
	struct rt2x00_usb *rt2x00usb, enum dev_state state)
{
	u16 reg;
	unsigned int i;
	char put_to_sleep;
	char bbp_state;
	char rf_state;

	put_to_sleep = (state != STATE_AWAKE);

	rt2x00_register_read(rt2x00usb, MAC_CSR17, &reg);
	rt2x00_set_field16_nb(&reg, MAC_CSR17_SET_STATE, 1);
	rt2x00_set_field16_nb(&reg, MAC_CSR17_BBP_DESIRE_STATE, state);
	rt2x00_set_field16_nb(&reg, MAC_CSR17_RF_DESIRE_STATE, state);
	rt2x00_set_field16_nb(&reg, MAC_CSR17_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00usb, MAC_CSR17, reg);

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, MAC_CSR17, &reg);
		bbp_state = rt2x00_get_field16_nb(reg,
			MAC_CSR17_BBP_CURR_STATE);
		rf_state = rt2x00_get_field16_nb(reg,
			MAC_CSR17_RF_CURR_STATE);
		if (bbp_state == state && rf_state == state)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state: bbp %d and rf %d.\n",
		state, bbp_state, rf_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt2500usb_alloc_dma_ring(
	struct rt2x00_usb *rt2x00usb,
	struct data_ring *ring,
	void (*handler)(void *),
	const u16 max_entries,
	const u16 data_size,
	const u16 desc_size)
{
	unsigned int i;
	int status;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kmalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = usb_buffer_alloc(
		interface_to_usbdev(rt2x00usb->usb_intf),
		rt2x00_get_ring_size(ring), GFP_KERNEL, &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	status = 0;
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].ring = ring;
		ring->entry[i].priv =
			(!status) ? usb_alloc_urb(0, GFP_KERNEL) : NULL;
		if (!ring->entry[i].priv)
			status = -ENOMEM;
		ring->entry[i].skb = NULL;
		ring->entry[i].data_addr = ring->data_addr
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
	}

	return status;
}

static void rt2500usb_free_ring(
	struct rt2x00_usb *rt2x00usb, struct data_ring *ring)
{
	unsigned int i;

	if (!ring->entry)
		goto exit;

	for (i = 0; i < ring->stats.limit; i++) {
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));
		usb_free_urb(rt2x00usb_urb(&ring->entry[i]));
	}

	kfree(ring->entry);
	ring->entry = NULL;

exit:
	if (ring->data_addr)
		usb_buffer_free(
			interface_to_usbdev(rt2x00usb->usb_intf),
			rt2x00_get_ring_size(ring), ring->data_addr,
			ring->data_dma);
	ring->data_addr = NULL;
}

static int rt2500usb_allocate_dma_rings(struct rt2x00_usb *rt2x00usb)
{
	if (rt2500usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_RX], &rt2500usb_rxdone,
		RX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct rxd))
	|| rt2500usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_TX], &rt2500usb_txdone,
		 TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt2500usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_ATIM], &rt2500usb_txdone,
		ATIM_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt2500usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_PRIO], &rt2500usb_txdone,
		TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt2500usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_BEACON], &rt2500usb_beacondone,
		BEACON_ENTRIES, MGMT_FRAME_SIZE, sizeof(struct txd))) {
		return -ENOMEM;
	}

	return 0;
}

static void rt2500usb_free_rings(struct rt2x00_usb *rt2x00usb)
{
	rt2500usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_RX]);
	rt2500usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_TX]);
	rt2500usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_ATIM]);
	rt2500usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_PRIO]);
	rt2500usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_BEACON]);
}

static void rt2500usb_init_rxring(
	struct rt2x00_usb *rt2x00usb, unsigned short type)
{
	struct data_ring *ring = &rt2x00usb->ring[type];
	struct usb_device *usb_dev = interface_to_usbdev(rt2x00usb->usb_intf);
	struct urb *urb;
	unsigned int i;

	ring->type = type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00usb_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		usb_fill_bulk_urb(
			urb,
			usb_dev,
			usb_rcvbulkpipe(usb_dev, 1),
			ring->entry[i].data_addr,
			ring->data_size + ring->desc_size,
			rt2500usb_interrupt,
			&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt2500usb_init_txring(
	struct rt2x00_usb *rt2x00usb, unsigned short type)
{
	struct data_ring *ring = &rt2x00usb->ring[type];
	struct urb *urb;
	unsigned int i;

	ring->type = type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00usb_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		CLEAR_FLAGS(&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt2500usb_init_rings(struct rt2x00_usb *rt2x00usb)
{
	rt2500usb_init_rxring(rt2x00usb, RING_RX);
	rt2500usb_init_txring(rt2x00usb, RING_TX);
	rt2500usb_init_txring(rt2x00usb, RING_ATIM);
	rt2500usb_init_txring(rt2x00usb, RING_PRIO);
	rt2500usb_init_txring(rt2x00usb, RING_BEACON);

	return 0;
}

static int rt2500usb_init_registers(struct rt2x00_usb *rt2x00usb)
{
	u16 reg;

	if (rt2500usb_set_state(rt2x00usb, STATE_AWAKE))
		return -EBUSY;

	rt2x00_vendor_request(rt2x00usb, USB_DEVICE_MODE,
		USB_VENDOR_REQUEST_OUT, 0x00, USB_MODE_TEST, NULL, 0);
	rt2x00_vendor_request(rt2x00usb, USB_SINGLE_WRITE,
		USB_VENDOR_REQUEST_OUT, 0x0308, 0xf0, NULL, 0);

	rt2x00_register_write(rt2x00usb, MAC_CSR13, 0x1111);
	rt2x00_register_write(rt2x00usb, MAC_CSR14, 0x1e11);

	rt2x00_register_write(rt2x00usb, TXRX_CSR2, 0x0001);
	rt2x00_register_write(rt2x00usb, MAC_CSR1, 0x0003);
	rt2x00_register_write(rt2x00usb, MAC_CSR1, 0x0000);
	rt2x00_register_write(rt2x00usb, TXRX_CSR5, 0x8c8d);
	rt2x00_register_write(rt2x00usb, TXRX_CSR6, 0x8b8a);
	rt2x00_register_write(rt2x00usb, TXRX_CSR7, 0x8687);
	rt2x00_register_write(rt2x00usb, TXRX_CSR8, 0x0085);
	rt2x00_register_write(rt2x00usb, TXRX_CSR21, 0xe78f);
	rt2x00_register_write(rt2x00usb, MAC_CSR9, 0xff1d);
	rt2x00_register_write(rt2x00usb, MAC_CSR1, 0x0004);

	reg = 0;
	rt2x00_register_read(rt2x00usb, MAC_CSR0, &reg);
	if (reg >= 0x0003) {
		rt2x00_register_read(rt2x00usb, PHY_CSR2, &reg);
		reg &= ~0x0002;
	} else {
		reg = 0x3002;
	}
	rt2x00_register_write(rt2x00usb, PHY_CSR2, reg);

	rt2x00_register_write(rt2x00usb, MAC_CSR11, 0x0002);
	rt2x00_register_write(rt2x00usb, MAC_CSR22, 0x0053);
	rt2x00_register_write(rt2x00usb, MAC_CSR15, 0x01ee);
	rt2x00_register_write(rt2x00usb, MAC_CSR16, 0x0000);

	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
	rt2x00_set_field16_nb(&reg, TXRX_CSR0_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field16_nb(&reg, TXRX_CSR0_KEY_ID, 0xff);
	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR8, &reg);
	rt2x00_set_field16_nb(&reg, MAC_CSR8_MAX_FRAME_UNIT,
		rt2x00usb->ring[RING_RX].data_size);
	rt2x00_register_write(rt2x00usb, MAC_CSR8, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR18, &reg);
	rt2x00_set_field16_nb(&reg, MAC_CSR18_DELAY_AFTER_BEACON, 0x5a);
	rt2x00_register_write(rt2x00usb, MAC_CSR18, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR1, &reg);
	rt2x00_set_field16_nb(&reg, TXRX_CSR1_AUTO_SEQUENCE, 1);
	rt2x00_register_write(rt2x00usb, TXRX_CSR1, reg);

	rt2x00_register_read(rt2x00usb, PHY_CSR4, &reg);
	rt2x00_register_write(rt2x00usb, PHY_CSR4, reg | 0x0001);

	return 0;
}

static int rt2500usb_init_bbp(struct rt2x00_usb *rt2x00usb)
{
	u8 value;
	u8 reg_id;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00usb, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00usb, 3, 0x02);
	rt2x00_bbp_write(rt2x00usb, 4, 0x19);
	rt2x00_bbp_write(rt2x00usb, 14, 0x1c);
	rt2x00_bbp_write(rt2x00usb, 15, 0x30);
	rt2x00_bbp_write(rt2x00usb, 16, 0xac);
	rt2x00_bbp_write(rt2x00usb, 17, 0x48);
	rt2x00_bbp_write(rt2x00usb, 18, 0x18);
	rt2x00_bbp_write(rt2x00usb, 19, 0xff);
	rt2x00_bbp_write(rt2x00usb, 20, 0x1e);
	rt2x00_bbp_write(rt2x00usb, 21, 0x08);
	rt2x00_bbp_write(rt2x00usb, 22, 0x08);
	rt2x00_bbp_write(rt2x00usb, 23, 0x08);
	rt2x00_bbp_write(rt2x00usb, 24, 0x80);
	rt2x00_bbp_write(rt2x00usb, 25, 0x50);
	rt2x00_bbp_write(rt2x00usb, 26, 0x08);
	rt2x00_bbp_write(rt2x00usb, 27, 0x23);
	rt2x00_bbp_write(rt2x00usb, 30, 0x10);
	rt2x00_bbp_write(rt2x00usb, 31, 0x2b);
	rt2x00_bbp_write(rt2x00usb, 32, 0xb9);
	rt2x00_bbp_write(rt2x00usb, 34, 0x12);
	rt2x00_bbp_write(rt2x00usb, 35, 0x50);
	rt2x00_bbp_write(rt2x00usb, 39, 0xc4);
	rt2x00_bbp_write(rt2x00usb, 40, 0x02);
	rt2x00_bbp_write(rt2x00usb, 41, 0x60);
	rt2x00_bbp_write(rt2x00usb, 53, 0x10);
	rt2x00_bbp_write(rt2x00usb, 54, 0x18);
	rt2x00_bbp_write(rt2x00usb, 56, 0x08);
	rt2x00_bbp_write(rt2x00usb, 57, 0x10);
	rt2x00_bbp_write(rt2x00usb, 58, 0x08);
	rt2x00_bbp_write(rt2x00usb, 61, 0x60);
	rt2x00_bbp_write(rt2x00usb, 62, 0x10);
	rt2x00_bbp_write(rt2x00usb, 75, 0xff);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		if (rt2x00usb->eeprom[i] != 0xffff
		&& rt2x00usb->eeprom[i] != 0x0000) {
			reg_id = rt2x00_get_field16(
				rt2x00usb->eeprom[i], EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(
				rt2x00usb->eeprom[i], EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00usb, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt2500usb_init_channel_time(struct rt2x00_usb *rt2x00usb)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt2500usb_config_channel(rt2x00usb,
		rt2x00usb->hw.modes[0].channels[0].val,
		rt2x00usb->hw.modes[0].channels[0].chan,
		rt2x00usb->hw.modes[0].channels[0].freq,
		rt2x00usb->hw.modes[0].channels[0].power_level);
	jiffies_end = jiffies;

	rt2x00usb->hw.channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00usb->hw.channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt2500usb_initialize(struct rt2x00_usb *rt2x00usb)
{
	if (GET_FLAG(rt2x00usb, DEVICE_INITIALIZED))
		return 0;

	/*
	 * Allocate all data rings.
	 */
	if (rt2500usb_allocate_dma_rings(rt2x00usb)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Initialize all registers.
	 */
	if (rt2500usb_init_rings(rt2x00usb)
	|| rt2500usb_init_registers(rt2x00usb)
	|| rt2500usb_init_bbp(rt2x00usb)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt2500usb_init_channel_time(rt2x00usb))
		goto exit_fail;

	SET_FLAG(rt2x00usb, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt2500usb_free_rings(rt2x00usb);

	return -EIO;
}

static void rt2500usb_uninitialize(struct rt2x00_usb *rt2x00usb)
{
	if (!GET_FLAG(rt2x00usb, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00usb->scan) {
		rt2x00usb->scan->status = SCANNING_CANCELLED;
		complete_all(&rt2x00usb->scan->completion);
	}

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00usb->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt2500usb_free_rings(rt2x00usb);

	CLEAR_FLAG(rt2x00usb, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt2500usb_enable_radio(struct rt2x00_usb *rt2x00usb)
{
	struct data_ring *ring;
	u16 reg;
	unsigned int i;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return 0;

	/*
	 * Check if the hardware has been initialized,
	 * if not then do it now.
	 */
	if (!GET_FLAG(rt2x00usb, DEVICE_INITIALIZED))
		if (rt2500usb_initialize(rt2x00usb))
			return -EIO;

	rt2x00_register_write(rt2x00usb, MAC_CSR13, 0x1111);
	rt2x00_register_write(rt2x00usb, MAC_CSR14, 0x1e11);

	/*
	 * Enable RX.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR2, &reg);
	rt2x00_set_field16_nb(&reg, TXRX_CSR2_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00usb, TXRX_CSR2, reg);

	ring = &rt2x00usb->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++) {
		SET_FLAG(&ring->entry[i], ENTRY_OWNER_NIC);
		usb_submit_urb(rt2x00usb_urb(&ring->entry[i]), GFP_ATOMIC);
	}

	/*
	 * Enable LED
	 */
	rt2500usb_enable_led(rt2x00usb);

	SET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO);

	return 0;
}

static void rt2500usb_disable_radio(struct rt2x00_usb *rt2x00usb)
{
	struct data_ring *ring;
	u16 reg;
	unsigned int i;

	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return;

	rt2x00_register_write(rt2x00usb, MAC_CSR13, 0x2121);
	rt2x00_register_write(rt2x00usb, MAC_CSR14, 0x2121);

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR2, &reg);
	rt2x00_set_field16_nb(&reg, TXRX_CSR2_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00usb, TXRX_CSR2, reg);

	rt2x00_vendor_request(rt2x00usb, USB_RX_CONTROL,
		USB_VENDOR_REQUEST_OUT, 0x00, 0x00, NULL, 0);

	ring = &rt2x00usb->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_TX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_ATIM];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_PRIO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_BEACON];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	/*
	 * Disable LED
	 */
	rt2500usb_disable_led(rt2x00usb);

	CLEAR_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO);
}

/*
 * TX descriptor initialization
 */
static void rt2500usb_write_tx_desc(
	struct rt2x00_usb *rt2x00usb, struct txd *txd,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	u32 length;
	u32 residual;
	u8 length_high;
	u8 length_low;
	u8 signal;
	u8 service;
	u8 rate;

	rt2x00_set_field32(&txd->word0, TXD_W0_ACK, !control->no_ack);

	ring = rt2x00usb_get_ring(rt2x00usb, control->queue);
	if (unlikely(!ring))
		return;

	if (control->queue == IEEE80211_TX_QUEUE_BEACON)
		rt2x00_set_field32(&txd->word0, TXD_W0_TIMESTAMP, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_TIMESTAMP, 0);

	/*
	 * TODO: IFS can be various values, where can we find
	 * which one we want to use?
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_IFS, 0);

	rt2x00_set_field32(&txd->word0, TXD_W0_RETRY_LIMIT,
		control->retry_limit);

	if (ieee80211_get_morefrag(ieee80211hdr))
		rt2x00_set_field32(&txd->word0, TXD_W0_MORE_FRAG, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_MORE_FRAG, 0);

	if (WLAN_GET_SEQ_FRAG(ieee80211hdr->seq_ctrl) == 0)
		rt2x00_set_field32(&txd->word0, TXD_W0_NEW_SEQ, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_NEW_SEQ, 0);

	rt2x00_set_field32(&txd->word0, TXD_W0_DATABYTE_COUNT, skb->len);

	rt2x00_set_field32(&txd->word1, TXD_W1_AIFS, ring->tx_params.aifs);
	rt2x00_set_field32(&txd->word1, TXD_W1_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&txd->word1, TXD_W1_CWMAX, ring->tx_params.cw_max);

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * Check if we are working with an OFDM rate,
	 * this can be done by checking if bit 4 or higher
	 * is set in the ratemask.
	 */
	if (DEVICE_RATE_FIELD(control->tx_rate, RATEMASK) & 0x0ff0) {
		rt2x00_set_field32(&txd->word0, TXD_W0_OFDM, 1);
		residual = 0;

		/*
		 * Convert length to microseconds.
		 */
		length_high = (length >> 6) & 0x3f;
		length_low = (length & 0x3f);

	} else {
		rate = DEVICE_RATE_FIELD(control->tx_rate, RATE);

		rt2x00_set_field32(&txd->word0, TXD_W0_OFDM, 0);

		residual = get_duration_res(length, rate);
		length = get_duration(length, rate);

		if (residual != 0)
			length++;

		length_high = length >> 8;
		length_low = length & 0xff;
	}

	signal = DEVICE_RATE_FIELD(control->tx_rate, PLCP);
	if (DEVICE_RATE_FIELD(control->tx_rate, PREAMBLE))
		signal |= 0x08;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_SERVICE, service);
	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_LENGTH_HIGH, length_high);
}

/*
 * Interrupt functions.
 */
static void rt2500usb_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(ring->net_dev);
	struct sk_buff *skb;
	struct ieee80211_tx_control beacon;

	memset(&beacon, 0x00, sizeof(beacon));

	skb = ieee80211_beacon_get(ring->net_dev,
		rt2x00usb->interface_id, &beacon);
	if (!skb)
		return;

	rt2500usb_beacon_update(ring->net_dev, skb, &beacon);

	dev_kfree_skb_any(skb);
}

static void rt2500usb_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct sk_buff *skb;
	struct rxd *rxd;
	struct urb *urb;
	u16 size;
	u8 rssi_count;
	char total_rssi;

	/*
	 * Initialize variable for average RSSI calculation.
	 */
	rssi_count = 0;
	total_rssi = 0;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00usb_rxdesc_addr(entry);
		urb = rt2x00usb_urb(entry);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		/*
		 * There has been a problem. Ignore packet.
		 */
		if (urb->status) {
			SET_FLAG(entry, ENTRY_OWNER_NIC);
			usb_submit_urb(urb, GFP_ATOMIC);
			rt2x00_ring_index_inc(ring);
			continue;
		}

		/*
		 * Received USB packets have 4 bytes of extra data.
		 */
		size = rt2x00_get_field32(rxd->word0, RXD_W0_DATABYTE_COUNT);
		size -= FCS_LEN;

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(rxd->word0, RXD_W0_CRC)
		&& !rt2x00_get_field32(rxd->word0, RXD_W0_PHYSICAL_ERROR)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				break;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size),
				rt2x00usb_rxdata_addr(entry), size);

			rt2x00usb->rx_params.ssi =
				rt2x00_get_field32(rxd->word1, RXD_W1_RSSI);

			__ieee80211_rx(ring->net_dev,
				skb, &rt2x00usb->rx_params);

			rssi_count++;
			total_rssi += rt2x00usb->rx_params.ssi;
		}

		SET_FLAG(entry, ENTRY_OWNER_NIC);
		usb_submit_urb(urb, GFP_ATOMIC);

		rt2x00_ring_index_inc(ring);
	}

	/*
	 * Tune link for optimal performance.
	 */
	if (total_rssi && rssi_count)
		rt2500usb_link_tuner(rt2x00usb, total_rssi / rssi_count);

	/*
	 * Update LED.
	 */
	rt2500usb_activity_led(rt2x00usb, 0);
}

static void rt2500usb_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct txd *txd;
	struct urb *urb;
	int ack;

	 while (!rt2x00_ring_empty(ring)) {
		entry = rt2x00_get_data_entry_done(ring);
		txd = rt2x00usb_txdesc_addr(entry);
		urb = rt2x00usb_urb(entry);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		ack = rt2x00_get_field32(txd->word0, TXD_W0_ACK);
	
		/*
		 * TODO: How can te below field be set correctly?
		 */
		entry->tx_status.tx_filtered = 0;
	
		entry->tx_status.queue_length = entry->ring->stats.limit;
	
		/*
		 * Check if we have received an
		 * ACK response when ACK was requested and status
		 * was succesfull.
		 */
		entry->tx_status.ack = 0;
		if (ack && (urb->status == TX_SUCCESS))
			entry->tx_status.ack = 1;
		else if (ack && urb->status == TX_FAIL_OTHER) {
			rt2x00usb->low_level_stats.dot11ACKFailureCount++;
			entry->tx_status.excessive_retries++;
		}
	
		rt2x00_bbp_read(rt2x00usb, 0,
			(u8*)&entry->tx_status.ack_signal);
	
		ieee80211_tx_status(ring->net_dev,
			entry->skb, &entry->tx_status);
	
		entry->skb = NULL;
	
		rt2x00_ring_index_done_inc(entry->ring);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00usb->scan
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_TX])
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_ATIM])
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_PRIO])) {
		rt2x00usb->scan->status = SCANNING_READY;
		complete(&rt2x00usb->scan->completion);
	}
}

static void rt2500usb_interrupt(struct urb *urb, struct pt_regs *regs)
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct rt2x00_usb *rt2x00usb =
		ieee80211_dev_hw_data(entry->ring->net_dev);

	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return;

	CLEAR_FLAG(entry, ENTRY_OWNER_NIC);

	if (urb->status)
		return;

	if (entry->ring->type == RING_RX)
		rt2500usb_activity_led(rt2x00usb, 1);

	queue_work(rt2x00usb->workqueue, &entry->ring->irq_work);
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt2500usb_tx(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	struct usb_device *usb_dev = interface_to_usbdev(rt2x00usb->usb_intf);
	struct data_ring *ring;
	struct data_entry *entry;
	struct txd *txd;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00usb_get_ring(rt2x00usb, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		return NET_XMIT_DROP;
	}

	if (rt2x00_ring_full(ring))
		return NET_XMIT_DROP;

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00usb_txdesc_addr(entry);

	memcpy(rt2x00usb_txdata_addr(entry), skb->data, skb->len);
	rt2500usb_write_tx_desc(rt2x00usb, txd, skb, control);
	entry->skb = skb;

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00usb_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		skb->len + ring->desc_size,
		rt2500usb_interrupt,
		entry);
	usb_submit_urb(rt2x00usb_urb(entry), GFP_ATOMIC);

	rt2x00_ring_index_inc(ring);

	return 0;
}

static int rt2500usb_reset(struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	rt2500usb_disable_radio(rt2x00usb);

	rt2500usb_init_rings(rt2x00usb);

	return rt2500usb_enable_radio(rt2x00usb);
}

static int rt2500usb_add_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * We support muliple monitor mode interfaces.
	 * All we need to do is keep increase the
	 * monitor_counter.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		rt2x00usb->monitor_count++;
		rt2x00usb->interface_promisc = 1;
		goto exit;
	}

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (GET_FLAG(rt2x00usb, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	rt2x00usb->interface_id = conf->if_id;

	rt2500usb_config_type(rt2x00usb, conf->type);
	rt2500usb_config_mac_address(rt2x00usb, conf->mac_addr);

	SET_FLAG(rt2x00usb, INTERFACE_INITIALIZED);

exit:
	/*
	 * Enable radio when this is the first
	 * interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return rt2500usb_enable_radio(rt2x00usb);
	return 0;
}

static void rt2500usb_remove_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * We support muliple monitor mode interfaces.
	 * Decrease the counter to remove one monitor
	 * interface.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		rt2x00usb->monitor_count--;
		goto exit;
	}

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (!GET_FLAG(rt2x00usb, INTERFACE_INITIALIZED))
		return;

	rt2x00usb->interface_id = 0;
	memset(&rt2x00usb->interface_bssid, 0x00, ETH_ALEN);
	rt2x00usb->interface_type = -EINVAL;

	CLEAR_FLAG(rt2x00usb, INTERFACE_INITIALIZED);

exit:
	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!rt2x00usb->monitor_count)
		rt2500usb_disable_radio(rt2x00usb);
}

static void rt2500usb_config_update(void *data)
{
	struct rt2x00_usb *rt2x00usb = data;
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(net_dev);
	u16 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt2500usb_disable_radio(rt2x00usb);
		else {
			rt2x00_register_read(rt2x00usb, TXRX_CSR2, &reg);
			rt2x00_set_field16_nb(&reg, TXRX_CSR2_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00usb, TXRX_CSR2, reg);
		}
	}

	rt2500usb_config_channel(rt2x00usb,
		conf->channel_val, conf->channel, conf->freq,
		conf->power_level);
	rt2500usb_config_txpower(rt2x00usb, conf->power_level);
	rt2500usb_config_antenna(rt2x00usb, conf->antenna_sel);
	rt2500usb_config_duration(rt2x00usb, conf->short_slot_time);
	rt2500usb_config_phymode(rt2x00usb, conf->phymode);

	/*
	 * Reenable RX only if the radio should is on.
	 */
	if (GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00usb, TXRX_CSR2, &reg);
		rt2x00_set_field16_nb(&reg, TXRX_CSR2_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00usb, TXRX_CSR2, reg);
	} else if (conf->radio_enabled) {
		if (rt2500usb_enable_radio(rt2x00usb))
			return;
	}
}

static int rt2500usb_config(
	struct net_device *net_dev, struct ieee80211_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Queue work.
	 */
	return !queue_work(rt2x00usb->workqueue, &rt2x00usb->config_work);
}

static void rt2500usb_interface_update(void *data)
{
	struct rt2x00_usb *rt2x00usb = data;

	rt2500usb_config_bssid(rt2x00usb, &rt2x00usb->interface_bssid[0]);
	rt2500usb_config_promisc(rt2x00usb, rt2x00usb->interface_promisc);
}

static int rt2500usb_config_interface(struct net_device *net_dev, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * When configuring monitor mode, we are done now.
	 * but if we are configuring another mode it must be
	 * equal to the interface that has been added.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00usb->interface_type)
		return -EINVAL;

	if (conf->bssid) {
		memcpy(&rt2x00usb->interface_bssid, conf->bssid, ETH_ALEN);
		return !queue_work(rt2x00usb->workqueue,
				&rt2x00usb->config_work);
	}

	return 0;
}

static void rt2500usb_set_multicast_list(struct net_device *net_dev,
	unsigned short flags, int mc_count)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (rt2x00usb->monitor_count)
		return;

	if (GET_FLAG(rt2x00usb, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC)) {
			rt2x00usb->interface_promisc = 0;
			queue_work(rt2x00usb->workqueue,
				&rt2x00usb->config_work);
		}
	} else {
		if (flags & IFF_PROMISC) {
			rt2x00usb->interface_promisc = 1;
			queue_work(rt2x00usb->workqueue,
				&rt2x00usb->config_work);
		}
	}
}

static void rt2500usb_scan(void *data)
{
	struct rt2x00_usb *rt2x00usb = data;

	if (unlikely(!rt2x00usb->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00usb->scan->status != SCANNING_READY)
		wait_for_completion(&rt2x00usb->scan->completion);

	/*
	 * Check if this scan has been cancelled while
	 * work was still scheduled.
	 */
	if (rt2x00usb->scan->status == SCANNING_CANCELLED)
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00usb->scan->state == IEEE80211_SCAN_START) {
		rt2500usb_config_phymode(rt2x00usb,
			rt2x00usb->scan->conf.scan_phymode);
	
		rt2500usb_config_channel(rt2x00usb,
			rt2x00usb->scan->conf.scan_channel_val,
			rt2x00usb->scan->conf.scan_channel,
			rt2x00usb->scan->conf.scan_freq,
			rt2x00usb->scan->conf.scan_power_level);
	} else {
		rt2500usb_config_phymode(rt2x00usb,
			rt2x00usb->scan->conf.running_phymode);
	
		rt2500usb_config_channel(rt2x00usb,
			rt2x00usb->scan->conf.running_channel_val,
			rt2x00usb->scan->conf.running_channel,
			rt2x00usb->scan->conf.running_freq,
			rt2x00usb->scan->conf.scan_power_level);
	}

exit:
	kfree(rt2x00usb->scan);
	rt2x00usb->scan = NULL;
}

static int rt2500usb_passive_scan(struct net_device *net_dev,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00usb->scan)
		return -EBUSY;

	/*
	 * Check if we need to disable the radio
	 */
	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00usb->scan = kmalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00usb->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt2500usb_tx(net_dev, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	init_completion(&rt2x00usb->scan->completion);

	memcpy(&rt2x00usb->scan->conf, conf, sizeof(*conf));

	rt2x00usb->scan->state = state;

	rt2x00usb->scan->status = 0;

	/*
	 * Queue work.
	 */
	if (!queue_work(rt2x00usb->workqueue, &rt2x00usb->scan_work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00usb->scan);
	rt2x00usb->scan = NULL;

	return -EIO;
}

static int rt2500usb_get_stats(struct net_device *net_dev,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	u16 reg;

	/*
	 * Update FCS error count from register.
	 * The dot11ACKFailureCount is updated in interrupt time.
	 * TODO: dot11RTSFailureCount and dot11RTSSuccessCount
	 * are never updated, we need to find a method to see
	 * where we can update those statistics from.
	 */
	rt2x00_register_read(rt2x00usb, STA_CSR0, &reg);
	rt2x00usb->low_level_stats.dot11FCSErrorCount +=
		rt2x00_get_field16(reg, STA_CSR0_FCS_ERROR);

	memcpy(stats, &rt2x00usb->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt2500usb_conf_tx(struct net_device *net_dev,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	struct data_ring *ring;

	ring = rt2x00usb_get_ring(rt2x00usb, queue);
	if (unlikely(!ring)) {
		NOTICE("Ignoring configuration for queue %d.\n", queue);
		return -EINVAL;
	}

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT2400 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 5; /* cw_min: 2^5 = 32. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	if (!params->aifs)
		ring->tx_params.aifs = 2;

	return 0;
}

static int rt2500usb_get_tx_stats(struct net_device *net_dev,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00usb->ring[RING_PRIO].stats,
		sizeof(rt2x00usb->ring[RING_PRIO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00usb->ring[RING_TX].stats,
		sizeof(rt2x00usb->ring[RING_TX].stats));

	return 0;
}

static int rt2500usb_beacon_update(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	struct usb_device *usb_dev = interface_to_usbdev(rt2x00usb->usb_intf);
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00usb->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	memcpy(rt2x00usb_txdata_addr(entry), skb->data, skb->len);
	rt2500usb_write_tx_desc(rt2x00usb,
		rt2x00usb_txdesc_addr(entry), skb, control);

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00usb_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		skb->len + rt2x00usb->ring[RING_BEACON].desc_size,
		rt2500usb_interrupt,
		entry);
	usb_submit_urb(rt2x00usb_urb(entry), GFP_ATOMIC);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt2500usb_alloc_rings(struct rt2x00_usb *rt2x00usb)
{
	unsigned int i;

	rt2x00usb->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00usb->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	SET_FLAG(rt2x00usb, DEVICE_SUPPORT_ATIM);

	for (i = 0; i < RING_NUM; i++) {
		/*
		 *Set device structure.
		 */
		rt2x00usb->ring[i].net_dev =
			usb_get_intfdata(rt2x00usb->usb_intf);
	
		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00usb->ring[i].tx_params.aifs = 2;
		rt2x00usb->ring[i].tx_params.cw_min = 5;
		rt2x00usb->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt2500usb_init_eeprom(struct rt2x00_usb *rt2x00usb)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
			usb_get_intfdata(rt2x00usb->usb_intf));
	u16 eeprom;

	/*
	 * 1 - Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00usb, EEPROM_ANTENNA, &eeprom);

	/*
	 * 2 - Identify RF chipset.
	 */
	rt2x00_set_chip(&rt2x00usb->chip, RT2570,
		rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE));

	if (!rt2x00_rf(&rt2x00usb->chip, RF2522)
	&& !rt2x00_rf(&rt2x00usb->chip, RF2523)
	&& !rt2x00_rf(&rt2x00usb->chip, RF2524)
	&& !rt2x00_rf(&rt2x00usb->chip, RF2525)
	&& !rt2x00_rf(&rt2x00usb->chip, RF2525E)
	&& !rt2x00_rf(&rt2x00usb->chip, RF5222))
		return -ENODEV;

	/*
	 * 3 - Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	conf->antenna_sel = rt2x00_get_field16(eeprom,
				EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * 4 - Store led mode, for correct led behaviour.
	 */
	rt2x00usb->led_mode = rt2x00_get_field16(eeprom,
				EEPROM_ANTENNA_LED_MODE);

	/*
	 * 5 - Check if the BBP tuning should be disabled.
	 */
	rt2x00_eeprom_read(rt2x00usb, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_DISABLE_BBP_TUNE))
		SET_FLAG(rt2x00usb, CONFIG_DISABLE_BBP_TUNING);

	/*
	 * 6 - Read BBP data from EEPROM and store in private structure.
	 */
	rt2x00usb->eeprom = kzalloc(EEPROM_BBP_SIZE * sizeof(u16), GFP_KERNEL);
	if (!rt2x00usb->eeprom) 
		return -ENOMEM;

	rt2x00_eeprom_multiread(rt2x00usb, EEPROM_BBP_START,
		rt2x00usb->eeprom, EEPROM_BBP_SIZE * sizeof(u16));

	return 0;
}

static int rt2500usb_init_mac(struct rt2x00_usb *rt2x00usb)
{
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	u16 eeprom[3] = {0, 0, 0};

	if (GET_FLAG(rt2x00usb, DEVICE_INITIALIZED_MAC))
		return 0;

	/*
	 * Read MAC address from EEPROM.
	 */
	rt2x00_eeprom_multiread(rt2x00usb, EEPROM_MAC_ADDR, &eeprom[0], 6);

	net_dev->dev_addr[0] = rt2x00_get_field16(eeprom[0],
					EEPROM_MAC_ADDR_BYTE0);
	net_dev->dev_addr[1] = rt2x00_get_field16(eeprom[0],
					EEPROM_MAC_ADDR_BYTE1);
	net_dev->dev_addr[2] = rt2x00_get_field16(eeprom[1],
					EEPROM_MAC_ADDR_BYTE2);
	net_dev->dev_addr[3] = rt2x00_get_field16(eeprom[1],
					EEPROM_MAC_ADDR_BYTE3);
	net_dev->dev_addr[4] = rt2x00_get_field16(eeprom[2],
					EEPROM_MAC_ADDR_BYTE4);
	net_dev->dev_addr[5] = rt2x00_get_field16(eeprom[2],
					EEPROM_MAC_ADDR_BYTE5);

	net_dev->addr_len = 6;

	if (is_valid_ether_addr(&net_dev->dev_addr[0])) {
		SET_FLAG(rt2x00usb, DEVICE_INITIALIZED_MAC);
		return 0;
	}

	return -EINVAL;
}

static int rt2500usb_init_hw(struct rt2x00_usb *rt2x00usb)
{
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	struct ieee80211_hw *hw = &rt2x00usb->hw;

	if (GET_FLAG(rt2x00usb, DEVICE_INITIALIZED_HW))
		return 0;

	/*
	 * IEEE80211 Function callbacks.
	 */
	hw->tx			= rt2500usb_tx;
	hw->reset		= rt2500usb_reset;
	hw->add_interface	= rt2500usb_add_interface;
	hw->remove_interface	= rt2500usb_remove_interface;
	hw->config		= rt2500usb_config;
	hw->config_interface	= rt2500usb_config_interface;
	hw->set_multicast_list	= rt2500usb_set_multicast_list;
	hw->passive_scan	= rt2500usb_passive_scan;
	hw->get_stats		= rt2500usb_get_stats;
	hw->conf_tx		= rt2500usb_conf_tx;
	hw->get_tx_stats	= rt2500usb_get_tx_stats;
	hw->beacon_update	= rt2500usb_beacon_update;

	/*
	 * IEEE80211 Variables.
	 */
	hw->version = IEEE80211_VERSION;
	hw->name = DRV_NAME;
	hw->host_gen_beacon = 1;
	hw->device_hides_wep = 0;
	hw->rx_includes_fcs = 0;
	hw->host_broadcast_ps_buffering = 1;
	hw->wep_include_iv = 1;
	hw->data_nullfunc_ack = 1;
	hw->no_tkip_wmm_hwaccel = 1;
	hw->extra_hdr_room = 0;
	hw->device_strips_mic = 0;
	hw->monitor_during_oper = 1;
	hw->fraglist = 0;

	/*
	 * We have 2 TX queues: TX and PRIO.
	 * PRIO should be capable of sending
	 * regular TX frames as well. By doing this
	 * we can increase performance since we
	 * will have support for high priority TX frames.
	 */
	hw->queues = RING_NUM_TX;

	if (ieee80211_register_hw(net_dev, hw))
		return -EIO;

	SET_FLAG(rt2x00usb, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt2500usb_init_hw_channels(struct rt2x00_usb *rt2x00usb,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u32 rf2_base;
	u16 eeprom;
	static const struct {
		unsigned int chip;
		u32 val[3];
	} rf[] = {
		{ RF2522,	{ 0x00002050, 0x00000101, 0x00000000 } },
		{ RF2523,	{ 0x00022010, 0x000e0111, 0x00000a1b } },
		{ RF2524,	{ 0x00032020, 0x00000101, 0x00000a1b } },
		{ RF2525,	{ 0x00022020, 0x00060111, 0x00000a1b } },
		{ RF2525E,	{ 0x00022010, 0x00060111, 0x00000000 } },
		{ RF5222,	{ 0x00000000, 0x00000101, 0x00000000 } }
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS
			| IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS
		| IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00usb->chip, RF5222)) {
		for (i = 14; i < 37; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS
				| IEEE80211_CHAN_W_ACTIVE_SCAN
				| IEEE80211_CHAN_W_SCAN;
			channels[i].power_level = 0x18;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	rf2_base = 0;
	if (rt2x00_rf(&rt2x00usb->chip, RF2525))
		rf2_base = 0x00080000;

	if (rt2x00_rf(&rt2x00usb->chip, RF2522)) {
		static const u32 vals[] = {
			0x000c1fda, 0x000c1fee, 0x000c2002, 0x000c2016,
			0x000c202a, 0x000c203e, 0x000c2052, 0x000c2066,
			0x000c207a, 0x000c208e, 0x000c20a2, 0x000c20b6,
			0x000c20ca, 0x000c20fa
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	} else if (rt2x00_rf(&rt2x00usb->chip, RF2523)
	|| rt2x00_rf(&rt2x00usb->chip, RF2524)
	|| rt2x00_rf(&rt2x00usb->chip, RF2525)) {
		static const u32 vals[] = {
			0x00000c9e, 0x00000ca2, 0x00000ca6, 0x00000caa,
			0x00000cae, 0x00000cb2, 0x00000cb6, 0x00000cba,
			0x00000cbe, 0x00000d02, 0x00000d06, 0x00000d0a,
			0x00000d0e, 0x00000d1a
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i] | rf2_base;
	} else if (rt2x00_rf(&rt2x00usb->chip, RF2525E)) {
		static const u32 vals[] = {
			0x0000089a, 0x0000089e, 0x0000089e, 0x000008a2,
			0x000008a2, 0x000008a6, 0x000008a6, 0x000008aa,
			0x000008aa, 0x000008ae, 0x000008ae, 0x000008b2,
			0x000008b2, 0x000008b6
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	} else if (rt2x00_rf(&rt2x00usb->chip, RF5222)) {
		static const u32 vals[] = {
			0x00001136, 0x0000113a, 0x0000113e, 0x00001182,
			0x00001186, 0x0000118a, 0x0000118e, 0x00001192,
			0x00001196, 0x0000119a, 0x0000119e, 0x000011a2,
			0x000011a6, 0x000011ae, 0x0001889a, 0x0001889a,
			0x0001889e, 0x000188a2, 0x000188a6, 0x000188aa,
			0x000188ae, 0x000188b2, 0x00008802, 0x00008806,
			0x0000880a, 0x0000880e, 0x00008812, 0x00008816,
			0x0000881a, 0x0000881e, 0x00008822, 0x00008826,
			0x0000882a, 0x000090a6, 0x000090ae, 0x000090b6,
			0x000090be
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00usb,
			EEPROM_TXPOWER_START + i, &eeprom);

		channels[(i * 2)].power_level =
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_1);
		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			channels[(i * 2)].power_level);

		channels[(i * 2) + 1].power_level =
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_2);
		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			channels[(i * 2) + 1].power_level);
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	for (i = 0; i < ARRAY_SIZE(rf); i++) {
		if (rt2x00_rf(&rt2x00usb->chip, rf[i].chip)) {
			rt2x00usb->rf1 = rf[i].val[0];
			rt2x00usb->rf3 = rf[i].val[1];
			rt2x00usb->rf4 = rf[i].val[2];
		}
	}
}

static void rt2500usb_init_hw_rates(struct rt2x00_usb *rt2x00usb,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	rates[0].rate = 10;
	rates[0].val = DEVICE_RATE_1MB;
	rates[0].flags = IEEE80211_RATE_CCK;
	rates[0].val2 = DEVICE_RATE_1MB;
	rates[0].min_rssi_ack = 0;
	rates[0].min_rssi_ack_delta = 0;

	rates[1].rate = 20;
	rates[1].val = DEVICE_RATE_2MB;
	rates[1].flags = IEEE80211_RATE_CCK_2;
	rates[1].val2 = DEVICE_RATE_2MB_PREAMBLE;
	rates[1].min_rssi_ack = 0;
	rates[1].min_rssi_ack_delta = 0;

	rates[2].rate = 55;
	rates[2].val = DEVICE_RATE_55MB;
	rates[2].flags = IEEE80211_RATE_CCK_2;
	rates[2].val2 = DEVICE_RATE_55MB_PREAMBLE;
	rates[2].min_rssi_ack = 0;
	rates[2].min_rssi_ack_delta = 0;

	rates[3].rate = 110;
	rates[3].val = DEVICE_RATE_11MB;
	rates[3].flags = IEEE80211_RATE_CCK_2;
	rates[3].val2 = DEVICE_RATE_11MB_PREAMBLE;
	rates[3].min_rssi_ack = 0;
	rates[3].min_rssi_ack_delta = 0;

	rates[4].rate = 60;
	rates[4].val = DEVICE_RATE_6MB;
	rates[4].flags = IEEE80211_RATE_OFDM;
	rates[4].val2 = DEVICE_RATE_6MB;
	rates[4].min_rssi_ack = 0;
	rates[4].min_rssi_ack_delta = 0;

	rates[5].rate = 90;
	rates[5].val = DEVICE_RATE_9MB;
	rates[5].flags = IEEE80211_RATE_OFDM;
	rates[5].val2 = DEVICE_RATE_9MB;
	rates[5].min_rssi_ack = 0;
	rates[5].min_rssi_ack_delta = 0;

	rates[6].rate = 120;
	rates[6].val = DEVICE_RATE_12MB;
	rates[6].flags = IEEE80211_RATE_OFDM;
	rates[6].val2 = DEVICE_RATE_12MB;
	rates[6].min_rssi_ack = 0;
	rates[6].min_rssi_ack_delta = 0;

	rates[7].rate = 180;
	rates[7].val = DEVICE_RATE_18MB;
	rates[7].flags = IEEE80211_RATE_OFDM;
	rates[7].val2 = DEVICE_RATE_18MB;
	rates[7].min_rssi_ack = 0;
	rates[7].min_rssi_ack_delta = 0;

	rates[8].rate = 240;
	rates[8].val = DEVICE_RATE_24MB;
	rates[8].flags = IEEE80211_RATE_OFDM;
	rates[8].val2 = DEVICE_RATE_24MB;
	rates[8].min_rssi_ack = 0;
	rates[8].min_rssi_ack_delta = 0;

	rates[9].rate = 360;
	rates[9].val = DEVICE_RATE_36MB;
	rates[9].flags = IEEE80211_RATE_OFDM;
	rates[9].val2 = DEVICE_RATE_36MB;
	rates[9].min_rssi_ack = 0;
	rates[9].min_rssi_ack_delta = 0;

	rates[10].rate = 480;
	rates[10].val = DEVICE_RATE_48MB;
	rates[10].flags = IEEE80211_RATE_OFDM;
	rates[10].val2 = DEVICE_RATE_48MB;
	rates[10].min_rssi_ack = 0;
	rates[10].min_rssi_ack_delta = 0;

	rates[11].rate = 540;
	rates[11].val = DEVICE_RATE_54MB;
	rates[11].flags = IEEE80211_RATE_OFDM;
	rates[11].val2 = DEVICE_RATE_54MB;
	rates[11].min_rssi_ack = 0;
	rates[11].min_rssi_ack_delta = 0;
}

static int rt2500usb_init_hw_modes(struct rt2x00_usb *rt2x00usb)
{
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	struct ieee80211_hw *hw = &rt2x00usb->hw;
	int num_modes;
	int num_channels;

	/*
	 * RT2500 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5222 also supports 802.11a, so allocate an
	 * additional 23 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00usb->chip, RF5222)) {
		num_modes = 3;
		num_channels = 37;
	}

	hw->num_modes = num_modes;
	hw->modes =
		kzalloc((sizeof(struct ieee80211_hw_modes) * num_modes),
		GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->channels =
		kzalloc((sizeof(struct ieee80211_channel) * num_channels),
		GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->rates =
		kzalloc((sizeof(struct ieee80211_rate) * 12),
		GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	hw->modes[0].mode = MODE_IEEE80211G;
	hw->modes[0].num_channels = 14;
	hw->modes[0].num_rates = 12;

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	hw->modes[1].mode = MODE_IEEE80211B;
	hw->modes[1].num_channels = 14;
	hw->modes[1].num_rates = 4;
	hw->modes[1].channels = hw->modes[0].channels;
	hw->modes[1].rates = hw->modes[0].rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (rt2x00_rf(&rt2x00usb->chip, RF5222)) {
		hw->modes[2].mode = MODE_IEEE80211A;
		hw->modes[2].num_channels = 37;
		hw->modes[2].num_rates = 8;
		hw->modes[2].channels = &hw->modes[0].channels[14];
		hw->modes[2].rates = &hw->modes[0].rates[4];
	}

	rt2500usb_init_hw_channels(rt2x00usb, hw->modes[0].channels);
	rt2500usb_init_hw_rates(rt2x00usb, hw->modes[0].rates);

	/*
	 * xr_end is only used on Atheros cards.
	 */
	hw->modes->xr_end = 0;

	return ieee80211_update_hw(net_dev, hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static void rt2500usb_free_dev(struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Free ring structures.
	 */
	kfree(rt2x00usb->ring);
	rt2x00usb->ring = NULL;

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00usb->eeprom);

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00usb->workqueue)) {
		destroy_workqueue(rt2x00usb->workqueue);
		rt2x00usb->workqueue = NULL;
	}

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00usb->hw.modes)) {
		kfree(rt2x00usb->hw.modes->channels);
		kfree(rt2x00usb->hw.modes->rates);
		kfree(rt2x00usb->hw.modes);
		rt2x00usb->hw.modes = NULL;
	}
}

static int rt2500usb_alloc_dev(struct usb_interface *usb_intf,
	struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	rt2x00usb->usb_intf = usb_intf;

	rt2x00usb->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00usb->workqueue)
		return -ENODEV;

	/*
	 * Initialize cofniguration work.
	 */
	INIT_WORK(&rt2x00usb->config_work, rt2500usb_config_update, rt2x00usb);
	INIT_WORK(&rt2x00usb->interface_work,
		rt2500usb_interface_update, rt2x00usb);

	/*
	 * Reset current working type.
	 */
	rt2x00usb->interface_type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	INIT_WORK(&rt2x00usb->scan_work, rt2500usb_scan, rt2x00usb);
	rt2x00usb->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt2500usb_alloc_rings(rt2x00usb))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt2500usb_init_eeprom(rt2x00usb)
	|| rt2500usb_init_mac(rt2x00usb)
	|| rt2500usb_init_hw(rt2x00usb)
	|| rt2500usb_init_hw_modes(rt2x00usb)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	return 0;

exit:
	rt2500usb_free_dev(net_dev);

	return -ENODEV;
}

/*
 * USB driver handlers.
 */
static int rt2500usb_probe(
	struct usb_interface *usb_intf, const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(usb_intf);
	struct net_device *net_dev;
	int status;

	if (unlikely(id->driver_info != RT2570)) {
		ERROR("Detected device not supported.\n");
		return -ENODEV;
	}

	usb_dev = usb_get_dev(usb_dev);

	net_dev = ieee80211_alloc_hw(sizeof(struct rt2x00_usb), NULL);
	if (!net_dev) {
		ERROR("Failed to allocate device.\n");
		status = -ENOMEM;
		goto exit_put_device;
	}

	SET_ETHTOOL_OPS(net_dev, &rt2500usb_ethtool_ops);

	usb_set_intfdata(usb_intf, net_dev);

	status = rt2500usb_alloc_dev(usb_intf, net_dev);
	if (status) {
		ERROR("Failed to allocate device.\n");
		status = -ENOMEM;
		goto exit_free_device;
	}

	return 0;

exit_free_device:
	ieee80211_free_hw(net_dev);

exit_put_device:
	usb_put_dev(usb_dev);

	return status;
}

static void rt2500usb_disconnect(struct usb_interface *usb_intf)
{
	struct net_device *net_dev = usb_get_intfdata(usb_intf);
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	rt2500usb_disable_radio(rt2x00usb);

	rt2500usb_uninitialize(rt2x00usb);

	rt2500usb_free_dev(net_dev);

	ieee80211_unregister_hw(net_dev);

	ieee80211_free_hw(net_dev);

	usb_set_intfdata(usb_intf, NULL);

	usb_put_dev(interface_to_usbdev(usb_intf));
}

#ifdef CONFIG_PM
static int rt2500usb_suspend(
	struct usb_interface *usb_intf, pm_message_t state)
{
	struct net_device *net_dev = usb_get_intfdata(usb_intf);
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	NOTICE("Going to sleep.\n");

	/*
	 * Disable the radio.
	 */
	rt2500usb_disable_radio(rt2x00usb);

	/*
	 * Set device mode to sleep for power management.
	 */
	if (rt2500usb_set_state(rt2x00usb, STATE_SLEEP))
		return -EBUSY;

	/*
	 * Uninitialize device.
	 */
	rt2500usb_uninitialize(rt2x00usb);

	/*
	 * Uninitialize hardware.
	 */
	rt2500usb_free_dev(net_dev);

	/*
	 * Decrease usbdev refcount.
	 */
	usb_put_dev(interface_to_usbdev(usb_intf));

	return 0;
}

static int rt2500usb_resume(struct usb_interface *usb_intf)
{
	struct net_device *net_dev = usb_get_intfdata(usb_intf);
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	NOTICE("Waking up.\n");

	/*
	 * Increase usbdev refcount.
	 */
	usb_get_dev(interface_to_usbdev(usb_intf));

	/*
	 * Initialize hardware.
	 */
	if (rt2500usb_alloc_dev(usb_intf, net_dev)) {
		ERROR("Failed to allocate device.\n");
		return -ENOMEM;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	return rt2500usb_set_state(rt2x00usb, STATE_AWAKE);
}
#endif /* CONFIG_PM */

/*
 * rt2500usb module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct usb_device_id rt2500usb_device_table[] = {
	/* ASUS */
	{ USB_DEVICE(0x0b05, 0x1706), .driver_info = RT2570},
	{ USB_DEVICE(0x0b05, 0x1707), .driver_info = RT2570},
	/* Belkin */
	{ USB_DEVICE(0x050d, 0x7050), .driver_info = RT2570},
	{ USB_DEVICE(0x050d, 0x7051), .driver_info = RT2570},
	{ USB_DEVICE(0x050d, 0x705a), .driver_info = RT2570},
	/* Cisco Systems */
	{ USB_DEVICE(0x13b1, 0x000d), .driver_info = RT2570},
	{ USB_DEVICE(0x13b1, 0x0011), .driver_info = RT2570},
	{ USB_DEVICE(0x13b1, 0x001a), .driver_info = RT2570},
	/* Conceptronic */
	{ USB_DEVICE(0x14b2, 0x3c02), .driver_info = RT2570},
	/* D-LINK */
	{ USB_DEVICE(0x2001, 0x3c00), .driver_info = RT2570},
	/* Gigabyte */
	{ USB_DEVICE(0x1044, 0x8001), .driver_info = RT2570},
	{ USB_DEVICE(0x1044, 0x8007), .driver_info = RT2570},
	/* Hercules */
	{ USB_DEVICE(0x06f8, 0xe000), .driver_info = RT2570},
	/* Melco */
	{ USB_DEVICE(0x0411, 0x0066), .driver_info = RT2570},
	{ USB_DEVICE(0x0411, 0x0067), .driver_info = RT2570},
	{ USB_DEVICE(0x0411, 0x008b), .driver_info = RT2570},
	/* MSI */
	{ USB_DEVICE(0x0db0, 0x6861), .driver_info = RT2570},
	{ USB_DEVICE(0x0db0, 0x6865), .driver_info = RT2570},
	{ USB_DEVICE(0x0db0, 0x6869), .driver_info = RT2570},
	/* Ralink */
	{ USB_DEVICE(0x148f, 0x1706), .driver_info = RT2570},
	{ USB_DEVICE(0x148f, 0x2570), .driver_info = RT2570},
	{ USB_DEVICE(0x148f, 0x2573), .driver_info = RT2570},
	{ USB_DEVICE(0x148f, 0x9020), .driver_info = RT2570},
	/* SMC */
	{ USB_DEVICE(0x0707, 0xee13), .driver_info = RT2570},
	/* Spairon */
	{ USB_DEVICE(0x114b, 0x0110), .driver_info = RT2570},
	/* Trust */
	{ USB_DEVICE(0x0eb0, 0x9020), .driver_info = RT2570},
	/* Zinwell */
	{ USB_DEVICE(0x5a57, 0x0260), .driver_info = RT2570},
	{0,}
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT2500 USB Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2570 USB chipset based cards");
MODULE_DEVICE_TABLE(usb, rt2500usb_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT2500USB_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT2500USB_DEBUG */

static struct usb_driver rt2500usb_driver = {
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15)
	.owner		= THIS_MODULE,
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15) */
	.name		= DRV_NAME,
	.id_table	= rt2500usb_device_table,
	.probe		= rt2500usb_probe,
	.disconnect	= rt2500usb_disconnect,
#ifdef CONFIG_PM
	.suspend	= rt2500usb_suspend,
	.resume		= rt2500usb_resume,
#endif /* CONFIG_PM */
};

static int __init rt2500usb_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return usb_register(&rt2500usb_driver);
}

static void __exit rt2500usb_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	usb_deregister(&rt2500usb_driver);
}

module_init(rt2500usb_init);
module_exit(rt2500usb_exit);
