/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2400pci
	Abstract: rt2400pci device specific routines.
	Supported chipsets: RT2460.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/ethtool.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

#include <asm/io.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME			"rt2400pci"

#ifdef CONFIG_RT2400PCI_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT2400PCI_DEBUG */

#ifdef CONFIG_RT2400PCI_BUTTON
#define CONFIG_RT2X00_BUTTON
#endif /* CONFIG_RT2400PCI_BUTTON */

#include "rt2x00.h"
#include "rt2x00pci.h"
#include "rt2400pci.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 * The caller to these register access functions, should take precautions
 * for the correct byte ordering of the values.
 */
static inline void rt2x00_register_read(
	const struct rt2x00_pci *rt2x00pci,
	const unsigned long offset, u32 *value)
{
	*value = readl((void*)(rt2x00pci->csr_addr + offset));
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_pci *rt2x00pci,
	const unsigned long offset, u32 *value, const u16 length)
{
	memcpy_fromio(
		(void*)value, (void*)(rt2x00pci->csr_addr + offset), length);
}

static inline void rt2x00_register_write(
	const struct rt2x00_pci *rt2x00pci,
	const unsigned long offset, const u32 value)
{
	writel(value, (void*)(rt2x00pci->csr_addr + offset));
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_pci *rt2x00pci,
	const unsigned long offset, u32 *value, const u16 length)
{
	memcpy_toio(
		(void*)(rt2x00pci->csr_addr + offset), (void*)value, length);
}

static void rt2x00_bbp_write(
	const struct rt2x00_pci *rt2x00pci, const u8 reg_id, const u8 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00pci, BBPCSR, &reg);
		if (!rt2x00_get_field32(reg, BBPCSR_BUSY))
			goto bbp_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("BBPCSR register busy. Write failed.\n");
	return;

bbp_write:
	reg = 0;
	rt2x00_set_field32(&reg, BBPCSR_VALUE, value);
	rt2x00_set_field32(&reg, BBPCSR_REGNUM, reg_id);
	rt2x00_set_field32(&reg, BBPCSR_BUSY, 1);
	rt2x00_set_field32(&reg, BBPCSR_WRITE_CONTROL, 1);

	rt2x00_register_write(rt2x00pci, BBPCSR, reg);
}

static void rt2x00_bbp_read(
	const struct rt2x00_pci *rt2x00pci, const u8 reg_id, u8 *value)
{
	u32 reg;
	unsigned int i;

	/*
	 * First request the register we wish to read from.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, BBPCSR_REGNUM, reg_id);
	rt2x00_set_field32(&reg, BBPCSR_BUSY, 1);
	rt2x00_set_field32(&reg, BBPCSR_WRITE_CONTROL, 0);

	rt2x00_register_write(rt2x00pci, BBPCSR, reg);

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00pci, BBPCSR, &reg);
		if (!rt2x00_get_field32(reg, BBPCSR_BUSY)) {
			*value = rt2x00_get_field32(reg, BBPCSR_VALUE);
			return;
		}
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("BBPCSR register busy. Read failed.\n");
	*value = 0xff;
}

static void rt2x00_rf_write(
	const struct rt2x00_pci *rt2x00pci, const u32 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00pci, RFCSR, &reg);
		if (!rt2x00_get_field32(reg, RFCSR_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("RFCSR register busy. Write failed.\n");
	return;

rf_write:
	reg = value;
	rt2x00_set_field32(&reg, RFCSR_NUMBER_OF_BITS, 20);
	rt2x00_set_field32(&reg, RFCSR_IF_SELECT, 0);
	rt2x00_set_field32(&reg, RFCSR_BUSY, 1);

	rt2x00_register_write(rt2x00pci, RFCSR, reg);
}

/*
 * EEPROM access.
 * The EEPROM is being accessed by word index.
 * rt2x00_eeprom_read is the main access function,
 * which should be called from the rest of the module.
 * It will take the index number of the eeprom word
 * and the value in which the data should be stored.
 * This function does assume the bus_width attribute
 * within rt2x00_pci has been correctly set.
 */
static inline void rt2x00_eeprom_pulse_high(
	const struct rt2x00_pci *rt2x00pci, u32 *flags)
{
	rt2x00_set_field32(flags, CSR21_EEPROM_DATA_CLOCK, 1);
	rt2x00_register_write(rt2x00pci, CSR21, *flags);
	udelay(1);
}

static inline void rt2x00_eeprom_pulse_low(
	const struct rt2x00_pci *rt2x00pci, u32 *flags)
{
	rt2x00_set_field32(flags, CSR21_EEPROM_DATA_CLOCK, 0);
	rt2x00_register_write(rt2x00pci, CSR21, *flags);
	udelay(1);
}

static void rt2x00_eeprom_shift_out_bits(
	const struct rt2x00_pci *rt2x00pci,
	const u16 data, const u16 count)
{
	u32 flags;
	u32 mask = 1 << (count - 1);

	rt2x00_register_read(rt2x00pci, CSR21, &flags);

	/*
	 * Clear data flags.
	 */
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_OUT, 0);

	/*
	 * Start writing all bits.
	 */
	do {
		/*
		 * Set the data_in bit only when required.
		 */
		if (data & mask)
			rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 1);
		else
			rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);

		rt2x00_register_write(rt2x00pci, CSR21, flags);

		rt2x00_eeprom_pulse_high(rt2x00pci, &flags);
		rt2x00_eeprom_pulse_low(rt2x00pci, &flags);

		/*
		 * Shift to next bit.
		 */
		mask >>= 1;
	} while (mask);

	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);
	rt2x00_register_write(rt2x00pci, CSR21, flags);
}

static void rt2x00_eeprom_shift_in_bits(
	const struct rt2x00_pci *rt2x00pci, u16 *data)
{
	u32 flags;
	unsigned int i;

	rt2x00_register_read(rt2x00pci, CSR21, &flags);

	/*
	 * Clear data flags.
	 */
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_OUT, 0);

	/*
	 * Start reading all 16 bits.
	 */
	for (i = 0; i < 16; i++) {
		/*
		 * Shift to the next bit.
		 */
		*data <<= 1;

		rt2x00_eeprom_pulse_high(rt2x00pci, &flags);

		rt2x00_register_read(rt2x00pci, CSR21, &flags);

		/*
		 * Clear data_in flag.
		 */
		rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);

		/*
		 * Set the data bit to 1 when the data_out flag is set.
		 */
		if (rt2x00_get_field32(flags, CSR21_EEPROM_DATA_OUT))
			*data |= 1;

		rt2x00_eeprom_pulse_low(rt2x00pci, &flags);
	}
}

static void rt2x00_eeprom_read(
	const struct rt2x00_pci *rt2x00pci, const u8 word, u16 *data)
{
	u32 flags;

	/*
	 * Clear all flags, and enable chip select.
	 */
	rt2x00_register_read(rt2x00pci, CSR21, &flags);
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_OUT, 0);
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_CLOCK, 0);
	rt2x00_set_field32(&flags, CSR21_EEPROM_CHIP_SELECT, 1);
	rt2x00_register_write(rt2x00pci, CSR21, flags);

	/*
	 * kick a pulse.
	 */
	rt2x00_eeprom_pulse_high(rt2x00pci, &flags);
	rt2x00_eeprom_pulse_low(rt2x00pci, &flags);

	/*
	 * Select the read opcode and bus_width.
	 */
	rt2x00_eeprom_shift_out_bits(rt2x00pci, EEPROM_READ_OPCODE, 3);
	rt2x00_eeprom_shift_out_bits(rt2x00pci, word, rt2x00pci->eeprom_width);

	rt2x00_eeprom_shift_in_bits(rt2x00pci, data);

	/*
	 * Clear chip_select and data_in flags.
	 */
	rt2x00_register_read(rt2x00pci, CSR21, &flags);
	rt2x00_set_field32(&flags, CSR21_EEPROM_DATA_IN, 0);
	rt2x00_set_field32(&flags, CSR21_EEPROM_CHIP_SELECT, 0);
	rt2x00_register_write(rt2x00pci, CSR21, flags);

	/*
	 * kick a pulse.
	 */
	rt2x00_eeprom_pulse_high(rt2x00pci, &flags);
	rt2x00_eeprom_pulse_low(rt2x00pci, &flags);
}

static void rt2x00_eeprom_multiread(
	const struct rt2x00_pci *rt2x00pci,
	const u8 word, u16 *data, const u16 length)
{
	unsigned int i;

	for (i = 0; i < (length / sizeof(u16)); i++)
		rt2x00_eeprom_read(rt2x00pci, word + i, data++);
}

static void rt2400pci_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->bus_info, pci_name(rt2x00pci->pci_dev));
}

static int rt2400pci_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt2400pci_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	regs->version = rt2x00_get_chip_id(&rt2x00pci->chip);

	regs->len = CSR_REG_SIZE;

	rt2x00_register_multiread(rt2x00pci, CSR_REG_BASE, data, CSR_REG_SIZE);
}

#ifdef CONFIG_RT2400PCI_DEBUG
static u32 rt2400pci_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt2400pci_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT2400PCI_DEBUG */

static int rt2400pci_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt2400pci_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	eeprom->magic = rt2x00_get_chip_id(&rt2x00pci->chip);

	rt2x00_eeprom_multiread(rt2x00pci,
		eeprom->offset / sizeof(u16), (void*)data, eeprom->len);

	return 0;
}

static struct ethtool_ops rt2400pci_ethtool_ops = {
	.get_drvinfo	= rt2400pci_get_drvinfo,
	.get_regs_len	= rt2400pci_get_regs_len,
	.get_regs	= rt2400pci_get_regs,
#ifdef CONFIG_RT2400PCI_DEBUG
	.get_msglevel	= rt2400pci_get_msglevel,
	.set_msglevel	= rt2400pci_set_msglevel,
#endif /* CONFIG_RT2400PCI_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt2400pci_get_eeprom_len,
	.get_eeprom	= rt2400pci_get_eeprom,
};

#ifdef CONFIG_RT2400PCI_BUTTON
static int rt2400pci_button_poll(unsigned long data)
{
	struct rt2x00_pci *rt2x00pci = (struct rt2x00_pci*)data;
	u32 reg;

	rt2x00_register_read(rt2x00pci, GPIOCSR, &reg);
	return rt2x00_get_field32(reg, GPIOCSR_BIT0);
}

static void rt2400pci_button_enable_radio(unsigned long data)
{
	rt2400pci_enable_radio((struct rt2x00_pci*)data);
}

static void rt2400pci_button_disable_radio(unsigned long data)
{
	rt2400pci_disable_radio((struct rt2x00_pci*)data);
}

static void rt2400pci_button_start(struct rt2x00_pci *rt2x00pci)
{
	struct radio_button *button = &rt2x00pci->button;

	/*
	 * Only start the button polling when
	 * the hardware button is present and the
	 * poll_delay module parameter has been set.
	 */
	if(!GET_FLAG(rt2x00pci, DEVICE_SUPPORT_HW_BUTTON)
	|| !rt2x00_poll_delay)
		return;

	button->dev_name	= "rt2400pci";
	button->data		= (unsigned long)rt2x00pci;
	button->button_poll	= rt2400pci_button_poll;
	button->enable_radio	= rt2400pci_button_enable_radio;
	button->disable_radio	= rt2400pci_button_disable_radio;
	button->poll_delay	= rt2x00_poll_delay;

	if (radiobtn_register_device(button))
		ERROR("Failed to register button handler.\n");
}

static void rt2400pci_button_stop(struct rt2x00_pci *rt2x00pci)
{
	if(!GET_FLAG(rt2x00pci, DEVICE_SUPPORT_HW_BUTTON)
	|| !rt2x00_poll_delay)
		return;

	radiobtn_unregister_device(&rt2x00pci->button);
}
#else /* CONFIG_RT2400PCI_BUTTON */
static void rt2400pci_button_start(struct rt2x00_pci *rt2x00pci){}
static void rt2400pci_button_stop(struct rt2x00_pci *rt2x00pci){}
#endif /* CONFIG_RT2400PCI_BUTTON */

/*
 * Configuration handlers.
 */
static void rt2400pci_config_bssid(struct rt2x00_pci *rt2x00pci, u8 *bssid)
{
	u32 reg[2] = {0, 0};

	rt2x00_set_field32(&reg[0], CSR5_BYTE0, *(bssid));
	rt2x00_set_field32(&reg[0], CSR5_BYTE1, *(bssid + 1));
	rt2x00_set_field32(&reg[0], CSR5_BYTE2, *(bssid + 2));
	rt2x00_set_field32(&reg[0], CSR5_BYTE3, *(bssid + 3));
	rt2x00_set_field32(&reg[1], CSR6_BYTE4, *(bssid + 4));
	rt2x00_set_field32(&reg[1], CSR6_BYTE5, *(bssid + 5));

	rt2x00_register_multiwrite(rt2x00pci, CSR5, &reg[0], sizeof(reg));
}

static void rt2400pci_config_promisc(struct rt2x00_pci *rt2x00pci, int promisc)
{
	u32 reg;

	rt2x00_register_read(rt2x00pci, RXCSR0, &reg);

	if (promisc) {
		rt2x00_set_field32(&reg, RXCSR0_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00pci, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field32(&reg, RXCSR0_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00pci, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00pci, RXCSR0, reg);
}

static void rt2400pci_config_type(struct rt2x00_pci *rt2x00pci, int type)
{
	u32 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (type == rt2x00pci->interface_type
	|| (rt2x00pci->monitor_count
	^ GET_FLAG(rt2x00pci, INTERFACE_ENABLED_MONITOR)))
		return;

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00pci, RXCSR0, &reg);

	if (type == IEEE80211_IF_TYPE_IBSS
	|| type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field32(&reg, RXCSR0_DROP_TODS, 1);
	else
		rt2x00_set_field32(&reg, RXCSR0_DROP_TODS, 0);

	rt2x00_set_field32(&reg, RXCSR0_DROP_CRC, 1);
	if (rt2x00pci->monitor_count) {
		rt2x00_set_field32(&reg, RXCSR0_DROP_PHYSICAL, 0);
		rt2x00_set_field32(&reg, RXCSR0_DROP_CONTROL, 0);
		rt2x00_set_field32(&reg, RXCSR0_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field32(&reg, RXCSR0_DROP_PHYSICAL, 1);
		rt2x00_set_field32(&reg, RXCSR0_DROP_CONTROL, 1);
		rt2x00_set_field32(&reg, RXCSR0_DROP_VERSION_ERROR, 1);
	}

	rt2x00_register_write(rt2x00pci, RXCSR0, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (rt2x00pci->monitor_count)
		rt2400pci_config_promisc(rt2x00pci, 1);

	/*
	 * Enable TSF counter.
	 */
	rt2x00_register_read(rt2x00pci, CSR14, &reg);
	rt2x00_set_field32(&reg, CSR14_TSF_COUNT, 1);
	if (type == IEEE80211_IF_TYPE_IBSS)
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 2);
	else if (type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 1);
	else
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 0);
	rt2x00_register_write(rt2x00pci, CSR14, reg);

	/*
	 * Enable beacon config
	 */
	rt2x00_register_read(rt2x00pci, BCNCSR1, &reg);
	rt2x00_set_field32(&reg, BCNCSR1_PRELOAD,
		PREAMBLE + get_duration(IEEE80211_HEADER, 2));
	rt2x00_register_write(rt2x00pci, BCNCSR1, reg);

	/*
	 * Update working mode.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR)
		rt2x00pci->interface_type = type;

	if (rt2x00pci->monitor_count)
		SET_FLAG(rt2x00pci, INTERFACE_ENABLED_MONITOR);
	else
		CLEAR_FLAG(rt2x00pci, INTERFACE_ENABLED_MONITOR);
}

static void rt2400pci_config_channel(struct rt2x00_pci *rt2x00pci,
	int rf2, int channel, int freq)
{
	u32 rf1 = rt2x00pci->rf1;
	u32 rf3 = rt2x00pci->rf3;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00pci->rx_params.channel)
		return;

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x.\n",
		rf1, rf2, rf3);

	rt2x00_rf_write(rt2x00pci, rf1);
	rt2x00_rf_write(rt2x00pci, rf2);
	rt2x00_rf_write(rt2x00pci, rf3);

	/*
	 * RF2420 chipset don't need any additional actions.
	 */
	if (rt2x00_rf(&rt2x00pci->chip, RF2420))
		return;

	/*
	 * For the RT2421 chipsets we need to write an invalid
	 * reference clock rate to activate auto_tune.
	 * After that we set the value back to the correct channel.
	 */
	rt2x00_rf_write(rt2x00pci, rf1);
	rt2x00_rf_write(rt2x00pci, cpu_to_le32(0x000c2a32));
	rt2x00_rf_write(rt2x00pci, rf3);

	msleep(1);

	rt2x00_rf_write(rt2x00pci, rf1);
	rt2x00_rf_write(rt2x00pci, rf2);
	rt2x00_rf_write(rt2x00pci, rf3);

	msleep(1);

	/*
	 * Switch off tuning bits.
	 */
	rt2x00_set_field32(&rf1, RF1_TUNER, 0);
	rt2x00_set_field32(&rf3, RF3_TUNER, 0);

	rt2x00_rf_write(rt2x00pci, rf1);
	rt2x00_rf_write(rt2x00pci, rf3);

	/*
	 * Update active info for RX.
	 */
	rt2x00pci->rx_params.freq = freq;
	rt2x00pci->rx_params.channel = channel;

	/*
	 * Update rf fields
	 */
	rt2x00pci->rf1 = rf1;
	rt2x00pci->rf2 = rf2;
	rt2x00pci->rf3 = rf3;

	/*
	 * Clear false CRC during channel switch.
	 */
	rt2x00_register_read(rt2x00pci, CNT0, &rf1);
}

static void rt2400pci_config_txpower(struct rt2x00_pci *rt2x00pci, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00pci->tx_power)
		return;

	rt2x00_bbp_write(rt2x00pci, 3, txpower);

	rt2x00pci->tx_power = txpower;
}

static void rt2400pci_config_antenna(struct rt2x00_pci *rt2x00pci, int antenna)
{
	u8 reg_rx;
	u8 reg_tx;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00pci->rx_params.antenna == antenna)
		return;

	rt2x00_bbp_read(rt2x00pci, 4, &reg_rx);
	rt2x00_bbp_read(rt2x00pci, 1, &reg_tx);

	/*
	 * Clear current config antenna bits.
	 */
	reg_rx &= ~0x06;
	reg_tx &= ~0x03;

	/*
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	if (antenna == 0) {
		/* Diversity. */
		reg_rx |= 0x02;
		reg_tx |= 0x01;
	} else if (antenna == 1) {
		/* RX: Antenna B */
		reg_rx |= 0x04;
		/* TX: Antenna A */
		reg_tx |= 0x00;
	} else if (antenna == 2) {
		/* RX: Antenna A */
		reg_rx |= 0x00;
		/* TX: Antenna B */
		reg_tx |= 0x02;
	}

	rt2x00_bbp_write(rt2x00pci, 4, reg_rx);
	rt2x00_bbp_write(rt2x00pci, 1, reg_tx);

	/*
	 * Update active info for RX.
	 */
	rt2x00pci->rx_params.antenna = antenna;
}

static void rt2400pci_config_cw(struct rt2x00_pci *rt2x00pci,
	struct ieee80211_tx_queue_params *params)
{
	u32 reg;

	rt2x00_register_read(rt2x00pci, CSR11, &reg);
	rt2x00_set_field32(&reg, CSR11_CWMIN, params->cw_min);
	rt2x00_set_field32(&reg, CSR11_CWMAX, params->cw_max);
	rt2x00_register_write(rt2x00pci, CSR11, reg);
}

static void rt2400pci_config_duration(
	struct rt2x00_pci *rt2x00pci, int short_slot_time)
{
	u32 reg;
	u32 value;

	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_read(rt2x00pci, CSR11, &reg);
	rt2x00_set_field32(&reg, CSR11_SLOT_TIME, short_slot_time);
	rt2x00_register_write(rt2x00pci, CSR11, reg);

	rt2x00_register_read(rt2x00pci, CSR18, &reg);
	rt2x00_set_field32(&reg, CSR18_SIFS, SIFS);
	rt2x00_set_field32(&reg, CSR18_PIFS, SIFS + short_slot_time);
	rt2x00_register_write(rt2x00pci, CSR18, reg);

	rt2x00_register_read(rt2x00pci, CSR19, &reg);
	value = SIFS + (2 * short_slot_time);
	rt2x00_set_field32(&reg, CSR19_DIFS, value);
	value = SIFS + get_duration(IEEE80211_HEADER + ACK_SIZE, 10);
	rt2x00_set_field32(&reg, CSR19_EIFS, value);
	rt2x00_register_write(rt2x00pci, CSR19, reg);

	rt2x00_register_read(rt2x00pci, TXCSR1, &reg);
	rt2x00_set_field32(&reg, TXCSR1_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&reg, TXCSR1_AUTORESPONDER, 1);
	rt2x00_register_write(rt2x00pci, TXCSR1, reg);
}

static void rt2400pci_config_rate(struct rt2x00_pci *rt2x00pci, const int rate)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
		pci_get_drvdata(rt2x00pci->pci_dev));
	u32 reg[4];
	u32 value;
	u32 preamble;

	preamble = DEVICE_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg[0] = DEVICE_RATE_FIELD(rate, RATEMASK);

	rt2x00_register_write(rt2x00pci, ARCSR1, cpu_to_le32(reg[0]));

	rt2x00_register_read(rt2x00pci, TXCSR1, &reg[0]);
	value = SIFS + PLCP
		+ (2 * (conf->short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME))
		+ preamble
		+ get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg[0], TXCSR1_ACK_TIMEOUT, value);
	value = SIFS + PLCP
		+ preamble
		+ get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg[0], TXCSR1_ACK_CONSUME_TIME, value);
	rt2x00_register_write(rt2x00pci, TXCSR1, reg[0]);

	value = DEVICE_RATE_FIELD(rate, PREAMBLE) ? 8 : 0;

	reg[0] = cpu_to_le32(0x00700400 | value);	/* ARCSR2 */
	reg[1] = cpu_to_le32(0x00380401 | value);	/* ARCSR3 */
	reg[2] = cpu_to_le32(0x00150402 | value);	/* ARCSR4 */
	reg[3] = cpu_to_le32(0x000b8403 | value);	/* ARCSR5 */

	rt2x00_register_multiwrite(rt2x00pci, ARCSR2, &reg[0], sizeof(reg));
}

static void rt2400pci_config_phymode(
	struct rt2x00_pci *rt2x00pci, const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00pci->rx_params.phymode == phymode)
		return;

	rate = &rt2x00pci->hw.modes[0].rates[
		rt2x00pci->hw.modes[0].num_rates - 1];

	rt2400pci_config_rate(rt2x00pci, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00pci->rx_params.phymode = phymode;
}

static void rt2400pci_config_mac_address(
	struct rt2x00_pci *rt2x00pci, void *addr)
{
	u32 reg[2] = {0, 0};

	rt2x00_set_field32(&reg[0], CSR3_BYTE0, ((u8*)addr)[0]);
	rt2x00_set_field32(&reg[0], CSR3_BYTE1, ((u8*)addr)[1]);
	rt2x00_set_field32(&reg[0], CSR3_BYTE2, ((u8*)addr)[2]);
	rt2x00_set_field32(&reg[0], CSR3_BYTE3, ((u8*)addr)[3]);
	rt2x00_set_field32(&reg[1], CSR4_BYTE4, ((u8*)addr)[4]);
	rt2x00_set_field32(&reg[1], CSR4_BYTE5, ((u8*)addr)[5]);

	rt2x00_register_multiwrite(rt2x00pci, CSR3, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt2400pci_link_tuner(struct rt2x00_pci *rt2x00pci)
{
	u8 reg;
	char false_cca_delta;

	/*
	 * Read false CCA counter.
	 */
	rt2x00_bbp_read(rt2x00pci, 39, &reg);

	/*
	 * Determine difference with previous CCA counter.
	 */
	false_cca_delta = reg - rt2x00pci->false_cca;
	rt2x00pci->false_cca = reg;

	/*
	 * Check if the difference is higher than the
	 * threshold and if so, tune the link.
	 */
	if (false_cca_delta >= 8) {
		/*
		 * Read and update RX AGC VGC.
		 */
		rt2x00_bbp_read(rt2x00pci, 13, &reg);
		reg += 2;
		if (reg < 0x20)
			rt2x00_bbp_write(rt2x00pci, 13, reg);
	}
}

/*
 * LED functions.
 */
static void rt2400pci_enable_led(struct rt2x00_pci *rt2x00pci)
{
	u32 reg;

	rt2x00_register_read(rt2x00pci, LEDCSR, &reg);

	rt2x00_set_field32(&reg, LEDCSR_ON_PERIOD, 30);
	rt2x00_set_field32(&reg, LEDCSR_OFF_PERIOD, 70);

	if (rt2x00pci->led_mode == LED_MODE_TXRX_ACTIVITY) {
		rt2x00_set_field32(&reg, LEDCSR_LINK, 1);
		rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 0);
	} else if (rt2x00pci->led_mode == LED_MODE_ASUS) {
		rt2x00_set_field32(&reg, LEDCSR_LINK, 0);
		rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 1);
	} else {
		rt2x00_set_field32(&reg, LEDCSR_LINK, 1);
		rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 1);
	}

	rt2x00_register_write(rt2x00pci, LEDCSR, reg);
}

static void rt2400pci_disable_led(struct rt2x00_pci *rt2x00pci)
{
	u32 reg;

	rt2x00_register_read(rt2x00pci, LEDCSR, &reg);
	rt2x00_set_field32(&reg, LEDCSR_LINK, 0);
	rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 0);
	rt2x00_register_write(rt2x00pci, LEDCSR, reg);
}

static void rt2400pci_activity_led(struct rt2x00_pci *rt2x00pci, char activity)
{
	u32 reg;

	if (rt2x00pci->led_mode != LED_MODE_TXRX_ACTIVITY)
		return;

	rt2x00_register_read(rt2x00pci, LEDCSR, &reg);
	rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, activity);
	rt2x00_register_write(rt2x00pci, LEDCSR, reg);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt2400pci_set_state(
	struct rt2x00_pci *rt2x00pci, enum dev_state state)
{
	u32 reg;
	unsigned int i;
	char put_to_sleep;
	char bbp_state;
	char rf_state;

	put_to_sleep = (state != STATE_AWAKE);

	rt2x00_register_read(rt2x00pci, PWRCSR1, &reg);
	rt2x00_set_field32(&reg, PWRCSR1_SET_STATE, 1);
	rt2x00_set_field32(&reg, PWRCSR1_BBP_DESIRE_STATE, state);
	rt2x00_set_field32(&reg, PWRCSR1_RF_DESIRE_STATE, state);
	rt2x00_set_field32(&reg, PWRCSR1_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00pci, PWRCSR1, reg);

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00pci, PWRCSR1, &reg);
		bbp_state = rt2x00_get_field32(reg, PWRCSR1_BBP_CURR_STATE);
		rf_state = rt2x00_get_field32(reg, PWRCSR1_RF_CURR_STATE);
		if (bbp_state == state && rf_state == state)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state: bbp %d and rf %d.\n",
		state, bbp_state, rf_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt2400pci_alloc_dma_ring(
	struct rt2x00_pci *rt2x00pci,
	struct data_ring *ring,
	void (*handler)(void *),
	const u16 max_entries,
	const u16 data_size,
	const u16 desc_size)
{
	unsigned int i;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kmalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = pci_alloc_consistent(rt2x00pci->pci_dev,
		rt2x00_get_ring_size(ring), &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].ring = ring;
		ring->entry[i].skb = NULL;
		ring->entry[i].priv = ring->data_addr
			+ (i * ring->desc_size);
		ring->entry[i].data_addr = ring->data_addr
			+ (ring->stats.limit * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (ring->stats.limit * ring->desc_size)
			+ (i * ring->data_size);
	}

	return 0;
}

static void rt2400pci_free_ring(
	struct rt2x00_pci *rt2x00pci, struct data_ring *ring)
{
	if (ring->data_addr)
		pci_free_consistent(rt2x00pci->pci_dev,
			rt2x00_get_ring_size(ring),
			ring->data_addr, ring->data_dma);
	ring->data_addr = NULL;

	kfree(ring->entry);
	ring->entry = NULL;
}

static int rt2400pci_allocate_dma_rings(struct rt2x00_pci *rt2x00pci)
{
	if (rt2400pci_alloc_dma_ring(
		rt2x00pci, &rt2x00pci->ring[RING_RX], rt2400pci_rxdone,
		RX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct rxd))
	|| rt2400pci_alloc_dma_ring(
		rt2x00pci, &rt2x00pci->ring[RING_TX], rt2400pci_txdone,
		 TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt2400pci_alloc_dma_ring(
		rt2x00pci, &rt2x00pci->ring[RING_ATIM], rt2400pci_txdone,
		ATIM_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt2400pci_alloc_dma_ring(
		rt2x00pci, &rt2x00pci->ring[RING_PRIO], rt2400pci_txdone,
		TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt2400pci_alloc_dma_ring(
		rt2x00pci, &rt2x00pci->ring[RING_BEACON], rt2400pci_beacondone,
		BEACON_ENTRIES, MGMT_FRAME_SIZE, sizeof(struct txd))) {
		return -ENOMEM;
	}

	return 0;
}

static void rt2400pci_free_rings(struct rt2x00_pci *rt2x00pci)
{
	rt2400pci_free_ring(rt2x00pci, &rt2x00pci->ring[RING_RX]);
	rt2400pci_free_ring(rt2x00pci, &rt2x00pci->ring[RING_TX]);
	rt2400pci_free_ring(rt2x00pci, &rt2x00pci->ring[RING_ATIM]);
	rt2400pci_free_ring(rt2x00pci, &rt2x00pci->ring[RING_PRIO]);
	rt2400pci_free_ring(rt2x00pci, &rt2x00pci->ring[RING_BEACON]);
}

static void rt2400pci_init_rxring(
	struct rt2x00_pci *rt2x00pci, unsigned short type)
{
	struct data_ring *ring = &rt2x00pci->ring[type];
	struct rxd *rxd;
	unsigned int i;

	memset(ring->data_addr, 0x00, rt2x00_get_ring_size(ring));

	ring->type = type;

	for (i = 0; i < ring->stats.limit; i++) {
		rxd = rt2x00pci_desc_addr(&ring->entry[i]);

		rt2x00_set_field32(&rxd->word2, RXD_W2_BUFFER_LENGTH,
			ring->data_size);
		rt2x00_set_field32(&rxd->word1, RXD_W1_BUFFER_ADDRESS,
			ring->entry[i].data_dma);
		rt2x00_set_field32(&rxd->word0, RXD_W0_OWNER_NIC, 1);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt2400pci_init_txring(
	struct rt2x00_pci *rt2x00pci, unsigned short type)
{
	struct data_ring *ring = &rt2x00pci->ring[type];
	struct txd *txd;
	unsigned int i;

	memset(ring->data_addr, 0x00, rt2x00_get_ring_size(ring));

	ring->type = type;

	for (i = 0; i < ring->stats.limit; i++) {
		txd = rt2x00pci_desc_addr(&ring->entry[i]);

		rt2x00_set_field32(&txd->word2, TXD_W2_BUFFER_LENGTH,
			ring->data_size);
		rt2x00_set_field32(&txd->word1, TXD_W1_BUFFER_ADDRESS,
			ring->entry[i].data_dma);
		rt2x00_set_field32(&txd->word0, TXD_W0_VALID, 0);
		rt2x00_set_field32(&txd->word0, TXD_W0_OWNER_NIC, 0);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt2400pci_init_rings(struct rt2x00_pci *rt2x00pci)
{
	u32 reg;

	/*
	 * Initialize rings.
	 */
	rt2400pci_init_rxring(rt2x00pci, RING_RX);
	rt2400pci_init_txring(rt2x00pci, RING_TX);
	rt2400pci_init_txring(rt2x00pci, RING_ATIM);
	rt2400pci_init_txring(rt2x00pci, RING_PRIO);
	rt2400pci_init_txring(rt2x00pci, RING_BEACON);

	/*
	 * Initialize registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, TXCSR2_TXD_SIZE,
		rt2x00pci->ring[RING_TX].desc_size);
	rt2x00_set_field32(&reg, TXCSR2_NUM_TXD,
		rt2x00pci->ring[RING_TX].stats.limit);
	rt2x00_set_field32(&reg, TXCSR2_NUM_ATIM,
		rt2x00pci->ring[RING_ATIM].stats.limit);
	rt2x00_set_field32(&reg, TXCSR2_NUM_PRIO,
		rt2x00pci->ring[RING_PRIO].stats.limit);
	rt2x00_register_write(rt2x00pci, TXCSR2, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR3_TX_RING_REGISTER,
		rt2x00pci->ring[RING_TX].data_dma);
	rt2x00_register_write(rt2x00pci, TXCSR3, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR5_PRIO_RING_REGISTER,
		rt2x00pci->ring[RING_PRIO].data_dma);
	rt2x00_register_write(rt2x00pci, TXCSR5, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR4_ATIM_RING_REGISTER,
		rt2x00pci->ring[RING_ATIM].data_dma);
	rt2x00_register_write(rt2x00pci, TXCSR4, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR6_BEACON_RING_REGISTER,
		rt2x00pci->ring[RING_BEACON].data_dma);
	rt2x00_register_write(rt2x00pci, TXCSR6, reg);

	reg = 0;
	rt2x00_set_field32(&reg, RXCSR1_RXD_SIZE,
		rt2x00pci->ring[RING_RX].desc_size);
	rt2x00_set_field32(&reg, RXCSR1_NUM_RXD,
		rt2x00pci->ring[RING_RX].stats.limit);
	rt2x00_register_write(rt2x00pci, RXCSR1, reg);

	reg = 0;
	rt2x00_set_field32(&reg, RXCSR2_RX_RING_REGISTER,
		rt2x00pci->ring[RING_RX].data_dma);
	rt2x00_register_write(rt2x00pci, RXCSR2, reg);

	return 0;
}

static int rt2400pci_init_registers(struct rt2x00_pci *rt2x00pci)
{
	u32 reg;

	if (rt2400pci_set_state(rt2x00pci, STATE_AWAKE))
		return -EBUSY;

	rt2x00_register_write(rt2x00pci, PWRCSR0, cpu_to_le32(0x3f3b3100));

	rt2x00_register_write(rt2x00pci, PSCSR0, cpu_to_le32(0x00020002));
	rt2x00_register_write(rt2x00pci, PSCSR1, cpu_to_le32(0x00000002));
	rt2x00_register_write(rt2x00pci, PSCSR2, cpu_to_le32(0x00020002));
	rt2x00_register_write(rt2x00pci, PSCSR3, cpu_to_le32(0x00000002));

	rt2x00_register_read(rt2x00pci, TIMECSR, &reg);
	rt2x00_set_field32(&reg, TIMECSR_US_COUNT, 33);
	rt2x00_set_field32(&reg, TIMECSR_US_64_COUNT, 63);
	rt2x00_set_field32(&reg, TIMECSR_BEACON_EXPECT, 0);
	rt2x00_register_write(rt2x00pci, TIMECSR, reg);

	rt2x00_register_read(rt2x00pci, CSR9, &reg);
	rt2x00_set_field32(&reg, CSR9_MAX_FRAME_UNIT,
		(rt2x00pci->ring[RING_RX].data_size / 128));
	rt2x00_register_write(rt2x00pci, CSR9, reg);

	rt2x00_register_write(rt2x00pci, CNT3, cpu_to_le32(0x3f080000));

	rt2x00_register_read(rt2x00pci, RXCSR0, &reg);
	rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00pci, RXCSR0, reg);

	rt2x00_register_write(rt2x00pci, MACCSR0, cpu_to_le32(0x00217223));

	rt2x00_register_read(rt2x00pci, MACCSR1, &reg);
	rt2x00_set_field32(&reg, MACCSR1_AUTO_TXBBP, 1);
	rt2x00_set_field32(&reg, MACCSR1_AUTO_RXBBP, 1);
	rt2x00_register_write(rt2x00pci, MACCSR1, reg);

	rt2x00_register_read(rt2x00pci, MACCSR2, &reg);
	rt2x00_set_field32(&reg, MACCSR2_DELAY, 64);
	rt2x00_register_write(rt2x00pci, MACCSR2, reg);

	rt2x00_register_read(rt2x00pci, RXCSR3, &reg);
	/*
	 * Tx power.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID0, 3);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID0_VALID, 1);
	/*
	 * Signal.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID1, 32);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID1_VALID, 1);
	/*
	 * Rssi.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID2, 36);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID2_VALID, 1);
	rt2x00_register_write(rt2x00pci, RXCSR3, reg);

	rt2x00_register_read(rt2x00pci, RALINKCSR, &reg);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_DATA0, 17);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_ID0, 154);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_DATA1, 0);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_ID1, 154);
	rt2x00_register_write(rt2x00pci, RALINKCSR, reg);

	rt2x00_register_read(rt2x00pci, CSR1, &reg);
	rt2x00_set_field32(&reg, CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, CSR1_BBP_RESET, 0);
	rt2x00_set_field32(&reg, CSR1_HOST_READY, 0);
	rt2x00_register_write(rt2x00pci, CSR1, reg);

	rt2x00_register_read(rt2x00pci, CSR1, &reg);
	rt2x00_set_field32(&reg, CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00pci, CSR1, reg);

	/*
	 * We must clear the FCS and FIFO error count.
	 * These registers are cleared on read,
	 * so we may pass a useless variable to store the value.
	 */
	rt2x00_register_read(rt2x00pci, CNT0, &reg);
	rt2x00_register_read(rt2x00pci, CNT4, &reg);

	return 0;
}

static int rt2400pci_init_bbp(struct rt2x00_pci *rt2x00pci)
{
	u8 reg_id;
	u8 value;
	unsigned int	i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00pci, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00pci, 1, 0x00);
	rt2x00_bbp_write(rt2x00pci, 3, 0x27);
	rt2x00_bbp_write(rt2x00pci, 4, 0x08);
	rt2x00_bbp_write(rt2x00pci, 10, 0x0f);
	rt2x00_bbp_write(rt2x00pci, 13, 0x08);
	rt2x00_bbp_write(rt2x00pci, 15, 0x72);
	rt2x00_bbp_write(rt2x00pci, 16, 0x74);
	rt2x00_bbp_write(rt2x00pci, 17, 0x20);
	rt2x00_bbp_write(rt2x00pci, 18, 0x72);
	rt2x00_bbp_write(rt2x00pci, 19, 0x0b);
	rt2x00_bbp_write(rt2x00pci, 20, 0x00);
	rt2x00_bbp_write(rt2x00pci, 28, 0x11);
	rt2x00_bbp_write(rt2x00pci, 29, 0x04);
	rt2x00_bbp_write(rt2x00pci, 30, 0x21);
	rt2x00_bbp_write(rt2x00pci, 31, 0x00);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		if (rt2x00pci->eeprom[i] != 0xffff
		&& rt2x00pci->eeprom[i] != 0x0000) {
			reg_id = rt2x00_get_field16(
				rt2x00pci->eeprom[i], EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(
				rt2x00pci->eeprom[i], EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00pci, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt2400pci_init_channel_time(struct rt2x00_pci *rt2x00pci)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt2400pci_config_channel(rt2x00pci,
		rt2x00pci->hw.modes[0].channels[0].val,
		rt2x00pci->hw.modes[0].channels[0].chan,
		rt2x00pci->hw.modes[0].channels[0].freq);
	jiffies_end = jiffies;

	rt2x00pci->hw.channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00pci->hw.channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt2400pci_initialize(struct rt2x00_pci *rt2x00pci)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00pci->pci_dev);

	if (GET_FLAG(rt2x00pci, DEVICE_INITIALIZED))
		return 0;

	/*
	 * Allocate all data rings.
	 */
	if (rt2400pci_allocate_dma_rings(rt2x00pci)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Initialize all registers.
	 */
	if (rt2400pci_init_rings(rt2x00pci)
	|| rt2400pci_init_registers(rt2x00pci)
	|| rt2400pci_init_bbp(rt2x00pci)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt2400pci_init_channel_time(rt2x00pci))
		goto exit_fail;

	/*
	 * Register interrupt handler.
	 */
	if (request_irq(rt2x00pci->pci_dev->irq, rt2400pci_interrupt,
		SA_SHIRQ, net_dev->name, rt2x00pci)) {
		ERROR("IRQ %d allocation failed.\n", rt2x00pci->pci_dev->irq);
		goto exit_fail;
	}

	SET_FLAG(rt2x00pci, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt2400pci_free_rings(rt2x00pci);

	return -EIO;
}

static void rt2400pci_uninitialize(struct rt2x00_pci *rt2x00pci)
{
	if (!GET_FLAG(rt2x00pci, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00pci->scan) {
		rt2x00pci->scan->status = SCANNING_CANCELLED;
		complete_all(&rt2x00pci->scan->completion);
	}

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00pci->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt2400pci_free_rings(rt2x00pci);

	/*
	 * Free irq line.
	 */
	free_irq(rt2x00pci->pci_dev->irq, rt2x00pci);

	CLEAR_FLAG(rt2x00pci, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt2400pci_enable_radio(struct rt2x00_pci *rt2x00pci)
{
	u32 reg;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO))
		return 0;

	/*
	 * Check if the hardware has been initialized,
	 * if not then do it now.
	 */
	if (!GET_FLAG(rt2x00pci, DEVICE_INITIALIZED))
		if (rt2400pci_initialize(rt2x00pci))
			return -EIO;

	rt2x00_register_write(rt2x00pci, PWRCSR0, cpu_to_le32(0x3f3b3100));

	/*
	 * Clear interrupts.
	 */
	rt2x00_register_read(rt2x00pci, CSR7, &reg);
	rt2x00_register_write(rt2x00pci, CSR7, reg);

	/*
	 * Enable interrupts.
	 */
	rt2x00_register_read(rt2x00pci, CSR8, &reg);
	rt2x00_set_field32(&reg, CSR8_TBCN_EXPIRE, 0);
	rt2x00_set_field32(&reg, CSR8_TXDONE_TXRING, 0);
	rt2x00_set_field32(&reg, CSR8_TXDONE_ATIMRING, 0);
	rt2x00_set_field32(&reg, CSR8_TXDONE_PRIORING, 0);
	rt2x00_set_field32(&reg, CSR8_RXDONE, 0);
	rt2x00_register_write(rt2x00pci, CSR8, reg);

	/*
	 * Enable RX.
	 */
	rt2x00_register_read(rt2x00pci, RXCSR0, &reg);
	rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00pci, RXCSR0, reg);

	/*
	 * Enable LED
	 */
	rt2400pci_enable_led(rt2x00pci);

	SET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO);

	return 0;
}

static void rt2400pci_disable_radio(struct rt2x00_pci *rt2x00pci)
{
	u32 reg;

	if (!GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO))
		return;

	rt2x00_register_write(rt2x00pci, PWRCSR0, 0);

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00pci, TXCSR0, &reg);
	rt2x00_set_field32(&reg, TXCSR0_ABORT, 1);
	rt2x00_register_write(rt2x00pci, TXCSR0, reg);

	rt2x00_register_read(rt2x00pci, RXCSR0, &reg);
	rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00pci, RXCSR0, reg);

	/*
	 * Disable interrupts.
	 */
	rt2x00_register_read(rt2x00pci, CSR8, &reg);
	rt2x00_set_field32(&reg, CSR8_TBCN_EXPIRE, 1);
	rt2x00_set_field32(&reg, CSR8_TXDONE_TXRING, 1);
	rt2x00_set_field32(&reg, CSR8_TXDONE_ATIMRING, 1);
	rt2x00_set_field32(&reg, CSR8_TXDONE_PRIORING, 1);
	rt2x00_set_field32(&reg, CSR8_RXDONE, 1);
	rt2x00_register_write(rt2x00pci, CSR8, reg);

	/*
	 * Disable LED
	 */
	rt2400pci_disable_led(rt2x00pci);

	CLEAR_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO);
}

/*
 * TX descriptor initialization
 */
static void rt2400pci_write_tx_desc(
	struct rt2x00_pci *rt2x00pci, struct txd *txd,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	u32 length;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 signal;
	u16 service;

	rt2x00_set_field32(&txd->word0, TXD_W0_VALID, 1);
	rt2x00_set_field32(&txd->word0, TXD_W0_ACK, !control->no_ack);

	/*
	 * TODO: IFS can be various values, where can we find
	 * which one we want to use?
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_IFS, 0);

	if (control->queue == IEEE80211_TX_QUEUE_BEACON)
		rt2x00_set_field32(&txd->word0, TXD_W0_TIMESTAMP, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_TIMESTAMP, 0);

	/*
	 * TODO: How can we determine if we want long retry or short retry?
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_RETRY_MODE, 0);

	if (ieee80211_get_morefrag(ieee80211hdr))
		rt2x00_set_field32(&txd->word0, TXD_W0_MORE_FRAG, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_MORE_FRAG, 0);

	/*
	 * TODO: Does this field mean device will send RTS, or that this
	 * frame is an RTS frame?
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_RTS, control->use_rts_cts);

	rt2x00_set_field32(&txd->word2, TXD_W2_DATABYTE_COUNT, skb->len);

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * Convert length to microseconds.
	 */
	residual = get_duration_res(length,
		DEVICE_RATE_FIELD(control->tx_rate, RATE));
	length = get_duration(length,
		DEVICE_RATE_FIELD(control->tx_rate, RATE));

	if (residual)
		length++;

	/*
	 * PLCP_SIGNAL, PLCP_SERVICE, PLCP_LENGTH_LOW and PLCP_LENGTH_HIGH
	 * are BBP registers. This means that we have to treat this value
	 * according to the BBP register rules.
	 * We have to set the busy (0x8000), and the register number (0x0f00).
	 * The value we want to write is stored in 0x00ff.
	 * For PLCP_SIGNAL we can optionally enable SHORT_PREAMBLE.
	 * For PLCP_SERVICE we can set the length extension bit according to
	 * 802.11b standard 18.2.3.5.
	 */
	length_high = 0x8000 | 0x0700 | (length >> 8);
	length_low = 0x8000 | 0x0800 | (length & 0xff);

	signal = 0x8500 | DEVICE_RATE_FIELD(control->tx_rate, PLCP);
	if (DEVICE_RATE_FIELD(control->tx_rate, PREAMBLE))
		signal |= 0x0008;

	service = 0x0600 | 0x0004;
	if (residual <= (8 % 11))
		service |= 0x0080;

	rt2x00_set_field32(&txd->word3, TXD_W3_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&txd->word3, TXD_W3_PLCP_SERVICE, service);
	rt2x00_set_field32(&txd->word4, TXD_W4_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&txd->word4, TXD_W4_PLCP_LENGTH_HIGH, length_high);

	/*
	 * Set this last, after this the device can start transmitting the packet.
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_OWNER_NIC, 1);
}

/*
 * Interrupt functions.
 */
static void rt2400pci_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(ring->net_dev);
	struct sk_buff *skb;
	struct ieee80211_tx_control beacon;

	memset(&beacon, 0x00, sizeof(beacon));

	skb = ieee80211_beacon_get(ring->net_dev,
		rt2x00pci->interface_id, &beacon);
	if (!skb)
		return;

	rt2400pci_beacon_update(ring->net_dev, skb, &beacon);

	dev_kfree_skb_any(skb);
}

static void
rt2400pci_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct sk_buff *skb;
	struct rxd *rxd;
	u16 size;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00pci_desc_addr(entry);

		if (rt2x00_get_field32(rxd->word0, RXD_W0_OWNER_NIC))
			break;

		size = rt2x00_get_field32(rxd->word0, RXD_W0_DATABYTE_COUNT);

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(rxd->word0, RXD_W0_CRC)
		&& !rt2x00_get_field32(rxd->word0, RXD_W0_PHYSICAL_ERROR)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				break;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size),
				rt2x00pci_data_addr(entry), size);

			rt2x00pci->rx_params.ssi =
				rt2x00_get_field32(rxd->word2, RXD_W2_RSSI);

			__ieee80211_rx(ring->net_dev,
				skb, &rt2x00pci->rx_params);
		}

		rt2x00_set_field32(&rxd->word0, RXD_W0_OWNER_NIC, 1);

		rt2x00_ring_index_inc(ring);
	}

	/*
	 * Tune link for optimal performance.
	 */
	rt2400pci_link_tuner(rt2x00pci);

	/*
	 * Update LED.
	 */
	rt2400pci_activity_led(rt2x00pci, 0);
}

static void
rt2400pci_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct txd *txd;
	int tx_status;
	int ack;

	while (!rt2x00_ring_empty(ring)) {
		entry = rt2x00_get_data_entry_done(ring);
		txd = rt2x00pci_desc_addr(entry);

		if (rt2x00_get_field32(txd->word0, TXD_W0_OWNER_NIC)
		|| !rt2x00_get_field32(txd->word0, TXD_W0_VALID))
			break;

		ack = rt2x00_get_field32(txd->word0, TXD_W0_ACK);

		/*
		 * TODO: How can te below field be set correctly?
		 */
		entry->tx_status.tx_filtered = 0;

		entry->tx_status.queue_length = ring->stats.limit;

		/*
		 * The TXD_W0_RESULT field will only be set when
		 * we had requested an ACK. So we have received an
		 * ACK response when ACK was requested and status
		 * was succesfull.
		 */
		tx_status = rt2x00_get_field32(txd->word0, TXD_W0_RESULT);
		entry->tx_status.ack = 0;
		if (ack && (tx_status == TX_SUCCESS
		|| tx_status == TX_SUCCESS_RETRY))
			entry->tx_status.ack = 1;
		else if (ack && tx_status == TX_FAIL_RETRY) {
			rt2x00pci->low_level_stats.dot11ACKFailureCount++;
			entry->tx_status.excessive_retries++;
		}

		rt2x00_bbp_read(rt2x00pci, 32,
			(u8*)&entry->tx_status.ack_signal);

		entry->tx_status.retry_count = rt2x00_get_field32(
			txd->word0, TXD_W0_RETRY_COUNT);

		ieee80211_tx_status(ring->net_dev,
			entry->skb, &entry->tx_status);

		rt2x00_set_field32(&txd->word0, TXD_W0_VALID, 0);
		entry->skb = NULL;

		rt2x00_ring_index_done_inc(ring);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00pci->scan
	&& rt2x00_ring_empty(&rt2x00pci->ring[RING_TX])
	&& rt2x00_ring_empty(&rt2x00pci->ring[RING_ATIM])
	&& rt2x00_ring_empty(&rt2x00pci->ring[RING_PRIO])) {
		rt2x00pci->scan->status = SCANNING_READY;
		complete(&rt2x00pci->scan->completion);
	}
}

static irqreturn_t rt2400pci_interrupt(
	int irq, void *dev_instance, struct pt_regs *regs)
{
	struct rt2x00_pci *rt2x00pci = dev_instance;
	u32 reg;

	/*
	 * Get the interrupt sources & saved to local variable.
	 * Write register value back to clear pending interrupts.
	 */
	rt2x00_register_read(rt2x00pci, CSR7, &reg);
	rt2x00_register_write(rt2x00pci, CSR7, reg);

	if (!reg)
		return IRQ_NONE;

	if (!GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO))
		return IRQ_HANDLED;

	/*
	 * Handle interrupts, walk through all bits
	 * and run the tasks, the bits are checked in order of
	 * priority.
	 */

	/*
	 * 1 - Beacon timer expired interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TBCN_EXPIRE))
		queue_work(rt2x00pci->workqueue,
			&rt2x00pci->ring[RING_BEACON].irq_work);

	/*
	 * 2 - Rx ring done interrupt.
	 * Enable the TXRX activity led.
	 */
	if (rt2x00_get_field32(reg, CSR7_RXDONE)) {
		queue_work(rt2x00pci->workqueue,
			&rt2x00pci->ring[RING_RX].irq_work);
		rt2400pci_activity_led(rt2x00pci, 1);
	}

	/*
	 * 3 - Atim ring transmit done interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TXDONE_ATIMRING))
		queue_work(rt2x00pci->workqueue,
			&rt2x00pci->ring[RING_ATIM].irq_work);

	/*
	 * 4 - Priority ring transmit done interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TXDONE_PRIORING))
		queue_work(rt2x00pci->workqueue,
			&rt2x00pci->ring[RING_PRIO].irq_work);

	/*
	 * 5 - Tx ring transmit done interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TXDONE_TXRING))
		queue_work(rt2x00pci->workqueue,
			&rt2x00pci->ring[RING_TX].irq_work);

	return IRQ_HANDLED;
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt2400pci_tx(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	struct data_ring *ring;
	struct data_entry *entry;
	struct txd *txd;
	u32 reg;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00pci_get_ring(rt2x00pci, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		return NET_XMIT_DROP;
	}

	if (rt2x00_ring_full(ring))
		return NET_XMIT_DROP;

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00pci_desc_addr(entry);

	if (rt2x00_get_field32(txd->word0, TXD_W0_OWNER_NIC)
	|| rt2x00_get_field32(txd->word0, TXD_W0_VALID))
		return NET_XMIT_DROP;

	/*
	 * Set the software sequence number.
	 */
	rt2x00_set_sequence(skb, &rt2x00pci->seq_ctrl);

	memcpy(rt2x00pci_data_addr(entry), skb->data, skb->len);
	rt2400pci_write_tx_desc(rt2x00pci, txd, skb, control);
	entry->skb = skb;

	rt2x00_ring_index_inc(ring);

	rt2x00_register_read(rt2x00pci, TXCSR0, &reg);
	if (control->queue == IEEE80211_TX_QUEUE_DATA0)
		rt2x00_set_field32(&reg, TXCSR0_KICK_PRIO, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_DATA1)
		rt2x00_set_field32(&reg, TXCSR0_KICK_TX, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_AFTER_BEACON)
		rt2x00_set_field32(&reg, TXCSR0_KICK_ATIM, 1);
	rt2x00_register_write(rt2x00pci, TXCSR0, reg);

	return 0;
}

static int rt2400pci_reset(struct net_device *net_dev)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	rt2400pci_disable_radio(rt2x00pci);

	rt2400pci_init_rings(rt2x00pci);

	return rt2400pci_enable_radio(rt2x00pci);
}

static int rt2400pci_add_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * We support muliple monitor mode interfaces.
	 * All we need to do is keep increase the
	 * monitor_counter.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		rt2x00pci->monitor_count++;
		goto exit;
	}

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (GET_FLAG(rt2x00pci, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	rt2x00pci->seq_ctrl = 0;
	rt2x00pci->interface_id = conf->if_id;

	rt2400pci_config_type(rt2x00pci, conf->type);
	rt2400pci_config_mac_address(rt2x00pci, conf->mac_addr);

	SET_FLAG(rt2x00pci, INTERFACE_INITIALIZED);

exit:
	/*
	 * Enable radio when this is the first
	 * interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO))
		return rt2400pci_enable_radio(rt2x00pci);
	return 0;
}

static void rt2400pci_remove_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * We support muliple monitor mode interfaces.
	 * Decrease the counter to remove one monitor
	 * interface.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		rt2x00pci->monitor_count--;
		goto exit;
	}

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (!GET_FLAG(rt2x00pci, INTERFACE_INITIALIZED))
		return;

	rt2x00pci->interface_id = 0;
	rt2x00pci->interface_type = -EINVAL;

	CLEAR_FLAG(rt2x00pci, INTERFACE_INITIALIZED);

exit:
	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!rt2x00pci->monitor_count)
		rt2400pci_disable_radio(rt2x00pci);
}

static void rt2400pci_config_update(void *data)
{
	struct rt2x00_pci *rt2x00pci = data;
	struct net_device *net_dev = pci_get_drvdata(rt2x00pci->pci_dev);
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(net_dev);
	u32 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt2400pci_disable_radio(rt2x00pci);
		else {
			rt2x00_register_read(rt2x00pci, RXCSR0, &reg);
			rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00pci, RXCSR0, reg);
		}
	}

	rt2400pci_config_channel(rt2x00pci,
		conf->channel_val, conf->channel, conf->freq);
	rt2400pci_config_txpower(rt2x00pci, conf->power_level);
	rt2400pci_config_antenna(rt2x00pci, conf->antenna_sel);
	rt2400pci_config_duration(rt2x00pci, conf->short_slot_time);
	rt2400pci_config_phymode(rt2x00pci, conf->phymode);

	/*
	 * Reenable RX only if the radio should is on.
	 */
	if (GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00pci, RXCSR0, &reg);
		rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00pci, RXCSR0, reg);
	} else if (conf->radio_enabled) {
		if (rt2400pci_enable_radio(rt2x00pci))
			return;
	}
}

static int rt2400pci_config(struct net_device *net_dev,
	struct ieee80211_conf *conf)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * Queue work.
	 */
	return !queue_work(rt2x00pci->workqueue, &rt2x00pci->config_work);
}

static int rt2400pci_config_interface(struct net_device *net_dev, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * When configuring monitor mode, we are done now.
	 * but if we are configuring another mode it must be
	 * equal to the interface that has been added.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00pci->interface_type)
		return -EINVAL;

	if (conf->bssid)
		rt2400pci_config_bssid(rt2x00pci, conf->bssid);

	return 0;
}

static void rt2400pci_set_multicast_list(struct net_device *net_dev,
	unsigned short flags, int mc_count)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (rt2x00pci->monitor_count)
		return;

	if (GET_FLAG(rt2x00pci, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC))
			rt2400pci_config_promisc(rt2x00pci, 0);
	} else {
		if (flags & IFF_PROMISC)
			rt2400pci_config_promisc(rt2x00pci, 1);
	}
}

static void rt2400pci_scan(void *data)
{
	struct rt2x00_pci *rt2x00pci = data;

	if (unlikely(!rt2x00pci->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00pci->scan->status != SCANNING_READY)
		wait_for_completion(&rt2x00pci->scan->completion);

	/*
	 * Check if this scan has been cancelled while
	 * work was still scheduled.
	 */
	if (rt2x00pci->scan->status == SCANNING_CANCELLED)
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00pci->scan->state == IEEE80211_SCAN_START) {
		rt2400pci_config_phymode(rt2x00pci,
			rt2x00pci->scan->conf.scan_phymode);

		rt2400pci_config_channel(rt2x00pci,
			rt2x00pci->scan->conf.scan_channel_val,
			rt2x00pci->scan->conf.scan_channel,
			rt2x00pci->scan->conf.scan_freq);

		rt2400pci_config_txpower(rt2x00pci,
			rt2x00pci->scan->conf.scan_power_level);
	} else {
		rt2400pci_config_phymode(rt2x00pci,
			rt2x00pci->scan->conf.running_phymode);

		rt2400pci_config_channel(rt2x00pci,
			rt2x00pci->scan->conf.running_channel_val,
			rt2x00pci->scan->conf.running_channel,
			rt2x00pci->scan->conf.running_freq);

		rt2400pci_config_txpower(rt2x00pci,
			rt2x00pci->scan->conf.running_power_level);
	}

exit:
	kfree(rt2x00pci->scan);
	rt2x00pci->scan = NULL;
}

static int rt2400pci_passive_scan(struct net_device *net_dev,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00pci->scan)
		return -EBUSY;

	/*
	 * Check if we need to disable the radio
	 */
	if (!GET_FLAG(rt2x00pci, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00pci->scan = kmalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00pci->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt2400pci_tx(net_dev, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	init_completion(&rt2x00pci->scan->completion);

	memcpy(&rt2x00pci->scan->conf, conf, sizeof(*conf));

	rt2x00pci->scan->state = state;

	rt2x00pci->scan->status = 0;

	/*
	 * Queue work.
	 */
	if (!queue_work(rt2x00pci->workqueue, &rt2x00pci->scan_work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00pci->scan);
	rt2x00pci->scan = NULL;

	return -EIO;
}

static int rt2400pci_get_stats(struct net_device *net_dev,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	u32 reg;

	/*
	 * Update FCS error count from register.
	 * The dot11ACKFailureCount is updated in interrupt time.
	 * TODO: dot11RTSFailureCount and dot11RTSSuccessCount
	 * are never updated, we need to find a method to see
	 * where we can update those statistics from.
	 */
	rt2x00_register_read(rt2x00pci, CNT0, &reg);
	rt2x00pci->low_level_stats.dot11FCSErrorCount +=
		rt2x00_get_field32(reg, CNT0_FCS_ERROR);

	memcpy(stats, &rt2x00pci->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt2400pci_set_retry_limit(struct net_device *net_dev,
	u32 short_retry, u32 long_retry)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	u32 reg;

	rt2x00_register_read(rt2x00pci, CSR11, &reg);
	rt2x00_set_field32(&reg, CSR11_LONG_RETRY, long_retry);
	rt2x00_set_field32(&reg, CSR11_SHORT_RETRY, short_retry);
	rt2x00_register_write(rt2x00pci, CSR11, reg);

	return 0;
}

static int rt2400pci_conf_tx(struct net_device *net_dev,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	struct data_ring *ring = &rt2x00pci->ring[RING_TX];

	/*
	 * We don't support variating cw_min and cw_max variables
	 * per queue. So by default we only configure the TX queue,
	 * and ignore all other configurations.
	 */
	if (queue != IEEE80211_TX_QUEUE_DATA0) {
		NOTICE("Ignoring configuration for queue %d.\n", queue);
		return -EINVAL;
	}

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT2400 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 5; /* cw_min: 2^5 = 32. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	/*
	 * Write configuration to register.
	 */
	rt2400pci_config_cw(rt2x00pci, &ring->tx_params);

	return 0;
}

static int rt2400pci_get_tx_stats(struct net_device *net_dev,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00pci->ring[RING_PRIO].stats,
		sizeof(rt2x00pci->ring[RING_PRIO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00pci->ring[RING_TX].stats,
		sizeof(rt2x00pci->ring[RING_TX].stats));

	return 0;
}

static u64 rt2400pci_get_tsf(struct net_device *net_dev)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	u64 tsf;
	u32 reg;

	rt2x00_register_read(rt2x00pci, CSR17, &reg);
	tsf = (u64)rt2x00_get_field32(reg, CSR17_HIGH_TSFTIMER) << 32;
	rt2x00_register_read(rt2x00pci, CSR16, &reg);
	tsf |= rt2x00_get_field32(reg, CSR16_LOW_TSFTIMER);

	return tsf;
}

static void rt2400pci_reset_tsf(struct net_device *net_dev)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	u32 reg = 0;

	rt2x00_register_write(rt2x00pci, CSR16, reg);
	rt2x00_register_write(rt2x00pci, CSR17, reg);
}

static int rt2400pci_beacon_update(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00pci->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	/*
	 * Set the software sequence number.
	 */
	rt2x00_set_sequence(skb, &rt2x00pci->seq_ctrl);

	memcpy(rt2x00pci_data_addr(entry), skb->data, skb->len);
	rt2400pci_write_tx_desc(rt2x00pci,
		rt2x00pci_desc_addr(entry), skb, control);

	return 0;
}

static int rt2400pci_tx_last_beacon(struct net_device *net_dev)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);
	u32 reg;

	rt2x00_register_read(rt2x00pci, CSR15, &reg);
	return rt2x00_get_field32(reg, CSR15_BEACON_SENT);
}

/*
 * Device initialization functions.
 */
static int rt2400pci_alloc_rings(struct rt2x00_pci *rt2x00pci)
{
	unsigned int i;

	rt2x00pci->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00pci->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	SET_FLAG(rt2x00pci, DEVICE_SUPPORT_ATIM);

	for (i = 0; i < RING_NUM; i++) {
		/*
		 *Set device structure.
		 */
		rt2x00pci->ring[i].net_dev =
			pci_get_drvdata(rt2x00pci->pci_dev);
	
		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00pci->ring[i].tx_params.cw_min = 5;
		rt2x00pci->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt2400pci_init_eeprom(struct rt2x00_pci *rt2x00pci)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
			pci_get_drvdata(rt2x00pci->pci_dev));
	u32 reg;
	u16 value;
	u16 eeprom;

	/*
	 * 1 - Detect EEPROM width.
	 */
	rt2x00_register_read(rt2x00pci, CSR21, &reg);
	if (rt2x00_get_field32(reg, CSR21_TYPE_93C46))
		rt2x00pci->eeprom_width = EEPROM_WIDTH_93C46;
	else
		rt2x00pci->eeprom_width = EEPROM_WIDTH_93C66;

	/*
	 * 2 - Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00pci, EEPROM_ANTENNA, &eeprom);

	/*
	 * 3 - Identify RF chipset.
	 */
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_set_chip(&rt2x00pci->chip, RT2460, value);

	if (!rt2x00_rf(&rt2x00pci->chip, RF2420)
	&& !rt2x00_rf(&rt2x00pci->chip, RF2421))
		return -ENODEV;

	/*
	 * 4 - Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	conf->antenna_sel = rt2x00_get_field16(eeprom,
				EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * 5 - Store led mode, for correct led behaviour.
	 */
	rt2x00pci->led_mode = rt2x00_get_field16(eeprom,
				EEPROM_ANTENNA_LED_MODE);

	/*
	 * 6 - Detect if this device has an hardware controlled radio.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_HARDWARE_RADIO))
		SET_FLAG(rt2x00pci, DEVICE_SUPPORT_HW_BUTTON);

	/*
	 * 7 - Read BBP data from EEPROM and store in private structure.
	 */
	rt2x00pci->eeprom = kzalloc(EEPROM_BBP_SIZE * sizeof(u16), GFP_KERNEL);
	if (!rt2x00pci->eeprom) 
		return -ENOMEM;

	rt2x00_eeprom_multiread(rt2x00pci, EEPROM_BBP_START,
		rt2x00pci->eeprom, EEPROM_BBP_SIZE * sizeof(u16));

	return 0;
}

static int rt2400pci_init_mac(struct rt2x00_pci *rt2x00pci)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00pci->pci_dev);
	u32 reg[2] = {0, 0};

	if (GET_FLAG(rt2x00pci, DEVICE_INITIALIZED_MAC))
		return 0;

	rt2x00_register_multiread(rt2x00pci, CSR3, &reg[0], sizeof(reg));

	net_dev->dev_addr[0] = rt2x00_get_field32(reg[0], CSR3_BYTE0);
	net_dev->dev_addr[1] = rt2x00_get_field32(reg[0], CSR3_BYTE1);
	net_dev->dev_addr[2] = rt2x00_get_field32(reg[0], CSR3_BYTE2);
	net_dev->dev_addr[3] = rt2x00_get_field32(reg[0], CSR3_BYTE3);
	net_dev->dev_addr[4] = rt2x00_get_field32(reg[1], CSR4_BYTE4);
	net_dev->dev_addr[5] = rt2x00_get_field32(reg[1], CSR4_BYTE5);

	net_dev->addr_len = 6;

	if (is_valid_ether_addr(&net_dev->dev_addr[0])) {
		SET_FLAG(rt2x00pci, DEVICE_INITIALIZED_MAC);
		return 0;
	}

	return -EINVAL;
}

static int rt2400pci_init_hw(struct rt2x00_pci *rt2x00pci)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00pci->pci_dev);
	struct ieee80211_hw *hw = &rt2x00pci->hw;

	if (GET_FLAG(rt2x00pci, DEVICE_INITIALIZED_HW))
		return 0;

	/*
	 * IEEE80211 Function callbacks.
	 */
	hw->tx			= rt2400pci_tx;
	hw->reset		= rt2400pci_reset;
	hw->add_interface	= rt2400pci_add_interface;
	hw->remove_interface	= rt2400pci_remove_interface;
	hw->config		= rt2400pci_config;
	hw->config_interface	= rt2400pci_config_interface;
	hw->set_multicast_list	= rt2400pci_set_multicast_list;
	hw->passive_scan	= rt2400pci_passive_scan;
	hw->get_stats		= rt2400pci_get_stats;
	hw->set_retry_limit	= rt2400pci_set_retry_limit;
	hw->conf_tx		= rt2400pci_conf_tx;
	hw->get_tx_stats	= rt2400pci_get_tx_stats;
	hw->get_tsf		= rt2400pci_get_tsf;
	hw->reset_tsf		= rt2400pci_reset_tsf;
	hw->beacon_update	= rt2400pci_beacon_update;
	hw->tx_last_beacon	= rt2400pci_tx_last_beacon;

	/*
	 * IEEE80211 Variables.
	 */
	hw->version = IEEE80211_VERSION;
	hw->name = DRV_NAME;
	hw->host_gen_beacon = 1;
	hw->device_hides_wep = 0;
	hw->rx_includes_fcs = 0;
	hw->host_broadcast_ps_buffering = 1;
	hw->wep_include_iv = 1;
	hw->data_nullfunc_ack = 1;
	hw->no_tkip_wmm_hwaccel = 1;
	hw->extra_hdr_room = 0;
	hw->device_strips_mic = 0;
	hw->monitor_during_oper = 1;
	hw->fraglist = 0;

	/*
	 * We have 2 TX queues: TX and PRIO.
	 * PRIO should be capable of sending
	 * regular TX frames as well. By doing this
	 * we can increase performance since we
	 * will have support for high priority TX frames.
	 */
	hw->queues = RING_NUM_TX;

	if (ieee80211_register_hw(net_dev, hw))
		return -EIO;

	SET_FLAG(rt2x00pci, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt2400pci_init_hw_channels(struct rt2x00_pci *rt2x00pci,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u16 eeprom;
	static const u32 vals[] = {
		0x000c1fda, 0x000c1fee, 0x000c2002, 0x000c2016,
		0x000c202a, 0x000c203e, 0x000c2052, 0x000c2066,
		0x000c207a, 0x000c208e, 0x000c20a2, 0x000c20b6,
		0x000c20ca, 0x000c20fa
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS
			| IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].val = cpu_to_le32(vals[i]);
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS
		| IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].val = cpu_to_le32(vals[13]);
	channels[13].antenna_max = 0xff;

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00pci,
			EEPROM_TXPOWER_START + i, &eeprom);

		channels[(i * 2)].power_level =
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_1);
		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			channels[(i * 2)].power_level);

		channels[(i * 2) + 1].power_level =
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_2);
		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			channels[(i * 2) + 1].power_level);
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	rt2x00pci->rf1 = cpu_to_le32(0x00022058);
	if (rt2x00_rf(&rt2x00pci->chip, RF2420))
		rt2x00pci->rf3 = cpu_to_le32(0x00000111);
	else
		rt2x00pci->rf3 = cpu_to_le32(0x00000101);
}

static void rt2400pci_init_hw_rates(struct rt2x00_pci *rt2x00pci,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	rates[0].rate = 10;
	rates[0].val = DEVICE_RATE_1MB;
	rates[0].flags = IEEE80211_RATE_CCK;
	rates[0].val2 = DEVICE_RATE_1MB;
	rates[0].min_rssi_ack = 0;
	rates[0].min_rssi_ack_delta = 0;

	rates[1].rate = 20;
	rates[1].val = DEVICE_RATE_2MB;
	rates[1].flags = IEEE80211_RATE_CCK_2;
	rates[1].val2 = DEVICE_RATE_2MB_PREAMBLE;
	rates[1].min_rssi_ack = 0;
	rates[1].min_rssi_ack_delta = 0;

	rates[2].rate = 55;
	rates[2].val = DEVICE_RATE_55MB;
	rates[2].flags = IEEE80211_RATE_CCK_2;
	rates[2].val2 = DEVICE_RATE_55MB_PREAMBLE;
	rates[2].min_rssi_ack = 0;
	rates[2].min_rssi_ack_delta = 0;

	rates[3].rate = 110;
	rates[3].val = DEVICE_RATE_11MB;
	rates[3].flags = IEEE80211_RATE_CCK_2;
	rates[3].val2 = DEVICE_RATE_11MB_PREAMBLE;
	rates[3].min_rssi_ack = 0;
	rates[3].min_rssi_ack_delta = 0;
}

static int rt2400pci_init_hw_modes(struct rt2x00_pci *rt2x00pci)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00pci->pci_dev);
	struct ieee80211_hw *hw = &rt2x00pci->hw;

	/*
	 * RT2400 only supports 802.11b.
	 * Allocate memory for 14 OFDM channels and 4 CCK rates.
	 */
	hw->num_modes = 1;
	hw->modes =
		kzalloc(sizeof(struct ieee80211_hw_modes), GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->num_channels = 14;
	hw->modes->channels =
		kzalloc(sizeof(struct ieee80211_channel) * 14, GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->num_rates = 4;
	hw->modes->rates =
		kzalloc(sizeof(struct ieee80211_rate) * 4, GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Initialize modes.
	 */
	hw->modes->mode = MODE_IEEE80211B;

	rt2400pci_init_hw_channels(rt2x00pci, hw->modes->channels);
	rt2400pci_init_hw_rates(rt2x00pci, hw->modes->rates);

	/*
	 * xr_end is only used on Atheros cards.
	 */
	hw->modes->xr_end = 0;

	return ieee80211_update_hw(net_dev, hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static void rt2400pci_free_dev(struct net_device *net_dev)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	/*
	 * Shutdown poll_timer for hardware button.
	 */
	rt2400pci_button_stop(rt2x00pci);

	/*
	 * Free ring structures.
	 */
	kfree(rt2x00pci->ring);
	rt2x00pci->ring = NULL;

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00pci->eeprom);

	/*
	 * Release CSR memory.
	 */
	if (likely(rt2x00pci->csr_addr)) {
		iounmap(rt2x00pci->csr_addr);
		rt2x00pci->csr_addr = NULL;
	}

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00pci->workqueue)) {
		destroy_workqueue(rt2x00pci->workqueue);
		rt2x00pci->workqueue = NULL;
	}

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00pci->hw.modes)) {
		kfree(rt2x00pci->hw.modes->channels);
		kfree(rt2x00pci->hw.modes->rates);
		kfree(rt2x00pci->hw.modes);
		rt2x00pci->hw.modes = NULL;
	}
}

static int rt2400pci_alloc_dev(
	struct pci_dev *pci_dev, struct net_device *net_dev)
{
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	rt2x00pci->pci_dev = pci_dev;

	rt2x00pci->csr_addr = ioremap(
		pci_resource_start(rt2x00pci->pci_dev, 0),
		pci_resource_len(rt2x00pci->pci_dev, 0));
	if (!rt2x00pci->csr_addr) {
		ERROR("Ioremap failed.\n");
		return -ENOMEM;
	}

	rt2x00pci->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00pci->workqueue)
		goto exit;

	/*
	 * Initialize cofniguration work.
	 */
	INIT_WORK(&rt2x00pci->config_work, rt2400pci_config_update, rt2x00pci);

	/*
	 * Reset current working type.
	 */
	rt2x00pci->interface_type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	INIT_WORK(&rt2x00pci->scan_work, rt2400pci_scan, rt2x00pci);
	rt2x00pci->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt2400pci_alloc_rings(rt2x00pci))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt2400pci_init_eeprom(rt2x00pci)
	|| rt2400pci_init_mac(rt2x00pci)
	|| rt2400pci_init_hw(rt2x00pci)
	|| rt2400pci_init_hw_modes(rt2x00pci)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	/*
	 * If required start hardware button polling.
	 */
	rt2400pci_button_start(rt2x00pci);

	return 0;

exit:
	rt2400pci_free_dev(net_dev);

	return -ENODEV;
}

/*
 * PCI driver handlers.
 */
static int rt2400pci_probe(
	struct pci_dev *pci_dev, const struct pci_device_id *id)
{
	struct net_device *net_dev;
	int status;

	if (unlikely(id->driver_data != RT2460)) {
		ERROR("Detected device not supported.\n");
		return -ENODEV;
	}

	if (pci_enable_device(pci_dev)) {
		ERROR("Enable device failed.\n");
		return -EIO;
	}

	pci_set_master(pci_dev);

	if (pci_set_mwi(pci_dev))
		NOTICE("MWI not available.\n");

	if (pci_set_dma_mask(pci_dev, DMA_64BIT_MASK)
	&& pci_set_dma_mask(pci_dev, DMA_32BIT_MASK)) {
		ERROR("PCI DMA not supported.\n");
		status = -EIO;
		goto exit_disable_device;
	}

	if (pci_request_regions(pci_dev, pci_name(pci_dev))) {
		ERROR("PCI request regions failed.\n");
		status = -EBUSY;
		goto exit_disable_device;
	}

	net_dev = ieee80211_alloc_hw(sizeof(struct rt2x00_pci), NULL);
	if (!net_dev) {
		ERROR("Failed to allocate device.\n");
		status = -ENOMEM;
		goto exit_release_regions;
	}

	SET_ETHTOOL_OPS(net_dev, &rt2400pci_ethtool_ops);

	pci_set_drvdata(pci_dev, net_dev);

	status = rt2400pci_alloc_dev(pci_dev, net_dev);
	if (status) {
		ERROR("Failed to allocate device.\n");
		status = -ENOMEM;
		goto exit_free_device;
	}

	return 0;

exit_free_device:
	ieee80211_free_hw(net_dev);

exit_release_regions:
	pci_release_regions(pci_dev);

exit_disable_device:
	if (status != -EBUSY)
		pci_disable_device(pci_dev);

	pci_set_drvdata(pci_dev, NULL);

	return status;
}

static void rt2400pci_remove(struct pci_dev *pci_dev)
{
	struct net_device *net_dev = pci_get_drvdata(pci_dev);
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	rt2400pci_disable_radio(rt2x00pci);

	rt2400pci_uninitialize(rt2x00pci);

	rt2400pci_free_dev(net_dev);

	ieee80211_unregister_hw(net_dev);

	ieee80211_free_hw(net_dev);

	pci_set_drvdata(pci_dev, NULL);

	pci_release_regions(pci_dev);

	pci_disable_device(pci_dev);
}

#ifdef CONFIG_PM
static int rt2400pci_suspend(struct pci_dev *pci_dev, pm_message_t state)
{
	struct net_device *net_dev = pci_get_drvdata(pci_dev);
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	NOTICE("Going to sleep.\n");

	/*
	 * Disable the radio.
	 */
	rt2400pci_disable_radio(rt2x00pci);

	/*
	 * Set device mode to sleep for power management.
	 */
	if (rt2400pci_set_state(rt2x00pci, STATE_SLEEP))
		return -EBUSY;

	/*
	 * Uninitialize device.
	 */
	rt2400pci_uninitialize(rt2x00pci);

	/*
	 * Uninitialize hardware.
	 */
	rt2400pci_free_dev(net_dev);

	/*
	 * Disable PCI.
	 */
	pci_save_state(pci_dev);
	pci_disable_device(pci_dev);
	return pci_set_power_state(pci_dev, pci_choose_state(pci_dev, state));
}

static int rt2400pci_resume(struct pci_dev *pci_dev)
{
	struct net_device *net_dev = pci_get_drvdata(pci_dev);
	struct rt2x00_pci *rt2x00pci = ieee80211_dev_hw_data(net_dev);

	NOTICE("Waking up.\n");

	/*
	 * Enable PCI.
	 */
	if (pci_set_power_state(pci_dev, PCI_D0)
	|| pci_enable_device(pci_dev)
	|| pci_restore_state(pci_dev)) {
		ERROR("Failed to resume device.\n");
		return -EIO;
	}

	/*
	 * Initialize hardware.
	 */
	if (rt2400pci_alloc_dev(pci_dev, net_dev)) {
		ERROR("Failed to allocate device.\n");
		return -ENOMEM;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	return rt2400pci_set_state(rt2x00pci, STATE_AWAKE);
}
#endif /* CONFIG_PM */

/*
 * RT2400pci module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct pci_device_id rt2400pci_device_table[] = {
	{ PCI_DEVICE(0x1814, 0x0101), .driver_data = RT2460},
	{0,}
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT2400 PCI & PCMCIA Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2460 PCI & PCMCIA chipset based cards");
MODULE_DEVICE_TABLE(pci, rt2400pci_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT2400PCI_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT2400PCI_DEBUG */

#ifdef CONFIG_RT2400PCI_BUTTON
module_param_named(poll_delay, rt2x00_poll_delay, short, S_IRUGO);
MODULE_PARM_DESC(poll_delay, "Delay between WiFi button pollings (in ms).");
#endif /* CONFIG_RT2400PCI_BUTTON */

static struct pci_driver rt2400pci_driver = {
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 14)
	.owner		= THIS_MODULE,
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 14) */
	.name		= DRV_NAME,
	.id_table	= rt2400pci_device_table,
	.probe		= rt2400pci_probe,
	.remove		= __devexit_p(rt2400pci_remove),
#ifdef CONFIG_PM
	.suspend	= rt2400pci_suspend,
	.resume		= rt2400pci_resume,
#endif /* CONFIG_PM */
};

static int __init rt2400pci_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return pci_register_driver(&rt2400pci_driver);
}

static void __exit rt2400pci_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	pci_unregister_driver(&rt2400pci_driver);
}

module_init(rt2400pci_init);
module_exit(rt2400pci_exit);
