/*
 * Copyright (c) 2006 Jiri Benc <jbenc@suse.cz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/if.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/rtnetlink.h>
#include <net/d80211.h>
#include "ieee80211_i.h"

#define to_ieee80211_local(class)	container_of(class, struct ieee80211_local, class_dev)


static ssize_t store_add_iface(struct class_device *dev,
			       const char *buf, size_t len)
{
	struct ieee80211_local *local = to_ieee80211_local(dev);
	struct net_device *new_dev;
	int res;

	if (!capable(CAP_NET_ADMIN))
		return -EPERM;
	if (len > IFNAMSIZ)
		return -EINVAL;
	rtnl_lock();
	res = ieee80211_if_add(local->mdev, buf, 0, &new_dev);
	if (res == 0)
		ieee80211_if_set_type(new_dev, IEEE80211_IF_TYPE_STA);
	rtnl_unlock();
	return res < 0 ? res : len;
}

static ssize_t store_remove_iface(struct class_device *dev,
				  const char *buf, size_t len)
{
	struct ieee80211_local *local = to_ieee80211_local(dev);
	int res;

	if (!capable(CAP_NET_ADMIN))
		return -EPERM;
	if (len > IFNAMSIZ)
		return -EINVAL;
	rtnl_lock();
	res = ieee80211_if_remove(local->mdev, buf, -1);
	rtnl_unlock();
	return res < 0 ? res : len;
}

#ifdef CONFIG_HOTPLUG
static int ieee80211_uevent(struct class_device *cd, char **envp,
			    int num_envp, char *buf, int size)
{
	struct ieee80211_local *local = to_ieee80211_local(cd);

	if (num_envp < 2)
		return -ENOMEM;
	envp[0] = buf;
	if (snprintf(buf, size, "IEEE80211_DEV=phy%d",
		     local->dev_index) + 1 >= size)
		return -ENOMEM;
	envp[1] = NULL;
	return 0;
}
#endif

static struct class_device_attribute ieee80211_class_dev_attrs[] = {
	__ATTR(add_iface, S_IWUSR, NULL, store_add_iface),
	__ATTR(remove_iface, S_IWUSR, NULL, store_remove_iface),
	{}
};

static struct class ieee80211_class = {
	.name = "ieee80211",
	.class_dev_attrs = ieee80211_class_dev_attrs,
#ifdef CONFIG_HOTPLUG
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15)
	.hotplug = ieee80211_uevent,
#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15) */
	.uevent = ieee80211_uevent,
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15) */
#endif
};

int ieee80211_register_sysfs(struct ieee80211_local *local)
{
	local->class_dev.class = &ieee80211_class;
	local->class_dev.class_data = local;
	snprintf(local->class_dev.class_id, BUS_ID_SIZE,
		 "phy%d", local->dev_index);
	return class_device_register(&local->class_dev);
}

void ieee80211_unregister_sysfs(struct ieee80211_local *local)
{
	class_device_del(&local->class_dev);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 15)
static int ieee80211_add_netdevice(struct class_device *cd)
#else /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 15) */
static int ieee80211_add_netdevice(struct class_device *cd,
				   struct class_interface *cintf)
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 15) */
{
	struct net_device *dev = container_of(cd, struct net_device, class_dev);
	struct ieee80211_local *local = dev->priv;

	if (ieee80211_dev_find_index(local) < 0)
		return 0;
	return sysfs_create_link(&cd->kobj, &local->class_dev.kobj, "wiphy");
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 15)
static void ieee80211_remove_netdevice(struct class_device *cd)
#else /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 15) */
static void ieee80211_remove_netdevice(struct class_device *cd,
				   struct class_interface *cintf)
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 15) */
{
	struct net_device *dev = container_of(cd, struct net_device, class_dev);
	struct ieee80211_local *local = dev->priv;

	if (ieee80211_dev_find_index(local) >= 0)
		sysfs_remove_link(&cd->kobj, "wiphy");
}

static struct class_interface ieee80211_wiphy_cintf = {
	.add = ieee80211_add_netdevice,
	.remove = ieee80211_remove_netdevice,
};

/* Adds class interface watching for new network devices and adding "wiphy"
 * attribute (symlink) to them. */
static int ieee80211_register_wiphy_cintf(void)
{
	ieee80211_wiphy_cintf.class = loopback_dev.class_dev.class;
	return class_interface_register(&ieee80211_wiphy_cintf);
}

static void ieee80211_unregister_wiphy_cintf(void)
{
	class_interface_unregister(&ieee80211_wiphy_cintf);
}

int ieee80211_sysfs_init(void)
{
	int result;

	result = class_register(&ieee80211_class);
	if (result)
		return result;
	result = ieee80211_register_wiphy_cintf();
	if (result)
		class_unregister(&ieee80211_class);
	return result;
}

void ieee80211_sysfs_deinit(void)
{
	ieee80211_unregister_wiphy_cintf();
	class_unregister(&ieee80211_class);
}
