/* -*-Mode: C++;-*-
 * $Id: shared.h 1.7 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _SHARED_H_
#define _SHARED_H_

extern pthread_mutexattr_t ERRORCHECK_MUTEX_ATTR;
extern pthread_mutexattr_t RECURSIVE_MUTEX_ATTR;

//////////////////////////////////////////////////////////////////////
//			     TXN, DESCRIPTORS
//////////////////////////////////////////////////////////////////////

class STXN
{
public:

    STXN ();

    virtual ~STXN ();

    virtual int   begin       (BASIC_DBENV &dbfs, int flags);
    bool          active      () const { return _dbtxn != NULL; }
    DbTxn*        dbtxn       () const { g_assert (active ()); return _dbtxn; }
    BASIC_DBENV&  dbenv       () const { g_assert (_dbfs); return *_dbfs; }
    DBFS&         dbfs        () const;
    DbEnv*        env         () const;
    int           commit      ();
    void          abort       ();

    // @@@ Where's this stuff go
    //bool    async       () const { return (_flags & DBFS_TXN_ASYNC); }
//  #ifdef HAVE_FDATASYNC
//  #define FSYNC fdatasync
//  #else
//  #define FSYNC fsync
//  #endif

protected:

    virtual int  pre_commit  ();
    virtual int  post_commit ();
    virtual void pre_abort   ();
    virtual void post_abort  ();

private:

    int         _flags;
    BASIC_DBENV       *_dbfs;
    DbTxn      *_dbtxn;
};

class MUTEX
{
public:
    // @@ Clean up mutexes
};

class MONITOR
{
public:

    MONITOR (pthread_mutex_t *mutex)
	: _mutex (NULL)
    {
	lock (mutex);
    }

    MONITOR ()
	: _mutex (NULL)
    {
    }

    ~MONITOR ()
    {
	release ();
    }

    int trylock (pthread_mutex_t *mutex)
    {
	int ret;

	g_assert (_mutex == NULL);

	if ((ret = pthread_mutex_trylock (mutex))) {
	    return ret;
	}

	_mutex = mutex;

	return 0;
    }

    void lock (pthread_mutex_t *mutex)
    {
	int ret;

	g_assert (_mutex == NULL);

	if ((ret = pthread_mutex_lock (mutex))) {
	    g_assert_not_reached ();
	}

	_mutex = mutex;
    }

    void release ()
    {
	if (_mutex != NULL) {

	    int ret;

	    if ((ret = pthread_mutex_unlock (_mutex))) {
		g_assert_not_reached ();
	    }

	    _mutex = NULL;
	}
    }

private:

    MONITOR            (const MONITOR& copy);
    MONITOR& operator= (const MONITOR& copy);

    pthread_mutex_t *_mutex;
};

class CLOSE_AFTER
{
public:

    CLOSE_AFTER ()
	: fd     (-1),
	  dbp    (NULL),
	  mem    (NULL),
	  onpage (0) { }

    int close ();

    int           fd;
    Db           *dbp;
    const guint8 *mem;
    guint         onpage;
};

template <class K, class D> class SHARED_DSET;

template <class K, class D>
class SHARED_DESC
{
public:

    SHARED_DESC ()
	: _dset (NULL)
    {
	pthread_mutex_init (& mutex, & RECURSIVE_MUTEX_ATTR);
    }

    virtual ~SHARED_DESC () { }

    virtual void close_after (CLOSE_AFTER& ca, BASIC_DBENV &dbfs) = 0;

    virtual const char* type () const = 0;

    virtual void reset (const K& nkey)
    {
	refs  = 0;
	key   = nkey;
    }

    void use ();

    class Hash {
    public:
	uint operator () (const SHARED_DESC<K,D> *a) const { return a->key.hash (); }
	uint operator () (const K                &k) const { return k.hash (); }
    };

    class Equals {
    public:
	bool operator () (const K& k, const SHARED_DESC<K,D> *b) const { return k == b->key; }
    };

    K                  key;

    guint16            refs;
    pthread_mutex_t    mutex;

    elink<D> use_link;
    elink<D> map_link;

    SHARED_DSET<K,D> *_dset;
};

template <class K, class D>
inline ostream&
operator<< (ostream &os, const SHARED_DESC<K,D>& sd)
{
    os << sd.type () << " (" << sd.key << ")";
    return os;
}

template <class K, class D>
class SHARED_DSET
{
    void reset ()
    {
	_free.clear ();
	_lru.clear ();
	_map.clear ();

	for (size_t i = 0; i < _size; i += 1) {
	    DEBUG_DBFS ("%p._dset = %p", &_array[i],this);
	    _array[i]._dset = this;
	    _free.push_back (& _array[i]);
	}
    }

public:

    typedef ehash_map<D,OFFSETOF(D,map_link), typename D::Hash, typename D::Equals> DESC_MAP;
    typedef elist<D,OFFSETOF(D,use_link)>                                           DESC_LIST;

    friend class SHARED_DESC<K,D>;

    SHARED_DSET (BASIC_DBENV& dbfs, const char *name, size_t size)
	: _dbfs   (dbfs),
	  _name   (name),
	  _size   (size),
	  _array  (new D [size]),
	  _hits   (0),
	  _misses (0)
    {
	pthread_mutex_init (& _mutex, NULL);

	reset ();
    }

    ~SHARED_DSET ()
    {
	delete [] _array;
    }

    DESC_LIST& lru          (MONITOR &mon) { lock (mon); return _lru; }

    int        static_alloc (D*      &descp);

    int        find_shared  (const K &key,
			     D*      &descp,
			     MONITOR &desc_mon);

    int        remove       (const K &key);

    void       stat         (guint64 &hits, guint64 &misses) { hits = _hits; misses = _misses; }

    void       debug_cache  ();

    int        close        ();
    int        assert_clear ();

private:

    void       lock         (MONITOR &mon) { mon.lock (& _mutex); }

public:

    BASIC_DBENV       &_dbfs;
    const char *_name;

private:

    size_t      _size;
    D          *_array;

    guint64   _hits;
    guint64   _misses;

    pthread_mutex_t _mutex;

    DESC_MAP      _map;
    DESC_LIST     _free;
    DESC_LIST     _lru;
};

struct XFIDPGNO
{
    XFIDPGNO () { }

    XFIDPGNO (const XFID& fid, const XPGNO& pgno)
	: fid  (fid),
	  pgno (pgno) { }

    uint hash () const { return fid.hash () ^ pgno.hash (); }

    bool operator== (const XFIDPGNO& o) const { return o.fid == fid && o.pgno == pgno; }

    XFID  fid;
    XPGNO pgno;
};

class SHDB_DESC
    : public SHARED_DESC<XFID,SHDB_DESC>
{
public:

    typedef SHARED_DESC<XFID,SHDB_DESC> PARENT;

    SHDB_DESC ()
	: dbp    (NULL) { }

    void close_after (CLOSE_AFTER& ca, BASIC_DBENV &dbfs);
    void reset       (const XFID& fid);
    const char* type () const { return "SHDB"; }

    Db *dbp;
};

class SHFD_DESC;

class SHPG_DESC
    : public SHARED_DESC<XFIDPGNO,SHPG_DESC>
{
public:

    typedef SHARED_DESC<XFIDPGNO,SHPG_DESC> PARENT;

    SHPG_DESC ()
	: mem    (NULL) { }

    void close_after (CLOSE_AFTER& ca, BASIC_DBENV &dbfs);
    void reset       (const XFIDPGNO& xpgno);
    const char* type () const { return "SHFD"; }

    guint8 *mem;
    guint   onpage;

    elink<SHPG_DESC>  shfd_link;
    SHFD_DESC        *shfd_desc;
};

typedef ELIST(SHPG_DESC,shfd_link) SHPG_LIST;

class SHFD_DESC
    : public SHARED_DESC<XFID,SHFD_DESC>
{
public:

    typedef SHARED_DESC<XFID,SHFD_DESC> PARENT;

    SHFD_DESC ()
	: fd     (-1) { }

    void close_after (CLOSE_AFTER& ca, BASIC_DBENV &dbfs);
    void reset       (const XFID& fid);
    const char* type () const { return "SHPG"; }

    // Throw out invalid pages:
    int  truncate    (BASIC_DBENV &dbfs, const XSIZ& size);

    int fd;

    SHPG_LIST pgs;
};

typedef SHARED_DSET<XFID,SHDB_DESC>     SHDB_DSET;
typedef SHARED_DSET<XFID,SHFD_DESC>     SHFD_DSET;
typedef SHARED_DSET<XFIDPGNO,SHPG_DESC> SHPG_DSET;

template <class K, class D>
inline int
SHARED_DSET<K,D>::close ()
{
    int ret = 0, t_ret;
    MONITOR this_mon;

    lock (this_mon);

    for (DESC_LIST::iterator siter = _lru.begin ();
	 siter != _lru.end ();
	 ++ siter) {

	D* descp = & (* siter);

	if (descp->refs != 0) {
	    DBFS_WARN ("SHARED_DESC still referenced: %d (ignore): ", descp->refs) () << (*descp);
	}

	CLOSE_AFTER ca;

	descp->close_after (ca, _dbfs);

	if ((t_ret = ca.close ())) {
	    PROP_ERROR (t_ret) ("SHARED_DESC close") () << (*descp);

	    if (ret == 0) {
		ret = t_ret;
	    }
	}
    }

    reset ();

    return ret;
}

template <class K, class D>
inline int
SHARED_DSET<K,D>::assert_clear ()
{
    if (! _lru.empty ()) {
	DBFS_WARN ("SHARED_DESC should not be open: internal error: %d pages", _lru.size ());
    }

    if (! _map.empty ()) {
	DBFS_WARN ("SHARED_DESC should not be open: internal error: %d entries", _map.size ());
    }

    reset ();

    return 0;
}

template <class K, class D>
inline int
SHARED_DSET<K,D>::remove (const K& key)
{
    int     ret;
    MONITOR this_mon;

    lock (this_mon);

    D *descp = _map.find (key);

    if (descp == NULL) {
	return DBFS_NOTFOUND;
    }

    DEBUG_SHARED ("find_shared: remove: ") (descp);

    // @@@ DTXN: Need changes to deal with removing in-use descriptors...
    g_assert (descp->refs == 0);

    _lru.erase      (descp);
    _map.erase      (descp);
    _free.push_back (descp);

    CLOSE_AFTER ca;

    descp->close_after (ca, _dbfs);

    // Release SHARED_DSET mutex prior to close()
    this_mon.release ();

    if ((ret = ca.close ())) {
	PROP_ERROR (ret) ("descp_close");
	return ret;
    }

    return 0;
}

template <class K, class D>
inline int
SHARED_DSET<K,D>::static_alloc (D*      &descp)
{
    if (_free.empty ()) {
	DBFS_ERROR ("no free descriptors in static_alloc");
	return DBFS_NOSPACE;
    }

    // Note: these are NOT mapped
    descp = & _free.pop_front ();

    _lru.push_back (descp);

    return 0;
}

template <class K, class D>
inline int
SHARED_DSET<K,D>::find_shared (const K& key, D*& descp, MONITOR &descp_mon)
{
    int ret;

    MONITOR     this_mon;
    CLOSE_AFTER ca;

    lock (this_mon);

    descp = _map.find (key);

    if (descp == NULL) {

	_misses += 1;

	// Now try to allocate a new descriptor
	if (! _free.empty ()) {

	    // From the free list
	    descp = & _free.pop_front ();

	} else {

	    // Find LRU
	    for (DESC_LIST::iterator it = _lru.begin ();
		 it != _lru.end ();
		 ++it) {

		D* lelt = & (* it);

		if (lelt->refs == 0) {
		    descp = lelt;
		    break;
		}
	    }

	    if (descp == NULL) {
		DBFS_ERROR ("all shared descriptors are currently referenced");
		return DBFS_NOSPACE;
	    }

	    DEBUG_SHARED ("find_shared: reuse descriptor: ") (descp);

	    _lru.erase (descp);
	    _map.erase (descp);

	    descp->close_after (ca, _dbfs);
	}

	descp->reset (key);

	_lru.push_back (descp);
	_map.insert    (descp);

	DEBUG_SHARED ("find_shared: allocate: ") (descp);

    } else {

	DEBUG_SHARED ("find_shared: hit cache: ") (descp);

	_hits += 1;
    }

    descp_mon.lock (& descp->mutex);

    // Release SHARED_DSET mutex prior to close()
    this_mon.release ();

    if ((ret = ca.close ())) {
	PROP_ERROR (ret) ("descp_close");
	return ret;
    }

    return 0;
}

template <class K,class D>
inline void
SHARED_DSET<K,D>::debug_cache ()
{
    DEBUG_CACHE ("%s cache hit rate %.2f", _name, (double) _hits / (double) (_hits + _misses));
}

template <class K,class D>
inline void
SHARED_DESC<K,D>::use ()
{
    MONITOR mon;

    _dset->lock (mon);

    _dset->_lru.erase     (static_cast<D*> (this));
    _dset->_lru.push_back (static_cast<D*> (this));
}

template <class T>
class DESC_REF
{
public:

    DESC_REF ()
	: _sdesc (NULL)
    {
    }

    ~DESC_REF ()
    {
	decr ();
    }

    void decr ()
    {
	if (_sdesc != NULL) {
	    MONITOR mon (& _sdesc->mutex);

	    _sdesc->refs -= 1;

	    DEBUG_REFS ("decr -> %d: key ", _sdesc->refs) () << (*_sdesc);

	    _sdesc = NULL;
	}
    }

    void copy (T *ndesc)
    {
	g_assert (_sdesc == NULL && ndesc != NULL);

	_sdesc = ndesc;

	MONITOR mon (& _sdesc->mutex);

	_sdesc->refs += 1;

	DEBUG_REFS ("incr -> %d: key ", _sdesc->refs) () << (*_sdesc);
    }

    T *_sdesc;
};

#endif
