/* -*-Mode: C++;-*-
 * $Id: seg.cc 1.14 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <edsiostdio.h>
#include <sys/mman.h>

/**********************************************************************/
/*			     Write Handle                             */
/**********************************************************************/

class WRH
{
public:

    WRH (const NODEC &node,
	 int          flags);
    ~WRH ();

    FileHandle*  fh    () { return & _fh; }

    int          close ();
    gssize       map   (guint pgno, const guint8** mem);
    int          unmap (guint pgno, const guint8** mem);

    int          open_long ();

private:

    FileHandle   _fh;

    TXN         &_txn;
    NODEC        _node;
    XFID         _fid;
    int          _flags;
    guint8      *_pagebuf;

    DESC_REF<SHFD_DESC> _dr;
};

static gboolean     dbfs_write_close (FileHandle    *fh) { return ((WRH*) fh)->close () == 0; }
static void         dbfs_write_abort (FileHandle    *fh) { g_assert_not_reached (); }
static const gchar* dbfs_write_name  (FileHandle    *fh) { g_assert_not_reached (); return NULL; }
static void         dbfs_write_free  (FileHandle    *fh) { delete ((WRH*) fh); }
static gboolean     dbfs_write_unmap (FileHandle    *fh,
				      guint          pgno,
				      const guint8 **mem) { return ((WRH*) fh)->unmap (pgno, mem) == 0; }
static gssize       dbfs_write_map   (FileHandle    *fh,
				      guint          pgno,
				      const guint8 **mem) { return ((WRH*) fh)->map (pgno, mem); }

static const HandleFuncTable write_func_table =
{
    dbfs_write_close,
    dbfs_write_abort,
    dbfs_write_name,
    dbfs_write_free,
    dbfs_write_map,
    dbfs_write_unmap,
    NULL,
    NULL,
    NULL
};

WRH::WRH (const NODEC &node,
	  int          flags)
    : _txn     (node.txn ()),
      _node    (node),
      _fid     (XFID_NONE),
      _flags   (flags),
      _pagebuf ((guint8*) g_malloc (DBFS_PAGE_SIZE))
{
    memset (& _fh, 0, sizeof (_fh));

    _fh.table         = & write_func_table;
    _fh.fh_open_flags = _fh.fh_open_mode = HV_Replace;
    _fh.fh_has_len    = TRUE;

    file_position_from_abs (DBFS_PAGE_SIZE, 0, & _fh.fh_cur_pos);
    file_position_from_abs (DBFS_PAGE_SIZE, 0, & _fh.fh_file_len);
}

FileHandle*
dbfs_seg_replace (const NODEC  &repl,
		  int           flags)
{
    WRH *wh = new WRH (repl, flags);

    return wh->fh ();
}

int
WRH::open_long ()
{
    int ret;
    DXN dxn (_txn, ret);

    SHFD_DESC *sfd;
    MONITOR    sfd_mon;

    if ((ret = dxn.allocate_fid (_fid))) {
	PROP_ERROR (ret) ("allocate_fid");
	return ret;
    }

    if ((ret = _txn.dbfs ()._shared_fds.find_shared (_fid, sfd, sfd_mon))) {
	PROP_ERROR (ret) ("shared_fds_find");
	return ret;
    }

    // It shouldn't already exist, if it does its this error handling
    // stuff (should see aborts)
    if (sfd->fd >= 0) {
	DBFS_ERROR ("shared-fd still open: ") (sfd);
	return DBFS_INVAL;
    }

    string name;

    _txn.dbfs ().absolute_fname (_fid, name);

    if ((sfd->fd = open (name.c_str (), O_RDWR | O_CREAT | O_TRUNC, 0666)) < 0) {
	// @@ Shared-dset error handling
	ret = errno;
	SYS_ERROR (ret) ("open_write");
	return ret;
    }

    _dr.copy (sfd);

    return 0;
}

int
WRH::unmap (guint pgno, const guint8** mem)
{
    int   ret;
    guint rem = file_position_rem_on_page_no (& _fh.fh_file_len, pgno);

    g_assert ((* mem) == _pagebuf);

    _fh.fh_cur_page = NULL;

    // Assume that first unmap short of pagesize means EOF, and
    // SHORT_THRESH must be less than pagesize.  Skip file creation,
    // actually written in close ().
    if (pgno == 0 && rem <= DBFS_FS_SHORT_THRESH) {
	return 0;
    }

    if ((_dr._sdesc == NULL) && (ret = open_long ())) {
	PROP_ERROR (ret) ("open_long");
	return ret;
    }

    if ((ret = write (_dr._sdesc->fd, _pagebuf, rem)) < 0) {
	ret = errno;
	SYS_ERROR (ret) ("write");
	return ret;
    }

    if ((guint) ret != rem) {
	ret = EIO;
	SYS_ERROR (ret) ("unexpected short write");
	return ret;
    }

    return 0;
}

gssize
WRH::map (guint pgno, const guint8** mem)
{
    gssize res = file_position_rem_on_page_no (& _fh.fh_file_len, pgno);

    g_assert (res >= 0);

    (* mem) = _pagebuf;

    return res;
}

int
WRH::close ()
{
    int  ret;
    DXN  dxn (_txn, ret);
    XSIZ length = XSIZ (handle_length (& _fh));

    if (_dr._sdesc == NULL) {

	if ((ret = dxn.create_short (_node, _pagebuf, length, _flags))) {
	    PROP_ERROR (ret) (_node) ("create_short");
	    return ret;
	}

    } else {

	if ((ret = dxn.create_long (_node, _fid, length, _flags))) {
	    PROP_ERROR (ret) (_node) ("create_long");
	    return ret;
	}
    }

    _dr.decr ();

    return 0;
}

WRH::~WRH ()
{
    g_free (_pagebuf);
}

/**********************************************************************/
/*			      Read Handle                             */
/**********************************************************************/

class RDH
{
public:

    RDH  (const NODEC& node, int flags);
    ~RDH ();

    FileHandle*  fh        () { return & _fh; }

    int          open_long (const NODEC& node);
    int          open_view (const NODEC& node);
    int          open      (const NODEC& node);
    int          close     ();
    int          map       (guint pgno, const guint8** mem, gssize &size);
    int          unmap     (guint pgno, const guint8** mem);

    int          sys_mmap   (guint pgno, guint onpage, const guint8 **mem);
    int          sys_munmap (guint pgno, guint onpage, const guint8 *mem);

private:

    FileHandle    _fh;
    TXN          &_txn;
    int           _flags;

    // Short
    guint8       *_short_buf;

    // View
    VIEWIMPL     *_view;
    DbMpoolFile  *_view_mpf;

    DESC_REF<SHFD_DESC> _dr;
};

static gboolean     dbfs_read_close (FileHandle* fh) { return ((RDH*) fh)->close () == 0; }
static void         dbfs_read_abort (FileHandle* fh) { g_assert_not_reached (); }
static const gchar* dbfs_read_name  (FileHandle *fh) { g_assert_not_reached (); return NULL; }
static void         dbfs_read_free  (FileHandle* fh) { delete ((RDH*) fh); }
static gssize       dbfs_read_map   (FileHandle *fh, guint pgno, const guint8** mem)
{
    gssize size;
    int    ret;

    if ((ret = ((RDH*) fh)->map (pgno, mem, size))) {
	return -1;
    }

    return size;
}
static gboolean     dbfs_read_unmap (FileHandle *fh, guint pgno, const guint8** mem)
{
    return ((RDH*) fh)->unmap (pgno, mem) == 0;
}

static const HandleFuncTable dbfs_read_func_table =
{
    dbfs_read_close,
    dbfs_read_abort,
    dbfs_read_name,
    dbfs_read_free,
    dbfs_read_map,
    dbfs_read_unmap,
    NULL,
    NULL,
    NULL
};

RDH::RDH (const NODEC& node, int flags)
    : _txn         (node.txn ()),
      _flags       (flags),
      _short_buf   (NULL),
      _view        (NULL),
      _view_mpf    (NULL)
{
    memset (& _fh, 0, sizeof (_fh));

    _fh.table         = & dbfs_read_func_table;
    _fh.fh_open_flags = _fh.fh_open_mode = HV_Read;
    _fh.fh_has_len    = TRUE;

    file_position_from_abs (DBFS_PAGE_SIZE,                     0, & _fh.fh_cur_pos);
    file_position_from_abs (DBFS_PAGE_SIZE, node.length ().key (), & _fh.fh_file_len);
}

RDH::~RDH ()
{
    if (_short_buf) {
	g_free (_short_buf);
    }
}

int
dbfs_seg_read (const NODEC  &read,
	       FileHandle  **fhp,
	       int           flags)
{
    RDH *rh = new RDH (read, flags);
    int  ret;

    if ((ret = rh->open (read))) {
	delete rh;
	PROP_ERROR (ret) (read) ("read_handle_open");
	return ret;
    }

    (*fhp) = rh->fh ();

    return 0;
}

int
RDH::open_long (const NODEC &node)
{
    int ret;

    SHFD_DESC *sfd;
    MONITOR    sfd_mon;

    if ((ret = _txn.dbfs ()._shared_fds.find_shared (node.cont_id ().file_id (), sfd, sfd_mon))) {
	PROP_ERROR (ret) ("shared_fds_find");
	return ret;
    }

    if (sfd->fd < 0) {

	string name;

	_txn.dbfs ().absolute_fname (node.cont_id ().file_id (), name);

	if ((sfd->fd = ::open (name.c_str (), O_RDONLY)) < 0) {
	    ret = errno;
	    SYS_ERROR (ret) (node) ("open_read: ") () << name;
	    return ret;
	}
    }

    _dr.copy (sfd);

    return 0;
}

int
RDH::open_view (const NODEC &node)
{
    int ret;
    VIEWDEF        *vdef;

    if ((ret = dbfs_viewdef_find (node.cont_id ().view_id (), & vdef))) {
	PROP_ERROR (ret) (node) ("viewdef_find");
	return ret;
    }

    _view = vdef->make ();

    if ((ret = _view->begin (node))) {
	DBFS_ERROR (ret) (node) ("view_begin");
	return ret;
    }

    if ((ret = node.txn ().env ()->memp_fcreate (& _view_mpf, 0))) {
	DB_ERROR (ret) (node) ("mpool_fcreate");
	return ret;
    }
    
    if ((ret = _view_mpf->open (NULL, 0, 0666, DBFS_PAGE_SIZE))) {
	DB_ERROR (ret) (node) ("mpool_fopen_temp");
	return ret;
    }

    if ((ret = _view_mpf->set_clear_len (1))) {
	PROP_ERROR (ret) (node) ("set_clear_len");
	return ret;
    }

    return 0;
}

int
RDH::open (const NODEC& node)
{
    int   ret;
    XTYPE type (node.type ());

    if (type == XTYPE_SHORTSEG) {

	DXN dxn (_txn, ret);

	_short_buf = (guint8*) g_malloc (node.length ().key ());

	if ((ret = dxn.get_short (node, _short_buf))) {
	    PROP_ERROR (ret) (node) ("get_short");
	    return ret;
	}

    } else if (type == XTYPE_LONGSEG) {

	if ((ret = open_long (node))) {
	    PROP_ERROR (ret) (node) ("RDH::open_long");
	    return ret;
	}

    } else if (type == XTYPE_VIEWSEG) {

	if ((ret = open_view (node))) {
	    PROP_ERROR (ret) (node) ("open_view");
	    return ret;
	}
    }

    return 0;
}

int
RDH::map (guint pgno_u, const guint8** mem, gssize &size_out)
{
    int       ret;
    db_pgno_t pgno   = pgno_u;
    guint     onpage = file_position_rem_on_page_no (& _fh.fh_file_len, pgno_u);

    g_assert (onpage > 0);

    if (_short_buf) {

	(* mem) = _short_buf;

	goto done;

    } else if (_view) {

	if ((ret = _view_mpf->get (& pgno, DB_MPOOL_CREATE, (void**) mem))) {
	    DB_ERROR (ret) ("mpool_get");
	    return ret;
	}

	if ((ret = _view->pgin ((guint8*) (* mem),
				XSIZ (pgno * DBFS_PAGE_SIZE),
				onpage))) {
	    DBFS_ERROR (ret) ("view_pgin");
	    return ret;
	}

    } else {

	if ((ret = sys_mmap (pgno_u, onpage, mem))) {
	    PROP_ERROR (ret) ("sys_mmap");
	    return ret;
	}
    }

  done:

    size_out = onpage;

    return 0;
}

int
RDH::unmap (guint pgno, const guint8** mem)
{
    guint onpage = file_position_rem_on_page_no (& _fh.fh_file_len, pgno);
    int   ret;

    if (_view_mpf && (ret = _view_mpf->put ((void*) *mem, 0))) {
	DB_ERROR (ret) ("view_mpf_put");
	return ret;
    }

    if (_dr._sdesc) {

	if ((ret = sys_munmap (pgno, onpage, *mem))) {
	    PROP_ERROR (ret) ("sys_munmap");
	    return ret;
	}
    }

    (* mem) = NULL;

    return 0;
}

int
RDH::close ()
{
    int ret;

    if (_view) {
	delete _view;
	_view = NULL;
    }

    if (_view_mpf && (ret = _view_mpf->close (0))) {
	DB_ERROR (ret) ("view_mpf_close");
	return ret;
    }

    _view_mpf = NULL;

    _dr.decr ();

    return 0;
}

int
RDH::sys_mmap (guint pgno, guint onpage, const guint8 **mem)
{
    int ret;

    SHPG_DSET &dset = _txn.dbfs ()._shared_pgs;
    MONITOR    spg_mon;
    SHPG_DESC *spg;

    XFIDPGNO   key (_dr._sdesc->key, XPGNO (pgno));

    if ((ret = dset.find_shared (key, spg, spg_mon))) {
	PROP_ERROR (ret) ("find_shared_page");
	return ret;
    }

    if (spg->mem == NULL) {

	if ((spg->mem = (guint8*) mmap (NULL, onpage, PROT_READ,
					MAP_PRIVATE, _dr._sdesc->fd,
					(pgno * DBFS_PAGE_SIZE))) == MAP_FAILED) {
	    ret = errno;
	    SYS_ERROR (ret) ("mmap");
	    return ret;
	}

	spg->onpage    = onpage;
	spg->shfd_desc = _dr._sdesc;

	_dr._sdesc->pgs.push_front (spg);
    }

    spg->use ();

    spg->refs += 1;

    (*mem) = spg->mem;

    return 0;
}

int
RDH::sys_munmap (guint pgno, guint onpage, const guint8 *mem)
{
    int ret;

    SHPG_DSET &dset = _txn.dbfs ()._shared_pgs;
    MONITOR    spg_mon;
    SHPG_DESC *spg;

    XFIDPGNO   key (_dr._sdesc->key, XPGNO (pgno));

    if ((ret = dset.find_shared (key, spg, spg_mon))) {
	PROP_ERROR (ret) ("find_shared_page");
	return ret;
    }

    if (spg->onpage != onpage) {
	DBFS_ERROR ("munmap size disagreement");
	return DBFS_INVAL;
    }

    if (spg->refs == 0) {
	DBFS_ERROR ("munmap refcount disagreement");
	return DBFS_INVAL;
    }

    if (spg->mem != mem) {
	DBFS_ERROR ("munmap region disagreement");
	return DBFS_INVAL;
    }

    g_assert (spg->refs > 0);

    spg->refs -= 1;

    return 0;
}
