#ifndef __ELIST_H__
#define __ELIST_H__


#include <algorithm>
#include <iterator>
//#include "types.h"


//RCSID (elist_h, "$Id: elist.h 1.1 Tue, 19 Dec 2000 15:34:11 +0300 jmacd $");

#define OFFSETOF(t,f) ((int) ((char*) &((t*) 0)->f))

#define ELIST(type,field)   elist<type, OFFSETOF(type,field)>
#define ESLIST(type,field)  eslist<type, OFFSETOF(type,field)>


// Models: Assignable
template <class T>
struct elink
{
    // Constructor
    elink () : next (next), prev (prev) { }
    // Copy constructor
    elink (const elink<T> &link) : next (link.next), prev (link.prev) { }

    T *next;
    T *prev;
};


template <class T>
inline elink<T>*
__elist_link (T *node, int offset)
{
    return (elink<T>*) ((char*) node + offset);
}

template <class T>
inline T*
__elist_unlink (const elink<T> *link, int offset)
{
    return (T*) ((char*) link - offset);
}


// Models: Assignable, Bidirectional Iterator
template <class T, class Ref, class Ptr, int offset>
struct elist_iterator
{
    typedef elist_iterator<T, T&, T*, offset> iterator;
    typedef elist_iterator<T, const T&, const T*, offset> const_iterator;
    typedef elist_iterator<T, Ref, Ptr, offset> self;

    typedef bidirectional_iterator_tag iterator_category;
    typedef T value_type;
    typedef value_type *pointer;
    typedef const value_type *const_pointer;
    typedef value_type &reference;
    typedef const value_type &const_reference;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;
    typedef T *link_type;

    link_type node;

    elist_iterator () { }
    elist_iterator (link_type x) : node (x) { }
    elist_iterator (const iterator &x) : node (x.node) { }

    bool operator== (const self &x) const { return node == x.node; }
    bool operator!= (const self &x) const { return node != x.node; }
    reference operator* () const { return *node; }
    pointer operator-> () const { return node; }

    self& operator++ () { node = __elist_link (node, offset)->next; return *this; }
    self  operator++ (int) { self tmp = *this; ++*this; return tmp; }
    self& operator-- () { node = __elist_link (node, offset)->prev; return *this; }
    self  operator-- (int) { self tmp = *this; --*this; return tmp; }
};


template <class T, class Ref, class Ptr, int offset>
inline void
iter_swap (elist_iterator<T, Ref, Ptr, offset> a, elist_iterator<T, Ref, Ptr, offset> b)
{
    elink<T> a_link = *__elist_link (a.node, offset);
    elink<T> b_link = *__elist_link (b.node, offset);;
    T tmp = *a;
    *a = *b;
    *b = tmp;
    *__elist_link (a.node, offset) = a_link;
    *__elist_link (b.node, offset) = b_link;
}


// Models: Assignable, Container, Sequence
template <class T, int offset>
struct elist
    : public elink<T>
{
    typedef T value_type;
    typedef value_type *pointer;
    typedef const value_type *const_pointer;
    typedef value_type &reference;
    typedef const value_type &const_reference;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;

    typedef elist_iterator<T, T&, T*, offset> iterator;
    typedef elist_iterator<T, const T&, const T*, offset> const_iterator;
    typedef reverse_iterator<const_iterator> const_reverse_iterator;
    typedef reverse_iterator<iterator> reverse_iterator;

    void clear ()
    {
        next = __elist_unlink (this, offset);
        prev = __elist_unlink (this, offset);
    }

    // Constructor
    elist () { clear (); }
    // Copy constructor
    elist (const elist<T,offset> &list) : elink<T> (list) { clear (); }
    // No-init constructor
    elist (bool) : elink<T> () { }

    iterator begin () { return iterator (next); }
    const_iterator begin () const { return const_iterator (next); }
    iterator end () { return iterator (__elist_unlink (this, offset)); }
    const_iterator end () const { return const_iterator (__elist_unlink (this, offset)); }
    reverse_iterator rbegin () { return reverse_iterator (end ()); }
    const_reverse_iterator rbegin () const { return const_reverse_iterator (end ()); }
    reverse_iterator rend () { return reverse_iterator (begin ()); }
    const_reverse_iterator rend () const { return const_reverse_iterator (begin ()); }

    bool empty () const
    {
        return (next == __elist_unlink (this, offset));
    }
    size_type size () const
    {
        return distance (begin (), end ());
    }
    size_type max_size () const { return size_type (-1); }
    reference front () { return *begin (); }
    const_reference front() const { return *begin (); }
    reference back () { return *(--end ()); }
    const_reference back () const { return *(--end ()); }
    void copy (elist<T,offset> &x)
    {
        if (!x.empty ()) {
            next = x.next;
            prev = x.prev;
            __elist_link (next, offset)->prev = __elist_unlink (this, offset);
            __elist_link (prev, offset)->next = __elist_unlink (this, offset);
            x.clear ();
        }
    }
    void swap (elist<T,offset> &x)
    {
        elist<T,offset> tmp;
        tmp.copy (*this);
        copy (x);
        x.copy (tmp);
    }
    void insert (iterator position, T *obj)
    {
        T *next = &(*position);
        T *prev = &(*--position);
        __elist_link (next, offset)->prev = obj;
        __elist_link (obj, offset)->next = next;
        __elist_link (obj, offset)->prev = prev;
        __elist_link (prev, offset)->next = obj;
    }
    void push_front (T *obj)
    {
        insert (begin (), obj);
    }
    void push_back (T *obj)
    {
        insert (end (), obj);
    }
    iterator erase (iterator position)
    {
        T *prev = __elist_link (position.node, offset)->prev;
        T *next = __elist_link (position.node, offset)->next;
        __elist_link (next, offset)->prev = prev;
        __elist_link (prev, offset)->next = next;
        return iterator (next);
    }
    iterator erase (T *obj)
    {
        return erase (iterator (obj));
    }
    bool remove (T *obj)
    {
	for (iterator it = begin ();
	     it != end ();
	     ++it) {
	    if (obj == &(*it)) {
		erase (it);
		return true;
	    }
	}

	return false;
    }
    reference pop_front () { reference it = *begin (); erase (begin ()); return it; }
    reference pop_back () { reference it = *(--end()); erase (--end ()); return it; }
};


// Models: Assignable
template <class T>
struct eslink
{
    // Constructor
    eslink () : next (next) { }
    // Copy Constructor
    eslink (const eslink<T> &link) : next (link.next) { }

    T *next;
};


template <class T>
inline eslink<T>*
__eslist_link (T *node, int offset)
{
    return (eslink<T>*) ((char*) node + offset);
}

template <class T>
inline T*
__eslist_unlink (const eslink<T> *link, int offset)
{
    return (T*) ((char*) link - offset);
}


// Models: Assignable, Forward Iterator
template <class T, class Ref, class Ptr, int offset>
struct eslist_iterator
{
    typedef eslist_iterator<T, T&, T*, offset> iterator;
    typedef eslist_iterator<T, const T&, const T*, offset> const_iterator;
    typedef eslist_iterator<T, Ref, Ptr, offset> self;

    typedef forward_iterator_tag iterator_category;
    typedef T value_type;
    typedef value_type *pointer;
    typedef const value_type *const_pointer;
    typedef value_type &reference;
    typedef const value_type &const_reference;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;
    typedef T *link_type;

    link_type node;

    eslist_iterator () { }
    eslist_iterator (link_type x) : node (x) { }
    eslist_iterator (const iterator &x) : node (x.node) { }

    bool operator== (const self &x) const { return node == x.node; }
    bool operator!= (const self &x) const { return node != x.node; }
    reference operator* () const { return *node; }
    pointer operator-> () const { return node; }

    self& operator++ () { node = __eslist_link (node, offset)->next; return *this; }
    self  operator++ (int) { self tmp = *this; ++*this; return tmp; }
};


template <class T, class Ref, class Ptr, int offset>
inline void
iter_swap (eslist_iterator<T, Ref, Ptr, offset> a, eslist_iterator<T, Ref, Ptr, offset> b)
{
    eslink<T> a_link = *__eslist_link (a.node, offset);
    eslink<T> b_link = *__eslist_link (b.node, offset);;
    T tmp = *a;
    *a = *b;
    *b = tmp;
    *__eslist_link (a.node, offset) = a_link;
    *__eslist_link (b.node, offset) = b_link;
}


template <class T, int offset>
struct eslist
    : public eslink<T>
{
    typedef T value_type;
    typedef value_type *pointer;
    typedef const value_type *const_pointer;
    typedef value_type &reference;
    typedef const value_type &const_reference;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;

    typedef eslist_iterator<T, T&, T*, offset> iterator;
    typedef eslist_iterator<T, const T&, const T*, offset> const_iterator;

    void clear ()
    {
        next = __eslist_unlink (this, offset);
    }

    // Constructor
    eslist () { clear (); }
    // Copy constructor
    eslist (const eslist<T,offset> &list) : eslink<T> (list) { clear (); }
    // No-init constructor
    eslist (bool) : elink<T> () { }

    iterator begin () { return iterator (next); }
    const_iterator begin () const { return const_iterator (next); }
    iterator end () { return iterator (__eslist_unlink (this, offset)); }
    const_iterator end () const { return const_iterator (__eslist_unlink (this, offset)); }

    bool empty () const
    {
        return (next == __eslist_unlink (this, offset));
    }
    size_type size () const
    {
        return distance (begin (), end ());
    }
    size_type max_size () const { return size_type (-1); }
    reference front () { return *begin (); }
    const_reference front() const { return *begin (); }
    void swap (eslist<T,offset> &x)
    {
        iterator iter;

        iter = x.begin ();
        while (__eslist_link (iter.node, offset)->next != x.end ().node) {
            ++iter;
        }
        __eslist_link (iter.node, offset)->next = end ().node;

        iter = begin ();
        while (__eslist_link (iter.node, offset)->next != end ().node) {
            ++iter;
        }
        __eslist_link (iter.node, offset)->next = x.end ().node;

        ::swap (next, x.next);
    }
    void insert (T *obj, T *prev, T *next)
    {
        __eslist_link (obj, offset)->next = next;
        __eslist_link (prev, offset)->next = obj;
    }
    void push_front (T *obj)
    {
        insert (obj, __eslist_unlink (this, offset), next);
    }
    void pop_front ()
    {
        next = __eslist_link (next, offset)->next;
    }
    void reverse ()
    {
        T *s, *p, *t, *o;

        p = __eslist_unlink (this, offset);
        s = o = next;

        do {
            t = __eslist_link (o, offset)->next;
            __eslist_link (o, offset)->next = p;

            p = o;
            o = t;
        } while (o != s);
    }
};


#endif /* __ELIST_H__ */

/*
  Local Variables:
  mode: c++
  End:
 */
