/* -*-Mode: C++;-*-
 * $Id: dbfs_errs.cc 1.13 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

#include <unistd.h>

static pthread_mutex_t MESSAGE_MUTEX = PTHREAD_MUTEX_INITIALIZER;

static ostream* standard_out = & cout;
static ostream* standard_err = & cerr;

static pid_t PID = 0;
static bool  MESSAGE_WHERE = false;
static bool  MESSAGE_PID   = false;
static bool  MESSAGE_QUIET = false;

MESSAGE_CLASS MCLASS_LOG      ("LOG", standard_err,      false, false, false);
MESSAGE_CLASS MCLASS_INFO     ("INFO", standard_err,     false, false, false);
MESSAGE_CLASS MCLASS_DEBUG    ("DEBUG", standard_err,    false, false, false);
MESSAGE_CLASS MCLASS_WARN     ("WARN",  standard_err,    false, true,  false);
MESSAGE_CLASS MCLASS_ERROR    ("ERROR", standard_err,    false, true,  false);
MESSAGE_CLASS MCLASS_FATAL    ("FATAL", standard_err,    true,  true,  false);
MESSAGE_CLASS MCLASS_NOPREFIX ("NOPREFIX", standard_out, false, true,  true);

MESSAGE_TYPE  ERROR_DB_MESSAGE      ("DB",   MCLASS_ERROR);
MESSAGE_TYPE  ERROR_DBFS_MESSAGE    ("DBFS", MCLASS_ERROR);
MESSAGE_TYPE  ERROR_PROP_MESSAGE    ("PROP", MCLASS_ERROR);
MESSAGE_TYPE  ERROR_SYS_MESSAGE     ("SYS",  MCLASS_ERROR);
MESSAGE_TYPE  ERROR_XDFS_MESSAGE    ("XDFS", MCLASS_ERROR);

MESSAGE_TYPE  WARN_DBFS_MESSAGE     ("DBFS", MCLASS_WARN);
MESSAGE_TYPE  DBFS_FATAL_MESSAGE    ("DBFS", MCLASS_FATAL);

MESSAGE_TYPE  NOPREFIX_MESSAGE      ("NOPREFIX", MCLASS_NOPREFIX);
MESSAGE_TYPE  ERROR_TEST_MESSAGE    ("TEST",     MCLASS_FATAL);


MESSAGE_TYPE  DEBUG_FREELIST_MESSAGE    ("freelist",  MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_PRECOMMIT_MESSAGE   ("precommit", MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_TXN_MESSAGE         ("txn",       MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_SYNC_MESSAGE        ("sync",      MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_MAINT_MESSAGE       ("maint",     MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_XDFS_MESSAGE        ("xdfs",      MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_CURSOR_MESSAGE      ("cursor",    MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_REFS_MESSAGE        ("refs",      MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_LINK_MESSAGE        ("link",      MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_CACHE_MESSAGE       ("cache",     MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_ENLARGE_MESSAGE     ("enlarge",   MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_SHARED_MESSAGE      ("shared",    MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_DBFS_MESSAGE        ("dbfs",      MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_ALLOC_MESSAGE       ("alloc",     MCLASS_DEBUG);
MESSAGE_TYPE  DEBUG_SIGNAL_MESSAGE      ("signal",    MCLASS_DEBUG);

MESSAGE_TYPE  INFO_DBFS_MESSAGE         ("dbfs",      MCLASS_INFO);
MESSAGE_TYPE  INFO_XDFS_MESSAGE         ("xdfs",      MCLASS_INFO);
MESSAGE_TYPE  INFO_TEST_MESSAGE         ("test",      MCLASS_INFO);

char*
dbfs_strerror (int num)
{
    switch ((DBFS_ERRORS) num) {
    case DBFS_MAX_MAJOR:         return "Too many major file descriptors";
    case DBFS_MAX_MINOR:         return "Too many minor file descriptors";
    case DBFS_NOT_IMPL:          return "Function not implemented";
    case DBFS_EXISTS:            return "File exists";
    case DBFS_NEXISTS:           return "File does not exist";
    case DBFS_NOT_READABLE:      return "File not readable";
    case DBFS_VIEW_UNDEF:        return "View undefined";
    case DBFS_STACK_UNDEF:       return "Stack undefined";
    case DBFS_INVAL:             return "Invalid argument";
    case DBFS_LINK_INCONSISTENT: return "Link inconsistency";
    case DBFS_NOSPACE:           return "Descriptor table full";
    case DBFS_NOTFOUND:          return "Key or data not found";
    case DBFS_NOPOSITION:        return "Cursor not positioned";
    case DBFS_NOTDIR:            return "Not a directory";
    case DBFS_NOTSEQ:            return "Not a sequence directory";
    case DBFS_NEEDS_ATTENTION:   return "Database requires operator attention";
    }

    if (num == -1) {
	return "Code needs work (-1)";
    }

    return db_strerror (num);
}

void
dbfs_set_streams (ostream *out,
		  ostream *err)
{
    if (out) {
	standard_out = out;
    }

    if (err) {
	standard_err = err;
    }
}

MESSAGE_TYPE::MESSAGE_TYPE (const char    *kind,
			    MESSAGE_CLASS &mclass)
    : kind   (kind),
      mclass (mclass)

{
    mclass.register_type (this);
}

MESSAGE_CLOSURE::MESSAGE_CLOSURE (MESSAGE_TYPE &type,
				  const char   *file,
				  int           line)
    : _type        (type),
      _file        (file),
      _line        (line),
      _error       (INT_MAX),
      _op_count    (0)
{
}

MESSAGE_CLOSURE::~MESSAGE_CLOSURE ()
{
    g_assert (_type.active);

    if (_error != INT_MAX) {
	_msg << ": ";
	_msg << dbfs_strerror (_error);
    }

    _msg << '\n';
    _msg << '\0';

    string final_msg (_msg.str ());

    _msg.freeze (false);

    MONITOR monitor (& MESSAGE_MUTEX);

    (*_type.mclass.print_str) << final_msg;
    _type.mclass.print_str->flush ();

    if (_type.mclass.is_fatal) {
	abort ();
    }
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (int error)
{
    _error = error;
    return *this;
}

void
MESSAGE_CLOSURE::prefix ()
{
    if (_op_count++ == 0 && ! _type.mclass.no_prefix) {

	if (_type.mclass.is_prefix) {
	    _msg << _type.kind << '-';
	}

	_msg << _type.mclass.kind;

	if (! _type.mclass.is_prefix || MESSAGE_PID || MESSAGE_WHERE) {

	    bool colon = false;

	    _msg << " [";

#define MAYBE_COLON() do { if (colon) { _msg << ':'; } else { colon = true; } } while (0)

	    if (! _type.mclass.is_prefix) {

		MAYBE_COLON ();

		_msg << _type.kind;
	    }

	    if (MESSAGE_WHERE) {

		MAYBE_COLON ();

		_msg << _file << ':' << _line;
	    }

	    if (MESSAGE_PID) {

		MAYBE_COLON ();

		_msg << PID;
	    }

	    _msg << "] ";
	} else {
	    _msg << ": ";
	}
    }
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const string &s)
{
    prefix ();

    _msg << s;

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const char *fmt, ...)
{
    prefix ();

    va_list args;

    va_start (args, fmt);

    _msg.vform (fmt, args);

    va_end (args);

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const MAJOR_DESC &node)
{
    prefix ();

    _msg.form ("[%04x-%04x-maj] ",
	       node.num.area_id ().key (),
	       XNTOHL (node.num.cont_seq ()).key ());

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const SAREA_DESC &node)
{
    prefix ();

    _msg.form ("[%04x-area] ",
	       node.area_id.key ());

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const DBCREF *dbc)
{
    prefix ();

    _msg.form ("[dbc%p] ", dbc->_dbc);

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const MINOR_DESC &node)
{
    prefix ();

    _msg.form ("[%04x-%04x-",
	       node.major->num.area_id ().key (),
	       XNTOHL (node.major->num.cont_seq ()).key ());

    if (node.mkey == DBFS_DEF_MKEY) {
	_msg << "min] ";
    } else {
	_msg << node.mkey << "] ";
    }

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const COMDBT     &comdbt)
{
    Dbt *dbt = comdbt ();

    _msg << "#[";// << dbt->get_size () << ":" << dbt->get_ulen () << ":";

    for (uint i = 0; i < dbt->get_size (); i += 1) {

	_msg.form ("\\%02x", ((guint8*) dbt->get_data ())[i]);
    }

    _msg << "]";

    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const SHFD_DESC  *shfd)
{
    _msg << "[SHFD " << shfd->key << " refs " << shfd->refs << " fd " << shfd->fd << "]";
    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const SHDB_DESC  *shdb)
{
    _msg << "[SHDB " << shdb->key << " refs " << shdb->refs << "]";
    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const SHPG_DESC  *shpg)
{
    _msg << "[SHPG " << shpg->key.fid << " pgno " << shpg->key.pgno << " refs " << shpg->refs << "]";
    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const STXN &txn)
{
    if (txn.active ()) {
	return operator() (txn, txn.dbtxn ());
    } else {
	_msg << "[TXN not active]";
	return *this;
    }
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const STXN &txn, DbTxn *dbtxn)
{
    _msg << "[TXN ";

    _msg.form ("id %x", dbtxn->id ());

    _msg << "]";
    return *this;
}

MESSAGE_CLOSURE&
MESSAGE_CLOSURE::operator () (const DKEY       &dkey)
{
    dkey.ckey_write (_msg, 0);
    return *this;
}

//////////////////////////////////////////////////////////////////////
// MCLASS and CKEY
//////////////////////////////////////////////////////////////////////

void
CKEY::ckey_write (ostream &os, size_t offset) const
{
    for (; offset < size (); offset += 1) {

	char c = _vec[offset];

	if (isgraph (c)) {
	    os << c;
	} else {
	    os.form ("\\%02x", c);
	}
    }
}

ostream&
operator<< (ostream &os, const DKEY& key)
{
    key.ckey_write (os, 0);

    return os;
}

ostream&
operator<< (ostream &os, const MKEY& key)
{
    XSTCK stck;

    key.template get_rec_prefix<XSTCK> (stck);

    switch (stck.key ()) {
    case DBFS_DEF_STACK_VAL:
	break;

    default:

	int ret;
	STACKDEF *sdef;

	if ((ret = dbfs_stackdef_find (stck, & sdef))) {
	    os << "unknown";
	} else {
	    os << sdef->abbrev ();
	}

	break;
    }

    if (key.size () > sizeof (XSTCK)) {
	os << "-";
	key.ckey_write (os, sizeof (XSTCK));
    }

    return os;
}

MESSAGE_CLASS::MESSAGE_CLASS (const char *kind,
			      ostream    *print_str,
			      bool        is_fatal,
			      bool        is_prefix,
			      bool        no_prefix)
    : kind      (kind),
      print_str (print_str),
      is_fatal  (is_fatal),
      is_prefix (is_prefix),
      no_prefix (no_prefix)
{
}

struct hash<string>
{
    size_t operator() (const string& __s) const { hash<const char*> h; return h (__s.c_str ()); }
};

void
MESSAGE_CLASS::register_type (MESSAGE_TYPE *mtype)
{
    type_list.push_front (mtype);

    typedef hash_map<string,MESSAGE_TYPE*> Dmap;
    static Dmap *mclass_map = NULL;

    if (mclass_map == NULL) {

	mclass_map = new Dmap;

	char *denv = getenv ("MCLASS");

	if (denv) {
	    denv = g_strdup (denv);

	    char *tok = strtok (denv, ",");

	    while (tok) {

		if (strcmp (tok, "where") == 0) {
		    MESSAGE_WHERE = true;
		} else if (strcmp (tok, "pid") == 0) {
		    MESSAGE_PID   = true;
		    PID           = getpid ();
		} else if (strcmp (tok, "quiet") == 0) {
		    MESSAGE_QUIET = true;
		} else {
		    string stok (tok);

		    (* mclass_map) [tok] = mtype;
		}

		tok = strtok (NULL, ",");
	    }

	    g_free (denv);
	}
    }

    if ((this == & MCLASS_ERROR) ||
	(this == & MCLASS_WARN) ||
	(this == & MCLASS_FATAL) ||
	(this == & MCLASS_NOPREFIX) ||
	(this == & MCLASS_INFO)) {
	mtype->active = ! MESSAGE_QUIET;
	return;
    }

    Dmap::iterator it = mclass_map->find (mtype->kind);

    if (it != mclass_map->end ()) {
	mtype->active = true;
    }
}

//////////////////////////////////////////////////////////////////////
//      PRINTFS
//////////////////////////////////////////////////////////////////////

int
TXN::printfs (int flags)
{
    int    ret;
    DBCREF areac;

    if ((ret = dbfs ()._area_db.cursor (*this, areac))) {
	PROP_ERROR (ret) ("area_cursor");
	return ret;
    }

    XAREA          area_id;
    RECDBT<XAREA>  area_dbt (area_id);
    DXN            dxn (*this, ret);
    NULLDBT        null;

    while ((ret = areac.move_pos (area_dbt, null, DB_NEXT, DBFS_NORMW)) == 0) {

	SAREA area;

	if ((ret = dxn.get_area (area_id, area))) {
	    PROP_ERROR (ret) ("get_area");
	    return ret;
	}

	if ((ret = area.printfs (flags))) {
	    PROP_ERROR (ret) ("area_printfs");
	    return ret;
	}
    }

    if (ret != DBFS_NOTFOUND) {
	PROP_ERROR (ret) ("area_move_pos");
	return ret;
    }

    return 0;
}

int
SAREA::printfs (int flags) const
{
    int ret;

    NOPREFIX_OUT ("####################################################################");
    NOPREFIX_OUT ("#");
    NOPREFIX_OUT ("# AREA ") (area_id ()) (" NODE SUMMARY");
    NOPREFIX_OUT ("#");

    DBCREF ncurs;

    if ((ret = ref ().container_dbr.cursor (txn (), ncurs))) {
	PROP_ERROR (ret) ("area_node_cursor");
	return ret;
    }

    CKEY    key;
    DKEY    value;

    while ((ret = ncurs.move_pos (key, value, DB_NEXT, DBFS_NORMW)) == 0) {

	MAJORC major;
	NODEC  minor;
	XNUM   majnum;
	XSTCK  stck;
	MKEY   mkey;
	XFID   dbid;
	NODEKEY_TYPE type;

	if ((ret = dbfs_area_key (*this, key, & major, & minor, majnum, stck, mkey, dbid, type))) {
	    PROP_ERROR (ret) ("area_key");
	    return ret;
	}

	switch (type) {
	case NODEKEY_SHORT: {
	    //NOPREFIX_OUT ("SHORT KEY: ") (key);
	    break;
	}

	case NODEKEY_INVLNK: {
	    NOPREFIX_OUT (major) ("inverse link from ") (dbid) (" key ") (value);
	    break;
	}

	case NODEKEY_MINOR: {

	    //NOPREFIX_OUT (minor);
	    switch (minor.type ()) {
	    case FT_Reflink: {
		MAJORC linkto;
		if ((ret = minor.read_reflink (linkto, 0))) {
		    PROP_ERROR (ret) ("read_reflink");
		    return ret;
		}
		NOPREFIX_OUT (minor) ("reflink to ") (linkto);
		break;
	    }
	    case FT_Arealink: {
		SAREA linkto;
		if ((ret = minor.read_arealink (linkto, 0))) {
		    PROP_ERROR (ret) ("read_arealink");
		    return ret;
		}
		NOPREFIX_OUT (minor) ("arealink to ") (linkto);
		break;
	    }
	    case FT_ShortSeg:
	    case FT_LongSeg:
	    case FT_ViewSeg:
		NOPREFIX_OUT (minor) ("%s length ", dbfs_type_to_string (minor.type ())) (minor.length ());
		break;
	    case FT_DirSeq:
	    case FT_DirHash:
	    case FT_DirBtree:
		NOPREFIX_OUT (minor) ("%s id ", dbfs_type_to_string (minor.type ())) (minor.cont_id ().file_id ());
		break;
	    case FT_NotPresent:
	    default:
		g_assert_not_reached ();
	    }

	    break;
	}

	case NODEKEY_MAJOR: {

	    NOPREFIX_OUT (major) ("refcount ") (major.nlinks ());
	    break;
	}
	default:
	    g_assert_not_reached ();
	}
    }

    return 0;
}

int
dbfs_area_key (const SAREA  &area,
	       const CKEY   &key,
	       MAJORC       *majorp,
	       NODEC        *minorp,
	       XNUM         &number,
	       XSTCK        &stck,
	       MKEY         &mkey,
	       XFID         &inv_dbid,
	       NODEKEY_TYPE &type)
{
    static const size_t __seqno_offset = (sizeof (XSEQNO));
    static const size_t __stack_offset = (sizeof (XSEQNO) + sizeof (XSTCK));

    size_t key_size = key.size ();

    g_assert (key_size == __seqno_offset ||
	      key_size >= __stack_offset);

    int    ret;
    XSEQNO cseq;

    key.get_rec_prefix (cseq);

    XSEQNO cseq_native (XNTOHL (cseq));

    if ((cseq_native.key () & DBFS_CONT_ID_MASK) != 0) {
	type = NODEKEY_SHORT;
	return 0;
    }

    number = XNUM (area.area_id (), cseq);

    DXN dxn (area.txn (), ret);

    if (majorp != NULL && (ret = dxn.get_major (number, *majorp))) {
	PROP_ERROR (ret) ("get_major");
	return ret;
    }

    if (key_size == __seqno_offset) {
	type = NODEKEY_MAJOR;
	return 0;
    }

    key.template get_rec_at<XSTCK> (__seqno_offset, stck);

    if (stck == DBFS_INV_STACK) {
	g_assert (key_size - __stack_offset == sizeof (XFID));

	key.template get_rec_at<XFID> (__stack_offset, inv_dbid);

	type = NODEKEY_INVLNK;
	return 0;
    }

    mkey.set_buf (key.data () + __seqno_offset, key_size - __seqno_offset);

    if (minorp != NULL && (ret = dxn.get_minor (*majorp, mkey, *minorp))) {
	PROP_ERROR (ret) ("get_minor");
	return ret;
    }

    type = NODEKEY_MINOR;
    return 0;
}
