 /*
 * file      : conv.c
 * project   : xcfa
 * with      : Gtk-2
 *
 * copyright : (C) 2003 - 2010 by Claude Bulin
 *
 * xcfa - GTK+ implementation of the GNU shell command
 * GNU General Public License
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * OLD ADRESS:
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * NEW ADRESS:
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * 
 */



#include <gtk/gtk.h>
#include "callbacks.h"

#include <pthread.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "file.h"
#include "poche.h"
#include "options.h"
#include "utils.h"
#include "info_song.h"
#include "conv.h"
#include "win_control.h"
#include "logs.h"
#include "prg_init.h"
#include "config_user.h"


/*
*---------------------------------------------------------------------------
* EXTERN
*---------------------------------------------------------------------------
*/
extern int kill (pid_t pid, int sig);
extern float roundf(float x);


/*
*---------------------------------------------------------------------------
* VARIABLES
*---------------------------------------------------------------------------
*/

#define CONV_MAX_CARS 1024
#define CONV_MAX_VALUE_PERCENT 0.978597

CONV conv = { FALSE }; /* conv.BoolUsed = FALSE */


/*
*---------------------------------------------------------------------------
* STOP THREAD AND TIMEOUT
*---------------------------------------------------------------------------
*/

void conv_inc_rip_completed (void)
{
	if (conv.rip_completed < conv.total_rip) {
		conv.rip_completed ++;
		conv.total_percent = (conv.rip_completed +
					conv.encode_completed +
					conv.extract_percent +
					conv.conversion_percent) /
					(conv.total_rip + conv.total_convert);
	}
}
void conv_inc_encode_completed (void)
{
	if (conv.encode_completed < conv.total_convert) {
		conv.encode_completed ++;
		// conv.total_percent = conv.encode_completed / conv.total_convert;
		conv.total_percent = (conv.rip_completed +
					conv.encode_completed +
					conv.extract_percent +
					conv.conversion_percent) /
					(conv.total_rip + conv.total_convert);
	}
}

void conv_stop_conversion (void)
{
	gint Ret_Kill;

	/* PRINT_FUNC_LF(); */

	conv.bool_stop = TRUE;

	if (conv.code_fork_conv > 0) {
		g_print ("kill (%d, SIGKILL)\n", conv.code_fork_conv);
		Ret_Kill = kill (conv.code_fork_conv, SIGKILL);
		if (Ret_Kill != 0) {
			g_print ("--------------------------\n");
			g_print ("ERREUR Ret_Kill = %d\n", Ret_Kill);
			g_print ("-------------------------\n");
		}
	}
	conv.code_fork_conv = -1;

	if (conv.code_fork_extract > 0) {
		g_print ("kill (%d, SIGKILL)\n", conv.code_fork_extract);
		Ret_Kill = kill (conv.code_fork_extract, SIGKILL);
		if (Ret_Kill != 0) {
			g_print ("--------------------------\n");
			g_print ("ERREUR Ret_Kill = %d\n", Ret_Kill);
			g_print ("-------------------------\n");
		}
	}
	conv.code_fork_extract = -1;
}

/*
*---------------------------------------------------------------------------
* WINDOWS DELETE
*---------------------------------------------------------------------------
*/

gboolean on_wind_extract_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	/* PRINT_FUNC_LF(); */
	conv_stop_conversion ();
	return TRUE;
}
gboolean on_wind_extract_destroy_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	/* PRINT_FUNC_LF(); */
	conv_stop_conversion ();
	return TRUE;
}

/*
*---------------------------------------------------------------------------
* WINDOWS REALIZE
*---------------------------------------------------------------------------
*/

void on_label_titre_extraction_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_label_extract = widget;
	gtk_label_set_use_underline (GTK_LABEL(widget), FALSE);
}
void on_progressbar_extraction_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_progressbar_extract = widget;
}
void on_progressbar_conversion_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_progressbar_conversion = widget;
}
void on_progressbar_total_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_progressbar_total = widget;
}
void on_label_titre_conversion_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_label_conversion = widget;
	gtk_label_set_use_underline (GTK_LABEL(widget), FALSE);
}
void on_label_total_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_label_total = widget;
}
void on_label_info_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_label_info_wind = widget;
}
void on_textview_expander_wind_extract_realize (GtkWidget *widget, gpointer user_data)
{
	conv.Adr_textview = widget;
}


/*
*---------------------------------------------------------------------------
* WINDOWS BUTTON STOP CLICKED
*---------------------------------------------------------------------------
*/

void on_button_stop_wind_extract_clicked (GtkButton *button, gpointer user_data)
{
	/* PRINT_FUNC_LF(); */
	conv_stop_conversion ();
}

/*
*---------------------------------------------------------------------------
* WINDOWS CREATE FOR EXTRACTIONS-CONVERSIONS
*---------------------------------------------------------------------------
*/

void conv_close_window (void)
{
	NEW_wincontrol_realize_window (&conv.Adr_wind, "wind_extract", &conv.Bool, WIND_IS_SHOW, WIND_USE_STRUCT, WIND_IS_MODAL, WIND_IS_TRANSIENT);
}

void conv_create_window_extract (gchar *title_wind, gboolean BoolModeExpert)
{
	gchar	*Str = NULL;
	
	NEW_wincontrol_realize_window (&conv.Adr_wind, "wind_extract", &conv.Bool, WIND_IS_SHOW, WIND_USE_STRUCT, WIND_IS_MODAL, WIND_IS_TRANSIENT);
	
	Str = g_strdup_printf ("<span font_desc=\"sans 15\"><span color=\"black\"><b>%s</b></span></span>", title_wind);
	gtk_label_set_label (GTK_LABEL (conv.Adr_label_info_wind), Str);
	g_free (Str);
	Str = NULL;
	
	gtk_widget_show (GTK_WIDGET (XCFA_GET_OBJECT("hbox_extraction_wind_extract")));
	if (BoolModeExpert == TRUE)
		gtk_widget_show (GTK_WIDGET (XCFA_GET_OBJECT("hbox_expander_extraction_wind_extract")));
	else	gtk_widget_hide (GTK_WIDGET (XCFA_GET_OBJECT("hbox_expander_extraction_wind_extract")));
	gtk_widget_show (GTK_WIDGET (XCFA_GET_OBJECT("hbox_conversion_wind_extract")));
	gtk_window_set_title (GTK_WINDOW (conv.Adr_wind), title_wind);
	gtk_expander_set_expanded (GTK_EXPANDER (GTK_WIDGET (XCFA_GET_OBJECT("expander_wind_extract"))), BoolModeExpert);
}

/*
*---------------------------------------------------------------------------
* WINDOWS CREATE
*---------------------------------------------------------------------------
*/

void conv_create_window (gchar *title_wind)
{
	gchar	*Str = NULL;
	
	NEW_wincontrol_realize_window (&conv.Adr_wind, "wind_extract", &conv.Bool, WIND_IS_SHOW, WIND_USE_STRUCT, WIND_IS_MODAL, WIND_IS_TRANSIENT);
	
	Str = g_strdup_printf ("<span font_desc=\"sans 14\"><span color=\"black\"><b>%s</b></span></span>", title_wind);
	gtk_label_set_label (GTK_LABEL (conv.Adr_label_info_wind), Str);
	g_free (Str);
	Str = NULL;
	gtk_widget_hide (GTK_WIDGET (XCFA_GET_OBJECT("hbox_extraction_wind_extract")));
	gtk_widget_hide (GTK_WIDGET (XCFA_GET_OBJECT("hbox_expander_extraction_wind_extract")));
	gtk_widget_show (GTK_WIDGET (XCFA_GET_OBJECT("hbox_conversion_wind_extract")));
	gtk_window_set_title (GTK_WINDOW (conv.Adr_wind), title_wind);
}

/*
*---------------------------------------------------------------------------
* PROGRESSBAR
*---------------------------------------------------------------------------
*/

void conv_activate_progressBar (GtkWidget *Adr_Progressbar)
{
	/* PRINT_FUNC_LF(); */
	gtk_widget_show (GTK_WIDGET (Adr_Progressbar));
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (Adr_Progressbar), 0.0);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (Adr_Progressbar), _("Waiting ..."));
}
void conv_set_data_progressBar (GtkWidget *Adr_Progressbar, double Etat)
{
	gchar     *Total= NULL;
	
	/* PRINT_FUNC_LF(); */

	if (Etat <= 0.0) Etat = 0.0;
	if (Etat >= 1.0) Etat = 1.0;

	if (Adr_Progressbar == conv.Adr_progressbar_extract) {
		switch (Config_User.Extract_cd_with) {
		case EXTRACT_WITH_CDPARANOIA :
			Total = g_strdup_printf ("cdparanoia : cd -> wav : %d%%", (gint)(Etat * 100));
			break;
		case EXTRACT_WITH_CDPARANOIA_EXPERT :
			Total = g_strdup_printf ("cdparanoia (expert) : cd -> wav : %d%%", (gint)(Etat * 100));
			break;
		case EXTRACT_WITH_CDDA2WAV :
			// Total = g_strdup_printf ("%s : cd -> wav : %d%%", PrgInit.name_cdda2wav_icedax, (gint)(Etat * 100));
			Total = g_strdup_printf ("%s : cd -> wav : %d%%", prginit_get_name (NMR_icedax), (gint)(Etat * 100));
			break;
		}
	}
	else if (Adr_Progressbar == conv.Adr_progressbar_total) {
		Total = g_strdup_printf ("Total : %d%%", (gint)(Etat * 100));
	}
	else {
		switch (conv.type_conv) {
		case NONE_CONV               : Total = g_strdup (_("Waiting ...")); break;
		case FLAC_FLAC_TO_WAV        : Total = g_strdup_printf ("flac : flac -> wav : %d%%", (int)(Etat*100)); break;
		case FLAC_WAV_TO_FLAC        : Total = g_strdup_printf ("flac : wav -> flac : %d%%", (int)(Etat*100)); break;
		case LAME_WAV_TO_MP3         : Total = g_strdup_printf ("lame : wav -> mp3 : %d%%", (int)(Etat*100)); break;
		case LAME_FLAC_TO_MP3        : Total = g_strdup_printf ("lame : flac -> mp3 : %d%%", (int)(Etat*100)); break;
		case OGGENC_WAV_TO_OGG       : Total = g_strdup_printf ("oggenc : wav -> ogg : %d%%", (int)(Etat*100)); break;
		case OGGENC_FLAC_TO_OGG      : Total = g_strdup_printf ("oggenc : flac -> ogg : %d%%", (int)(Etat*100)); break;
		case OGG123_OGG_TO_WAV       : Total = g_strdup_printf ("ogg123 : ogg -> wav : %d%%", (int)(Etat*100)); break;
		case MPG321_MP3_TO_WAV       : Total = g_strdup_printf ("mpg321 : mp3 -> wav : %d%%", (int)(Etat*100)); break;
		case SOX_WAV_TO_WAV          : Total = g_strdup_printf ("sox : wav -> wav : %d%%", (int)(Etat*100)); break;
		case SHORTEN_SHN_TO_WAV      : Total = g_strdup_printf ("shorten : shn -> wav : %d%%", (int)(Etat*100)); break;
		case FAAD_M4A_TO_WAV         : Total = g_strdup_printf ("faad : m4a -> wav : %d%%", (int)(Etat*100)); break;
		case FAAC_WAV_TO_M4A         : Total = g_strdup_printf ("faac : wav -> m4a : %d%%", (int)(Etat*100)); break;
		case MPLAYER_WAV_TO_WAV      : Total = g_strdup_printf ("mplayer : wav -> wav : %d%%", (int)(Etat*100)); break;
		case MPLAYER_WMA_TO_WAV      : Total = g_strdup_printf ("mplayer : wma -> wav : %d%%", (int)(Etat*100)); break;
		case MPLAYER_RM_TO_WAV       : Total = g_strdup_printf ("mplayer : rm -> wav : %d%%", (int)(Etat*100)); break;
		case MPLAYER_DTS_TO_WAV      : Total = g_strdup_printf ("mplayer : dts -> wav : %d%%", (int)(Etat*100)); break;
		case MPLAYER_AIFF_TO_WAV     : Total = g_strdup_printf ("mplayer : aif -> wav : %d%%", (int)(Etat*100)); break;
		case COPY_FILE               : Total = g_strdup_printf ("Copy : %d%%", (int)(Etat*100)); break;
		case NORMALISE_CALCUL        : Total = g_strdup_printf ("Normalise : Calcul niveau : %d%%", (int)(Etat*100)); break;
		case NORMALISE_APPLIQUE      : Total = g_strdup_printf ("Normalise : Applique niveau : %d%%", (int)(Etat*100)); break;
		
		// case MPPDEC_MPC_TO_WAV       : Total = g_strdup_printf ("mppdec : mpc -> wav : %d%%", (int)(Etat*100)); break;
		case MPPDEC_MPC_TO_WAV       : Total = g_strdup_printf ("%s : mpc -> wav : %d%%", prginit_get_name (NMR_musepack_tools_mppdec), (int)(Etat*100)); break;
		
		// case MPPENC_WAV_TO_MPC       : Total = g_strdup_printf ("mppenc : wav -> mpc : %d%%", (int)(Etat*100)); break;
		case MPPENC_WAV_TO_MPC       : Total = g_strdup_printf ("%s : wav -> mpc : %d%%", prginit_get_name (NMR_musepack_tools_mppenc), (int)(Etat*100)); break;
		
		case MAC_APE_TO_WAV          : Total = g_strdup_printf ("mac : ape -> wav : %d%%", (int)(Etat*100)); break;
		case MAC_WAV_TO_APE          : Total = g_strdup_printf ("mac : wav -> ape : %d%%", (int)(Etat*100)); break;
		case WAVPACK_WAV_TO_WAVPACK  : Total = g_strdup_printf ("wavpack : wav -> wavpack : %d%%", (int)(Etat*100)); break;
		case WVUNPACK_WAVPACK_TO_WAV : Total = g_strdup_printf ("wvunpack : wavpack -> wav : %d%%", (int)(Etat*100)); break;
		case MPLAYER_M4A_TO_WAV      : Total = g_strdup_printf ("mplayer : m4a -> wav : %d%%", (int)(Etat*100)); break;
		case MPLAYER_OGG_TO_WAV      : Total = g_strdup_printf ("mplayer : ogg -> wav : %d%%", (int)(Etat*100)); break;
		
		case MPLAYER_AUDIO_TO_WAV    :
		case CDPARANOIA_CD_TO_WAV    :
		case CDPARANOIA_CD_TO_WAV_EXPERT :
		case CDPARANOIA_CD_TO_WAV_EXPERT_SEGMENT :
		case CDDA2WAV_CD_TO_WAV      :
		case REPLAYGAIN              :
		case LSDVD                   :
		case NORMALISE_EXEC          : 
		case NORMALISE_GET_LEVEL     : 
		case SPLIT                   : 
		case CDDB_TOOL               :
		case ICEDAX		     : break;
		case AACPLUSENC_WAV_TO_AAC   : Total = g_strdup_printf ("aacplusenc : wav -> aac : %d%%", (int)(Etat*100)); break;
		}
	}
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (Adr_Progressbar), Etat);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (Adr_Progressbar), Total);
	g_free (Total);
	Total= NULL;
}

/*
*---------------------------------------------------------------------------
* LABEL PROGRESSBAR
*---------------------------------------------------------------------------
*/

void conv_set_label_extract (void)
{
	gchar *Str_Label = NULL;

	/* PRINT_FUNC_LF(); */
	if (!conv.name_extract) {
		Str_Label = g_strdup (_("<b>[Extraction]</b>"));
	}
	else {
		if (g_utf8_validate(conv.name_extract, -1, NULL) == TRUE) {
			Str_Label = g_strdup_printf ("<b>%s</b>", g_markup_escape_text (conv.name_extract, -1));
		}
		else {
			gchar *Dummy = NULL;
			Dummy = g_strdup_printf ("<b>%s</b>", conv.name_extract);
			Str_Label = utils_convert_string (Dummy, TRUE);
			g_free (Dummy);
			Dummy = NULL;
		}
		
	}
	gtk_label_set_markup_with_mnemonic (GTK_LABEL (conv.Adr_label_extract), Str_Label);
	g_free (Str_Label);
	Str_Label = NULL;
	gtk_label_set_use_underline (GTK_LABEL (conv.Adr_label_extract), FALSE);
}
void conv_set_label_conversion (void)
{
	gchar *Str_Label = NULL;

	/* PRINT_FUNC_LF(); */
	switch (conv.type_conv) {
	case NONE_CONV               : Str_Label = g_strdup (_("<b>[Conversion]</b> ")); break;
	case FLAC_FLAC_TO_WAV        :
	case FLAC_WAV_TO_FLAC        :
	case LAME_WAV_TO_MP3         :
	case LAME_FLAC_TO_MP3        :
	case OGGENC_WAV_TO_OGG       :
	case OGGENC_FLAC_TO_OGG      :
	case OGG123_OGG_TO_WAV       :
	case MPG321_MP3_TO_WAV       :
	case SOX_WAV_TO_WAV          :
	case SHORTEN_SHN_TO_WAV      :
	case FAAD_M4A_TO_WAV         :
	case FAAC_WAV_TO_M4A         :
	case MPLAYER_WMA_TO_WAV      :
	case MPLAYER_RM_TO_WAV       :
	case MPLAYER_DTS_TO_WAV      :
	case MPLAYER_AIFF_TO_WAV     :
	case MPLAYER_WAV_TO_WAV      :
	case COPY_FILE               :
	case NORMALISE_CALCUL        :
	case NORMALISE_APPLIQUE      :
	case MPPDEC_MPC_TO_WAV       :
	case MPPENC_WAV_TO_MPC       :
	case MAC_APE_TO_WAV          :
	case MAC_WAV_TO_APE          :
	case WAVPACK_WAV_TO_WAVPACK  :
	case WVUNPACK_WAVPACK_TO_WAV :
	case MPLAYER_M4A_TO_WAV      :	
	case MPLAYER_OGG_TO_WAV      :	
	case AACPLUSENC_WAV_TO_AAC   : 
					if (g_utf8_validate(conv.name_file, -1, NULL) == TRUE) {
						Str_Label = g_strdup_printf ("<b>%s   </b>", g_markup_escape_text (conv.name_file, -1));
					}
					else {
						gchar *Dummy = NULL;
						Dummy = g_strdup_printf ("<b>%s</b>", conv.name_file);
						Str_Label = utils_convert_string (Dummy, TRUE);
						g_free (Dummy);
						Dummy = NULL;
					}
					break;
	case MPLAYER_AUDIO_TO_WAV    :
	case CDPARANOIA_CD_TO_WAV    :
	case CDPARANOIA_CD_TO_WAV_EXPERT :
	case CDPARANOIA_CD_TO_WAV_EXPERT_SEGMENT :
	case CDDA2WAV_CD_TO_WAV      :
	case REPLAYGAIN              :
	case LSDVD                   :
	case NORMALISE_EXEC          :  
	case NORMALISE_GET_LEVEL     :  
	case SPLIT                   :  
	case CDDB_TOOL               :
	case ICEDAX                  :  break;
	}

	gtk_label_set_label (GTK_LABEL (conv.Adr_label_conversion), Str_Label);
	g_free (Str_Label);
	Str_Label = NULL;
	gtk_label_set_use_underline (GTK_LABEL (conv.Adr_label_conversion), FALSE);
}
void conv_set_label_total (TYPE_OPER type)
{
	/* PRINT_FUNC_LF(); */
return;
	if (type == _EXTRACT_) {
		gchar *Str_Label = NULL;
		gint   begin = conv.rip_completed + conv.encode_completed;
		gint   all = conv.total_rip + conv.total_convert;

		if (begin < all) begin ++;
		Str_Label = g_strdup_printf (_("<b>Total : %d/%d</b>"), begin, all);
		gtk_label_set_label (GTK_LABEL (conv.Adr_label_total), Str_Label);
		g_free (Str_Label);
		Str_Label = NULL;
	}
	else if (type == _CONV_) {
		gchar *Str_Label = NULL;
		gint   Begin = conv.encode_completed;

		if (Begin < conv.total_convert) Begin ++;
		Str_Label = g_strdup_printf (_("<b>Total : %d/%d</b>"), Begin, conv.total_convert);
		gtk_label_set_label (GTK_LABEL (conv.Adr_label_total), Str_Label);
		g_free (Str_Label);
		Str_Label = NULL;
	}
}

/*
*---------------------------------------------------------------------------
* SIGNAL ET EXTRACTION
*---------------------------------------------------------------------------
*/

void conv_sigchld_extract (gint signum)
{
        gint status;
        wait(&status);
	/* PRINT_FUNC_LF(); */
        /* if there are still children waiting
        *  re-install the signal handler
	*/
	conv.signal_numchildren_extract --;
        if (conv.signal_numchildren_extract > 0)
        {
                /* re-install the signal handler */
                signal (SIGCHLD, conv_sigchld_extract);
        }
}

/*
*---------------------------------------------------------------------------
* SIGNAL ET CONVERSIONS
*---------------------------------------------------------------------------
*/

void conv_sigchld_convert (gint signum)
{
	gint status;
        wait(&status);
	/* PRINT_FUNC_LF(); */
        /* if there are still children waiting
        *  re-install the signal handler
	*/
	conv.signal_numchildren_conv --;
        if (conv.signal_numchildren_conv > 0)
        {
                /* re-install the signal handler */
                signal (SIGCHLD, conv_sigchld_convert);
        }
}

/*
*---------------------------------------------------------------------------
* EXECUTE EXTERNAL
*---------------------------------------------------------------------------
*/
int conv_call_exec (ARGS_TO_EXEC *args, pid_t *p, gint p_output)
{
	gchar **ptr = (gchar **)args;

	conv.signal_numchildren_conv = 0;
	if (pipe (conv.tube_conv) != 0)
	{
		fprintf (stderr, "error: pipe\n");
		exit (1);
	}
	if ((*p = fork()) == 0)
	{
		dup2 (conv.tube_conv [ 1 ], p_output);
		close (conv.tube_conv [ 1 ]);
		execvp ((gchar *)*(ptr+0), ptr);
		fprintf (stderr, "error: exec");
		exit (2);
	}
	conv.signal_numchildren_conv ++;
	signal (SIGCHLD, conv_sigchld_convert);
	close (conv.tube_conv [ 1 ]);
	return (conv.tube_conv [ 0 ]);
}

int conv_extract_call_exec (ARGS_TO_EXEC *args, pid_t *p, gint p_output)
{
	gchar **ptr = (gchar **)args;

	conv.signal_numchildren_extract = 0;
	if (pipe (conv.tube_extract) != 0)
	{
		fprintf (stderr, "error: pipe\n");
		exit (1);
	}
	if ((*p = fork()) == 0)
	{
		dup2 (conv.tube_extract [1 ], p_output);
		close (conv.tube_extract [ 1 ]);
		execvp ((gchar *)*(ptr+0), ptr);
		fprintf (stderr, "error: exec");
		exit (2);
	}
	conv.signal_numchildren_extract ++;
	signal (SIGCHLD, conv_sigchld_extract);
	close(conv.tube_extract [ 1 ]);
	return (conv.tube_extract [ 0 ]);
}

gboolean conv_copy_src_to_dest (gchar *filesrc, gchar *filedest)
{
#define SIZE_COPY  4096
	FILE     *fpsrc = NULL;
	FILE     *fpdest = NULL;
	gint      nmr_fpsrc;
	gint      nmr_fpdest;
	gulong    cpt = 0;
	gulong    blk;
	size_t    size;
	// gint      modulo;
	gchar     buffer [ (SIZE_COPY + 256) ];
	
	if (infosong_file_is_reg (filesrc) == FALSE) {
		PRINT_FUNC_LF();
		g_print ("Le fichier [%s] n'existe pas !\n", filesrc);
		g_print ("\tRETOUR\n");
		return (FALSE);
	}
	
	g_print ("conv_copy_src_to_dest:\n\t%s\n\t%s\n", filesrc, filedest);

	// PRINT_FUNC_LF();

	size = infosong_get_size_file (filesrc);
	if (size == 0) {
		PRINT_FUNC_LF();
		g_print ("Le fichier [%s] a une taille de 0 octet!\n", filesrc);
		g_print ("\tRETOUR\n");
		return (FALSE);
	}
	
	blk = size / SIZE_COPY;

	if (blk <= 0) {
		blk = 1;
		PRINT_FUNC_LF();
		g_print ("\tRECTIF DE BLK -> %lu\n", blk);
	}

	fpsrc = fopen (filesrc, "r");
	if (!fpsrc) {
		g_print ("ERREUR: src = fopen (%s);\n", filesrc);
		return (FALSE);
	}
	fpdest = fopen (filedest, "wb+");
	if (!fpdest) {
		g_print ("ERREUR: dest = fopen (%s);\n", filedest);
		fclose (fpsrc);
		return (FALSE);
	}
	// flockfile (fpsrc);
	
	nmr_fpsrc  = fileno (fpsrc);
	nmr_fpdest = fileno (fpdest);
	
	cpt = 0;
	// modulo = 0;
	while (TRUE) {
		if ((size = fread (&buffer[0], sizeof(gchar), SIZE_COPY, fpsrc)) > 0) {

			fwrite (&buffer[0], sizeof(gchar), size, fpdest);
			// fflush (fpdest);
		} else {
			break;
		}
		if (feof (fpsrc)) {
			break;
		}
		// if (modulo ++ > 50) {
			// fflush (fpdest);
			// fsync (nmr_fpsrc);
			// fsync (nmr_fpdest);
		// 	modulo = 0;
		// }
		cpt ++;

		conv.conversion_percent = (double)cpt / (double)blk;
		if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.0;
		conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
		conv.bool_percent_conv = TRUE;
		
	}
	// funlockfile (fpsrc);
	fflush (fpdest);
	fclose (fpdest);
	fclose (fpsrc);
	
	conv_inc_encode_completed ();
	conv.conversion_percent = 0.0;
	/*conv.bool_reset_conv = TRUE;*/
#undef SIZE_COPY
	return (TRUE);
}
void conv_product_name_for_label_extract (gchar *NameFile)
{
	gchar *Ptr = NULL;

		if (conv.name_extract) {
			g_free (conv.name_extract);
			conv.name_extract = NULL;
		}
		conv.name_extract = g_strdup (NameFile);
		if ((Ptr = strrchr (conv.name_extract, '/'))) {
			Ptr ++;
			strcpy (conv.name_extract, Ptr);
			Ptr = conv.name_extract;
			if ((gint)strlen (Ptr) > MAX_LIMITE_NAME) {
				Ptr += MAX_LIMITE_NAME;
				*Ptr ++ = '.';
				*Ptr ++ = '.';
				*Ptr ++ = '.';
				*Ptr ++ = '\0';
			}
		}
}
void conv_product_name_for_label (gchar *NameFile)
{
	gchar *Ptr = NULL;

		if (conv.name_file) {
			g_free (conv.name_file);
			conv.name_file = NULL;
		}
		conv.name_file = g_strdup (NameFile);
		if ((Ptr = strrchr (conv.name_file, '/'))) {
			Ptr ++;
			strcpy (conv.name_file, Ptr);
			Ptr = conv.name_file;
			if ((gint)strlen (Ptr) > MAX_LIMITE_NAME) {
				Ptr += MAX_LIMITE_NAME;
				*Ptr ++ = '.';
				*Ptr ++ = '.';
				*Ptr ++ = '.';
				*Ptr ++ = '\0';
			}
		}
}
gchar *conv_remove_name_for_label_extract (TYPE_OPER typename)
{
		g_free (conv.name_extract);
		conv.name_extract = NULL;
	return (NULL);
}
gchar *conv_remove_name_for_label (TYPE_OPER typename)
{
		g_free (conv.name_file);
		conv.name_file = NULL;
	return (NULL);
}

void conv_set_struct_not_used (void)
{
	conv.BoolUsed = FALSE;
}
gboolean conv_struct_is_used (void)
{
	return (conv.BoolUsed);
}

void conv_reset_struct (void)
{
	/*PRINT_FUNC_LF();*/
	
	conv.BoolUsed = TRUE;
	
	conv.typeoper = _CONV_;

	/*
	conv.Adr_wind = NULL;
	conv.Adr_label_extract = NULL;
	conv.Adr_progressbar_extract = NULL;
	conv.Adr_label_conversion = NULL;
	conv.Adr_progressbar_conversion = NULL;
	conv.Adr_label_total = NULL;
	conv.Adr_progressbar_total = NULL;
	conv.Adr_textview = NULL;
	conv.BoolPutTextview = FALSE;
	conv.ListPutTextview = NULL;
	*/
	
	if (conv.name_extract) g_free (conv.name_extract);
	conv.name_extract = NULL;
	if (conv.name_file) g_free (conv.name_file);
	conv.name_file = NULL;
	conv.type_conv = NONE_CONV;
	conv.extract_percent = 0.0;
	conv.total_rip = 0;
	conv.rip_completed = 0;
	conv.conversion_percent = 0.0;
	conv.total_convert = 0;
	conv.encode_completed = 0;
	conv.total_percent = 0.0;
	conv.bool_percent_conv = FALSE;
	conv.bool_percent_extract = FALSE;
	conv.bool_reset_conv = FALSE;
	conv.bool_reset_extract = FALSE;
	conv.bool_set_log_wvunpack = FALSE;
	conv.bool_set_log_wavpack = FALSE;
	
	conv.BoolPercentCdparanoia = FALSE;
	conv.BoolPercentCdda2wav = FALSE;
	
	conv.BoolPercentWav = FALSE;
	conv.BoolPercentFlac = FALSE;
	conv.BoolPercentApe = FALSE;
	conv.BoolPercentWavPack = FALSE;
	conv.BoolPercentOgg = FALSE;
	conv.BoolPercentM4a = FALSE;
	conv.BoolPercentMpc = FALSE;
	conv.BoolPercentMp3 = FALSE;
		
	conv.bool_stop = FALSE;

	if (conv.TmpRep) g_free (conv.TmpRep);
	conv.TmpRep = NULL;

	conv.code_fork_conv = -1;
	conv.code_fork_extract = -1;
	conv.handler_timeout_conv = 0;
	conv.bool_thread_conv = FALSE;
	conv.bool_thread_extract = FALSE;
	conv.signal_numchildren_conv = 0;
	conv.signal_numchildren_extract = 0;

	conv.Bool_artist_tag   = optionstags_get_checkbutton_artist_tag ();
	conv.Bool_title_tag    = optionstags_get_checkbutton_title_tag ();
	conv.Bool_album_tag    = optionstags_get_checkbutton_album_tag ();
	conv.Bool_numerate_tag = optionstags_get_checkbutton_numerate_tag ();
	conv.Bool_genre_tag    = optionstags_get_checkbutton_genre_tag ();
	conv.Bool_year_tag     = optionstags_get_checkbutton_year_tag ();
	conv.Bool_comment_tag  = optionstags_get_checkbutton_comment_tag ();
	
	conv.Bool_MAJ_select = FALSE;
	
	conv.ArgExtract = NULL;
	conv.ArgConv = NULL;
}

void conv_set_type_oper (TYPE_OPER typeoper)
{
	conv.typeoper = typeoper;
}

gboolean conv_with_flac_ARGS (ARGS_TO_EXEC *args)
{
	gint		 pos = 0;
	gint		 fd;
	gint		 size;
	gint		 sector = 0;
	gchar		 buf [ CONV_MAX_CARS + 10 ];
	gchar		*Ptr = NULL;
	gboolean	 BoolNoErrorRet = TRUE;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		/*
		01.flac: 27% complete
		*/
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';
		
		if ((Ptr = strstr (buf, "ERROR"))) {
			/*
			coldplay_test.flac: ERROR, MD5 signature mismatch
			coldplay_test.wav: ERROR while decoding metadata
				state = FLAC__STREAM_DECODER_END_OF_STREAM
			*/
			BoolNoErrorRet = FALSE;
			break;
		}
		
		/* exemple du contenu de buf
		*
		a.wav: 5% complete, ratio=0,534
		a.wav: 41% complete, ratio=0,489
		a.wav: 91% complete, ratio=0,455
		*/
		if ((Ptr = strchr (buf, ':'))) {
			Ptr ++;
			Ptr ++;
			sscanf(Ptr, "%d", &sector);

			if (conv.typeoper == _CONV_) {
				conv.conversion_percent = (double)((sector * 0.1) / 10);
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
				conv.bool_reset_conv = TRUE;
			}
			else {
				/* conv.typeoper == _EXTRACT_ */
				conv.conversion_percent = (double)((sector * 0.1) / 10);
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);
				conv.bool_percent_conv = TRUE;
				conv.BoolPercentFlac = TRUE;
				conv.bool_reset_conv = TRUE;
			}
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);

	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentFlac = TRUE;
	
	return (BoolNoErrorRet);
}

void conv_with_lame_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gint   size;
	gint   sector;
	gint   end;
	gchar  buf [ CONV_MAX_CARS + 10 ];

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		sector = -1;
		end = -1;

		if (sscanf(buf, "%d/%d", &sector, &end) == 2) {
		
			if (conv.typeoper == _CONV_) {
				conv.conversion_percent = (double)sector/end;
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.encode_completed + (double)sector/end) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
			else {
				/* conv.typeoper == _EXTRACT_ */
				conv.conversion_percent = (double)sector/end;
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);
				conv.bool_percent_conv = TRUE;
				conv.BoolPercentMp3 = TRUE;
			}
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentMp3 = TRUE;
}

void conv_with_oggenc_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gint   size;
	gint   sector;
	gint   end;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	
	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	conv.conversion_percent = 0.0;

	do {
		size = -1;
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while (buf[pos] != '\b' && buf[pos] != '\r' && buf[pos] != '\n' && size > 0);
		pos ++;
		buf[ pos ] = '\0';

		sector = -1;
		end = -1;

		/*
		[  3,0%] [ 0m17s remaining]
		*/
		if (sscanf(buf, "\t[%d.%d%%]", &sector, &end) != 2) {
			sscanf(buf, "\t[%d,%d%%]", &sector, &end);
		}

		if (sector > -1 && end > -1)
		{
			if (conv.typeoper == _CONV_) {
				conv.conversion_percent = (double)(sector + (end * 0.1)) / 100;
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.encode_completed + (double)(sector + (end * 0.1)) / 100) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
			else {
				/* conv.typeoper == _EXTRACT_ */
				conv.conversion_percent = (double)(sector + (end * 0.1)) / 100;
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);
				conv.bool_percent_conv = TRUE;
				conv.BoolPercentOgg = TRUE;
			}
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentOgg = TRUE;
}

void conv_with_ogg123_ARGS (ARGS_TO_EXEC *args)
{
	gint   fd;
	gint   pos = 0;
	gint   size;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   Dummy;
	gint   Min = -1, Sec = -1, Cent = -1;
	gint   Code = -1, TT_Sector = -1;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		size = 0;
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		/*
		Time: 00:39,17 [03:57,86] of 04:37,03  (190,3 kbit/s)  Output Buffer 100,0%
		*/
		Code = -1;

		if (TT_Sector == -1) {
			sscanf(buf, "Time: %d:%d,%d [%d:%d,%d] of %d:%d,%d",
				&Dummy, &Dummy, &Dummy, &Dummy, &Dummy, &Dummy, &Min, &Sec, &Cent);
			if (Min > -1 && Sec > -1 && Cent > -1) {
				TT_Sector = ((Min * 60) + Sec);
			}
		} else {
			sscanf(buf, "Time: %d:%d,%d", &Min, &Sec, &Cent);
			if (Min > -1 && Sec > -1 && Cent > -1) {

				conv.conversion_percent = (double)((Min * 60) + Sec) / TT_Sector;
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.encode_completed + (double)((Min * 60) + Sec) / TT_Sector) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
				conv.BoolPercentOgg = TRUE;
			}
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentOgg = TRUE;
}

void conv_with_mpg321_ARGS (ARGS_TO_EXEC *args)
{
	gint   fd;
	gint   pos = 0;
	gint   size = -1;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gchar  type [ 20 ];
	gint   Code, TT_Sector = -1;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		/*
		Frame#   792 [12190], Time: 00:20.68 [05:18.45]
		sscanf(buf, "##: %d [%d]%s @ %d", &code, type, &sector);
		*/
		Code = -1;
		if (TT_Sector == -1) {
			sscanf(buf, "%s %d [%d]", type, &Code, &TT_Sector);
		} else {
			sscanf(buf, "%s %d", type, &Code);
			if (Code > -1 && Code <= TT_Sector) {
				conv.conversion_percent = (double)Code / TT_Sector;
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.encode_completed + (double)Code / TT_Sector) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

GList *conv_with_glist_sox_get_param (GList *p_list, gchar *filedest)
{
	GList	*PtrList = g_list_first (p_list);
	GList	*list = NULL;
	
	list = g_list_append (list, g_strdup ("nice"));
	list = g_list_append (list, g_strdup ("-n"));
	list = g_list_append (list, g_strdup_printf ("%d", Config_User.Nice));
	list = g_list_append (list, g_strdup ("sox"));
	while (PtrList) {
		if ((gchar *)PtrList->data != NULL) {
			list = g_list_append (list, g_strdup ((gchar *)PtrList->data));
		}
		PtrList = g_list_next (PtrList);
	}
	
	// -r 44100 -c 2 -S -t .wav -o ./all.wav
	
	// list = g_list_append (list, g_strdup ("-ql"));	// resample
	list = g_list_append (list, g_strdup ("-t"));		// type file
	list = g_list_append (list, g_strdup (".wav"));		// 
	list = g_list_append (list, g_strdup ("-S"));		// --show-progress
	list = g_list_append (list, g_strdup ("-r"));		// --rate 
	list = g_list_append (list, g_strdup ("44100"));	// 
	list = g_list_append (list, g_strdup ("-c"));		// --channels
	list = g_list_append (list, g_strdup ("2"));
	// list = g_list_append (list, g_strdup ("-w"));	// 
	list = g_list_append (list, g_strdup ("-o"));		// 
	list = g_list_append (list, g_strdup (filedest));	// 
	
	return ((GList *)list);
}

GList *conv_with_sox_float_get_param (gchar *filesrc, gchar *filedest)
{
	GList *list = NULL;
	
	list = g_list_append (list, g_strdup ("nice"));
	list = g_list_append (list, g_strdup ("-n"));
	list = g_list_append (list, g_strdup_printf ("%d", Config_User.Nice));
	list = g_list_append (list, g_strdup ("sox"));
	
	list = g_list_append (list, g_strdup (filesrc));
	list = g_list_append (list, g_strdup ("-t"));
	list = g_list_append (list, g_strdup ("wavpcm"));
	list = g_list_append (list, g_strdup ("-S"));
	list = g_list_append (list, g_strdup ("-f"));
	list = g_list_append (list, g_strdup ("-o"));
	list = g_list_append (list, g_strdup (filedest));
	
	return ((GList *)list);
}
	
GList *conv_with_sox_get_param (gchar *filesrc, gchar *filedest, gchar *frequence, gchar *voie, gchar *bits)
{
	GList *list = NULL;
	
	list = g_list_append (list, g_strdup ("nice"));
	list = g_list_append (list, g_strdup ("-n"));
	list = g_list_append (list, g_strdup_printf ("%d", Config_User.Nice));
	list = g_list_append (list, g_strdup ("sox"));
	
	// list = g_list_append (list, g_strdup ("-ql"));
	
	list = g_list_append (list, g_strdup ("-t"));
	list = g_list_append (list, g_strdup (".wav"));
	list = g_list_append (list, g_strdup (filesrc));
	list = g_list_append (list, g_strdup ("-S"));
	list = g_list_append (list, g_strdup ("-r"));
	list = g_list_append (list, g_strdup (frequence));
	list = g_list_append (list, g_strdup ("-c"));
	list = g_list_append (list, g_strdup (voie));
	
	/*
	switch (atoi (bits)) {
		case 8 :  list = g_list_append (list, g_strdup ("-b")); break;
		case 16 : list = g_list_append (list, g_strdup ("-w")); break;
		case 32 : list = g_list_append (list, g_strdup ("-l")); break;
		case 64 : list = g_list_append (list, g_strdup ("-d")); break;
		default : list = g_list_append (list, g_strdup ("-b")); break;
	}
	*/
	// AJOUT DU 24 BITS
	// Sat, 21 Nov 2009 23:24:17 +0100
	//
	// -1	8 bits
	// -2	16 bits
	// -3	24 bits
	// -4	32 bits
	// -8	64 bits
	switch (atoi (bits)) {
		case 8 :  list = g_list_append (list, g_strdup ("-1")); break;
		case 16 : list = g_list_append (list, g_strdup ("-2")); break;
		case 24 : list = g_list_append (list, g_strdup ("-3")); break;
		case 32 : list = g_list_append (list, g_strdup ("-4")); break;
		case 64 : list = g_list_append (list, g_strdup ("-8")); break;
		default : list = g_list_append (list, g_strdup ("-2")); break;
	}
	list = g_list_append (list, g_strdup ("-o"));
	list = g_list_append (list, g_strdup (filedest));
	
	return ((GList *)list);
}

void conv_with_sox_ARGS (ARGS_TO_EXEC *args)
{
	gint   fd;
	gint   size;
	gint   pos = 0;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   cent;
	gint   min;
	gint   sec;
	gint   TT_Sector = -1;
	gint   dummy;
	gboolean BoolTst = FALSE;
	
	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';
		
		/*
		Time: 04:31.62 [00:00.00] of 04:31.62 ( 100.0%) Output Buffer:  11.98M
		*/
		if (strstr (buf, "Time: ")) BoolTst = TRUE;
		if (BoolTst && (buf[0] == '\0' || strstr (buf, "Input File     :"))) {
			BoolTst = FALSE;
			if (conv.encode_completed < conv.total_convert) {
				// conv.encode_completed ++;
				// g_print("conv.encode_completed = %d -> %d\n", conv.encode_completed -1, conv.encode_completed);
				conv.bool_percent_conv = TRUE;
				conv.bool_reset_conv = TRUE;
				TT_Sector = -1;
			}
		}
		
		if (TT_Sector == -1) {
			min = sec = cent = -1;
			sscanf(buf, "Time: %d:%d.%d [%d:%d.%d] of %d:%d.%d",
				&dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &min, &sec, &cent);
			if (min > -1 && sec > -1 && cent > -1) {
				TT_Sector = ((min * 60) + sec);
				/*g_print ("SEARCH BEGIN %d %d %d\n", min, sec, cent);*/
			}
		}
		else /*if (strstr (buf, "Time: "))*/ {
			min = sec = cent = -1;
			sscanf(buf, "Time: %d:%d.%d", &min, &sec, &cent);
			if (min > -1 && sec > -1 && cent > -1) {
				conv.conversion_percent = (double)((min * 60) + sec) / TT_Sector;
				/*if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;*/
				conv.total_percent = (conv.encode_completed + (double)((min * 60) + sec) / TT_Sector) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
		}
		
		buf [ 0 ] = '\0';
		buf [ 1 ] = '\0';
		buf [ 2 ] = '\0';
		
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_percent_conv = TRUE;
	conv.bool_reset_conv = TRUE;
}

void conv_with_shorten_ARGS (ARGS_TO_EXEC *args)
{
	gint   fd;
	gint   size;
	gint   pos = 0;
	gchar  buf [ CONV_MAX_CARS + 10 ];

	/* PRINT_FUNC_LF(); */

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		conv.conversion_percent = 1.0;
		conv.total_percent = 1.0;
		conv.bool_percent_conv = TRUE;
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

void conv_with_faad_ARGS (ARGS_TO_EXEC *args)
{
	gint   fd;
	gint   size;
	gint   pos = 0;
	gchar  buf [ CONV_MAX_CARS + 10 ];

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	/*
	39% decoding 05.m4a.
	52% decoding 05.m4a.
	60% decoding 05.m4a.
	67% decoding 05.m4a.
	77% decoding 05.m4a.
	84% decoding 05.m4a.
	*/
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
			if (strstr(buf,"Audio: no sound")) {
				conv_stop_conversion ();
				break;
			}
			
		} while (!conv.bool_stop && (buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		
		if (buf[pos] != '\n') {
			pos ++;
			buf[pos++] = '\n';
			buf[pos] = '\0';
		}

	} while (!conv.bool_stop && (size > 0));

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

void conv_with_faac_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gint   size;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   dum = 0;
	gint   sector = 0;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	/*
	   53/9518  (  0%)|  108.2  |    0.1/18.0   |   12.31x | 17.9
	  250/9518  (  2%)|  117.7  |    0.5/17.5   |   12.62x | 17.1
	  450/9518  (  4%)|  122.8  |    0.8/17.9   |   12.38x | 17.0
	  650/9518  (  6%)|  124.8  |    1.2/17.9   |   12.37x | 16.6
	 2100/9518  ( 22%)|  120.5  |    3.8/17.2   |   12.89x | 13.4
	 9518/9518  (100%)|  124.0  |   17.2/17.2   |   12.86x | 0.0
	*/
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		sscanf(buf, "%d/%d (%d)", &dum, &dum, &sector);
		if (sector == 0) continue;

		if (conv.typeoper == _CONV_) {
			conv.conversion_percent = (double)((sector * 0.1) / 10);
				
			if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
			conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
			conv.bool_percent_conv = TRUE;
		}
		else {
			/* conv.typeoper == _EXTRACT_ */
			conv.conversion_percent = (double)((sector * 0.1) / 10);
				
			if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
			conv.total_percent = (conv.rip_completed +
						conv.encode_completed +
						conv.extract_percent +
						conv.conversion_percent) /
						(conv.total_rip + conv.total_convert);
			conv.bool_percent_conv = TRUE;
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

gboolean conv_with_mplayer_ARGS (ARGS_TO_EXEC *args)
{
	gint      pos = 0;
	gint      fd;
	gint      size;
	gchar     buf [ CONV_MAX_CARS + 10 ];
	gboolean  BoolRet = TRUE;
	gchar    *Ptr = NULL;
	gboolean  BoolFinfTotal = FALSE;
	gint      TimeTotal = 0;
	gint      Secs = 0;
	gint      Hundred = 0;
	gboolean  BoolErreurMplayer = FALSE;
	
	/* PRINT_FUNC_LF(); */

	conv.conversion_percent = 0.0;
	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDOUT_FILENO);

	/* Ne pas stopper mplayer */
	/* conv.code_fork_conv = -1; */
	
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
			if (strstr(buf,"Audio: no sound")) {
				/*conv_stop_conversion ();*/
				BoolRet = FALSE;
				BoolErreurMplayer = TRUE;
				break;
			}
			
		} while (!conv.bool_stop && (buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		
		if (BoolErreurMplayer == TRUE) {
			break;
		}
		
		/*if (strcmp (buf, "Starting playback...") == 0) {
			g_print ("Starting playback... TROUVé \n");
		}
		g_print ("%s", buf);*/
		
		/*
		A:   7.3 (07.2) of 276.0 (04:36.0)  0.6%
		*/
		if (BoolFinfTotal == FALSE) {
			if ((Ptr = strstr(buf,"A:")) != NULL) {
				if ((Ptr = strstr(buf,"of")) != NULL) {
					while (*Ptr != ' ') Ptr ++;
					while (*Ptr == ' ') Ptr ++;
					TimeTotal = atoi (Ptr);
					BoolFinfTotal = TRUE;
				}
			}
		}
		else if ((Ptr = strstr(buf,"A:")) != NULL) {
			Ptr += 2;
			while (*Ptr == ' ') Ptr++;
			Secs = atoi (Ptr);
			/*g_print ("2> Ptr=%s", Ptr);*/
			
			while (*Ptr != '.') Ptr++;
			if (*Ptr == '.') Ptr++;
			/*g_print ("2> Ptr=%s", Ptr);*/
			
			Hundred = atoi (Ptr);
			conv.TimeMplayer = (gdouble)Secs + ((gdouble)Hundred / 10.0);
			
			/*
			conv.conversion_percent = (gdouble)Secs / (gdouble)TimeTotal;
			
			conv.total_percent = (conv.rip_completed +
						conv.encode_completed +
						conv.extract_percent +
						conv.conversion_percent) /
						(conv.total_rip + conv.total_convert);
			conv.bool_percent_conv = TRUE;
			*/
			
			/* -- */
			if (conv.typeoper == _CONV_) {
				conv.conversion_percent = (gdouble)Secs / (gdouble)TimeTotal;
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
			else {
				conv.conversion_percent = (gdouble)Secs / (gdouble)TimeTotal;
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);
				conv.bool_percent_conv = TRUE;
			}
			/* -- */
			
			
			/*g_print ("PERCENT = %f\n", conv.conversion_percent);*/
		}
		
		if (buf[pos] != '\n') {
			pos ++;
			buf[pos++] = '\n';
			buf[pos] = '\0';
		}

	} while (!conv.bool_stop && (size > 0));
	close(fd);

	if (BoolErreurMplayer == TRUE) {
		g_print("\n");
		g_print("BoolErreurMplayer = TRUE\n");
		g_print("KILL Process mplayer = ");
		if ((kill (conv.code_fork_conv, SIGKILL) != 0))
			g_print ("ERREUR\n");
		else	g_print ("OK\n");
		g_print("\n");
	}

	conv.code_fork_conv = -1;
	conv.bool_percent_conv = TRUE;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	
	return (BoolRet);
}

void conv_with_mppdec_ARGS (ARGS_TO_EXEC *args)
{
	gint   fd;
	gint   size;
	gint   pos = 0;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   percent;
	gchar *ptr = NULL;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		/*
		0:00.02/    3:51.29 decoded ( 0.0%)
		1:40.64/    3:51.29 decoded (43.5%)
		*/
		/*sscanf(buf, "%d", &percent);*/

		if ((ptr = strstr( buf, "decoded ("))) {
			/*g_print ("-->%s\n", ptr);*/
			ptr += 9;
			/*g_print ("-->%s\n", ptr);*/
			sscanf(ptr, "%d", &percent);
			/*g_print ("---->%d\n", percent);*/
		}
		else {
			continue;
		}
		/*
		pos = sscanf(buf, "%d:%d.%d/\t%d:%d:.%d decoded (%d", &dummy,&dummy,&dummy,&dummy,&dummy,&dummy, &percent);
		g_print (">pos     = %d\n", pos);
		if (pos != 7) {
			continue;
		}
		g_print (">pos     = %d\n", pos);
		g_print (">percent = %d\n", percent);
		*/
		if (conv.typeoper == _CONV_) {
			conv.conversion_percent = (double)((percent * 0.1) / 10);
				
			if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
			conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
			conv.bool_percent_conv = TRUE;
		}
		else {
			/* conv.typeoper == _EXTRACT_ */
			conv.conversion_percent = (double)((percent * 0.1) / 10);
				
			if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
			conv.total_percent = (conv.rip_completed +
						conv.encode_completed +
						conv.extract_percent +
						conv.conversion_percent) /
						(conv.total_rip + conv.total_convert);
			conv.bool_percent_conv = TRUE;
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;

}

GList *conv_with_normalise_get_param (gchar *filesrc, gboolean bool_fix, gint value_fix)
{
	GList *list = NULL;
	
	list = g_list_append (list, g_strdup ("nice"));
	list = g_list_append (list, g_strdup ("-n"));
	/*list = g_list_append (list, g_strdup_printf ("%d", options_get_val_nice ()));*/
	list = g_list_append (list, g_strdup_printf ("%d", Config_User.Nice));
	// list = g_list_append (list, g_strdup (PrgInit.name_normalize));
	list = g_list_append (list, g_strdup (prginit_get_name (NMR_normalize)));
	
	if (bool_fix == TRUE) {
		/*
		!------------------------------!
		!  EXEMPLE - FIX               !
		!------------------------------!
		if (dd < 10) g_print ("dd = %0.1f\n",(double)dd / 10.0);
		if (dd >= 10) g_print ("dd = %0.02f\n",(double)dd / 100.0);
		normalise-audio
		--amplitude [0,1 .. 0,99]
		-T 0,2
		name_file

		if (value_fix < 10) 	  ptr_value = g_strdup_printf ("%0.1f",(double)value_fix / 10.0);
		else if (value_fix >= 10) ptr_value = g_strdup_printf ("%0.02f",(double)value_fix / 100.0);

		args [ pos++ ] = "-a";
		args [ pos++ ] = ptr_value;
		args [ pos++ ] = "-T";
		args [ pos++ ] = "0,2";
		args [ pos++ ] = filesrc;
		args [ pos++ ] = NULL;
		*/
		list = g_list_append (list, g_strdup ("-a"));
		if (value_fix < 10) 	  list = g_list_append (list, g_strdup_printf ("%0.1f",(double)value_fix / 10.0));
		else if (value_fix >= 10) list = g_list_append (list, g_strdup_printf ("%0.02f",(double)value_fix / 100.0));
	}
	else {
		/*
		!------------------------------!
		!  EXEMPLE - PEAK              !
		!------------------------------!
		normalise-audio
		--peak
		-T 0,2
		name_file
		*/
		/*
		args [ pos++ ] = "--peak";
		args [ pos++ ] = "-T";
		args [ pos++ ] = "0,2";
		args [ pos++ ] = filesrc;
		args [ pos++ ] = NULL;
		*/
		list = g_list_append (list, g_strdup ("--peak"));
	}
	
	/*
	list = g_list_append (list, g_strdup ("-T"));
	list = g_list_append (list, g_strdup ("0,2"));
	*/
	list = g_list_append (list, g_strdup (filesrc));
	
	return ((GList *)list);
}

#include <math.h>

void conv_with_normalise_get_PEAK_RMS_GROUP_ARGS (ARGS_TO_EXEC *args)
{
	gchar     *ptr = NULL;
	gchar      buf [ CONV_MAX_CARS + 10 ];
	gint       fd;
	gint       size;
	gint       pos = 0;
	gdouble    value = 0.0;
	gdouble    dmin = 0.0;
	gdouble    dmax = 0.0;
	gboolean   BoolPass = TRUE;
	
	fd = conv_call_exec (args, &conv.code_fork_conv, STDOUT_FILENO);
	do
	{
		pos = -1;
		do
		{
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos] = '\0';
		
		if ((ptr = strstr (&buf[0], "dBFS"))) {
			while (*ptr != ' ') ptr ++;
			while (*ptr == ' ') ptr ++;
			
			/*g_print ("buf = %s\n", buf);*/
			value = atof(ptr);
			if (BoolPass == TRUE) {
				dmin = dmax = value;
				BoolPass = FALSE;
			}
			if (dmin > value) dmin = value;
			if (dmax < value) dmax = value;
		}

	} while (size != 0);

	/*
	g_print ("value = %f, dmin = %f,  dmax = %f\n", value, dmin, dmax);
	g_print ("dmax - dmin - 0.1 = %f\n", (dmax - dmin) - 0.1);
	g_print ("dmin - dmax - 0.1 = %f\n", (dmin - dmax) - 0.1);
	conv.value_PEAK_RMS_GROUP_ARGS = (dmin - dmax) - 0.1;
	conv.value_PEAK_RMS_GROUP_ARGS = dmax - dmin - 0.1;
	g_print ("conv.value_PEAK_RMS_GROUP_ARGS = %f\n", conv.value_PEAK_RMS_GROUP_ARGS);
	g_print ("****\n");
	*/
	g_print ("\nSUR LES TRES BONS CONSEILS DE @Dzef  ;-)\n");
	g_print ("\tdmin = %f\n", dmin);
	g_print ("\tdmax = %f\n", dmax);
	g_print ("\t0.0 - dmax(%f) - 0.1 = %f\n", dmax, 0.0 - dmax - 0.1);
	conv.value_PEAK_RMS_GROUP_ARGS = 0.0 - dmax - 0.1;
	g_print ("\tconv.value_PEAK_RMS_GROUP_ARGS = %f\n\n", conv.value_PEAK_RMS_GROUP_ARGS);
	
	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

void conv_with_normalise_ARGS (ARGS_TO_EXEC *args)
{
	gint       end = 0;
	gchar     *ptr = NULL;
	gchar      buf [ CONV_MAX_CARS + 10 ];
	gint       fd;
	gint       size = 0;
	gint       pos = 0;
	gboolean   bool_pass_2 = TRUE;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDOUT_FILENO);

	do
	{
		/*
		$ normalize-audio --peak "/home/cat/Musique/CD/PEAK_Charles Aznavour/01.wav" --
		Computing levels...
		 01.wav            100% done, ETA 00:00:00 (batch 100% done, ETA 00:00:00) 
		Applying adjustment of 0,82dB to /home/cat/Musique/CD/PEAK_Charles Aznavour/01.wav...
		 01.wav            100% done, ETA 00:00:00 (batch 100% done, ETA 00:00:00) 
		*/

		pos = -1;
		do
		{
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos] = '\0';
		
		// g_print("buf = %s\n", buf);
		
		if ((ptr = strstr (&buf[0], "batch"))) {
			
			ptr += 6;
			while (*ptr == ' ') ptr ++;
			
			if (sscanf(ptr, "%d",&end) == 1) {

				if (bool_pass_2 == TRUE) {
					if (end >= 100) bool_pass_2 = FALSE;
					conv.conversion_percent = (double)((end * 0.1) / 10);
				
					if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
					conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
					conv.bool_percent_conv = TRUE;
				}
				else {
					conv.type_conv = NORMALISE_APPLIQUE;
					conv.bool_reset_conv = TRUE;
					conv.conversion_percent = (double)((end * 0.1) / 10);
				
					if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
					conv.bool_percent_conv = TRUE;
				}
			}
		}
	} while (size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

void conv_with_mppenc_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gint   size;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   percent = 0;
	gint   old_percent = 0;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		sscanf(buf, "%d", &percent);
		if (percent <= 0) continue;
		if (percent >= old_percent) {
			if (conv.typeoper == _CONV_) {
				conv.conversion_percent = (double)((percent * 0.1) / 10);
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
			else {
				/* conv.typeoper == _EXTRACT_ */
				conv.conversion_percent = (double)((percent * 0.1) / 10);
				
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);
				conv.bool_percent_conv = TRUE;
				conv.BoolPercentMpc = TRUE;
			}
			old_percent = percent;
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentMpc = TRUE;
}

void conv_with_mac_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gint   size;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   percent;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	/*
	mac ./01.wav ./02.ape -c2000
	--- Monkey's Audio Console Front End (v 3.99) (c) Matthew T. Ashland ---
	Compressing (normal)...
	Progress: 8.4% (8.6 seconds remaining, 0.8 seconds total)
	Progress: 13.3% (7.6 seconds remaining, 1.2 seconds total)
	Progress: 19.3% (6.9 seconds remaining, 1.6 seconds total)
	Progress: 24.1% (6.4 seconds remaining, 2.0 seconds total)
	Progress: 26.6% (6.1 seconds remaining, 2.2 seconds total)
	Progress: 29.0% (5.9 seconds remaining, 2.4 seconds total)
	Progress: 53.1% (3.8 seconds remaining, 4.3 seconds total)
	Progress: 79.7% (1.7 seconds remaining, 6.6 seconds total)
	Progress: 100.0% (0.0 seconds remaining, 8.2 seconds total)
	Success...
	*/
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		sscanf(buf, "Progress: %d)", &percent);
		if (percent == 0) continue;

		if (conv.typeoper == _CONV_) {
			conv.conversion_percent = (double)((percent * 0.1) / 10);
				
			if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
			conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
			conv.bool_percent_conv = TRUE;
		}
		else {
			/* conv.typeoper == _EXTRACT_ */
			conv.conversion_percent = (double)((percent * 0.1) / 10);
				
			if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
			conv.total_percent = (conv.rip_completed +
						conv.encode_completed +
						conv.extract_percent +
						conv.conversion_percent) /
						(conv.total_rip + conv.total_convert);
			conv.bool_percent_conv = TRUE;
			conv.BoolPercentApe = TRUE;
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentApe = TRUE;
}

void conv_with_wavpack_ARGS (ARGS_TO_EXEC *args)
{
	gint       pos = 0;
	gboolean   BoolPass = FALSE;
	
	gint       fd;
	gint       size;
	gchar      buf [ CONV_MAX_CARS + 10 ];
	gchar     *ptr = NULL;
	gint       percent;
	
	gint	   PassMess = 0;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

conv.conversion_percent = 1.0;
conv.bool_percent_conv = TRUE;
	do
	{
		pos = -1;
		do
		{
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos] = '\0';

		/*
		fish@debian:~/tmp$ wavpack -y -f -j0 ./1.wav

		 WAVPACK  Hybrid Lossless Audio Compressor  Linux Version 4.32  2006-04-05
		 Copyright (c) 1998 - 2005 Conifer Software.  All Rights Reserved.

		creating ./1.wv,  10% done...
		 14% done...
		 18% done...
		 32% done...
		 38% done...
		 41% done...
		 49% done...
		 53% done...
		created ./1.wv in 4.25 secs (lossless, 20.89%)
		*/

		if ((ptr = strstr (&buf[0], "done"))) {
			ptr -= 5;
			while (*ptr == ' ') ptr ++;

			if (sscanf(ptr, "%d",&percent) == 1) {

				if (percent == 0) continue;

				if (conv.typeoper == _CONV_) {
					conv.conversion_percent = (double)((percent * 0.1) / 10);
				
					if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
					conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
					conv.bool_percent_conv = TRUE;
					BoolPass = TRUE;
				}
				else {
					/* conv.typeoper == _EXTRACT_ */
					conv.conversion_percent = (double)((percent * 0.1) / 10);
				
					if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
					conv.total_percent = (conv.rip_completed +
								conv.encode_completed +
								conv.extract_percent +
								conv.conversion_percent) /
								(conv.total_rip + conv.total_convert);
					conv.bool_percent_conv = TRUE;
					conv.BoolPercentWavPack = TRUE;
				}
			}
		}
		else if ((ptr = strstr (&buf[0], "original md5 signature:"))) {
			
			PassMess = 1;
			logs_write_datas ("\t");
			logs_writeln_datas (buf);
		}
		else if ((ptr = strstr (&buf[0], "created"))) {
			
			PassMess = 2;
			logs_write_datas ("\t");
			logs_writeln_datas (buf);
		}
		else if (PassMess == 0) {
			PassMess = 3;
		}

	} while (size != 0);
	
	switch (PassMess) {
	case 0 :
	case 1 :
	case 2 :
		break;
	case 3 :
		logs_writeln_datas (_("Wav to Wavpack"));
		break;
	}
	
	if (BoolPass == FALSE) {
		
		percent = 100;
		conv.conversion_percent = 1.0;
		conv.total_percent = ((conv.encode_completed) + (double)conv.conversion_percent) / conv.total_convert;
		conv.bool_percent_conv = TRUE;
	}

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
	conv.BoolPercentWavPack = TRUE;
}

void conv_with_wvunpack_ARGS (ARGS_TO_EXEC *args)
{
	gint       fd;
	gint       size;
	gint       pos = 0;
	gchar      buf [ CONV_MAX_CARS + 10 ];
	gchar     *ptr = NULL;
	gint       percent;

	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);

	do
	{
		pos = -1;
		do
		{
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos] = '\0';

		/*
		fish@debian:~/Musique/aaa$ wvunpack -y ./1.wv

		 WVUNPACK  Hybrid Lossless Audio Decompressor  Linux Version 4.32  2006-04-05
		 Copyright (c) 1998 - 2005 Conifer Software.  All Rights Reserved.

		restoring ./1.wav,  49% done...
		 61% done...
		 82% done...
		restored ./1.wav in 3.21 secs (lossless, 20.89%)
		*/
		if ((ptr = strstr (&buf[0], "done"))) {
			ptr -= 5;
			while (*ptr == ' ') ptr ++;

			if (sscanf(ptr, "%d", &percent) == 1) {

				if (percent == 0) continue;

				if (conv.typeoper == _CONV_) {
					conv.conversion_percent = (double)((percent * 0.1) / 10);
				
					if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
					conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
					conv.bool_percent_conv = TRUE;
				}
				else {
					/* conv.typeoper == _EXTRACT_ */
					conv.conversion_percent = (double)((percent * 0.1) / 10);
				
					if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				
					conv.total_percent = (conv.rip_completed +
								conv.encode_completed +
								conv.extract_percent +
								conv.conversion_percent) /
								(conv.total_rip + conv.total_convert);
					conv.bool_percent_conv = TRUE;
				}
			}
		}
		/*
		fish@debian:~/Musique/aaa/new$ wvunpack -y -m ./20.wv

		 WVUNPACK  Hybrid Lossless Audio Decompressor  Linux Version 4.32  2006-04-05
		 Copyright (c) 1998 - 2005 Conifer Software.  All Rights Reserved.

		restoring ./20.wav,  34% done...

		original md5:  2b74aed5a5d62d1c3b0e6b96d5147cc9
		unpacked md5:  2b74aed5a5d62d1c3b0e6b96d5147cc9
		restored ./20.wav in 17.23 secs (lossless, 26.45%)
		*/
		else if ((ptr = strstr (&buf[0], "original md5:"))) {
			logs_writeln_datas (buf);
		}
		else if ((ptr = strstr (&buf[0], "unpacked md5:"))) {
			logs_writeln_datas (buf);
		}
		else if ((ptr = strstr (&buf[0], "restored"))) {
			logs_writeln_datas (buf);
		}
	} while (size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

void conv_with_cdparanoia_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   size = 1;
	gint   start;
	gint   end;
	gint   code;
	gchar  type[50];
	gint   sector;

	/* PRINT_FUNC_LF(); */

	conv.bool_percent_extract = FALSE;

	fd = conv_extract_call_exec (args, &conv.code_fork_extract, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		if ((buf[0] == 'R') && (buf[1] == 'i')) {
			sscanf(buf, "Ripping from sector %d", &start);
		}
		else if (buf[0] == '\t') {
			sscanf(buf, "\t to sector %d", &end);
		}
		else if (buf[0] == '#') {
			sscanf(buf, "##: %d %s @ %d", &code, type, &sector);
			sector /= 1176;
			if (strncmp("[wrote]", type, 7) == 0) {
				conv.extract_percent = (double)(sector-start) / (end-start);
				
				if (conv.extract_percent >= CONV_MAX_VALUE_PERCENT) conv.extract_percent = 1.000000;
				
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);

				conv.bool_percent_extract = TRUE;
				conv.BoolPercentCdparanoia = TRUE;
			}
		}
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_extract = -1;
	conv.extract_percent = 0.0;
	conv.bool_reset_extract = TRUE;
	conv.BoolPercentCdparanoia = TRUE;
}

void conv_with_cdparanoia_mode_expert_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gint   size = 1;
	/*
	gint   start;
	gint   end;
	gint   code;
	gchar  type[50];
	gint   sector;
	*/

	/* PRINT_FUNC_LF(); */

	conv.bool_percent_extract = FALSE;

	fd = conv_extract_call_exec (args, &conv.code_fork_extract, STDERR_FILENO);
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		pos ++;
		buf[ pos ] = '\0';

		/*
		if ((buf[0] == 'R') && (buf[1] == 'i')) {
			sscanf(buf, "Ripping from sector %d", &start);
		}
		else if (buf[0] == '\t') {
			sscanf(buf, "\t to sector %d", &end);
		}
		else if (buf[0] == '#') {
			sscanf(buf, "##: %d %s @ %d", &code, type, &sector);
			sector /= 1176;
			if (strncmp("[wrote]", type, 7) == 0) {
				conv.extract_percent = (double)(sector-start) / (end-start);
				
				if (conv.extract_percent >= CONV_MAX_VALUE_PERCENT) conv.extract_percent = 1.000000;
				
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);

				conv.bool_percent_extract = TRUE;
				conv.BoolPercentCdparanoia = TRUE;
			}
		}
		*/
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_extract = -1;
	conv.extract_percent = 0.0;
	conv.bool_reset_extract = TRUE;
	conv.BoolPercentCdparanoia = TRUE;
}

void conv_with_cdda2wav_ARGS (ARGS_TO_EXEC *args)
{
	gint   pos = 0;
	gint   fd;
	gchar  buf [ CONV_MAX_CARS + 10 ];
	gchar *ptr = NULL;
	gint   size = 1;
	gint   percent;

	conv.bool_percent_extract = FALSE;

	fd = conv_extract_call_exec (args, &conv.code_fork_extract, STDERR_FILENO);
	do {
		/*
		 93%
		 96%
		 99%
		100%  track  7 recorded successfully
		*/
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while (((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n')) && (size > 0));
		if (buf[ pos ] != '\n') {
			pos ++;
			buf[ pos ] = '\0';
		}
		pos ++;
		buf[ pos ] = '\0';

		if ((ptr = strchr (buf, '%')) == NULL) continue;
		ptr --;
		if (ptr > buf && (*ptr >= '0' && *ptr <= '9')) ptr --;
		if (ptr > buf && (*ptr >= '0' && *ptr <= '9')) ptr --;
		if (ptr > buf && (*ptr >= '0' && *ptr <= '9')) ptr --;
		if (*ptr == ' ') ptr ++;

		percent = atol (ptr);

		conv.extract_percent = (double)((percent * 0.1) / 10);
				
		if (conv.extract_percent >= CONV_MAX_VALUE_PERCENT) conv.extract_percent = 1.000000;
				
		conv.total_percent = (conv.rip_completed +
					conv.encode_completed +
					conv.extract_percent +
					conv.conversion_percent) /
					(conv.total_rip + conv.total_convert);
		conv.bool_percent_extract = TRUE;
		conv.BoolPercentCdda2wav = TRUE;
		
	} while (!conv.bool_stop && size != 0);

	close(fd);
	conv.code_fork_extract = -1;
	conv.extract_percent = 0.0;
	conv.bool_reset_extract = TRUE;
	conv.BoolPercentCdda2wav = TRUE;
}

void conv_with_replaygain_ARGS (ARGS_TO_EXEC *args)
{
	gint            pos = 0;
	gint            size = 0;
	gchar           buf[CONV_MAX_CARS + 10];
	gint            fd;
	
	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);
	
	do
	{
		size = -1;
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			conv.bool_percent_conv = TRUE;
			
		} while (buf[pos] != '\b' && buf[pos] != '\r' && buf[pos] != '\n' && size > 0);
		pos ++;
		buf[ pos ] = '\0';

	} while (size != 0);

	close(fd);
}

GList *conv_with_lsdvd_get_param (gchar *pathdev)
{
	GList *list = NULL;
	
	if (pathdev == NULL) {
		PRINT("pathdev == NULL");
	}
	list = g_list_append (list, g_strdup ("nice"));
	list = g_list_append (list, g_strdup ("-n"));
	/*list = g_list_append (list, g_strdup_printf ("%d", options_get_val_nice ()));*/
	list = g_list_append (list, g_strdup_printf ("%d", Config_User.Nice));
	list = g_list_append (list, g_strdup ("lsdvd"));
	list = g_list_append (list, g_strdup (pathdev));
	list = g_list_append (list, g_strdup ("-x"));
	list = g_list_append (list, g_strdup ("2>&1"));

	/*return ((GList *)g_list_first (list));*/
	return ((GList *)list);
}

void conv_with_lsdvd_ARGS (ARGS_TO_EXEC *args)
{
	gint       fd;
	gint       size;
	gint       pos = 0;
	gchar      buf [ CONV_MAX_CARS + 10 ];

	fd = conv_call_exec (args, &conv.code_fork_conv, STDOUT_FILENO);

	do
	{
		pos = -1;
		do
		{
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos++] = '\n';
		buf[pos] = '\0';
		utils_add_datas_on_disk (buf);

	} while (size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	conv.bool_reset_conv = TRUE;
}

void conv_with_split_ARGS (ARGS_TO_EXEC *args)
{
	gint            pos = 0;
	gint            size = 0;
	gchar           buf[CONV_MAX_CARS + 10];
	gint            fd;

	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);
	
	do
	{
		size = -1;
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			conv.bool_percent_conv = TRUE;
			
		} while (buf[pos] != '\b' && buf[pos] != '\r' && buf[pos] != '\n' && size > 0);
		pos ++;
		buf[ pos ] = '\0';

	} while (size != 0);

	conv.code_fork_conv = -1;
	close(fd);
}

ARGS_TO_EXEC *conv_alloc_arg (GList *p_list)
{
	/*GList *List = NULL;*/
	ARGS_TO_EXEC *New = NULL;
	gint          elements = 0;
	gint          pos;

	/*PRINT_FUNC_LF();*/

	/*
	for (List=p_list; List->prev != NULL; List=List->prev);
	for (elements=1; List->next != NULL; List=List->next, elements ++);
g_print("elements = %d\n", elements);
	*/
	
	elements = g_list_length (p_list);
	/*
g_print("elements = %d\n", elements);
	*/
	 
	/* Cherche le nombre d'elements */
	if (elements <= 0) {
		elements = 50;
		g_print ("\nATTENTION, MAUVAISE ALLOCATION MEMOIRE !!!\n");
	}
	
	/* Allocation */
	New = (ARGS_TO_EXEC *)g_malloc0(sizeof(ARGS_TO_EXEC) * (elements +5));
	
	/* Init a NULL la structure */
	for (pos = 0; pos < elements +4; pos ++) New[ pos ].str = NULL;
	
	/* Infos */
	g_print ("\n");
	g_print ("!-----------------------------------------------------------!\n");
	g_print ("!  [ 0 .. %d .. %d ]\n", elements-1, elements +1);

	return ((ARGS_TO_EXEC *)New);
}

gint conv_g_list_length (GList *p_list, gint *p_NbrList)
{
	/*GList *List = NULL;*/
	gint   cpt_file;
	
	/*
	for (List=p_list; List->prev != NULL; List=List->prev);
	for (cpt_file=1; List->next != NULL; List=List->next, cpt_file ++);
	*/
	cpt_file = g_list_length (p_list);
	
	if (cpt_file >= *p_NbrList) {
	
		if (conv.ArgConv != NULL) {
			g_free (conv.ArgConv);
			conv.ArgConv = NULL;
		}
		
		*p_NbrList += 10;
		if (cpt_file >= *p_NbrList) *p_NbrList += 50;
		conv.ArgConv = (ARGS_TO_EXEC *)g_malloc0(sizeof(ARGS_TO_EXEC) * *p_NbrList);
	}
	g_print ("\n");
	g_print ("!-----------------------------------------------------------!\n");
	g_print ("!  [ 0 .. %d .. %d ]\n", cpt_file-1, *p_NbrList-1);
	
	return (*p_NbrList);
}

void conv_with_cddb_tool_ARGS (ARGS_TO_EXEC *args)
{
	gint             pos = 0;
	gint             size = 0;
	gchar            buf[CONV_MAX_CARS + 10];
	gint             fd;

	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDOUT_FILENO);
	
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos ++ ] = '\n';
		buf[pos] = '\0';
		utils_add_datas_on_disk (buf);
		
	} while (size != 0);

	conv.code_fork_conv = -1;
	close(fd);
}

void conv_with_icedax_tool_ARGS (ARGS_TO_EXEC *args)
{
	gint             pos = 0;
	gint             size = 0;
	gchar            buf[CONV_MAX_CARS + 10];
	gint             fd;

	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);
		
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos ++ ] = '\n';
		buf[pos] = '\0';
		
		if (buf [ 0 ] != '\n') g_print("BUF = %s", buf);
		
	} while (size != 0);

	conv.code_fork_conv = -1;
	close(fd);
}

void conv_with_aacplusenc_ARGS (ARGS_TO_EXEC *args)
{
	gint             pos = 0;
	gint             size = 0;
	gchar            buf[CONV_MAX_CARS + 10];
	gint             fd;
	gchar		*Ptr = NULL;
	
	conv.conversion_percent = 0.0;
	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDERR_FILENO);
	
	do {
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			
		} while ((buf[pos] != '\b') && (buf[pos] != '\r') && (buf[pos] != '\n') && (size > 0));
		buf[pos ++ ] = '\n';
		buf[pos] = '\0';
		
		if (conv.typeoper == _CONV_) {
			if ((Ptr = strchr (buf, '['))) {
				Ptr ++;
				conv.conversion_percent = (double)((atoi(Ptr) * 0.1) / 10);
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				conv.total_percent = (conv.encode_completed + (double)conv.conversion_percent) / conv.total_convert;
				conv.bool_percent_conv = TRUE;
			}
		}
		else {
			if ((Ptr = strchr (buf, '['))) {
				Ptr ++;
				conv.conversion_percent = (double)((atoi(Ptr) * 0.1) / 10);
				if (conv.conversion_percent >= CONV_MAX_VALUE_PERCENT) conv.conversion_percent = 1.000000;
				conv.total_percent = (conv.rip_completed +
							conv.encode_completed +
							conv.extract_percent +
							conv.conversion_percent) /
							(conv.total_rip + conv.total_convert);
				conv.bool_percent_conv = TRUE;
			}
		}
		
	} while (size != 0);

	conv.code_fork_conv = -1;
	conv.conversion_percent = 0.0;
	close(fd);
}

gboolean conv_exec (gboolean WithParameter, TYPE_CONV type_conv, GList *p_list, gchar *info)
{
	GList          *List = NULL;
	gchar          *Ptr = NULL;
	gint            pos = 0;
	gboolean        RetBool = TRUE;
		
	/* Chercher le debut de la liste */
	for (List=p_list; List->prev != NULL; List=List->prev);
	
	/* Affectation */
	pos = 0;
	while (List) {
		if ((Ptr = (gchar *)List->data)) {
			conv.ArgConv[ pos++ ].str = (gchar *)Ptr;
		}
		List=List->next;
	}
	conv.ArgConv[ pos++ ].str = NULL;
	conv.ArgConv[ pos++ ].str = NULL;
	
	/* Infos */
	g_print ("!-----------------------------------------------------------!\n");
	if (WithParameter == TRUE)
	g_print ("!  --With Expert Parameters--\n");
	else
	g_print ("!  --Without Expert Parameters--\n");
	g_print ("!  %s\n", info);
	g_print ("!-----------------------------------------------------------!\n");

	for (pos = 0; conv.ArgConv[ pos ].str != NULL; pos ++) {
		g_print ("ArgConv[ %02d ] = %s\n", pos, conv.ArgConv[ pos ].str);
	}
	g_print ("\n");

	/* Autorise les appel a:
	 *       GList *filelc_get_command_line (PARAM_FILELC *param_filelc)
	 */
	switch (type_conv) {
	case NONE_CONV :
		break;
	case FLAC_FLAC_TO_WAV :
	case FLAC_WAV_TO_FLAC :
		RetBool = conv_with_flac_ARGS (conv.ArgConv);
		break;
	case LAME_WAV_TO_MP3 :
	case LAME_FLAC_TO_MP3 :
		conv_with_lame_ARGS (conv.ArgConv);
		break;
	case OGGENC_WAV_TO_OGG :
	case OGGENC_FLAC_TO_OGG :
		conv_with_oggenc_ARGS (conv.ArgConv);
		break;
	case OGG123_OGG_TO_WAV :
		conv_with_ogg123_ARGS (conv.ArgConv);
		break;
	case MPG321_MP3_TO_WAV :
		conv_with_mpg321_ARGS (conv.ArgConv);
		break;
	case SOX_WAV_TO_WAV :
		conv_with_sox_ARGS (conv.ArgConv);
		break;
	case SHORTEN_SHN_TO_WAV :
		conv_with_shorten_ARGS (conv.ArgConv);
		break;
	case FAAD_M4A_TO_WAV :
		conv_with_faad_ARGS (conv.ArgConv);
		break;
	case FAAC_WAV_TO_M4A :
		conv_with_faac_ARGS (conv.ArgConv);
		break;
	case MPLAYER_WMA_TO_WAV :
	case MPLAYER_RM_TO_WAV :
	case MPLAYER_DTS_TO_WAV :
	case MPLAYER_AIFF_TO_WAV :
	case MPLAYER_WAV_TO_WAV :
	case MPLAYER_AUDIO_TO_WAV :
	case MPLAYER_M4A_TO_WAV :
	case MPLAYER_OGG_TO_WAV :
		RetBool = conv_with_mplayer_ARGS (conv.ArgConv);
		break;
	case COPY_FILE :
		break;
	case NORMALISE_CALCUL :
	case NORMALISE_APPLIQUE :
	case NORMALISE_EXEC :
		conv_with_normalise_ARGS (conv.ArgConv);
		break;
	case NORMALISE_GET_LEVEL:
		conv_with_normalise_get_PEAK_RMS_GROUP_ARGS (conv.ArgConv);
		break;
	case MPPDEC_MPC_TO_WAV :
		conv_with_mppdec_ARGS (conv.ArgConv);
		break;
	case MPPENC_WAV_TO_MPC :
		conv_with_mppenc_ARGS (conv.ArgConv);
		break;
	case MAC_APE_TO_WAV :
	case MAC_WAV_TO_APE :
		conv_with_mac_ARGS (conv.ArgConv);
		break;
	case WAVPACK_WAV_TO_WAVPACK :
		conv_with_wavpack_ARGS (conv.ArgConv);
		break;
	case WVUNPACK_WAVPACK_TO_WAV :
		conv_with_wvunpack_ARGS (conv.ArgConv);
		break;
	
	case CDPARANOIA_CD_TO_WAV :
	case CDPARANOIA_CD_TO_WAV_EXPERT :
	case CDPARANOIA_CD_TO_WAV_EXPERT_SEGMENT :
	case CDDA2WAV_CD_TO_WAV :
		break;
	
	case REPLAYGAIN :
		conv_with_replaygain_ARGS (conv.ArgConv);
		break;
	case LSDVD :
		conv_with_lsdvd_ARGS (conv.ArgConv);
		break;
	case SPLIT :
		conv_with_split_ARGS (conv.ArgConv);
		break;
	case CDDB_TOOL :
		conv_with_cddb_tool_ARGS (conv.ArgConv);
		break;
	
	case ICEDAX :
		conv_with_icedax_tool_ARGS (conv.ArgConv);
		break;

	case AACPLUSENC_WAV_TO_AAC :
		conv_with_aacplusenc_ARGS (conv.ArgConv);
		break;
	}
	
	return (RetBool);
}

gboolean conv_exec_extract (gboolean WithParameter, TYPE_CONV type_conv, GList *p_list, gchar *info)
{
	/*ARGS_TO_EXEC   *argument = NULL;*/
	GList          *List = NULL;
	gchar          *Ptr = NULL;
	gint            pos = 0;
	gboolean        RetBool = TRUE;
	
	/* Chercher le debut de la liste */
	for (List=p_list; List->prev != NULL; List=List->prev);
	
	/* Affectation */
	pos = 0;
	while (List) {
		if ((Ptr = (gchar *)List->data)) {
			conv.ArgExtract[ pos++ ].str = (gchar *)Ptr;
		}
		List=List->next;
	}
	conv.ArgExtract[ pos++ ].str = NULL;
	conv.ArgExtract[ pos++ ].str = NULL;
	
	/* Infos */
	g_print ("!-----------------------------------------------------------!\n");
	if (WithParameter == TRUE)
	g_print ("!  --With Expert Parameters--\n");
	else
	g_print ("!  --Without Expert Parameters--\n");
	g_print ("!  %s\n", info);
	g_print ("!-----------------------------------------------------------!\n");

	for (pos = 0; conv.ArgExtract[ pos ].str != NULL; pos ++) {
		g_print ("ArgExtract[ %02d ] = %s\n", pos, conv.ArgExtract[ pos ].str);
	}
	g_print ("\n");

	if (type_conv == CDPARANOIA_CD_TO_WAV || type_conv == CDPARANOIA_CD_TO_WAV_EXPERT)
		conv_with_cdparanoia_ARGS (conv.ArgExtract);
	else if (type_conv == CDPARANOIA_CD_TO_WAV_EXPERT_SEGMENT)
		conv_with_cdparanoia_mode_expert_ARGS (conv.ArgExtract);
	else if (type_conv == CDDA2WAV_CD_TO_WAV)
		conv_with_cdda2wav_ARGS (conv.ArgExtract);
		
	return (RetBool);
}



gboolean conv_bash (gchar *PathNameFileBash)
{
	gboolean	 RetBool = TRUE;
	gint		 elements = 10;
	gint		 pos;
	ARGS_TO_EXEC	*args = NULL;
	gint             size = 0;
	gchar            buf[CONV_MAX_CARS + 10];
	gint             fd;

	/* PRINT_FUNC_LF(); */
	
	args = (ARGS_TO_EXEC *)g_malloc0(sizeof(ARGS_TO_EXEC) * (elements +2));
	for (pos = 0; pos < elements +2; pos ++) args[ pos ].str = NULL;

	pos = 0;
	args[ pos++ ].str = (gchar *)PathNameFileBash;
	args[ pos++ ].str = NULL;
	
	g_print ("\n");
	g_print ("!-----------------------------------------------------------!\n");
	g_print ("!  [ 0 .. %d .. %d ]\n", elements-1, elements +1);
	g_print ("!-----------------------------------------------------------!\n");
	for (pos = 0; args[ pos ].str != NULL; pos ++) {
		g_print ("args[ %02d ] = %s\n", pos, args[ pos ].str);
	}
	g_print ("\n");

	conv.bool_percent_conv = TRUE;
	fd = conv_call_exec (args, &conv.code_fork_conv, STDOUT_FILENO);
	pos = 0;
	do
	{
		size = -1;
		pos = -1;
		do {
			pos++;
			if (pos >= CONV_MAX_CARS) {
				PRINT_FUNC_LF();
				g_print ("pos(%d) >= CONV_MAX_CARS(%d)\n", pos, CONV_MAX_CARS);
				pos --;
			}
			size = read(fd, &buf[pos], 1);
			conv.bool_percent_conv = TRUE;
			
		} while (buf[pos] != '\b' && buf[pos] != '\r' && buf[pos] != '\n' && size > 0);
		pos ++;
		buf[ pos ] = '\0';
	} while (size != 0);

	close(fd);
	conv.code_fork_conv = -1;
	
	g_free (args);
	args = NULL;

	return (RetBool);
}












