/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libwps/libwps.h>
#include <libodfgen/libodfgen.hxx>

#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "wks2ods"

using namespace libwps;

class OdsOutputFileHelper : public OutputFileHelper
{
public:
	OdsOutputFileHelper(const char *outFileName,const char *password, const char *encoding) :
		OutputFileHelper(outFileName, password), mpPassword(password), mpEncoding(encoding) {};
	~OdsOutputFileHelper() {};

	bool convertDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdsGenerator collector;
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		libwps::WPSResult result=WPSDocument::parse(&input, &collector, mpPassword, mpEncoding);
		if (result == WPS_ENCRYPTION_ERROR)
		{
			fprintf(stderr, "ERROR : encrypted file, problem with password\n");
			return false;
		}
		else if (result != WPS_OK)
		{
			fprintf(stderr, "ERROR : Couldn't convert the document\n");
			return false;
		}

		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.spreadsheet";

		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}

	bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		WPSCreator creator;
		WPSKind kind;
		bool needEncoding;
		WPSConfidence confidence = WPSDocument::isFileFormatSupported(&input, kind, creator, needEncoding);
		if (confidence == WPS_CONFIDENCE_NONE ||
		        (kind != WPS_SPREADSHEET && kind != WPS_DATABASE))
			return false;
		if (!mpEncoding && needEncoding)
			fprintf(stderr, "WARNING: called without encoding, try to decode with basic character set encoding.\n");
		return true;
	}
private:
	char const *mpPassword;
	char const *mpEncoding;
};

int printUsage(const char *name)
{
	UsageHelper usage(name, "converts MS Workds database/spreadsheet to ODF.");
	usage.addToDescription("If OUTPUT is omitted, the result is printed as Flat ODF to standard output.\n");
	usage.addStdoutOption();
	usage.addEncodingOption();
	usage.addListEncodingsOption();
	usage.addPasswordOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	char *encoding = 0;
	char *password = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "--stdout")==0)
			stdOutput = true;
		else if (strcmp(argv[i], "--encoding")==0)
		{
			if (i+1>=argc)
				return printUsage(TOOLNAME);
			encoding=argv[++i];
			continue;
		}
		else if (strcmp(argv[i], "--password")==0)
		{
			if (i+1>=argc)
				return printUsage(TOOLNAME);
			password=argv[++i];
			continue;
		}
		else if (strcmp(argv[i], "--version")==0)
			return UsageHelper::printVersion(TOOLNAME);
		else if (strcmp(argv[i], "--help")==0)
			return printUsage(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdsOutputFileHelper helper(szOutFile, password, encoding);
	librevenge::RVNGFileStream input(szInputFile);
	if (!helper.isSupportedFormat(input))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Microsoft Works spreadsheet document.\n");
		return 1;
	}

	if (!helper.convertDocument(input, szOutFile==0))
		return 1;

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
