/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libwps/libwps.h>
#include <libodfgen/libodfgen.hxx>

#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"

using namespace libwps;

class OdtOutputFileHelper : public OutputFileHelper
{
public:
	OdtOutputFileHelper(const char *outFileName,const char *password) :
		OutputFileHelper(outFileName, password) {};
	~OdtOutputFileHelper() {};
	bool convertDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdtGenerator collector;
		collector.registerEmbeddedObjectHandler("image/wks-ods", &handleEmbeddedWKSObject);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (WPS_OK != WPSDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.text";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}
	bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		WPSKind kind = WPS_TEXT;
		WPSConfidence confidence = WPSDocument::isFileFormatSupported(&input, kind);
		if (confidence == WPS_CONFIDENCE_NONE || kind != WPS_TEXT)
			return false;
		return true;
	}

private:

	static bool handleEmbeddedWKSObject(const librevenge::RVNGBinaryData &data, OdfDocumentHandler *pHandler,  const OdfStreamType streamType)
	{
		if (!data.size()) return false;
		OdsGenerator exporter;
		exporter.addDocumentHandler(pHandler, streamType);
		return WPSDocument::parse(data.getDataStream(), &exporter)==WPS_OK;
	}
};


int printUsage(char *name)
{
	fprintf(stderr, "USAGE : %s [--stdout] --password <password> <infile> [outfile]\n", name);
	fprintf(stderr, "USAGE : Where <infile> is the MS Works source document\n");
	fprintf(stderr, "USAGE : and [outfile] is the odt target document. Alternately,\n");
	fprintf(stderr, "USAGE : pass '--stdout' or simply omit the [outfile] to pipe the\n");
	fprintf(stderr, "USAGE : resultant document as flat XML to standard output\n");
	fprintf(stderr, "USAGE : \n");
	return 1;
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(argv[0]);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(argv[0]);
	}

	if (!szInputFile)
		return printUsage(argv[0]);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdtOutputFileHelper helper(szOutFile, 0);
	librevenge::RVNGFileStream input(szInputFile);
	if (!helper.isSupportedFormat(input))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Microsoft Works document.\n");
		return 1;
	}

	if (!helper.convertDocument(input, szOutFile==0))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
