/* pdatei.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2002-2004 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: pdatei.cc,v 1.5 2004/03/18 07:10:53 ralf Exp $ */

#include "pdatei.h"
#include <aguix/lowlevelfunc.h>

#ifdef HAVE_LIBXPKMASTER
extern "C" {
int XpkOpen( struct XpkBuffer **, struct TagItem *);
int XpkClose( struct XpkBuffer * );
int XpkRead( struct XpkBuffer *, char *, int );
}
#endif

PDatei::PDatei()
{
  type = PDATEI_UNKNOWN;
  name = NULL;
  readbuf = NULL;
}

PDatei::~PDatei()
{
  if ( name != NULL ) {
    close();
  }
}

int PDatei::open( const char *filename )
{
  close();
#ifdef HAVE_LIBBZ2
  int bzerror;
#endif
  char tbuf[5];
  unsigned char tbuf2[2];
  int r;
  enum pdatei_handler candidate = PDATEI_UNKNOWN;

  fp = fopen ( filename, "r" );
  if ( fp != NULL ) {
    r = fread( tbuf, 1, 4, fp );
    if ( r == 4 ) {
      if ( ( tbuf[0] == 'X' ) &&
	   ( tbuf[1] == 'P' ) &&
	   ( tbuf[2] == 'K' ) &&
	   ( tbuf[3] == 'F' ) ) {
	candidate = PDATEI_XPK;
      } else if ( ( tbuf[0] == 'B' ) &&
		  ( tbuf[1] == 'Z' ) &&
		  ( tbuf[2] == 'h' ) ) {
	candidate = PDATEI_BZIP2;
      } else {
	memcpy( tbuf2, tbuf, 2 );
	if ( ( tbuf2[0] == 0x1f ) &&
	     ( tbuf2[1] == 0x8b ) ) {
	  candidate = PDATEI_GZIP;
	}
      }
    }
    fclose( fp );

#ifdef HAVE_LIBXPKMASTER
    if ( candidate == PDATEI_XPK ) {
      if ( XpkOpenTags( (struct XpkBuffer**)&xpkfp, XPK_InName, (unsigned int)filename, TAG_DONE ) == 0 ) {
	type = PDATEI_XPK;
	name = dupstring( filename );
	readbuf_size = xpkfp->xf_NLen + XPK_MARGIN + 1;
	readbuf = (char*)_allocsafe( readbuf_size );
	readbuf_start = readbuf_used = 0;
	return 0;
      }
    }
#endif
    
#ifdef HAVE_LIBBZ2
    if ( candidate == PDATEI_BZIP2 ) {
      fp = fopen( filename, "r" );
      if ( fp != NULL ) {
	bzfp = BZ2_bzReadOpen( &bzerror, fp, 0, 0, NULL, 0 );
	if ( bzerror == BZ_OK ) {
	  readbuf_size = 4096;
	  readbuf = (char*)_allocsafe( readbuf_size );
	  readbuf_used = BZ2_bzRead( &bzerror, bzfp, readbuf, readbuf_size );
	  if ( ( ( bzerror != BZ_OK ) && ( bzerror != BZ_STREAM_END ) )
	       || ( readbuf_used < 1 ) ) {
	    _freesafe( readbuf );
	  } else {
	    readbuf_start = 0;
	    type = PDATEI_BZIP2;
	    name = dupstring( filename );
	    fileend = ( bzerror == BZ_STREAM_END ) ? true : false;
	    return 0;
	  }
	}
	BZ2_bzReadClose( &bzerror, bzfp );
	fclose( fp );
      }
    }
#endif

#ifdef HAVE_LIBZ
    gzfp = gzopen( filename, "r" );
    if ( gzfp != NULL ) {
      type = PDATEI_GZIP;
      name = dupstring( filename );
      readbuf_size = 4096;
      readbuf = (char*)_allocsafe( readbuf_size );
      readbuf_start = readbuf_used = 0;
      return 0;
    }
#endif

    fp = fopen( filename, "r" );
    if ( fp != NULL ) {
      type = PDATEI_NORMAL;
      name = dupstring( filename );
      readbuf_size = 4096;
      readbuf = (char*)_allocsafe( readbuf_size );
      readbuf_start = readbuf_used = 0;
      return 0;
    }
  }
  return 1;
}

int PDatei::read( void *buf, int size )
{
#if defined HAVE_LIBXPKMASTER || defined HAVE_LIBBZ2
  int reads;
  int offset;
#endif
#ifdef HAVE_LIBBZ2
  int bzerror;
#endif
  
  switch ( type ) {
  case PDATEI_XPK:
#ifdef HAVE_LIBXPKMASTER
    offset = 0;
    reads = 0;
    do {
      if ( readbuf_used > 0 ) {
	if ( readbuf_used >= size ) {
	  memcpy( (char*)buf + offset, readbuf + readbuf_start, size );
	  readbuf_start += size;
	  readbuf_used -= size;
	  if ( readbuf_used < 1 ) readbuf_start = 0;
	  reads += size;
	  break;
	} else {
	  memcpy( (char*)buf + offset, readbuf + readbuf_start, readbuf_used );
	  reads += readbuf_used;
	  size -= readbuf_used;
	  offset += readbuf_used;
	  readbuf_start = readbuf_used = 0;
	}
      }
      readbuf_used = XpkRead( (struct XpkBuffer*)xpkfp, readbuf, xpkfp->xf_NLen );
      readbuf_start = 0;
    } while( readbuf_used > 0 );
    return reads;
    break;
#else
    fprintf( stderr, "PDatei: reading from unsupported type \"xpkmaster\"!\n" );
    break;
#endif
  case PDATEI_BZIP2:
#ifdef HAVE_LIBBZ2
    offset = 0;
    if ( readbuf_used > 0 ) {
      if ( readbuf_used >= size ) {
	memcpy( (char*)buf, readbuf + readbuf_start, size );
	readbuf_start += size;
	readbuf_used -= size;
        if ( readbuf_used < 1 ) readbuf_start = 0;
	return size;
      } else {
	memcpy( (char*)buf, readbuf + readbuf_start, readbuf_used );
	size -= readbuf_used;
	offset = readbuf_used;
      }
    }
    if ( fileend == true ) {
      reads = readbuf_used;
    } else {
      reads = BZ2_bzRead( &bzerror, bzfp, (char*)buf + offset, size );
      if ( bzerror == BZ_STREAM_END ) {
	reads += readbuf_used;
	fileend = true;
      } else if ( bzerror != BZ_OK ) {
	reads = -1;
      } else {
	reads += readbuf_used;
      }
    }
    readbuf_start = readbuf_used = 0;
    return reads;
    break;
#else
    fprintf( stderr, "PDatei: reading from unsupported type \"bzip2\"!\n" );
    break;
#endif
  case PDATEI_GZIP:
#ifdef HAVE_LIBZ
    return gzread( gzfp, buf, size );
    break;
#else
    fprintf( stderr, "PDatei: reading from unsupported type \"gzip\"!\n" );
    break;
#endif
  case PDATEI_NORMAL:
    return fread( buf, 1, size, fp );
    break;
  default:
    fprintf( stderr, "PDatei: reading from unknown type!!\n" );
    break;
  }
  return -1;
}

void PDatei::close()
{
#ifdef HAVE_LIBBZ2
  int bzerror;
#endif
  
  if ( name != NULL ) {
    switch ( type ) {
#ifdef HAVE_LIBXPKMASTER
    case PDATEI_XPK:
      XpkClose( (struct XpkBuffer*)xpkfp );
      type = PDATEI_UNKNOWN;
      _freesafe( name );
      name = NULL;
      _freesafe( readbuf );
      break;
#endif
#ifdef HAVE_LIBBZ2
    case PDATEI_BZIP2:
      BZ2_bzReadClose( &bzerror, bzfp );
      fclose( fp );
      type = PDATEI_UNKNOWN;
      _freesafe( name );
      name = NULL;
      _freesafe( readbuf );
      break;
#endif
#ifdef HAVE_LIBZ
    case PDATEI_GZIP:
      gzclose( gzfp );
      type = PDATEI_UNKNOWN;
      _freesafe( name );
      name = NULL;
      _freesafe( readbuf );
      break;
#endif
    case PDATEI_NORMAL:
      fclose( fp );
      type = PDATEI_UNKNOWN;
      _freesafe( name );
      name = NULL;
      _freesafe( readbuf );
      break;
    default:
      fprintf( stderr, "PDatei: closing unknown or unsupported type!!\n" );
      type = PDATEI_UNKNOWN;
      if ( name != NULL ) _freesafe( name );
      name = NULL;
      break;
    }
  }
}

enum PDatei::pdatei_handler PDatei::getCruncher( const char *filename )
{
  PDatei *p;
  enum pdatei_handler erg = PDATEI_UNKNOWN;
  unsigned char buf[5];
  int r;
  
  p = new PDatei();
  if ( p->open( filename ) == 0 ) {
    erg = p->getHandler();
    p->close();
  }
  delete p;
  
  if ( erg == PDATEI_GZIP ) {
    FILE *tfp;
    tfp = fopen( filename, "r" );
    if ( tfp != NULL ) {
      r = fread( buf, 4, 1, tfp );
      if ( r == 1 ) {
	if ( ( buf[0] == 0x1f ) &&
	     ( buf[1] == 0x8b ) ) {
          erg = PDATEI_GZIP;
        } else {
          erg = PDATEI_NORMAL;
        }
      }
      fclose( tfp );
    }
  }
  
  return erg;
}

enum PDatei::pdatei_handler PDatei::getHandler()
{
  return type;
}

