/* acontainer.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2004 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: acontainer.cc,v 1.4 2004/02/02 22:25:01 ralf Exp $ */

#include "acontainer.h"
#include "awindow.h"
#include "guielement.h"
#include "ascaler.h"

AContainer::AContElem::AContElem()
{
  Guielem = NULL;
  Cont = NULL;
  Win = NULL;
  Width = Height = 0;
  MinWidth = MaxWidth = MinHeight = MaxHeight = -1;
  AutosizeMode = ACONT_NONE;
  WWeight = HWeight = 1;
  NoResize = false;
}

GUIElement *AContainer::AContElem::guielem() const
{
  return Guielem;
}

AContainer *AContainer::AContElem::cont() const
{
  return Cont;
}

AWindow *AContainer::AContElem::win() const
{
  return Win;
}

void AContainer::AContElem::guielem( GUIElement *newelem )
{
  Guielem = newelem;
  Cont = NULL;
  Win = NULL;
}

void AContainer::AContElem::cont( AContainer *newcont )
{
  Guielem = NULL;
  Cont = newcont;
  Win = NULL;
}

void AContainer::AContElem::win( AWindow *newwin )
{
  Guielem = NULL;
  Cont = NULL;
  Win = newwin;
}

int AContainer::AContElem::width() const
{
  return Width;
}

void AContainer::AContElem::width( int nv )
{
  if ( nv >= 0 ) Width = nv;
}

int AContainer::AContElem::height() const
{
  return Height;
}

void AContainer::AContElem::height( int nv )
{
  if ( nv >= 0 ) Height = nv;
}

int AContainer::AContElem::minWidth() const
{
  return MinWidth;
}

void AContainer::AContElem::minWidth( int nv )
{
  MinWidth = nv;
  if ( MinWidth < -1 ) MinWidth = -1;
}

int AContainer::AContElem::maxWidth() const
{
  return MaxWidth;
}

void AContainer::AContElem::maxWidth( int nv )
{
  MaxWidth = nv;
  if ( MaxWidth < -1 ) MaxWidth = -1;
}

int AContainer::AContElem::minHeight() const
{
  return MinHeight;
}

void AContainer::AContElem::minHeight( int nv )
{
  MinHeight = nv;
  if ( MinHeight < -1 ) MinHeight = -1;
}

int AContainer::AContElem::maxHeight() const
{
  return MaxHeight;
}

void AContainer::AContElem::maxHeight( int nv )
{
  MaxHeight = nv;
  if ( MaxHeight < -1 ) MaxHeight = -1;
}

int AContainer::AContElem::autosizeMode() const
{
  return AutosizeMode;
}

void AContainer::AContElem::autosizeMode( int nv )
{
  AutosizeMode = nv & ( ACONT_MAXW | ACONT_MAXH | ACONT_MINW | ACONT_MINH );
}

int AContainer::AContElem::wWeight() const
{
  return WWeight;
}

void AContainer::AContElem::wWeight( int nv )
{
  WWeight = nv;
  if ( WWeight < 1 ) WWeight = 1;
}

int AContainer::AContElem::hWeight() const
{
  return HWeight;
}

void AContainer::AContElem::hWeight( int nv )
{
  HWeight = nv;
  if ( HWeight < 1 ) HWeight = 1;
}

int AContainer::AContElem::noResize() const
{
  return NoResize;
}

void AContainer::AContElem::noResize( bool nv )
{
  NoResize = nv;
}

AContainer::AContainer( AWindow *twin, int txelems, int tyelems )
{
  win = twin;
  xelems = txelems;
  if ( xelems < 1 ) xelems = 1;
  yelems = tyelems;
  if ( yelems < 1 ) yelems = 1;

  elements = new AContElem [xelems*yelems];
  parent = NULL;
  x = y = width = height = 0;
  BorderWidth = 5;
  MinSpace = 5;
  MaxSpace = -1;
  ergWidths = ergHeights = NULL;
}

AContainer::~AContainer()
{
  int i;

  for ( i = 0; i < ( xelems * yelems ); i++ ) {
    if ( elements[i].cont() != NULL ) {
      delete elements[i].cont();
    }
  }
  if ( parent != NULL ) {
    parent->remove( this );
  } else {
    // top container
    if ( win != NULL ) win->removeContainer( this );
  }
  delete [] elements;
  if ( ergWidths != NULL ) delete [] ergWidths;
  if ( ergHeights != NULL ) delete [] ergHeights;
}

GUIElement *AContainer::add( GUIElement *newelement, int tx, int ty, int tautosizeMode )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].guielem( newelement );
  elements[tx + xelems * ty].autosizeMode( tautosizeMode );
  elements[tx + xelems * ty].noResize( ( ( tautosizeMode & ACONT_NORESIZE ) != 0 ) ? true : false );
  if ( win != NULL ) win->add( newelement );
  
  if ( ( tautosizeMode & ACONT_MINW ) != 0 ) {
    elements[tx + xelems * ty].minWidth( newelement->getWidth() );
  }
  if ( ( tautosizeMode & ACONT_MINH ) != 0 ) {
    elements[tx + xelems * ty].minHeight( newelement->getHeight() );
  }
  if ( ( tautosizeMode & ACONT_MAXW ) != 0 ) {
    elements[tx + xelems * ty].maxWidth( newelement->getWidth() );
  }
  if ( ( tautosizeMode & ACONT_MAXH ) != 0 ) {
    elements[tx + xelems * ty].maxHeight( newelement->getHeight() );
  }

  return newelement;
}

AContainer *AContainer::add( AContainer *newcont, int tx, int ty )
{
  int i;

  if ( newcont == NULL ) return NULL;

  for ( i = 0; i < ( xelems * yelems ); i++ ) {
    if ( elements[i].cont() == newcont ) {
      // already added so just return
      return newcont;
    }
  }

  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].cont( newcont );
  newcont->setParent( this );
  return newcont;
}

AWindow *AContainer::add( AWindow *newwin, int tx, int ty, int tautosizeMode )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].win( newwin );
  elements[tx + xelems * ty].autosizeMode( tautosizeMode );

  if ( ( tautosizeMode & ACONT_MINW ) != 0 ) {
    elements[tx + xelems * ty].minWidth( newwin->getWidth() );
  }
  if ( ( tautosizeMode & ACONT_MINH ) != 0 ) {
    elements[tx + xelems * ty].minHeight( newwin->getHeight() );
  }
  if ( ( tautosizeMode & ACONT_MAXW ) != 0 ) {
    elements[tx + xelems * ty].maxWidth( newwin->getWidth() );
  }
  if ( ( tautosizeMode & ACONT_MAXH ) != 0 ) {
    elements[tx + xelems * ty].maxHeight( newwin->getHeight() );
  }

  return newwin;
}

void AContainer::setParent( AContainer *newparent )
{
  AContainer *tc;

  if ( newparent != NULL ) {
    if ( parent != NULL ) {
      // only when different parent
      if ( parent != newparent ) {
	setParent( NULL );
	setParent( newparent );
      }
    } else {
      parent = newparent;
      parent->add( this ); /* add this element to the parent window
			    * when called from outside we have to do this
			    * add will do nothing if this is already in his list
			    * add will call setParent again, but because we already setted parent
			    * this won't be executed twice */
    }
  } else {
    if ( parent != NULL ) {
      tc = parent;
      parent = NULL;
      tc->remove( this );  /* remove from parent window because this can be called from outside
			    * because I set parent = NULL this won't be executed twice!
			    */
    }
  }
}

int AContainer::remove( const AContainer *v )
{
  int pos;
  AContainer *tc;

  if ( v == NULL ) return 1;

  for ( pos = 0; pos < ( xelems * yelems ); pos++ ) {
    if ( elements[pos].cont() == v ) break;
  }
  if ( ( pos < 0 ) || ( pos >= ( xelems * yelems ) ) ) return 1;

  tc = elements[pos].cont();
  elements[pos].cont( NULL );
  tc->setParent( NULL );
  return 0;
}

int AContainer::remove( const GUIElement *v )
{
  int pos;

  if ( v == NULL ) return 1;

  for ( pos = 0; pos < ( xelems * yelems ); pos++ ) {
    if ( elements[pos].guielem() == v ) break;
  }
  if ( ( pos < 0 ) || ( pos >= ( xelems * yelems ) ) ) {
    // we don't own it call it for all subcontainers
    for ( pos = 0; pos < ( xelems * yelems ); pos++ ) {
      if ( elements[pos].cont() != NULL ) {
	elements[pos].cont()->remove( v );
      }
    }
    return 1;
  }

  elements[pos].guielem( NULL );
  return 0;
}

int AContainer::remove( const AWindow *v )
{
  int pos;

  if ( v == NULL ) return 1;

  for ( pos = 0; pos < ( xelems * yelems ); pos++ ) {
    if ( elements[pos].win() == v ) break;
  }
  if ( ( pos < 0 ) || ( pos >= ( xelems * yelems ) ) ) {
    // we don't own it call it for all subcontainers
    for ( pos = 0; pos < ( xelems * yelems ); pos++ ) {
      if ( elements[pos].cont() != NULL ) {
	elements[pos].cont()->remove( v );
      }
    }
    return 1;
  }

  elements[pos].win( NULL );
  return 0;
}

int AContainer::resize( int nw, int nh )
{
  int *mins, *maxs, *weights, *erg;
  int i, ew, t1, t2;
  int *specialMinWidth, *specialMaxWidth, *specialMinHeight, *specialMaxHeight;
  bool repeatScale;
  int realContWidth, realContHeight;
  int v, j;
  bool *contDone;

  specialMinWidth = new int[xelems];
  specialMaxWidth = new int[xelems];
  specialMinHeight = new int[yelems];
  specialMaxHeight = new int[yelems];

  for ( i = 0; i < xelems; i++ ) specialMinWidth[i] = -1;
  for ( i = 0; i < xelems; i++ ) specialMaxWidth[i] = -1;
  for ( i = 0; i < yelems; i++ ) specialMinHeight[i] = -1;
  for ( i = 0; i < yelems; i++ ) specialMaxHeight[i] = -1;

  contDone = new bool[xelems * yelems * 2 * 2];
  for ( i = 0; i < ( xelems * yelems * 2 * 2 ); i++ ) contDone[i] = false;

  for ( repeatScale = true; repeatScale == true; ) {
    repeatScale = false;
    /* first scale all columns */
    mins = new int[ 2 * xelems + 1];
    for ( i = 0; i < ( 2 * xelems + 1 ); i++ ) mins[i] = 0;
    maxs = new int[ 2 * xelems + 1 ];
    for ( i = 0; i < ( 2 * xelems + 1 ); i++ ) maxs[i] = -1;
    weights = new int[ 2 * xelems + 1 ];
    for ( i = 0; i < ( 2 * xelems + 1 ); i++ ) weights[i] = 1;
    erg = new int[ 2 * xelems + 1 ];
    
    if ( BorderWidth > 0 ) t1 = BorderWidth;
    else t1 = 0;
    mins[0] = maxs[0] = mins[ 2 * xelems] = maxs[ 2 * xelems] = t1;
    
    if ( MinSpace > 0 ) t1 = MinSpace;
    else t1 = 0;
    if ( MaxSpace >= 0 ) t2 = MaxSpace;
    else t2 = -1;
    for ( i = 0; i < ( xelems - 1 ); i++ ) {
      mins[ 2 * i + 2 ] = t1;
      maxs[ 2 * i + 2 ] = t2;
    }
    
    for ( i = 0; i < xelems; i++ ) {
      t1 = 0;
      for ( j = 0; j < yelems; j++ ) {
	v = elements[i + j * xelems].minWidth();
	if ( v > t1 ) t1 = v;
      }
      
      if ( specialMinWidth[i] > t1 ) {
	t1 = specialMinWidth[i];
      }
      
      t2 = -1;
      for ( j = 0; j < yelems; j++ ) {
	v = elements[i + j * xelems].maxWidth();
	if ( v >= 0 ) {
	  if ( t2 < 0 ) t2 = v;
	  else if ( v < t2 ) t2 = v;
	}
      }
      
      if ( specialMaxWidth[i] > 0 ) {
        if ( ( t2 == -1 ) ||
	     ( specialMaxWidth[i] < t2 ) ) {
	  t2 = specialMaxWidth[i];
        }
      }
      
      if ( ( t1 > t2 ) && ( t2 >= 0 ) ) {
	// rather make the elements larger than making the smaller
	t2 = t1;
      }
      mins[ 2 * i + 1 ] = t1;
      maxs[ 2 * i + 1 ] = t2;
      
      if ( elements[i].wWeight() > 1 ) {
	weights[ 2 * i + 1 ] = elements[i].wWeight();
      }
    }
    ew = AScaler::scaleElements( nw, ( 2 * xelems + 1 ), mins, maxs, weights, erg );
    width = ew;

    if ( ergWidths == NULL ) ergWidths = new int[ xelems * 2 + 1 ];
    
    for ( i = 0; i < ( 2 * xelems + 1 ); i++ ) {
      ergWidths[i] = erg[i];
    }
    
    delete [] mins;
    delete [] maxs;
    delete [] weights;
    delete [] erg;
    
    /* now scale all rows */
    mins = new int[ 2 * yelems + 1];
    for ( i = 0; i < ( 2 * yelems + 1 ); i++ ) mins[i] = 0;
    maxs = new int[ 2 * yelems + 1 ];
    for ( i = 0; i < ( 2 * yelems + 1 ); i++ ) maxs[i] = -1;
    weights = new int[ 2 * yelems + 1 ];
    for ( i = 0; i < ( 2 * yelems + 1 ); i++ ) weights[i] = 1;
    erg = new int[ 2 * yelems + 1 ];
    
    if ( BorderWidth > 0 ) t1 = BorderWidth;
    else t1 = 0;
    mins[0] = maxs[0] = mins[ 2 * yelems] = maxs[ 2 * yelems] = t1;
    
    if ( MinSpace > 0 ) t1 = MinSpace;
    else t1 = 0;
    if ( MaxSpace >= 0 ) t2 = MaxSpace;
    else t2 = -1;
    for ( i = 0; i < ( yelems - 1 ); i++ ) {
      mins[ 2 * i + 2 ] = t1;
      maxs[ 2 * i + 2 ] = t2;
    }
    
    for ( i = 0; i < yelems; i++ ) {
      t1 = 0;
      for ( j = 0; j < xelems; j++ ) {
	v = elements[j + i * xelems].minHeight();
	if ( v > t1 ) t1 = v;
      }
      
      if ( specialMinHeight[i] > t1 ) {
	t1 = specialMinHeight[i];
      }
      
      t2 = -1;
      for ( j = 0; j < xelems; j++ ) {
	v = elements[j + i * xelems].maxHeight();
	if ( v >= 0 ) {
	  if ( t2 < 0 ) t2 = v;
	  else if ( v < t2 ) t2 = v;
	}
      }

      if ( specialMaxHeight[i] > 0 ) {
        if ( ( t2 == -1 ) ||
	     ( specialMaxHeight[i] < t2 ) ) {
	  t2 = specialMaxHeight[i];
        }
      }

      if ( ( t1 > t2 ) && ( t2 >= 0 ) ) {
	// rather make the elements larger than making the smaller
	t2 = t1;
      }
      
      mins[ 2 * i + 1 ] = t1;
      maxs[ 2 * i + 1 ] = t2;
      
      if ( elements[i * xelems].hWeight() > 1 ) {
	weights[ 2 * i + 1 ] = elements[i * xelems].hWeight();
      }
    }
    
    ew = AScaler::scaleElements( nh, ( 2 * yelems + 1 ), mins, maxs, weights, erg );
    height = ew;

    if ( ergHeights == NULL ) ergHeights = new int[ yelems * 2 + 1 ];
    
    for ( i = 0; i < ( 2 * yelems + 1 ); i++ ) {
      ergHeights[i] = erg[i];
    }
    
    delete [] mins;
    delete [] maxs;
    delete [] weights;
    delete [] erg;
    
    // now scale all sub containers
    for ( i = 0; i < ( xelems * yelems ); i++ ) {
      if ( elements[i].cont() != NULL ) {
	t1 = ergWidths[ 2 * ( i % xelems ) + 1];
	t2 = ergHeights[ 2 * ( i / xelems ) + 1];
	if ( ( t1 >= 0 ) && ( t2 >= 0 ) ) {
	  elements[i].cont()->resize( t1, t2 );
	  realContWidth = elements[i].cont()->getWidth();
	  realContHeight = elements[i].cont()->getHeight();
	  
	  /* update special widths/heights
	   * but only the first time for each container
	   * reason: When we fail to resize to wanted size
	   *         it will work the second time (no need to repeat)
	   *         or will fail just with the same values
	   * but use a special flag for each direction
	   * reason: when a container uses more space than wanted
	   *         and we give it more space in the next try
	   *         but it will than use less
	   */
	  if ( realContWidth > t1 ) {
	    if ( realContWidth > specialMinWidth[i % xelems] ) {
	      specialMinWidth[i % xelems] = realContWidth;
	      if ( contDone[i] == false ) {
		repeatScale = true;
	      }
	      contDone[i] = true;
	    }
	  } else if ( realContWidth < t1 ) {
	    if ( ( specialMaxWidth[i % xelems] < 0 ) ||
		 ( realContWidth < specialMaxWidth[i % xelems] ) ) {
	      specialMaxWidth[i % xelems] = realContWidth;
	      if ( contDone[i + 1] == false ) {
		repeatScale = true;
	      }
	      contDone[i + 1] = true;
	    }
	  }

	  if ( realContHeight > t2 ) {
	    if ( realContHeight > specialMinHeight[i / xelems] ) {
	      specialMinHeight[i / xelems] = realContHeight;
	      if ( contDone[i + 2] == false ) {
		repeatScale = true;
	      }
	      contDone[i + 2] = true;
	    }
	  } else if ( realContHeight < t2 ) {
	    if ( ( specialMaxHeight[i / xelems] < 0 ) ||
		 ( realContHeight < specialMaxHeight[i / xelems] ) ) {
	      specialMaxHeight[i / xelems] = realContHeight;
	      if ( contDone[i + 3] == false ) {
		repeatScale = true;
	      }
	      contDone[i + 3] = true;
	    }
	  }
	}
      }
    }
  }

  delete [] specialMinWidth;
  delete [] specialMaxWidth;
  delete [] specialMinHeight;
  delete [] specialMaxHeight;

  delete [] contDone;

  return 0;
}

int AContainer::move( int nx, int ny )
{
  x = nx;
  y = ny;
  return 0;
}

int AContainer::rearrange( int vhmode )
{
  int i, j, cx, cy, tw, th, ind;
  int ux, uy, uw, uh;

  if ( ( ergWidths == NULL ) || ( ergHeights == NULL ) ) return 1;

  cy = y;
  for ( j = 0; j < ( 2 * yelems + 1 ); j++ ) {
    cx = x;
    th = ergHeights[j];
    if ( ( j % 2 ) == 1 ) {
      for ( i = 0; i < ( 2 * xelems + 1 ); i++ ) {
	tw = ergWidths[i];
	if ( tw < 0 ) tw = 0;
	
	if ( ( i % 2 ) == 1 ) {
	  ind = i / 2 + ( j / 2 ) * xelems;
	  ux = cx;
	  uy = cy;
	  uw = tw;
	  uh = th;
	  if ( elements[ind].guielem() != NULL ) {
	    if ( vhmode == ACONT_ONLYH ) {
	      uy = elements[ind].guielem()->getY();
	      uh = elements[ind].guielem()->getHeight();
	    } else if ( vhmode == ACONT_ONLYV ) {
	      ux = elements[ind].guielem()->getX();
	      uw = elements[ind].guielem()->getWidth();
	    }
	    elements[ind].guielem()->move( ux, uy );
	    if ( elements[ind].noResize() == false ) elements[ind].guielem()->resize( uw, uh );
	  } else if ( elements[ind].win() != NULL ) {
	    if ( vhmode == ACONT_ONLYH ) {
	      uy = elements[ind].win()->getY();
	      uh = elements[ind].win()->getHeight();
	    } else if ( vhmode == ACONT_ONLYV ) {
	      ux = elements[ind].win()->getX();
	      uw = elements[ind].win()->getWidth();
	    }
	    elements[ind].win()->move( ux, uy );
	    if ( elements[ind].noResize() == false ) elements[ind].win()->resize( uw, uh );
	  } else if ( elements[ind].cont() != NULL ) {
	    if ( vhmode == ACONT_ONLYH ) {
	      uy = elements[ind].cont()->getY();
	    } else if ( vhmode == ACONT_ONLYV ) {
	      ux = elements[ind].cont()->getX();
	    }
	    elements[ind].cont()->move( ux, uy );
	    elements[ind].cont()->rearrange( vhmode );
	  }
	}
	cx += tw;
      }
    }
    cy += th;
  }
  return 0;
}

void AContainer::setBorderWidth( int nv )
{
  BorderWidth = nv;
  if ( BorderWidth < 0 ) BorderWidth = 0;
}

int AContainer::getBorderWidth() const
{
  return BorderWidth;
}

void AContainer::setMinSpace( int nv )
{
  MinSpace = nv;
  if ( MinSpace < 0 ) MinSpace = 0;
}

int AContainer::getMinSpace() const
{
  return MinSpace;
}

void AContainer::setMaxSpace( int nv )
{
  MaxSpace = nv;
  if ( MaxSpace < -1 ) MaxSpace = -1;
}

int AContainer::getMaxSpace() const
{
  return MaxSpace;
}

void AContainer::setMinWidth( int nv, int tx, int ty )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].minWidth( nv );
}

int AContainer::getMinWidth( int tx, int ty ) const
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  return elements[tx + xelems * ty].minWidth();
}

void AContainer::setMaxWidth( int nv, int tx, int ty )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].maxWidth( nv );
}

int AContainer::getMaxWidth( int tx, int ty ) const
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  return elements[tx + xelems * ty].maxWidth();
}

void AContainer::setMinHeight( int nv, int tx, int ty )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].minHeight( nv );
}

int AContainer::getMinHeight( int tx, int ty ) const
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  return elements[tx + xelems * ty].minHeight();
}

void AContainer::setMaxHeight( int nv, int tx, int ty )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].maxHeight( nv );
}

int AContainer::getMaxHeight( int tx, int ty ) const
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  return elements[tx + xelems * ty].maxHeight();
}

void AContainer::setWWeight( int nv, int tx, int ty )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].wWeight( nv );
}

int AContainer::getWWeight( int tx, int ty ) const
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  return elements[tx + xelems * ty].wWeight();
}

void AContainer::setHWeight( int nv, int tx, int ty )
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  elements[tx + xelems * ty].hWeight( nv );
}

int AContainer::getHWeight( int tx, int ty ) const
{
  if ( ( tx < 0 ) || ( tx >= xelems ) ) tx = 0;
  if ( ( ty < 0 ) || ( ty >= yelems ) ) ty = 0;
  return elements[tx + xelems * ty].hWeight();
}

int AContainer::getX() const
{
  return x;
}

int AContainer::getY() const
{
  return y;
}

void AContainer::readLimits()
{
  GUIElement *elem;
  AWindow *win;
  int pos;

  for ( pos = 0; pos < ( xelems * yelems ); pos++ ) {
    if ( elements[pos].win() != NULL ) {
      win = elements[pos].win();
      if ( ( elements[pos].autosizeMode() & ACONT_MINW ) != 0 ) {
	elements[pos].minWidth( win->getWidth() );
      }
      if ( ( elements[pos].autosizeMode() & ACONT_MINH ) != 0 ) {
	elements[pos].minHeight( win->getHeight() );
      }
      if ( ( elements[pos].autosizeMode() & ACONT_MAXW ) != 0 ) {
	elements[pos].maxWidth( win->getWidth() );
      }
      if ( ( elements[pos].autosizeMode() & ACONT_MAXH ) != 0 ) {
	elements[pos].maxHeight( win->getHeight() );
      }
    } else if ( elements[pos].guielem() != NULL ) {
      elem = elements[pos].guielem();
      if ( ( elements[pos].autosizeMode() & ACONT_MINW ) != 0 ) {
	elements[pos].minWidth( elem->getWidth() );
      }
      if ( ( elements[pos].autosizeMode() & ACONT_MINH ) != 0 ) {
	elements[pos].minHeight( elem->getHeight() );
      }
      if ( ( elements[pos].autosizeMode() & ACONT_MAXW ) != 0 ) {
	elements[pos].maxWidth( elem->getWidth() );
      }
      if ( ( elements[pos].autosizeMode() & ACONT_MAXH ) != 0 ) {
	elements[pos].maxHeight( elem->getHeight() );
      }
    }
  }
}

int AContainer::getWidth() const
{
  return width;
}

int AContainer::getHeight() const
{
  return height;
}
