/* calendar.cc
 * This file belongs to Worker, a file manager for UNIX/X11.
 * Copyright (C) 2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "calendar.hh"
#include <string>
#include <sstream>
#include <iomanip>
#include <aguix/awindow.h>
#include <aguix/button.h>
#include <aguix/stringgadget.h>
#include "worker.h"

Calendar::Calendar()
{
    time_t now = time( NULL );
    localtime_r( &now, &m_initial_date );
    m_current_date = m_initial_date;
    m_today = m_initial_date;

    for ( int week = 0; week < 6; week++ ) {
        for ( int day = 0; day < 7; day++ ) {
            m_day_buttons[week][day] = NULL;
        }
    }
}

Calendar::~Calendar()
{
}

void Calendar::setInitialDate( time_t t )
{
    localtime_r( &t, &m_initial_date );
}

void Calendar::setInitialDate( const std::string &t )
{
    struct tm t1;
    time_t ct1;
    int year = -1, month = -1, day = -1;

    //TODO use strptime if available
    //fill struct tm with negativ dst and call mktime
    
    t1.tm_sec = 0;
    t1.tm_min = 0;
    t1.tm_hour = 0;
    t1.tm_isdst = -1;

    if ( sscanf( t.c_str(), "%d-%d-%d", &year, &month, &day ) == 3 ) {
        t1.tm_mday = day;
        t1.tm_mon = month - 1;
        t1.tm_year = year - 1900;
        
        ct1 = mktime( &t1 );
        setInitialDate( ct1 );
    }
}

bool Calendar::showCalendar()
{
    AGUIX *aguix = Worker::getAGUIX();
    AGMessage *agmsg;
    AWindow *win;
    int endmode = -1;
    std::string str1;

    win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 819 ) );
    win->create();
    
    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );
    
    AContainer *ac1_2 = ac1->add( new AContainer( win, 5, 1 ), 0, 0 );
    ac1_2->setMinSpace( 0 );
    ac1_2->setMaxSpace( 0 );
    ac1_2->setBorderWidth( 0 );
    
    Button *prev_year_b = (Button*)ac1_2->add( new Button( aguix,
                                                           0,
                                                           0,
                                                           "<",
                                                           1,
                                                           0,
                                                           0 ), 1, 0, AContainer::CO_FIX );
    m_year_sg = (StringGadget*)ac1_2->add( new StringGadget( aguix, 0, 0, 100, "", 0 ), 2, 0, AContainer::CO_FIX );
    m_year_sg->resize( aguix->getTextWidth( "99999" ) + 20 , m_year_sg->getHeight() );
    Button *next_year_b = (Button*)ac1_2->add( new Button( aguix,
                                                           0,
                                                           0,
                                                           ">",
                                                           1,
                                                           0,
                                                           0 ), 3, 0, AContainer::CO_FIX );
    ac1_2->readLimits();
  
    AContainer *ac1_3 = ac1->add( new AContainer( win, 5, 1 ), 0, 1 );
    ac1_3->setMinSpace( 0 );
    ac1_3->setMaxSpace( 0 );
    ac1_3->setBorderWidth( 0 );
    
    Button *prev_month_b = (Button*)ac1_3->add( new Button( aguix,
                                                            0,
                                                            0,
                                                            "<",
                                                            1,
                                                            0,
                                                            0 ), 1, 0, AContainer::CO_FIX );
    m_month_sg = (StringGadget*)ac1_3->add( new StringGadget( aguix, 0, 0, 100, "", 0 ), 2, 0, AContainer::CO_FIX );
    m_month_sg->resize( aguix->getTextWidth( "99" ) + 20 , m_month_sg->getHeight() );
    Button *next_month_b = (Button*)ac1_3->add( new Button( aguix,
                                                            0,
                                                            0,
                                                            ">",
                                                            1,
                                                            0,
                                                            0 ), 3, 0, AContainer::CO_FIX );
    ac1_3->readLimits();
    
    AContainer *ac1_4 = ac1->add( new AContainer( win, 7, 7 ), 0, 2 );
    ac1_4->setMinSpace( 0 );
    ac1_4->setMaxSpace( 0 );
    ac1_4->setBorderWidth( 0 );

    for ( int week = 0; week < 6; week++ ) {
        for ( int day = 0; day < 7; day++ ) {
            m_day_buttons[week][day] = (Button*)ac1_4->add( new Button( aguix,
                                                                        0,
                                                                        0,
                                                                        "999",
                                                                        1,
                                                                        0,
                                                                        0 ), day, week + 1, AContainer::CO_INCW );
        }
    }

    {
        struct tm t1 = m_initial_date;
        for ( int day = 0; day < 7; day++ ) {
            t1.tm_isdst = -1;
            if ( mktime( &t1 ) != (time_t)-1 ) {
                char weekday_str[128];
                
                strftime( weekday_str, 128, "%a", &t1 );
                ac1_4->add( new Text( aguix,
                                      0,
                                      0,
                                      weekday_str,
                                      1 ), ( t1.tm_wday + 7 - 1 ) % 7, 0, AContainer::CO_FIXNR + AContainer::ACONT_CENTER );
            }
            t1.tm_mday++;
        }
    }
    
    win->setDoTabCycling( true );
    win->contMaximize( true, true );
    win->useStippleBackground();

    m_current_date = m_initial_date;
    updateCalendar( m_current_date );

    win->show();

    for (; endmode == -1 ; ) {
        agmsg = aguix->WaitMessage( win );
        if ( agmsg != NULL ) {
            switch ( agmsg->type ) {
                case AG_CLOSEWINDOW:
                    if ( agmsg->closewindow.window == win->getWindow() ) endmode = 0;
                    break;
                case AG_BUTTONCLICKED:
                    if ( agmsg->button.button == prev_year_b ) {
                        struct tm t1 = m_current_date;
                        t1.tm_year--;
                        t1.tm_mday = 1;
                        updateCalendar( t1 );
                    } else if ( agmsg->button.button == next_year_b ) {
                        struct tm t1 = m_current_date;
                        t1.tm_year++;
                        t1.tm_mday = 1;
                        updateCalendar( t1 );
                    } else if ( agmsg->button.button == prev_month_b ) {
                        struct tm t1 = m_current_date;
                        t1.tm_mon--;
                        t1.tm_mday = 1;
                        updateCalendar( t1 );
                    } else if ( agmsg->button.button == next_month_b ) {
                        struct tm t1 = m_current_date;
                        t1.tm_mon++;
                        t1.tm_mday = 1;
                        updateCalendar( t1 );
                    } else {
                        for ( int week = 0; week < 6; week++ ) {
                            for ( int day = 0; day < 7; day++ ) {
                                if ( agmsg->button.button == m_day_buttons[week][day] ) {
                                    if ( m_day_buttons[week][day]->getData() != -1 ) {
                                        m_current_date.tm_mday = m_day_buttons[week][day]->getData();
                                        m_current_date.tm_isdst = -1;
                                        if ( mktime( &m_current_date) != (time_t)-1 ) endmode = 1;
                                    }
                                }
                            }
                        }
                    }
                    break;
                case AG_STRINGGADGET_OK:
                    if ( agmsg->stringgadget.sg == m_year_sg ) {
                        int year = 0;
                        if ( sscanf( m_year_sg->getText(), "%d", &year ) == 1 ) {
                            struct tm t1 = m_current_date;
                            t1.tm_year = year - 1900;
                            t1.tm_mday = 1;
                            updateCalendar( t1 );
                        }
                    } else if ( agmsg->stringgadget.sg == m_month_sg ) {
                        int month = atoi( m_month_sg->getText() );
                        if ( month >= 1 && month <= 12 ) {
                            struct tm t1 = m_current_date;
                            t1.tm_mon = month - 1;
                            t1.tm_mday = 1;
                            updateCalendar( t1 );
                        }
                    }
                    break;
            }
            aguix->ReplyMessage( agmsg );
        }
    }

    delete win;

    return ( endmode == 1 ) ? true : false;
}

std::string Calendar::getDate() const
{
    std::stringstream d;
    d << m_current_date.tm_year + 1900;
    d << "-" << std::setw( 2 ) << std::setfill( '0' ) << m_current_date.tm_mon + 1;
    d << "-" << m_current_date.tm_mday;
    return d.str();
}

void Calendar::updateWindow()
{
    struct tm my_date = m_current_date;
    int offset, j;
    int my_month;

    my_month = my_date.tm_mon;

    my_date.tm_mday = 1;
    my_date.tm_isdst = -1;
    mktime( &my_date );

    offset = ( my_date.tm_wday + 7 - 1 ) % 7;

    for ( j = 0; j < offset; j++ ) {
        m_day_buttons[0][j]->hide();
        m_day_buttons[0][j]->setData( -1 );
    }

    for ( int day = 1; day < 32; day++ ) {
        std::stringstream dstr;
        dstr << day;

        my_date.tm_mday = day;
        my_date.tm_isdst = -1;
        if ( mktime( &my_date ) == (time_t)-1 ) break;
        if ( my_date.tm_mon != my_month ) break;

        m_day_buttons[offset / 7][offset % 7]->show();
        m_day_buttons[offset / 7][offset % 7]->setText( 0, dstr.str().c_str() );
        m_day_buttons[offset / 7][offset % 7]->setData( day );

        if ( m_today.tm_year == my_date.tm_year &&
             m_today.tm_mon == my_date.tm_mon &&
             m_today.tm_mday == my_date.tm_mday ) {
            m_day_buttons[offset / 7][offset % 7]->setFG( 0, 2 );
            m_day_buttons[offset / 7][offset % 7]->setBG( 0, 1 );
        } else {
            m_day_buttons[offset / 7][offset % 7]->setFG( 0, 1 );
            m_day_buttons[offset / 7][offset % 7]->setBG( 0, 0 );
        }

        offset++;
    }

    for ( ; offset < 6 * 7; offset++ ) {
        m_day_buttons[offset / 7][offset % 7]->hide();
        m_day_buttons[offset / 7][offset % 7]->setData( -1 );
    }
}

void Calendar::updateCalendar( const struct tm &new_time )
{
    struct tm t1 = new_time;

    t1.tm_isdst = -1;
    if ( mktime( &t1 ) != (time_t)-1 ) {
        m_current_date = t1;
    }

    std::stringstream dstr;

    dstr << m_current_date.tm_year + 1900;
    m_year_sg->setText( dstr.str().c_str() );

    dstr.str( "" );

    dstr << m_current_date.tm_mon + 1;
    m_month_sg->setText( dstr.str().c_str() );

    updateWindow();
}
