/* textstorage.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: textstorage.cc,v 1.3 2005/08/16 21:01:40 ralf Exp $ */

#include "textstorage.h"

TextStorageString::TextStorageString( const char *buf, int len, AWidth &lencalc ) : _lencalc( lencalc )
{
  std::string s1( buf, len );
  initBuffer( s1 );
}

TextStorageString::TextStorageString( std::string content, AWidth &lencalc ) : _lencalc( lencalc )
{
  initBuffer( content );
}

int TextStorageString::getNrOfLines() const
{
  return line_offsets.size();
}

int TextStorageString::getLine( int line_nr, unsigned int offset, int len, std::string &return_line ) const
{
  unsigned int start_offset, line_len;  // remember offset from line_offsets are not necessarly inside buffer

  if ( ( line_nr < 0 ) || ( line_nr >= getNrOfLines() ) ) return 1;

  start_offset = line_offsets[ line_nr ];
  line_len = getLineCharacters( line_nr );

  return_line = "";

  if ( offset < line_len ) {
    // offset given still inside line
    start_offset += offset;
    line_len -= offset;
    
    if ( len >= 0 ) {
      // line limiter given
      if ( (unsigned int )len < line_len ) line_len = (unsigned int)len;
    }
    if ( ( start_offset < buffer.length() ) && ( line_len > 0 ) )
      return_line.append( buffer, start_offset, line_len );
  }
  return 0;
}

int TextStorageString::getLine( int line_nr, unsigned int offset, std::string &return_line ) const
{
  return getLine( line_nr, offset, -1, return_line );
}

void TextStorageString::initBuffer( std::string &content, int line_limit_width )
{
  unsigned int i;

  buffer = content;
  line_offsets.clear();
  line_offsets.push_back( 0 );

  if ( line_limit_width < 1 ) {
    // the easiest stuff: no line limit
    for ( i = 0; i < buffer.length(); i++ ) {
      if ( buffer[i] == '\n' ) {
	line_offsets.push_back( i + 1 );  // if newline is last character
                                          // this line offset will be outside
                                          // the buffer
      }
    }
  } else {
    unsigned int cur_line_width, token_width, token_characters;
    bool newline;
    
    cur_line_width = 0;
    
    for ( i = 0; i < buffer.length();) {
      newline = false;
      if ( buffer[i] == '\n' ) {
	newline = true;
	i++;
      } else {
	token_characters = getTokenLen( i );
	if ( ( i + token_characters ) > buffer.length() ) {
	  token_characters = buffer.length() - i;
	}
	if ( token_characters < 1 ) token_characters = 1;
	token_width = _lencalc.getWidth( buffer.c_str() + i, token_characters );

	if ( ( cur_line_width + token_width ) <= (unsigned int)line_limit_width ) {
	  cur_line_width += token_width;
	  i += token_characters;
	} else if ( cur_line_width == 0 ) {
	  // no character in current line and no space for whole token
	  // so split it
	  token_characters = _lencalc.getStrlen4Width( buffer.c_str() + i, line_limit_width, NULL );

	  if ( token_characters < 1 ) {
	    // no room for at least one character at line start so we need to take
	    // this first one even this will break the line limit
	    token_characters = 1;
	  }
	  i += token_characters;
	  newline = true;
	} else {
	  newline = true;
	  // i is not advanced but for the newline cur_line_len will be 0
	  // in the next round so the previous case will catch on if it
	  // doesn't fit
	}
      }
      if ( newline == true ) {
	line_offsets.push_back( i );
	cur_line_width = 0;
      }
    }
  }
}

TextStorageString &TextStorageString::operator=( const TextStorageString &other )
{
  if ( this != &other ) {
    buffer = other.buffer;
    line_offsets = other.line_offsets;
  }
  return *this;
}

int TextStorageString::getMaxLineWidth() const
{
  int max_line_len = 0, l, i;

  for ( i = 0; i < getNrOfLines(); i++ ) {
    l = getLineWidth( i );
    if ( l > max_line_len ) max_line_len = l;
  }
  
  return max_line_len;
}

int TextStorageString::getLineWidth( int line_nr ) const
{
  unsigned int start_offset, line_len;  // remember offset from line_offsets are not necessarly inside buffer

  if ( ( line_nr < 0 ) || ( line_nr >= getNrOfLines() ) ) return -1;

  start_offset = line_offsets[ line_nr ];
  if ( ( line_nr + 1 ) < getNrOfLines() ) {
    // next line available
    line_len = line_offsets[ line_nr + 1 ] - start_offset;

    // if only newline results in new lines one can sub 1 above
    // but for word wrap it is possible to have a new line
    // without newline
    if ( ( line_len > 0 ) &&
	 ( buffer[ start_offset + line_len - 1 ] == '\n' ) )
      line_len--;
  } else {
    // no next line
    line_len = buffer.length() - start_offset;
  }
  return _lencalc.getWidth( buffer.c_str() + start_offset, line_len );
}

unsigned int TextStorageString::getTokenLen( unsigned int start_offset )
{
  unsigned int token_len;

  if ( start_offset >= buffer.length() ) return 0;
  
  for ( token_len = 0; start_offset < buffer.length(); start_offset++, token_len++ ) {
    if ( buffer[start_offset] == ' ' ) break;
    if ( buffer[start_offset] == '\n' ) break;
  }
  
  return token_len;
}

void TextStorageString::setLineLimit( int new_limit )
{
  if ( new_limit < -1 ) new_limit = -1;
  initBuffer( buffer, new_limit );
}

int TextStorageString::getLineCharacters( int line_nr ) const
{
  unsigned int start_offset, line_len;  // remember offset from line_offsets are not necessarly inside buffer

  if ( ( line_nr < 0 ) || ( line_nr >= getNrOfLines() ) ) return -1;

  start_offset = line_offsets[ line_nr ];
  if ( ( line_nr + 1 ) < getNrOfLines() ) {
    // next line available
    line_len = line_offsets[ line_nr + 1 ] - start_offset;

    // if only newline results in new lines one can sub 1 above
    // but for word wrap it is possible to have a new line
    // without newline
    if ( ( line_len > 0 ) &&
	 ( buffer[ start_offset + line_len - 1 ] == '\n' ) )
      line_len--;
    return line_len;
  } else {
    // no next line
    return buffer.length() - start_offset;
  }
}
