/* textview.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not^, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: textview.cc,v 1.2 2005/05/15 13:48:07 ralf Exp $ */

#include "textview.h"
#include "acontainer.h"

TVCallBack::TVCallBack( TextView *_tv )
{
  tv = _tv;
}

TVCallBack::~TVCallBack()
{
}

void TVCallBack::run( GUIElement *elem, int value )
{
  if ( tv != NULL ) tv->gui_callback( elem, value );
}

void TVCallBack::run( GUIElement *elem, void *p )
{
}

TextView::TextView( AGUIX *parent,
		    int x,
		    int y,
		    int width,
		    int height,
		    int bg,
		    std::string title,
		    TextStorage &_ts ) : AWindow( parent, x, y, width, height, bg, title ), ts( _ts ), tvcb( this )
{
  hbar = vbar = NULL;
  cont = NULL;
  font = NULL;
  last_w = last_h = -1;
  line_wrap = false;
  _display_focus = false;
  
  setCanHandleFocus();
  setAcceptFocus( true );
  
  gettimeofday( &_lastwheel, NULL );
}

TextView::~TextView()
{
  destroy();
}

void TextView::doCreateStuff()
{
  AWindow::doCreateStuff();
  if ( _created == false ) return;

  hbar = new Slider( _aguix, 0, 0, 30, 10, false, 0 );
  hbar->setCallbackHandler( &tvcb );
  hbar->setAcceptFocus( false );
  vbar = new Slider( _aguix, 0, 0, 10, 30, true, 1 );
  vbar->setCallbackHandler( &tvcb );
  vbar->setAcceptFocus( false );
  
  createContainer();
  updateBars();
  return;
}

void TextView::createContainer()
{
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cinch = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXW;

  if ( isCreated() == false ) return;
  
  if ( cont != NULL ) {
    setContainer( NULL );
    delete cont;
  }
  cont = new AContainer( this, 2, ( line_wrap == true ) ? 1 : 2 );
  cont->setMinSpace( 0 );
  cont->setMaxSpace( 0 );
  cont->setBorderWidth( ( _display_focus == true ) ? 2 : 1 );
  setContainer( cont, true );
  
  if ( vbar != NULL ) {
    cont->add( vbar, 1, 0, cinch );
    cont->setMinHeight( 30, 1, 0 );
  }
  if ( hbar != NULL ) {
    if ( line_wrap == false ) {
      cont->add( hbar, 0, 1, cincw );
      hbar->show();
    } else {
      hbar->hide();
    }
  }
  cont->setMinWidth( 2, 0, 0 );
  cont->setMinHeight( 2, 0, 0 );
  
  cont->resize( _w, _h );
  cont->rearrange();
}

void TextView::boxRedraw()
{
  _aguix->SetWindowBG( win, getBG() );
  _aguix->ClearWin( win );
  _aguix->setFG( 0, 2 );
  _aguix->DrawLine( win, 0, 0, _w - 1, 0 );
  _aguix->DrawLine( win, 0, 0, 0, _h - 1 );
  _aguix->setFG( 0, 1 );
  _aguix->DrawLine( win, 0, _h - 1, _w - 1, _h - 1 );
  _aguix->DrawLine( win, _w - 1, _h - 1, _w - 1, 1 );

  if ( getDisplayFocus() == true ) {
    if ( getHasFocus() == true ) {
      _aguix->setFG( 0, 2 );
    } else {
      _aguix->setFG( 0, 1 );
    }
    _aguix->DrawLine( win, 1, _h - 2, _w - 2, _h - 2 );
    _aguix->DrawLine( win, _w - 2, _h - 2, _w - 2, 2 );
    
    if ( getHasFocus() == true ) {
      _aguix->setFG( 0, 1 );
    } else {
      _aguix->setFG( 0, 2 );
    }
    _aguix->DrawLine( win, 1, 1, _w - 2, 1 );
    _aguix->DrawLine( win, 1, 1, 1, _h - 2 );
  }
}

bool TextView::handleMessage(XEvent *E,Message *msg)
{
  struct timeval t2;
  int dt, scrollspeed;
  
  if ( isCreated() == false ) return false;
  
  if ( ( msg->type == Expose ) &&
       ( msg->window == win ) ) {
    redraw();
  } else if ( msg->type == KeyPress ) {
    if ( ( getAcceptFocus() == true ) && ( getHasFocus() == true ) ) {
      if ( isVisible() == true ) {
	if ( isTopParent( msg->window ) == true ) {
	  // we have the focus so let's react to some keys
	  // lets call an extra handler so it can be overwritten
	  switch ( msg->key ) {
	    case XK_Up:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() - 1 );
		redraw();
	      }
	      break;
	    case XK_Left:
	      if ( hbar != NULL ) {
		hbar->setOffset( hbar->getOffset() - 1 );
		redraw();
	      }
	      break;
	    case XK_Down:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() + 1 );
		redraw();
	      }
	      break;
	    case XK_Right:
	      if ( hbar != NULL ) {
		hbar->setOffset( hbar->getOffset() + 1 );
		redraw();
	      }
	      break;
	    case XK_Home:
	      if ( vbar != NULL ) {
		vbar->setOffset( 0 );
		redraw();
	      }
	      break;
	    case XK_End:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getMaxLen() );
		redraw();
	      }
	      break;
	    case XK_Prior:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() - vbar->getMaxDisplay() + 1 );
		redraw();
	      }
	      break;
	    case XK_Next:
	      if ( vbar != NULL ) {
		vbar->setOffset( vbar->getOffset() + vbar->getMaxDisplay() - 1 );
		redraw();
	      }
	      break;
	    case XK_w:
	      setLineWrap( ( getLineWrap() == true ) ? false : true );
	      break;
	    default:
	      break;
	  }
	}
      }
    }
  } else if ( ( msg->type == ButtonPress ) && ( msg->window == win ) ) {
    if ( ( msg->button == Button4 ) || ( msg->button == Button5 ) ) {
      gettimeofday( &t2, NULL );
      dt = ldiffgtod_m( &t2, &_lastwheel );
      
      if ( dt < 200 ) scrollspeed = 5;
      else if ( dt < 400 ) scrollspeed = 2;
      else scrollspeed = 1;
      
      if ( msg->button == Button4 ) {
	if ( vbar != NULL ) {
	  int old_offset = vbar->getOffset();
	  vbar->setOffset( vbar->getOffset() - scrollspeed );
	  if ( vbar->getOffset() != old_offset )
	    redraw();
	}
      } else if ( msg->button == Button5 ) {
	if ( vbar != NULL ) {
	  int old_offset = vbar->getOffset();
	  vbar->setOffset( vbar->getOffset() + scrollspeed );
	  if ( vbar->getOffset() != old_offset )
	    redraw();
	}
      }
      
      _lastwheel = t2;
    } else if ( msg->button == Button1 ) {
      applyFocus( this );
    }
  }

  return AWindow::handleMessage( E, msg );
}

void TextView::doDestroyStuff()
{
  AWindow::doDestroyStuff();
  delete cont;
  cont = NULL;
  setContainer( NULL );
  hbar = vbar = NULL;
}

void TextView::gui_callback( GUIElement *elem, int value )
{
  if ( ( elem == vbar ) || ( elem == hbar ) ) {
    redraw();
  }
}

void TextView::redraw()
{
  int tx, ty, elementHeight, lines, chars;
  GC usegc;

  if ( isCreated() == false ) return;

  boxRedraw();

  tx = ty = ( _display_focus == true ) ? 3 : 2;

  if ( font == NULL ) usegc = 0;
  else usegc = font->getGC();

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();

  getLinesChars( lines, chars );

  if ( ( _w != last_w ) || ( _h != last_h ) ) {
    if ( line_wrap == true ) {
      ts.setLineLimit( chars );
    } else {
      ts.setLineLimit( -1 );
    }
    updateBars();
  }

  if ( font == NULL )
    _aguix->setFG( 1 );
  else
    _aguix->setFG( font->getGC(), 1 );

  for ( int i = 0; i < lines; i++ ) {
    std::string s1;
    ts.getLine( vbar->getOffset() + i, hbar->getOffset(), chars, s1 );
    if ( font == NULL )
      _aguix->DrawText( win, s1.c_str(), tx, i * elementHeight + ty );
    else
      _aguix->DrawText( win, font, s1.c_str(), tx, i * elementHeight + ty );
  }
}

int TextView::setFont( char *fontname )
{
  font = _aguix->getFont( fontname );
  last_w = last_h = -1;
  if ( font == NULL ) return -1;
  return 0;
}

void TextView::updateBars()
{
  int lines, chars;
  
  if ( isCreated() == false ) return;

  getLinesChars( lines, chars );
  
  vbar->setMaxDisplay( lines );
  hbar->setMaxDisplay( chars );
  vbar->setMaxLen( ts.getNrOfLines() );
  hbar->setMaxLen( ts.getMaxLineLen() );
}

void TextView::setLineWrap( bool nv )
{
  line_wrap = nv;
  last_w = last_h = -1; // force textstorage rebuild
  createContainer();
  redraw();
}

bool TextView::getLineWrap() const
{
  return line_wrap;
}

void TextView::getLinesChars( int &lines, int &chars ) const
{
  int tw ,th, elementHeight, elementWidth;

  if ( cont != NULL ) {
    tw = cont->getWidth( 0, 0 ) - 2;
    if ( tw < 0 ) tw = 0;
    th = cont->getHeight( 0, 0 ) - 2;
    if ( th < 0 ) th = 0;
  } else {
    tw = th = 0;
  }

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();
  if ( font == NULL ) elementWidth = _aguix->getCharWidth();
  else elementWidth = font->getCharWidth();

  lines = th / elementHeight;
  chars = tw / elementWidth;
}

void TextView::setDisplayFocus( bool nv )
{
  _display_focus = nv;
  redraw();
}

bool TextView::getDisplayFocus() const
{
  return _display_focus;
}

void TextView::maximizeX()
{
  int l, elementWidth;

  if ( line_wrap == true ) return;
  
  if ( font == NULL ) elementWidth = _aguix->getCharWidth();
  else elementWidth = font->getCharWidth();
  
  l = ( ts.getMaxLineLen() + 1 ) * elementWidth;
  l += 2 + 2 + ( vbar != NULL ) ? vbar->getWidth() : 0;

  if ( l >= _aguix->getRootWindowWidth() )
    l = _aguix->getRootWindowWidth();

  resize( l, getHeight() );
}

void TextView::maximizeYLines( int max_lines )
{
  int l, elementHeight;

  if ( font == NULL ) elementHeight = _aguix->getCharHeight();
  else elementHeight = font->getCharHeight();
  
  l = ts.getNrOfLines();
  if ( l > max_lines ) l = max_lines;
  l = ( l + 1 ) * elementHeight;
  l += 2 + 2 + ( ( hbar != NULL ) && ( line_wrap == false ) ) ? hbar->getHeight() : 0;

  if ( l >= _aguix->getRootWindowHeight() )
    l = _aguix->getRootWindowHeight();

  resize( getWidth(), l );
}
