/* button.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: button.cc,v 1.28 2005/05/15 13:45:12 ralf Exp $ */

#include "button.h"
#include "awindow.h"
#include "guielement.h"

const char *Button::type="Button";

Button::~Button()
{
  if(text[0]!=NULL) _freesafe(text[0]);
  if(text[1]!=NULL) _freesafe(text[1]);
}

Button::Button(AGUIX *taguix,int tx,int ty,int width,
               const char *ttext,int tfg,int tbg,int tdata):GUIElement(taguix)
{
  init_class();
  
  _x = tx;
  _y = ty;
  if ( width > 0 ) _w = width;
  _h = taguix->getCharHeight() + 4;
  fg[0] = tfg;
  fg[1] = 0;
  bg[0] = tbg;
  bg[1] = 0;

  data = tdata;
  _freesafe( text[0] );
  text[0] = dupstring( ttext );
}

Button::Button(AGUIX *taguix,int tx,int ty,
               const char *ttext,int tfg,int tbg,int tdata):GUIElement(taguix)
{
  init_class();
  
  _x = tx;
  _y = ty;
  _w = ( strlen( ttext ) + 2 ) * taguix->getCharWidth();
  _h = taguix->getCharHeight() + 4;

  data = tdata;
  _freesafe( text[0] );
  text[0] = dupstring( ttext );
  fg[0] = tfg;
  fg[1] = 0;
  bg[0] = tbg;
  bg[1] = 0;
}

Button::Button(AGUIX *taguix,int tx,int ty,int width,int height,
               const char *ttext,int tfg,int tbg,int tdata):GUIElement(taguix)
{
  init_class();
  
  _x = tx;
  _y = ty;
  if ( width > 0 ) _w = width;
  if ( height > 0 ) _h = height;

  data = tdata;
  _freesafe( text[0] );
  text[0] = dupstring( ttext );
  fg[0] = tfg;
  fg[1] = 0;
  bg[0] = tbg;
  bg[1] = 0;
}

Button::Button(AGUIX *taguix,int tx,int ty,int width,
               const char *text_normal,const char *text_high,
               int fg_normal,int fg_high,
               int bg_normal,int bg_high,int tdata):GUIElement(taguix)
{
  init_class();
  
  _x = tx;
  _y = ty;
  if ( width > 0 ) _w = width;
  _h = taguix->getCharHeight() + 4;

  dual = true;
  data = tdata;
  _freesafe( text[0] );
  _freesafe( text[1] );
  text[0] = dupstring( text_normal );
  text[1] = dupstring( text_high );
  fg[0] = fg_normal;
  fg[1] = fg_high;
  bg[0] = bg_normal;
  bg[1] = bg_high;
}

const char *Button::getText(int index2) const
{
  if((index2<0)||(index2>1)) return NULL;
  return dupstring(text[index2]);
}

void Button::setText(int index2,const char *new_text)
{
  if((index2<0)||(index2>1)) return;
  if(text[index2]!=NULL) _freesafe(text[index2]);
  text[index2]=dupstring(new_text);
  redraw();
}

bool Button::getDualState() const
{
  return dual;
}

void Button::setDualState(bool new_dual)
{
  dual=new_dual;
  redraw();
}

void Button::setFG(int index2,int tfg)
{
  if((index2<0)||(index2>1)) return;
  if((tfg<(_aguix->getMaxCols()))&&(tfg>=0)) {
    this->fg[index2]=tfg;
  }
  redraw();
}

int Button::getFG(int index2) const
{
  if((index2<0)||(index2>1)) return 0;
  return fg[index2];
}

void Button::setBG(int index2,int tbg)
{
  if((index2<0)||(index2>1)) return;
  if((tbg<(_aguix->getMaxCols()))&&(tbg>=0)) {
    this->bg[index2]=tbg;
  }
  redraw();
}

int Button::getBG(int index2) const
{
  if((index2<0)||(index2>1)) return 0;
  return bg[index2];
}

int Button::getData() const
{
  return data;
}

void Button::setData(int tdata)
{
  this->data=tdata;
}

void Button::redraw()
{
  int newbg;
  GC usegc;
  bool showe=false;
  int px1 = 0,
      py1 = 0,
      px2 = 0,
      py2 = 0,
      px3 = 0,
      py3 = 0;
  int dw,dw2;
  int dy;
  unsigned int maxdisplay;
  char *tstr;
  int ch,cw;

  if ( isCreated() == false ) return;
  if ( win == 0 ) return;

  /* Clear the window */

  if ( ( dual == false ) || ( state != 2 ) ) {
    newbg=bg[0];
  } else {
    newbg=bg[1];
  }
  _aguix->SetWindowBG(win,newbg);
  lastbg=newbg;
  _aguix->ClearWin(win);
  if(font==NULL) usegc=0; else usegc=font->getGC();

  /* calculate dimension of the turned-down corner */

  if ( ( dual == true ) && ( showdual == true ) && ( state != 2 ) ) {
    showe = true;

    dw=_w/4;
    dw2=(int)((double)(_h/3)*1.3);
    if( dw2 < dw ) dw = dw2;

    px1 = _w-dw;
    py1 = 0;
    px2 = _w-dw;
    py2 = dw-1;
    px3 = _w-1;
    py3 = dw-1;
  }

  /* draw the text */

  if ( ( dual == false ) || ( state != 2 ) ) {
    _aguix->setFG(usegc,fg[0]);
  } else {
    _aguix->setFG(usegc,fg[1]);
  }
  if(font==NULL) {
    ch=_aguix->getCharHeight();
    cw=_aguix->getCharWidth();
  } else {
    ch=font->getCharHeight();
    cw=font->getCharWidth();
  }

  dy=_h/2-ch/2;
  maxdisplay=(_w-2)/cw;
  if ( ( dual == false ) || ( state != 2 ) ) {
    tstr=dupstring(text[0]);
  } else {
    tstr=dupstring(text[1]);
  }
  if(strlen(tstr)>maxdisplay) tstr[maxdisplay]=0;
  dw=strlen(tstr)*cw;
  int dx=_w/2-dw/2;
  if(font==NULL) {
    _aguix->DrawText(win,tstr,dx,dy);
  } else {
    _aguix->DrawText(win,font,tstr,dx,dy);
  }
  _freesafe(tstr);

  // draw the focus
  if ( ( getHasFocus() == true ) && ( getAcceptFocus() == true ) ) {
    _aguix->setDottedFG( 1 );
    _aguix->DrawDottedRectangle( win, 1, 1, _w - 2, _h - 2 );
  }

  /* now draw the button */

  if( showe == true ) {
    _aguix->setFG( usegc, 2 );
    _aguix->DrawTriangleFilled( win, usegc, px1, py1, px2, py2, px3, py3 );
    _aguix->setFG( usegc, 0 );
    _aguix->DrawTriangleFilled( win, usegc, px1, py1, _w-1, 0, px3, py3 );
  }

  if( state == 0 )
    _aguix->setFG( usegc, 2 );
  else
    _aguix->setFG( usegc, 1 );

  _aguix->DrawLine( win, usegc, 0, _h-1, 0, 0 );

  if( showe == true ) {
    _aguix->DrawLine( win, usegc, 0, 0, px1, py1 );
    _aguix->setFG( usegc, 1 );
    _aguix->DrawLine( win, usegc, px1, py1, px2, py2 );
  } else
    _aguix->DrawLine( win, usegc, 0, 0, _w-1, 0 );

  if( state == 0 )
    _aguix->setFG( usegc, 1 );
  else
    _aguix->setFG( usegc, 2 );

  _aguix->DrawLine( win, usegc, 0, _h-1, _w-1, _h-1 );

  if( showe == true ) {
    _aguix->DrawLine( win, usegc, px3, py3, _w-1, _h-1 );
    _aguix->setFG( usegc, 1 );
    _aguix->DrawLine( win, usegc, px2, py2, px3, py3 );
    _aguix->DrawLine( win, usegc, px1, py1, px3, py3 );
  } else
    _aguix->DrawLine(win,usegc,_w-1,_h-1,_w-1,1);

  _aguix->Flush();
}

void Button::flush()
{
}

void Button::setState(int tstate)
{
  if((dual==false)&&(tstate==2)) {
    this->state=1;
  } else {
    this->state=tstate;
  }
  redraw(); 
}

int Button::getState() const
{
  return state;
}

bool Button::isInside(int px,int py) const
{
  if((px>0)&&(px<=_w)) {
    if((py>0)&&(py<=_h)) return true;
  }
  return false;
}

bool Button::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue;

  if ( isCreated() == false ) return false;

  returnvalue=false;
  if((msg->type==ButtonPress)||(msg->type==ButtonRelease)) {
    if ( msg->window == win ) {
      if ( ( msg->button == Button3 ) ||
	   ( msg->button == Button1 ) ||
	   ( ( msg->button == Button4 ) && ( allowWheel == true ) ) ||
	   ( ( msg->button == Button5 ) && ( allowWheel == true ) ) ) {
	int mx,my;
	mx=msg->mousex;
	my=msg->mousey;
	if(msg->type==ButtonPress) {
	  takeFocus();
	  if(msg->window==win) {
	    if( ( ( ( msg->button == Button1 ) || ( msg->button == Button4 ) || ( msg->button == Button5 ) ) && ( active[0] == true ) ) ||
		( ( msg->button == Button3 ) && ( active[1] == true ) ) ) {
	      AGMessage *agmsg = AGUIX_allocAGMessage();
	      agmsg->type=AG_BUTTONPRESSED;
	      agmsg->button.button=this;
	      if((msg->button==Button3)&&(dual==true)) {
		setState(2);
		instate=2;
		agmsg->button.state=2;
	      } else {
		switch ( msg->button ) {
		case Button4:
		  instate = 3;
		  break;
		case Button5:
		  instate = 4;
		  break;
		default:
		  instate=1;
		  break;
		}
		setState( instate );
		agmsg->button.state = instate;
	      }
	      _aguix->putAGMsg(agmsg);
	      returnvalue=true;
	    }
	  }
	} else {
	  if((state!=0)&&(instate!=0)) {
	    AGMessage *agmsg = AGUIX_allocAGMessage();
	    agmsg->type=AG_BUTTONRELEASED;
	    agmsg->button.button=this;
	    agmsg->button.state=instate;
	    _aguix->putAGMsg(agmsg);
	    agmsg = AGUIX_allocAGMessage();
	    agmsg->type=AG_BUTTONCLICKED;
	    agmsg->button.button=this;
	    agmsg->button.state=instate;
	    _aguix->putAGMsg(agmsg);
	  }
	  if(instate!=0) {
	    setState(0);
	    instate=0;
	    returnvalue=true;
	  }
	}
      }
    }
  } else if(msg->type==EnterNotify) {
    // alles hier und alles mit instate wird benutzt, damit Button sich anpat, wenn
    // Mauszeiger im Button oder auerhalb des Buttons ist
    if ( msg->window == win ) {
      if(instate!=0) {
	if(state!=instate) {
	  setState(instate);
	}
      }
    }
  } else if(msg->type==LeaveNotify) {
    // alles hier und alles mit instate wird benutzt, damit Button sich anpat, wenn
    // Mauszeiger im Button oder auerhalb des Buttons ist
    if ( msg->window == win ) {
      if(instate!=0) {
	setState(0);
      }
    }
  } else if(msg->type==Expose) {
    if ( msg->window == win ) {
      redraw();
    }
  } else if ( msg->type == KeyPress ) {
    if ( ( getAcceptFocus() == true ) && ( getHasFocus() == true ) ) {
      if ( msg->key == XK_space ) {
	if ( isVisible() == true ) {
	  if ( _parent->isTopParent( msg->window ) == true ) {
	    AGMessage *agmsg;

	    agmsg = AGUIX_allocAGMessage();
	    agmsg->type=AG_BUTTONCLICKED;
	    agmsg->button.button=this;
	    agmsg->button.state=1;
	    _aguix->putAGMsg(agmsg);
	    if(instate!=0) {
	      setState(0);
	      instate=0;
	      returnvalue=true;
	    }
	  }
	}
      }
    }
  }
  if(returnvalue==true) {
    // jetzt noch die Message mit den Werten fllen
    msg->gadget=this;
    msg->gadgettype=BUTTON_GADGET;
  }
//  return returnvalue;
  return false; // see cyclebutton.cc
}

bool Button::getShowDual() const
{
  return showdual;
}

void Button::setShowDual(bool new_val)
{
  if(showdual!=new_val) {
    showdual=new_val;
    redraw();
  } else showdual=new_val;
}

int Button::setFont(char *fontname)
{
  font=_aguix->getFont(fontname);
  if(font==NULL) return -1;
  return 0;
}

const char *Button::getType() const
{
  return type;
}

bool Button::isType(const char *qtype) const
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

void Button::deactivate()
{
  deactivate(0);
  deactivate(1);
}

void Button::activate()
{
  activate(0);
  activate(1);
}

void Button::deactivate(int mode)
{
  if( ( mode == 0 ) || ( mode == 1 ) ) {
    active[mode]=false;
    if ( mode == 0 ) setAcceptFocus( false );
  }
}

void Button::activate(int mode)
{
  if( ( mode == 0 ) || ( mode == 1 ) ) {
    active[mode]=true;
  }
}

bool Button::getAllowWheel() const
{
  return allowWheel;
}

void Button::setAllowWheel( bool nv )
{
  allowWheel = nv;
}

void Button::init_class()
{
  dual = false;
  data = 0;
  state = 0;
  instate = 0;
  lastbg = -1;
  showdual = true;
  font = NULL;
  active[0] = true;
  active[1] = true;
  fg[0] = 0;
  fg[1] = 0;
  bg[0] = 0;
  bg[1] = 0;
  text[0] = dupstring( "" );
  text[1] = dupstring( "" );
  allowWheel = false;
  
  setCanHandleFocus();
  setAcceptFocus( true );
}
