/*  Wordtrans, front-end for several dictionaries, for the Qt toolkit.
    Copyright (C) 2000 Ricardo Villalba <rvm@linuxfan.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "QTextDiccionario.h"
#include "qfunciones.h"
#include "progress.h"
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>
//#include <stdlib.h>
#include "intern.h"
#include <QConfig.h>

QTextDiccionario::QTextDiccionario() : DiccionarioBase() {
	setTipo( DiccionarioBase::Text_dict );
}

QTextDiccionario::~QTextDiccionario() {
	//delete conf;
}


void QTextDiccionario::extraeDatos(QString& output, bool invertir, 
                                   QString & resultado, bool con_formato,
                                   bool contabiliza) {
	//debug("QTextDiccionario::extraeDatos: Iniciando");

	resultado="";

	if (contabiliza) {
		n_matches=0;
		bestMatch="";
	}

	unsigned long_menor=1000;


	int long_separador=separador.length();

	//En output est la salida del comando ejecutado previamente.
	if (!output.isEmpty()) {
		QTextStream t( &output, IO_ReadOnly );
		QString linea;

		while ( !t.eof() ) {        // until end of file...
    		linea = t.readLine();       // line of text excluding '\n'

			//qDebug("%s", linea.latin1() );
			//UTF8
			if (getUTF8()) {
				linea = QString::fromUtf8( linea.latin1() );
			}

			if (invertir) {
				int p=linea.find(separador, 0, FALSE);
				QString izq=linea.left(p);
				QString der=linea.right(linea.length() - (p + long_separador) );
				//debug("izq: '%s' der: '%s'", izq.data(), der.data() );
				linea= der + separador + izq;
			}

			//Sustituye el separador por otro que quede mejor en pantalla.
			int p=linea.find(separador, 0, FALSE);
			if (p!=-1) linea.replace( p, separador.length(), " --> ");

			if (contabiliza) {
				//debug( linea.latin1() );
				n_matches++;
				if (linea.length() < long_menor) {
					long_menor=linea.length();
					bestMatch=linea;
				}
			}

			linea += "\n";
			//if (con_formato) linea += "<br>";
			if (con_formato) {
				if (DiccionarioBase::html_use_br)
					linea += "<br>";
				else
					linea = "<p>" + linea + "</p>";
			}
			resultado += linea;
		}
	}

	//debug("QTextDiccionario::extraeDatos: Finalizado");
}

bool QTextDiccionario::comprimido(const QString& diccionario) {
	QFileInfo fi(diccionario);
	return ( fi.extension(false) == "gz" );
}


QCString QTextDiccionario::creaComando(const QString& palabra, const QString& diccionario) { 
	QCString comando;

	if (comprimido(diccionario))
		comando="zegrep -h ";
	else
		comando="egrep -h ";

	if (palabras_completas) comando += "-w ";
	if (!case_sensitive) comando += "-i ";

	comando = comando + "-e \"" + palabra.latin1() + "\" " + diccionario.latin1();

	qDebug("Comando: '%s'", comando.data() );

	return comando;
}



void QTextDiccionario::formateaTexto(QString & texto, const QString& c_search) {
	//debug("QTextDiccionario::formateaTexto: Iniciando");

	QString rep= "<b>" + c_search + "</b>";

	if (c_search.isEmpty()) return;	

	int contador=0;

	Progress * progreso;
	if (!DiccionarioBase::no_gui) {
		progreso = new Progress();
		progreso->setLabel( _("Formating text...") );
	}

	int index=0;

	while (index != -1) {
		index = texto.find( c_search, index );
		if (index != -1) {
			texto.replace( index, c_search.length(), rep );
			index += c_search.length();
			index += 7; //La longitud de <b></b>
		}
		contador++;
		if (contador >= 100) {
			contador=0;
			if (!DiccionarioBase::no_gui) progreso->show( index, texto.length() );
		}
	}

	if (!DiccionarioBase::no_gui) {
		progreso->end();
		delete progreso;
	}

	//debug("QTextDiccionario::formateaTexto: Finalizado");
}



bool QTextDiccionario::busca(const QString& palabra, QString & resultado,
                             bool con_formato) {
	resultado="";

	QString word=palabra;

	word=word.stripWhiteSpace();

	if (ignora_acentos) word= getExprIgnorarAcentos( word );

	if (word.isEmpty()) {
		setError( _("Nothing to look for") );
		return false;
	}

	if (word.contains( "\"", FALSE)!=0 ) {
		setError( _("Quotation marks not allowed") );
		return false;
	}

	trabajando=true;

	QCString comando;
	QString output="";

	QString palabra_busqueda; //Palabra a buscar, con formato UTF8 o no

	if (getUTF8())
		palabra_busqueda=word.utf8();
	else
		palabra_busqueda=word;

	//Diccionario principal
	if ( QFile::exists(diccionario) ) {
		comando=creaComando( palabra_busqueda, diccionario);
		DiccionarioBase::exec_command( comando, output );
	} else {
		qDebug( "File '%s' doesn't exist!!!", diccionario.latin1() );
	}

	//Diccionario personal
	if ( QFile::exists(personal_dict) ) {
		comando=creaComando( palabra_busqueda, personal_dict);
		DiccionarioBase::exec_command( comando, output );
	} else {
		qDebug( "File '%s' doesn't exist!!!", personal_dict.latin1() );
	}

	extraeDatos(output, busqueda_inversa, resultado, con_formato, true);

	//Busca la mejor traduccin y el nmero de traducciones
	//analizaTexto( resultado );

	if (con_formato) 
		formateaTexto( resultado, word );

	trabajando=false;
	return true;
}

bool QTextDiccionario::chequea(const QString& texto_idioma1, const QString& texto_idioma2,
     						   QString & resultado) {
	QCString patron;

	if ( (texto_idioma1.isEmpty()) || (texto_idioma2.isEmpty()) )
		return false;

	patron += "^";

	if (getUTF8())
		patron += texto_idioma1.utf8();
	else
		patron += texto_idioma1.latin1();

	patron += separador.latin1();

	if (getUTF8())
		patron += texto_idioma2.utf8();
	else
		patron += texto_idioma2.latin1();

	patron += "$";


	resultado="";

	QCString comando;
	QString output="";

	//Diccionario principal
	if ( QFile::exists(diccionario) ) {
		if (comprimido(diccionario)) comando="zgrep"; else comando="grep";
		comando += " -i \"" + patron + "\" " + diccionario.latin1();
		DiccionarioBase::exec_command( comando, output );
	} else {
		qDebug( "File '%s' doesn't exist!!!", diccionario.latin1() );
	}

	//Diccionario personal
	if ( QFile::exists(personal_dict) ) {
		if (comprimido(personal_dict)) comando="zgrep"; else comando="grep";
		comando += " -i \"" + patron + "\" " + personal_dict.latin1();
		DiccionarioBase::exec_command( comando, output );
	} else {
		qDebug( "File '%s' doesn't exist!!!", personal_dict.latin1() );
	}

	extraeDatos(output, busqueda_inversa, resultado, false);

	return true;
}

bool QTextDiccionario::aprende(const QString& texto_idioma1, const QString& texto_idioma2) {
	QString texto1=texto_idioma1;
	QString texto2=texto_idioma2;

	texto1=texto1.stripWhiteSpace();
	texto2=texto2.stripWhiteSpace();

	QString resultado;

	if (!chequea(texto1, texto2, resultado)) {
		setError(
			_("There's been an error while trying to check if '%1 - %2' "
			  "are in the dictionary").arg(texto1).arg(texto2) );
		qDebug( error.latin1() );
		return false;
	}

	bool encontrado= !resultado.isEmpty();
	
	if (encontrado) {
		setError(
			_("%1 - %2 are already in the dictionary").arg(texto1).arg(texto2) );
		qDebug( error.latin1() );
		return false;
	}

	QCString nueva_linea;
	if (getUTF8())
		nueva_linea += texto1.utf8();
	else
		nueva_linea += texto1.latin1();

	nueva_linea += separador.latin1();

	if (getUTF8())
		nueva_linea += texto2.utf8();
	else
		nueva_linea += texto2.latin1();

	nueva_linea += "\n";
    //printf("Linea nueva: '%s'\n", nueva_linea.data());

	//Guarda lnea en el fichero
	QFile f( personal_dict );
	if (f.open( IO_WriteOnly | IO_Append ) ) {
		QTextStream t( &f );
        //La siguiente lnea es necesaria para guardat
        //correctamente el texto en UTF-8 (lo s es raro)
		t.setEncoding( QTextStream::Latin1 );
		t << nueva_linea.data(); // << "\n";
		f.close();
	} else {
		setError(
			_("Error when trying to open %1").arg(personal_dict) );
		qDebug( error.latin1() );
		return false;
	}

	return true;
}

/*
void QTextDiccionario::analizaTexto(QStringList & lista) {

	bestMatch="";

	unsigned int long_menor=1000;
	QString linea;
	for ( QStringList::Iterator it = lista.begin(); it != lista.end(); ++it ) {
		linea= (*it);
		if (linea.length() < long_menor) { 
			long_menor=linea.length();
			bestMatch=linea;
		}
	}

	n_matches=lista.count();
}
*/

QString QTextDiccionario::getExprIgnorarAcentos( const QString& palabra ) {
	QString s="";

	QChar c;
	for (unsigned int n=0; n <= palabra.length(); n++) {
		c = palabra[n];

		if ( (c == QChar('a')) || (c == QChar('')) ) s += "[a]";
		else
		if ( (c == QChar('e')) || (c == QChar('')) ) s += "[e]";
		else
		if ( (c == QChar('i')) || (c == QChar('')) ) s += "[i]";
		else
		if ( (c == QChar('o')) || (c == QChar('')) ) s += "[o]";
		else
		if ( (c == QChar('u')) || (c == QChar('')) || (c == QChar('')) ) s += "[u]";
		else
		s += c;
	}

	return s;
}

void QTextDiccionario::save() {
	conf->setGroup("General");

	conf->writeEntry("alias", alias.latin1() );
	conf->writeEntry("idioma1", idioma1.latin1() );
	conf->writeEntry("idioma2", idioma2.latin1() );
	conf->writeEntry("separador", separador.latin1() );
	conf->writeBoolEntry("utf8", getUTF8() );

	conf->setGroup("Diccionarios");
	conf->writeEntry("general", diccionario.latin1() );
	conf->writeEntry("personal", personal.latin1() );

	conf->setGroup("Opciones");
	conf->writeBoolEntry("busqueda_inversa", busqueda_inversa );
	conf->writeBoolEntry("case_sensitive", case_sensitive );
	conf->writeBoolEntry("palabras_completas", palabras_completas );
	conf->writeBoolEntry("ignorar_acentos", ignora_acentos );

	conf->setGroup("Iconos");
	conf->writeEntry("idioma1", getIconoIdioma1().latin1() );
	conf->writeEntry("idioma2", getIconoIdioma2().latin1() );

	save_desc();
	save_font();

	conf->sync();
}

void QTextDiccionario::load(const QString& fichero) {
	//debug("QTextDiccionario::load: leyendo %s", fichero.data() );

	conf->setFileName(fichero.latin1());

	conf->setGroup("General");

	alias = conf->readEntry("alias", "sin_alias");
	idioma1 = conf->readEntry("idioma1", "language1");
	idioma2 = conf->readEntry("idioma2", "language2");
	separador = conf->readEntry("separador", " : ");
	setUTF8( conf->readBoolEntry("utf8", false ) );

	conf->setGroup("Diccionarios");
	diccionario = conf->readEntry("general", "/usr/share/i2e/i2e.dict");
	setPersonal( conf->readEntry("personal", ".i2e.dict") );

	conf->setGroup("Opciones");
	busqueda_inversa= conf->readBoolEntry("busqueda_inversa", false );
	case_sensitive = conf->readBoolEntry("case_sensitive", false );
	palabras_completas= conf->readBoolEntry("palabras_completas", true );
	ignora_acentos= conf->readBoolEntry("ignorar_acentos", false );

	conf->setGroup("Iconos");
	setIconoIdioma1( conf->readEntry("idioma1","/usr/share/wordtrans/i2e.xpm") );
	setIconoIdioma2( conf->readEntry("idioma2","/usr/share/wordtrans/e2i.xpm") );

	load_desc();
	load_font();
}

