open Lexing
open Lex_engines

open Test
open Test_classes

let rec classify2 lexbuf =
  if lexbuf.lex_curr_pos >= lexbuf.lex_buffer_len then
    if lexbuf.lex_eof_reached then wrong_code_point
    else (lexbuf.refill_buff lexbuf; classify2 lexbuf)
  else let c = lexbuf.lex_buffer.[lexbuf.lex_curr_pos] in
  lexbuf.lex_curr_pos <- lexbuf.lex_curr_pos + 1;
  match c with
    | _ -> letter

let rec classify lexbuf =
  if lexbuf.lex_curr_pos >= lexbuf.lex_buffer_len then
    if lexbuf.lex_eof_reached then eof
    else (lexbuf.refill_buff lexbuf; classify lexbuf)
  else let c = lexbuf.lex_buffer.[lexbuf.lex_curr_pos] in
  lexbuf.lex_curr_pos <- lexbuf.lex_curr_pos + 1;
  match c with
    | '.' -> char_2e
    | '' -> char_e9
    | '+' -> char_2b
    | '-' -> char_2d
    | ' ' -> char_20
    | '\n'-> char_0a
    | '(' -> char_28
    | '*' -> char_2a
    | ')' -> char_29
    | 'a'..'z' | 'A'..'Z' -> letter
    | '0'..'9' -> digit
    | '!' -> classify2 lexbuf
    | _ -> wrong_code_point

let table_8bit =
  let v = Array.create 256 wrong_code_point in
  let set ch cl = v.(Char.code ch) <- cl in
  let set_int ch1 ch2 cl =
    for i = Char.code ch1 to Char.code ch2 do v.(i) <- cl done in
  set '.' char_2e;
  set '' char_e9;
  set '+' char_2b;
  set '-' char_2d;
  set ' ' char_20;
  set '\n' char_0a;
  set '(' char_28;
  set '*' char_2a;
  set ')' char_29;
  set_int 'a' 'z' letter;
  set_int 'A' 'Z' letter;
  set_int '0' '9' digit;
  v

let table_tiny_8bit =
  let v = String.create 256 in
  for i = 0 to 255 do
    v.[i] <- Char.chr (Obj.magic table_8bit.(i))
  done;
  v

let table_tiny_utf8 =
  let v = String.make 0xFFFF table_tiny_8bit.[0] in
  String.blit table_tiny_8bit 0 v 0 256;
  v.[0x1200] <- Char.chr (Obj.magic ethioH);
  v


let main () =
  let str_lat = "55 (* + 35 *)  O - xyz ()"
  and str_utf = "55 (* + 35 *) é O - xyz ሀ (éሀ)" in
  let test_engine str engine =
    let lexbuf = from_string str in
    let rec aux () =
      let s = token "EOF" engine lexbuf in 
      print_endline s; 
      flush stdout;
      if s <> "EOF" then aux () else print_newline ()
    in
    aux ()
  in

  print_endline "classify_fun:";
  test_engine str_lat (engine_classify_fun classify);
  print_endline "8bit:";
  test_engine str_lat (engine_8bit table_8bit);
  print_endline "tiny_8bit:";
  test_engine str_lat (engine_tiny_8bit table_tiny_8bit);
  print_endline "tiny_utf8:";
  test_engine str_utf (engine_tiny_utf8 table_tiny_utf8 (fun _ -> wrong_code_point))

let _ = main ()
