/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <boost/lexical_cast.hpp>

#include "Wt/WApplication"
#include "Wt/WAnchor"
#include "Wt/WContainerWidget"
#include "Wt/WEnvironment"
#include "Wt/WLabel"
#include "Wt/WMenuItem"
#include "Wt/WMenu"
#include "Wt/WTableCell"

#include "WtException.h"

namespace Wt {

WMenuItem::WMenuItem(const WString& text, WWidget *contents,
		     LoadPolicy policy)
  : text_(text),
    itemWidget_(0),
    contentsContainer_(0),
    contents_(contents)
{
  if (policy == PreLoading)
    // prelearn the stateless slot.
    implementStateless(&WMenuItem::selectVisual, &WMenuItem::undoSelectVisual);
  else {
    contentsContainer_ = new WContainerWidget();
  }
}

WMenuItem::~WMenuItem()
{
  delete contents_;
  delete contentsContainer_;
}

WWidget *WMenuItem::createItemWidget()
{
  WWidget *result;

  const std::vector<WMenuItem *>& items = menu_->items();
  WApplication *app = WApplication::instance();

  std::string url;

  if (!menu_->browserHistoryId().empty())
    url = app->bookmarkUrl(menu_->browserHistoryId(),
			   boost::lexical_cast<std::string>(items.size()-1));
  else
    url = "#";

  result = new WAnchor(url, text_);

  return result;
}

void WMenuItem::setText(const WString& text)
{
  text_ = text;

  WAnchor *a = dynamic_cast<WAnchor *>(itemWidget());
  if (a) {
    a->setText(text);
    return;
  }
}

WWidget *WMenuItem::itemWidget()
{
  if (!itemWidget_) {
    itemWidget_ = createItemWidget();

    if (contentsContainer_)
      // load contents (will only do something on the first activation).
      activateSignal().connectBase(SLOT(this, WMenuItem::loadContents));
    else {
      activateSignal().connectBase(SLOT(this, WMenuItem::selectVisual));
      activateSignal().connectBase(SLOT(this, WMenuItem::select));
    }
  }
  return itemWidget_;
}

SignalBase& WMenuItem::activateSignal()
{
  if (!itemWidget_)
    itemWidget_ = createItemWidget();

  WInteractWidget *wi
    = dynamic_cast<WInteractWidget *>(itemWidget_->webWidget());

  if (wi)
    return wi->clicked;
  else
    throw WtException("WMenuItem::activateSignal(): "
		      "could not dynamic_cast itemWidget() to a "
		      "WInteractWidget");
}

void WMenuItem::renderSelected(bool selected)
{
  itemWidget()->setStyleClass(selected ? "itemselected" : "item");
}

void WMenuItem::loadContents()
{
  if (contentsContainer_) {
    contentsContainer_->addWidget(contents_);
    contentsContainer_ = 0;

    select();

    implementStateless(&WMenuItem::selectVisual, &WMenuItem::undoSelectVisual);
    activateSignal().connectBase(SLOT(this, WMenuItem::selectVisual));
    activateSignal().connectBase(SLOT(this, WMenuItem::select));
  }
}

void WMenuItem::setMenu(WMenu *menu)
{
  menu_ = menu;
}

WWidget *WMenuItem::contents()
{
  if (contentsContainer_)
    return contentsContainer_;
  else
    return contents_;
}

void WMenuItem::select()
{
  menu_->select(this);
}

void WMenuItem::selectVisual()
{
  menu_->selectVisual(this);
}

void WMenuItem::undoSelectVisual()
{
  menu_->undoSelectVisual();
}

}
