/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */
#include <iostream>
#include <boost/lexical_cast.hpp>
#include <boost/tokenizer.hpp>

#include "Wt/WApplication"
#include "Wt/WContainerWidget"
#include "Wt/WMemoryResource"
#include "Wt/WServer"
#include "Wt/WText"

#include "WtException.h"
#include "WServerPushResource.h"
#include "WebSession.h"
#include "DomElement.h"
#include "Configuration.h"
#include "WebController.h"

namespace Wt {

//#define WTDEBUG
//#define WTDEBUG_STATE(x) x
#define WTDEBUG_STATE(x)

class Iframe : public WWebWidget
{
public:
  Iframe(const std::string& uri, WContainerWidget *parent = 0)
    : WWebWidget(parent),
      used(false)
  {
    if (uri.empty())
      uri_ = generateUrl();
    else
      uri_ = uri;
  }

  bool used;

  const std::string& uri() const { return uri_; }

private:
  std::string uri_;

  void updateDom(DomElement& element, bool all) {
    if (all) {
      element.setId(this, true);

      element.setAttribute("class", "Wt-resource");
      element.setAttribute("src", uri_);
    }

    WWebWidget::updateDom(element, all);
  }

  DomElementType domElementType() const { return DomElement_IFRAME;}

  void resourceHTML(std::ostream&) { }
};

WApplication::ScriptLibrary::ScriptLibrary(const std::string& anUri,
					   const std::string& aSymbol)
  : uri(anUri), symbol(aSymbol)
{ }

bool WApplication::ScriptLibrary::operator< (const ScriptLibrary& other) const
{
  return uri < other.uri;
}

bool WApplication::ScriptLibrary::operator== (const ScriptLibrary& other) const
{
  return uri == other.uri;
}

WApplication::StyleSheet::StyleSheet(const std::string& anUri,
				     const std::string& aDependency)
  : uri(anUri), dependency(aDependency)
{ }

WApplication::WApplication(const WEnvironment& env)
  : session_(env.session_),
    titleChanged_(false),
    serverPush_(0),
    javaScriptClass_("Wt"),
    dialogCover_(0),
    dialogWindow_(0),
    quited_(false),
    scriptLibrariesAdded_(0),
    styleSheetsAdded_(0),
    exposeSignals_(true),
    autoJavaScriptChanged_(false),
    javaScriptResponse_(this)
{
  locale_ = environment().locale();
  session_->app_ = this;

  domRoot_ = new WContainerWidget();
  domRoot_->load();


  if (session_->type() == WebSession::Application) {
    domRoot_->resize(WLength(), WLength(100, WLength::Percentage));

    /*
     * IE 6 weirdest bug ever: a margin of 19px at the top, because
     * of our standard layout stuff (19px is suspiciously close to
     * the width of a scrollbar (=18px) ?)
     */
    if (environment().agentIE6())
      domRoot_->setMargin(-19, Top);
  }

  timerRoot_ = new WContainerWidget(domRoot_);
  timerRoot_->resize(WLength(), WLength(0));

  if (session_->type() == WebSession::Application) {
    ajaxMethod_ = XMLHttpRequest;

    domRoot2_ = 0;
    widgetRoot_ = new WContainerWidget(domRoot_);
    widgetRoot_->resize(WLength(), WLength(100, WLength::Percentage));
  } else {
    ajaxMethod_ = DynamicScriptTag;

    domRoot2_ = new WContainerWidget();
    domRoot2_->load();
    widgetRoot_ = 0;
  }

  std::string resourcesURL = resourcesUrl();

  if (environment().ajax())
    for (unsigned i = 0; i < 10; ++i)
      iframes_.push_back
	(new Iframe(i == 0 ? std::string() : iframes_[0]->uri(), domRoot_));

  /*
   * Subset of typical CSS "reset" styles
   */
  styleSheet_.addRule("table", "border-collapse: collapse; border: 0px");
  styleSheet_.addRule("div, td, img", "margin: 0px; padding: 0px; border: 0px");
  styleSheet_.addRule("td", "vertical-align: top; text-align: left;");
  styleSheet_.addRule("button", "white-space: nowrap");

  if (environment().agentIE())
    styleSheet_.addRule("html, body", "overflow: auto;");
  else if (environment().agentGecko())
    styleSheet_.addRule("html", "overflow: auto;");

  /*
   * Standard Wt CSS styles: resources, button wrap and form validation
   */
  styleSheet_.addRule("iframe.Wt-resource",
		      "width: 0px; height: 0px; border: 0px;");
  styleSheet_.addRule("button.Wt-wrap",
		      "border: 0px; text-align: left;"
		      "background-color: transparent; "
		      "margin: 0px; padding: 0px; font-size: inherit; "
		      "pointer: hand; cursor: pointer; cursor: hand; "
		      "color: inherit;");
  styleSheet_.addRule("a.Wt-wrap", "text-decoration: none;");
  styleSheet_.addRule("input.Wt-invalid", "background-color: #f79a9a;");
  styleSheet_.addRule(".unselectable",
		      "-moz-user-select: none;"
		      "-khtml-user-select: none;"
		      "user-select: none;");

  /*
   * The "Loading" indicator (with workarounds for old IE)
   */
  styleSheet_.addRule("div#feedback",
		      "background-color: red; color: white;"
		      "font-family: Arial,Helvetica,sans-serif;"
		      "font-size: small;"
		      "position: absolute; right: 0px; top: 0px;");
  styleSheet_.addRule("body > div#feedback",
		      "position: fixed;");
  if (environment().userAgent().find("MSIE 5.5") != std::string::npos
      || environment().userAgent().find("MSIE 6") != std::string::npos)
    styleSheet_.addRule("div#feedback",
			"right: expression((("
			"ignoreMe2 = document.documentElement.scrollLeft ? "
			"document.documentElement.scrollLeft : "
			"document.body.scrollLeft )) + 'px' );"
			"top: expression((("
			"ignoreMe = document.documentElement.scrollTop ? "
			"document.documentElement.scrollTop : "
			"document.body.scrollTop)) + 'px' );");

  javaScriptResponse_.connect(SLOT(this,
				   WApplication::handleJavaScriptResponse));
}

void WApplication::initialize()
{ }

void WApplication::finalize()
{ }

std::string WApplication::onePixelGifUrl()
{
  if (onePixelGifUrl_.empty()) {
    WMemoryResource *w = new WMemoryResource("image/gif", this);

    static const char gifData[]
      = { 0x47, 0x49, 0x46, 0x38, 0x39, 0x61, 0x01, 0x00, 0x01, 0x00,
	  0x80, 0x00, 0x00, 0xdb, 0xdf, 0xef, 0x00, 0x00, 0x00, 0x21,
	  0xf9, 0x04, 0x01, 0x00, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00,
	  0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x02, 0x02, 0x44,
	  0x01, 0x00, 0x3b };

    w->setData(gifData, 43);
    onePixelGifUrl_ = w->generateUrl();
  }

  return onePixelGifUrl_;
}

WApplication::~WApplication()
{
  dialogCover_ = 0;
  dialogWindow_ = 0;

  WContainerWidget *tmp = domRoot_;
  domRoot_ = 0;
  delete tmp;

  tmp = domRoot2_;
  domRoot2_ = 0;
  delete tmp;

  if (serverPush_)
    delete serverPush_;

  session_->app_ = 0;
}

void WApplication::attachThread()
{
  WebSession::Handler::attachThreadToSession(*session_);
}

void WApplication::setAjaxMethod(AjaxMethod method)
{
  ajaxMethod_ = method;
}

std::string WApplication::resourcesUrl()
{
  std::string result = "resources/";
  readConfigurationProperty("resourcesURL", result);

  if (!result.empty() && result[result.length()-1] != '/')
    result += '/';

  return instance()->fixRelativeUrl(result);
}

void WApplication::bindWidget(WWidget *widget, const std::string& divId)
{
  if (session_->type() != WebSession::WidgetSet)
    throw WtException("WApplication::bind() can be used only "
		      "in WidgetSet mode.");

  widget->setId(divId);
  domRoot2_->addWidget(widget);
}

WContainerWidget *WApplication::dialogCover(bool create)
{
  if (dialogCover_ == 0 && create) {
    dialogCover_ = new WContainerWidget(domRoot_);
    dialogCover_->setStyleClass("Wt-dialogcover");
  }

  return dialogCover_;
}

bool WApplication::isExposed(WWidget *w) const
{
  if (dialogWindow_ && !dialogWindow_->isHidden()) {
    for (WWidget *p = w; p; p = p->parent())
      if (p == dialogWindow_)
	return true;
    return false;
  } else
    return true;
}

WContainerWidget *WApplication::dialogWindow(bool create)
{
  if (dialogWindow_ == 0 && create) {
    dialogWindow_ = new WContainerWidget(domRoot_);
    dialogWindow_->setStyleClass("Wt-dialogwindow");

    addAutoJavaScript
      ("{"
       "" "var w=" + dialogWindow_->jsRef() + ";"
       "" "if (w && w.style.display != 'none') {"
       ""   "var ml='-' + Math.round(w.clientWidth/2) + 'px';"
       ""   "if (w.style.marginLeft != ml)"
       ""     "w.style.marginLeft=ml;"
       ""   "var mt='-' + Math.round(w.clientHeight/2) + 'px';"
       ""   "if (w.style.marginTop != mt)"
       ""     "w.style.marginTop=mt;"
       "" "}"
       "}");
  }

  return dialogWindow_;
}

std::string WApplication::sessionId() const
{
  return session_->sessionId();
}

WWidget *WApplication::useIframe()
{
  for (unsigned i = 0; i < iframes_.size(); ++i) {
    if (!iframes_[i]->used) {
      iframes_[i]->used = true;
      return iframes_[i];
    }
  }

  throw WtException("Reached maximum file upload count");
}

void WApplication::releaseIframe(WWidget *iframe)
{
  ((Iframe *)iframe)->used = false;
}

void WApplication::useStyleSheet(const std::string& uri, const std::string& dep)
{
  styleSheets_.push_back(StyleSheet(uri, dep));
  ++styleSheetsAdded_;
}

const WEnvironment& WApplication::environment() const
{
  return session_->env();
}

WEnvironment& WApplication::env()
{
  return session_->env();
}

void WApplication::setTitle(const WString& title)
{
  if (session()->renderer().preLearning() || title_ != title) {
    title_ = title;
    titleChanged_ = true;
  }
}

std::string WApplication::url() const
{
  return session_->applicationUrl();
}

const std::string &WApplication::applicationName() const
{
  return session_->applicationName_;
}

std::string WApplication::fixRelativeUrl(const std::string& url) const
{
  if (ajaxMethod_ == XMLHttpRequest)
    return url;

  if (url.find("://") != std::string::npos)
    return url;

  if (!url.empty()) {
    if (url[0] != '/')
      return session_->absoluteBaseUrl() + url;
    else
      return environment().urlScheme() + "://" + environment().hostName() + url;
  } else
    return url;
}

void WApplication::quit()
{
  quited_ = true;
}

void WApplication::addExposedSignal(Wt::EventSignalBase *signal)
{
  std::string s = signal->encodeCmd();
  exposedSignals_[s] = signal;

#ifdef WTDEBUG
  std::cerr << "WApplication::addExposedSignal: " << s << std::endl;
#endif
}

void WApplication::removeExposedSignal(Wt::EventSignalBase *signal)
{
  std::string s = signal->encodeCmd();
  SignalMap::iterator i = exposedSignals_.find(s);

  if (i == exposedSignals_.end()) {
    std::cerr << " WApplication::removeExposedSignal of non-exposed "
	      << s << "??" << std::endl;    
  } else {
#ifdef WTDEBUG
    std::cerr << " WApplication::removeExposedSignal: " << s << std::endl;    
#endif
    exposedSignals_.erase(i);
  }
}

EventSignalBase *
WApplication::decodeExposedSignal(const std::string& signalName) const
{
  SignalMap::const_iterator i = exposedSignals_.find(signalName);
  
  if (i != exposedSignals_.end())
    if (isExposed(dynamic_cast<WWidget *>(i->second->sender())))
      return i->second;
    else
      return 0;
  else
    return 0;
}

EventSignalBase *
WApplication::decodeExposedSignal(const std::string& objectId,
				  const std::string& name)
{
  std::string signalName
    = (objectId == "app" ? formName() : objectId) + '.' + name;

  return decodeExposedSignal(signalName);
}

const WApplication::SignalMap& WApplication::exposedSignals() const
{
  return exposedSignals_;
}

std::string WApplication::addExposedResource(WResource *resource)
{
  exposedResources_[resource->formName()] = resource;

  return session_->mostRelativeUrl(resource->suggestedFileName())
    + "&resource=" + resource->formName()
    + "&rand=" + boost::lexical_cast<std::string>
    (WtRandom::getUnsigned());
}

void WApplication::removeExposedResource(WResource *resource)
{
  ResourceMap::iterator i = exposedResources_.find(resource->formName());

  if (i != exposedResources_.end())
    exposedResources_.erase(i);
}

WResource *WApplication::decodeExposedResource(const std::string& resourceName) 
  const
{
  ResourceMap::const_iterator i = exposedResources_.find(resourceName);
  
  if (i != exposedResources_.end())
    return i->second;
  else
    return 0;
}

std::string WApplication::encodeObject(WObject *object)
{
  std::string result = "w" + object->uniqueId();

  encodedObjects_[result] = object;

  return result;
}

WObject *WApplication::decodeObject(const std::string& objectId) const
{
  ObjectMap::const_iterator i = encodedObjects_.find(objectId);

  if (i != encodedObjects_.end()) {
    return i->second;
  } else
    return 0;
}

void WApplication::setLocale(const std::string& locale)
{
  locale_ = locale;
  refresh();
}

void WApplication::refresh()
{
  messageResourceBundle_.refresh();
  widgetRoot_->refresh();

  if (title_.refresh())
    titleChanged_ = true;
}

void WApplication::redirect(const std::string& url)
{
  session_->redirect(url);
}

void WApplication::setTwoPhaseRenderingThreshold(int bytes)
{
  session_->renderer().setTwoPhaseThreshold(bytes);
}

void WApplication::setCookie(const std::string& name, const std::string& value,
			     int maxAge, const std::string& domain,
			     const std::string& path)
{
  session_->renderer().setCookie(name, value, maxAge, domain, path);
}

WApplication *WApplication::instance()
{
  WebSession *session = WebSession::instance();

  return session ? session->app() : 0;
}

int WApplication::maximumRequestSize() const
{
  return WebController::conf().maxRequestSize() * 1024;
}

std::string WApplication::docType() const
{
  const bool xhtml = environment().contentType() == WEnvironment::XHTML1;

  if (xhtml)
    /*
     * This would be what we want, but it is too strict (does not
     * validate iframe's and target attribute for links):
     
     "\"-//W3C//DTD XHTML 1.1 plus MathML 2.0 plus SVG 1.1//EN\" "
     "\"http://www.w3.org/2002/04/xhtml-math-svg/xhtml-math-svg.dtd\">"
     * so instead we use transitional xhtml -- it will fail to
     * validate properly when we have svg !
     */ 
    return "<!DOCTYPE html PUBLIC "
      "\"-//W3C//DTD XHTML 1.0 Transitional//EN\" "
      "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">";
  else
    return "<!DOCTYPE html PUBLIC "
      "\"-//W3C//DTD HTML 4.01 Transitional//EN\" "
      "\"http://www.w3.org/TR/html4/loose.dtd\">";
}

bool WApplication::loadRsh()
{
  if (environment().userAgent().find("WebKit") == std::string::npos
      && environment().userAgent().find("KHTML") == std::string::npos) {
    require(resourcesUrl() + "json2005.js");
    if (require(resourcesUrl() + "rsh.js")) {
      /*
       * for IE: we can only do this while reading the body,
       * but before onload fires -- therefore we specify 'false'
       */
      doJavaScript
	(std::string("window.dhtmlHistory.create(")
	 + (environment().inFrame() ? "window.parent" : "window") + ", {"
	 "url: '" + session_->mostRelativeUrl() + "',"
	 "toJSON: function(o) { return JSON.stringify(o); }, "
	 "fromJSON: function(s) { return JSON.parse(s); } });"
	 "window.dhtmlHistory.initialize();"
	 "window.dhtmlHistory.addListener(" + javaScriptClass_
	 + ".private.historyChange);", false);
    }

    return true;
  } else
    return false;
}

void WApplication::setState(const std::string& scope, const std::string& state)
{
  loadRsh();

  WTDEBUG_STATE(std::cerr << "setState: " << scope << ": " << state << std::endl);

  if (session()->renderer().preLearning() || state_[scope] != state) {
    state_[scope] = state;
    statesChanged_.insert(scope);
  }
}

std::string WApplication::state(const std::string& scope) const
{
  const_cast<WApplication *>(this)->loadRsh();

  StateMap::const_iterator i = state_.find(scope);

  if (i != state_.end())
    return i->second;
  else
    return std::string();
}

std::string WApplication::getCurrentHistoryKey() const
{
  std::string result;
  for (WApplication::StateMap::const_iterator i = state_.begin();
       i != state_.end(); ++i) {
    if (i->second.empty())
      continue;
    if (!result.empty())
      result += ';';
    result += i->first + ":" + i->second;
  }

  WTDEBUG_STATE(std::cerr << "getCHK: " << result << std::endl);

  return result;
}

std::string WApplication::bookmarkUrl() const
{
  std::string result;

  if (!environment().javaScript())
    if (environment().agentIsSpiderBot())
      result = "?historyKey=";
    else
      result = session_->mostRelativeUrl() + "&historyKey=";
  else
    result = session_->bookmarkBaseUrl() + '#';

  return result + getCurrentHistoryKey();
}

std::string WApplication::bookmarkUrl(const std::string& scope,
				      const std::string& newState)
{
  std::string s = state(scope);
  setState(scope, newState);
  std::string result = bookmarkUrl();
  setState(scope, s);

  return result;
}

void WApplication::setCurrentHistoryKey(const std::string& stateStr)
{
  typedef boost::tokenizer<boost::char_separator<char> > semicolon_tok;
  semicolon_tok tok(stateStr, boost::char_separator<char>(";"));

  StateMap oldState = state_;
  state_.clear();

  WTDEBUG_STATE(std::cerr << "setCHK: " << stateStr << std::endl);

  for (semicolon_tok::iterator i = tok.begin(); i != tok.end(); ++i) {
    std::string pair = *i;

    std::string::size_type colonPos = pair.find(':');

    if (colonPos != std::string::npos) {
      std::string key = pair.substr(0, colonPos);
      std::string value  = pair.substr(colonPos + 1);
      state_[key] = value;

      if (oldState.find(key) == oldState.end())
	stateChanged.emit(key, value);
    }
  }

  for (StateMap::const_iterator i = oldState.begin();
       i != oldState.end(); ++i) {
    StateMap::const_iterator j = state_.find(i->first);

    if (j == state_.end())
      stateChanged.emit(i->first, "");
    else
      if (i->second != j->second)
	stateChanged.emit(i->first, j->second);
  }

  statesChanged_.clear();
}

WLogEntry WApplication::log(const std::string& type)
{
  return session_->log(type);
}

void WApplication::enableUpdates()
{
  if (!serverPush_) {
    serverPush_ = new WServerPushResource(this);

    require(resourcesUrl() + "orbited.js");
    doJavaScript("setTimeout(\"Orbited.connect(eval,'none','"
		 + serverPush_->generateUrl() + "','0');\",1000);");
  }
}

bool WApplication::updatesEnabled() const
{
  return serverPush_ != 0;
}

void WApplication::triggerUpdate()
{
  if (serverPush_)
    serverPush_->triggerUpdate();
  else
    throw WtException("WApplication::update() called but server-triggered "
		      "updates not enabled using "
		      "WApplication::enableUpdates()"); 
}

WApplication::UpdateLock WApplication::getUpdateLock()
{
  return UpdateLock(*this);
}

class UpdateLockImpl
{
public:
  UpdateLockImpl(WApplication& app)
    : handler_(*(app.session()))
#ifdef THREADED
    ,sessionLock_(app.session()->mutex)
#endif // THREADED
  { 
#ifndef THREADED
    throw WtException("UpdateLock needs Wt with thread support");
#endif // THREADED
  }

private:
  WebSession::Handler handler_;

#ifdef THREADED
  boost::mutex::scoped_lock sessionLock_;
#endif // THREADED
};

WApplication::UpdateLock::UpdateLock(WApplication& app)
  : impl_(0)
{
  /*
   * If we are already handling this application, then we already have
   * exclusive access.
   */
  if (WApplication::instance() != &app)
    impl_ = new UpdateLockImpl(app);
}

WApplication::UpdateLock::UpdateLock(const UpdateLock& other)
{
  impl_ = other.impl_;
  other.impl_ = 0;
}

WApplication::UpdateLock::~UpdateLock()
{
  delete impl_;
}

void WApplication::doJavaScript(const std::string& javascript,
				bool afterLoaded)
{
  if (afterLoaded) {
    afterLoadJavaScript_ += javascript;
  } else {
    beforeLoadJavaScript_ += javascript;
    newBeforeLoadJavaScript_ += javascript;
  }
}

void WApplication::addAutoJavaScript(const std::string& javascript)
{
  autoJavaScript_ += javascript;
  autoJavaScriptChanged_ = true;
}

void WApplication::declareJavaScriptFunction(const std::string& name,
					     const std::string& function)
{
  doJavaScript(javaScriptClass_ + '.' + name + '=' + function + ';', false);
}

std::string WApplication::afterLoadJavaScript()
{
  std::string result = afterLoadJavaScript_;
  afterLoadJavaScript_.clear();
  return result;
}

std::string WApplication::newBeforeLoadJavaScript()
{
  std::string result = newBeforeLoadJavaScript_;
  newBeforeLoadJavaScript_.clear();
  return result;
}

std::string WApplication::beforeLoadJavaScript()
{
  newBeforeLoadJavaScript_.clear();
  return beforeLoadJavaScript_;
}

void WApplication::notify(const WEvent& e)
{
  WebController::instance()->notify(e);
}

void WApplication::processEvents()
{
  /* set timeout to allow other events to be interleaved */
  session_->doRecursiveEventLoop
    ("setTimeout(\"" + javaScriptClass_
     + ".private.update(null,'" + javaScriptResponse_.encodeCmd()
     + "',null,false);\",0);");
}

void WApplication::handleJavaScriptResponse(WResponseEvent event)
{
  session_->unlockRecursiveEventLoop();
}

bool WApplication::require(const std::string& uri, const std::string& symbol)
{
  ScriptLibrary sl(uri, symbol);

  if (std::find(scriptLibraries_.begin(), scriptLibraries_.end(), sl)
      == scriptLibraries_.end()) {

    sl.beforeLoadJS = newBeforeLoadJavaScript_;
    newBeforeLoadJavaScript_.clear();

    scriptLibraries_.push_back(sl);
    ++scriptLibrariesAdded_;

    return true;
  } else
    return false;
}

bool WApplication::readConfigurationProperty(const std::string& name,
					     std::string& value)
{
  const Configuration::PropertyMap& properties
    = WebController::conf().properties();

  Configuration::PropertyMap::const_iterator i = properties.find(name);

  if (i != properties.end()) {
    value = i->second;
    return true;
  } else
    return false;
}

WServer::Exception::Exception(const std::string what)
  : what_(what)
{ }

WServer::Exception::~Exception() throw()
{ }

}
