/*
 * wintrust softpub functions tests
 *
 * Copyright 2007,2010 Juan Lang
 * Copyright 2010 Andrey Turkin
 * Copyright 2016 Mark Jansen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdio.h>
#include <stdarg.h>

#include <windef.h>
#include <winbase.h>
#include <winerror.h>
#include <wintrust.h>
#include <softpub.h>
#include <mssip.h>
#include <winuser.h>
#include "winnls.h"

#include "wine/test.h"

/* Just in case we're being built with borked headers, redefine function
 * pointers to have the correct calling convention.
 */
typedef void   *(WINAPI *SAFE_MEM_ALLOC)(DWORD);
typedef void    (WINAPI *SAFE_MEM_FREE)(void *);
typedef BOOL    (WINAPI *SAFE_ADD_STORE)(CRYPT_PROVIDER_DATA *,
 HCERTSTORE);
typedef BOOL    (WINAPI *SAFE_ADD_SGNR)(CRYPT_PROVIDER_DATA *,
 BOOL, DWORD, struct _CRYPT_PROVIDER_SGNR *);
typedef BOOL    (WINAPI *SAFE_ADD_CERT)(CRYPT_PROVIDER_DATA *,
 DWORD, BOOL, DWORD, PCCERT_CONTEXT);
typedef BOOL    (WINAPI *SAFE_ADD_PRIVDATA)(CRYPT_PROVIDER_DATA *,
 CRYPT_PROVIDER_PRIVDATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_INIT_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_OBJTRUST_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_SIGTRUST_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_CERTTRUST_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_FINALPOLICY_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_TESTFINALPOLICY_CALL)(
 CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_CLEANUP_CALL)(CRYPT_PROVIDER_DATA *);
typedef BOOL    (WINAPI *SAFE_PROVIDER_CERTCHKPOLICY_CALL)(
 CRYPT_PROVIDER_DATA *, DWORD, BOOL, DWORD);

typedef struct _SAFE_PROVIDER_FUNCTIONS
{
    DWORD                              cbStruct;
    SAFE_MEM_ALLOC                     pfnAlloc;
    SAFE_MEM_FREE                      pfnFree;
    SAFE_ADD_STORE                     pfnAddStore2Chain;
    SAFE_ADD_SGNR                      pfnAddSgnr2Chain;
    SAFE_ADD_CERT                      pfnAddCert2Chain;
    SAFE_ADD_PRIVDATA                  pfnAddPrivData2Chain;
    SAFE_PROVIDER_INIT_CALL            pfnInitialize;
    SAFE_PROVIDER_OBJTRUST_CALL        pfnObjectTrust;
    SAFE_PROVIDER_SIGTRUST_CALL        pfnSignatureTrust;
    SAFE_PROVIDER_CERTTRUST_CALL       pfnCertificateTrust;
    SAFE_PROVIDER_FINALPOLICY_CALL     pfnFinalPolicy;
    SAFE_PROVIDER_CERTCHKPOLICY_CALL   pfnCertCheckPolicy;
    SAFE_PROVIDER_TESTFINALPOLICY_CALL pfnTestFinalPolicy;
    struct _CRYPT_PROVUI_FUNCS        *psUIpfns;
    SAFE_PROVIDER_CLEANUP_CALL         pfnCleanupPolicy;
} SAFE_PROVIDER_FUNCTIONS;

static BOOL (WINAPI * pWTHelperGetKnownUsages)(DWORD action, PCCRYPT_OID_INFO **usages);
static BOOL (WINAPI * CryptSIPCreateIndirectData_p)(SIP_SUBJECTINFO *, DWORD *, SIP_INDIRECT_DATA *);
static VOID (WINAPI * CertFreeCertificateChain_p)(PCCERT_CHAIN_CONTEXT);

static void InitFunctionPtrs(void)
{
    HMODULE hWintrust = GetModuleHandleA("wintrust.dll");
    HMODULE hCrypt32 = GetModuleHandleA("crypt32.dll");

#define WINTRUST_GET_PROC(func) \
    p ## func = (void*)GetProcAddress(hWintrust, #func); \
    if(!p ## func) { \
      trace("GetProcAddress(%s) failed\n", #func); \
    }

    WINTRUST_GET_PROC(WTHelperGetKnownUsages)

#undef WINTRUST_GET_PROC

#define CRYPT32_GET_PROC(func) \
    func ## _p = (void*)GetProcAddress(hCrypt32, #func); \
    if(!func ## _p) { \
      trace("GetProcAddress(%s) failed\n", #func); \
    }

    CRYPT32_GET_PROC(CryptSIPCreateIndirectData)
    CRYPT32_GET_PROC(CertFreeCertificateChain)

#undef CRYPT32_GET_PROC
}

static const BYTE v1CertWithPubKey[] = {
0x30,0x81,0x95,0x02,0x01,0x01,0x30,0x02,0x06,0x00,0x30,0x15,0x31,0x13,0x30,
0x11,0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,
0x6e,0x67,0x00,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,0x30,0x31,
0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,
0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x15,0x31,0x13,0x30,0x11,
0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,0x6e,
0x67,0x00,0x30,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x01,0x05,0x00,0x03,0x11,0x00,0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0xa3,0x16,0x30,0x14,0x30,0x12,0x06,
0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,0x02,
0x01,0x01 };

static void test_utils(SAFE_PROVIDER_FUNCTIONS *funcs)
{
    CRYPT_PROVIDER_DATA data = { 0 };
    HCERTSTORE store;
    CRYPT_PROVIDER_SGNR sgnr = { 0 };
    BOOL ret;

    /* Crash
    ret = funcs->pfnAddStore2Chain(NULL, NULL);
    ret = funcs->pfnAddStore2Chain(&data, NULL);
     */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, X509_ASN_ENCODING, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    if (store)
    {
        ret = funcs->pfnAddStore2Chain(&data, store);
        ok(ret, "pfnAddStore2Chain failed: %08lx\n", GetLastError());
        ok(data.chStores == 1, "Expected 1 store, got %ld\n", data.chStores);
        ok(data.pahStores != NULL, "Expected pahStores to be allocated\n");
        if (data.pahStores)
        {
            ok(data.pahStores[0] == store, "Unexpected store\n");
            CertCloseStore(data.pahStores[0], 0);
            funcs->pfnFree(data.pahStores);
            data.pahStores = NULL;
            data.chStores = 0;
            CertCloseStore(store, 0);
            store = NULL;
        }
    }
    else
        skip("CertOpenStore failed: %08lx\n", GetLastError());

    /* Crash
    ret = funcs->pfnAddSgnr2Chain(NULL, FALSE, 0, NULL);
    ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, NULL);
     */
    ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
    ok(ret, "pfnAddSgnr2Chain failed: %08lx\n", GetLastError());
    ok(data.csSigners == 1, "Expected 1 signer, got %ld\n", data.csSigners);
    ok(data.pasSigners != NULL, "Expected pasSigners to be allocated\n");
    if (data.pasSigners)
    {
        PCCERT_CONTEXT cert;

        ok(!memcmp(&data.pasSigners[0], &sgnr, sizeof(sgnr)),
         "Unexpected data in signer\n");
        /* Adds into the location specified by the index */
        sgnr.cbStruct = sizeof(CRYPT_PROVIDER_SGNR);
        sgnr.sftVerifyAsOf.dwLowDateTime = 0xdeadbeef;
        ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 1, &sgnr);
        ok(ret, "pfnAddSgnr2Chain failed: %08lx\n", GetLastError());
        ok(data.csSigners == 2, "Expected 2 signers, got %ld\n", data.csSigners);
        ok(!memcmp(&data.pasSigners[1], &sgnr, sizeof(sgnr)),
         "Unexpected data in signer\n");
        /* This also adds, but the index is ignored */
        sgnr.cbStruct = sizeof(DWORD);
        ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
        ok(ret, "pfnAddSgnr2Chain failed: %08lx\n", GetLastError());
        ok(data.csSigners == 3, "Expected 3 signers, got %ld\n", data.csSigners);
        sgnr.sftVerifyAsOf.dwLowDateTime = 0;
        todo_wine
        ok(!memcmp(&data.pasSigners[2], &sgnr, sizeof(sgnr)),
           "Unexpected data in signer\n");
        /* But too large a thing isn't added */
        sgnr.cbStruct = sizeof(sgnr) + sizeof(DWORD);
        SetLastError(0xdeadbeef);
        ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
        ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
         "Expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());

        /* Crash
        ret = funcs->pfnAddCert2Chain(NULL, 0, FALSE, 0, NULL);
        ret = funcs->pfnAddCert2Chain(&data, 0, FALSE, 0, NULL);
         */
        cert = CertCreateCertificateContext(X509_ASN_ENCODING, v1CertWithPubKey,
         sizeof(v1CertWithPubKey));
        if (cert)
        {
            /* Notes on behavior that are hard to test:
             * 1. If pasSigners is invalid, pfnAddCert2Chain crashes
             * 2. An invalid signer index isn't checked.
             */
            ret = funcs->pfnAddCert2Chain(&data, 0, FALSE, 0, cert);
            ok(ret, "pfnAddCert2Chain failed: %08lx\n", GetLastError());
            ok(data.pasSigners[0].csCertChain == 1, "Expected 1 cert, got %ld\n",
             data.pasSigners[0].csCertChain);
            ok(data.pasSigners[0].pasCertChain != NULL,
             "Expected pasCertChain to be allocated\n");
            if (data.pasSigners[0].pasCertChain)
            {
                ok(data.pasSigners[0].pasCertChain[0].pCert == cert,
                 "Unexpected cert\n");
                CertFreeCertificateContext(
                 data.pasSigners[0].pasCertChain[0].pCert);
            }
            CertFreeCertificateContext(cert);
        }
        else
            skip("CertCreateCertificateContext failed: %08lx\n", GetLastError());
        funcs->pfnFree(data.pasSigners);
    }
}

static void testInitialize(SAFE_PROVIDER_FUNCTIONS *funcs, GUID *actionID)
{
    HRESULT ret;
    CRYPT_PROVIDER_DATA data = { 0 };
    WINTRUST_DATA wintrust_data = { 0 };

    if (!funcs->pfnInitialize)
    {
        skip("missing pfnInitialize\n");
        return;
    }

    /* Crashes
    ret = funcs->pfnInitialize(NULL);
     */
    memset(&data, 0, sizeof(data));
    ret = funcs->pfnInitialize(&data);
    ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
    data.padwTrustStepErrors =
     funcs->pfnAlloc(TRUSTERROR_MAX_STEPS * sizeof(DWORD));
    /* Without wintrust data set, crashes when padwTrustStepErrors is set */
    data.pWintrustData = &wintrust_data;
    if (data.padwTrustStepErrors)
    {
        /* Apparently, cdwTrustStepErrors does not need to be set. */
        memset(data.padwTrustStepErrors, 0,
         TRUSTERROR_MAX_STEPS * sizeof(DWORD));
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
        data.cdwTrustStepErrors = 1;
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
        memset(data.padwTrustStepErrors, 0xba,
         TRUSTERROR_MAX_STEPS * sizeof(DWORD));
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_WVTINIT] = 0;
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
        funcs->pfnFree(data.padwTrustStepErrors);
    }
}

static void getNotepadPath(WCHAR *notepadPathW, DWORD size)
{
    static const CHAR notepad[] = "\\notepad.exe";
    CHAR notepadPath[MAX_PATH];

    /* Workaround missing W-functions for win9x */
    GetWindowsDirectoryA(notepadPath, MAX_PATH);
    lstrcatA(notepadPath, notepad);
    MultiByteToWideChar(CP_ACP, 0, notepadPath, -1, notepadPathW, size);
}

/* Creates a test file and returns a handle to it.  The file's path is returned
 * in temp_file, which must be at least MAX_PATH characters in length.
 */
static HANDLE create_temp_file(WCHAR *temp_file)
{
    HANDLE file = INVALID_HANDLE_VALUE;
    WCHAR temp_path[MAX_PATH];

    if (GetTempPathW(ARRAY_SIZE(temp_path), temp_path))
    {
        static const WCHAR img[] = { 'i','m','g',0 };

        if (GetTempFileNameW(temp_path, img, 0, temp_file))
            file = CreateFileW(temp_file, GENERIC_READ | GENERIC_WRITE, 0, NULL,
             CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
    }
    return file;
}

static void testObjTrust(SAFE_PROVIDER_FUNCTIONS *funcs, GUID *actionID)
{
    HRESULT ret;
    CRYPT_PROVIDER_DATA data = { 0 };
    CRYPT_PROVIDER_SIGSTATE sig_state = { 0 };
    WINTRUST_DATA wintrust_data = { 0 };
    WINTRUST_CERT_INFO certInfo = { sizeof(WINTRUST_CERT_INFO), 0 };
    WINTRUST_FILE_INFO fileInfo = { sizeof(WINTRUST_FILE_INFO), 0 };

    if (!funcs->pfnObjectTrust)
    {
        skip("missing pfnObjectTrust\n");
        return;
    }

    /* Crashes
    ret = funcs->pfnObjectTrust(NULL);
     */

    data.cbStruct = sizeof(data);
    data.pSigState = &sig_state;
    data.pWintrustData = &wintrust_data;
    data.padwTrustStepErrors =
     funcs->pfnAlloc(TRUSTERROR_MAX_STEPS * sizeof(DWORD));
    if (data.padwTrustStepErrors)
    {
        WCHAR pathW[MAX_PATH];
        PROVDATA_SIP provDataSIP = { 0 };
        static const GUID unknown = { 0xC689AAB8, 0x8E78, 0x11D0, { 0x8C,0x47,
         0x00,0xC0,0x4F,0xC2,0x95,0xEE } };
        static GUID bogusGuid = { 0xdeadbeef, 0xbaad, 0xf00d, { 0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00 } };

        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         ERROR_INVALID_PARAMETER,
         "Expected ERROR_INVALID_PARAMETER, got %08lx\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        U(wintrust_data).pCert = &certInfo;
        wintrust_data.dwUnionChoice = WTD_CHOICE_CERT;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
        certInfo.psCertContext = (PCERT_CONTEXT)CertCreateCertificateContext(
         X509_ASN_ENCODING, v1CertWithPubKey, sizeof(v1CertWithPubKey));
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
        CertFreeCertificateContext(certInfo.psCertContext);
        certInfo.psCertContext = NULL;
        wintrust_data.dwUnionChoice = WTD_CHOICE_FILE;
        U(wintrust_data).pFile = NULL;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         ERROR_INVALID_PARAMETER,
         "Expected ERROR_INVALID_PARAMETER, got %08lx\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        U(wintrust_data).pFile = &fileInfo;
        /* Crashes
        ret = funcs->pfnObjectTrust(&data);
         */
        /* Create and test with an empty file */
        fileInfo.hFile = create_temp_file(pathW);
        /* pfnObjectTrust now crashes unless both pPDSip and psPfns are set */
        U(data).pPDSip = &provDataSIP;
        data.psPfns = (CRYPT_PROVIDER_FUNCTIONS *)funcs;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN,
         "expected TRUST_E_SUBJECT_FORM_UNKNOWN, got %08lx\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        CloseHandle(fileInfo.hFile);
        fileInfo.hFile = NULL;
        fileInfo.pcwszFilePath = pathW;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN,
         "expected TRUST_E_SUBJECT_FORM_UNKNOWN, got %08lx\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        DeleteFileW(pathW);
        /* Test again with a file we expect to exist, and to contain no
         * signature.
         */
        getNotepadPath(pathW, MAX_PATH);
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE ||
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN,
         "Expected TRUST_E_NOSIGNATURE or TRUST_E_SUBJECT_FORM_UNKNOWN, got %08lx\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        if (data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE)
        {
            ok(!memcmp(&provDataSIP.gSubject, &unknown, sizeof(unknown)),
             "Unexpected subject GUID\n");
            ok(provDataSIP.pSip != NULL, "Expected a SIP\n");
            ok(provDataSIP.psSipSubjectInfo != NULL,
             "Expected a subject info\n");
        }
        /* Specifying the GUID results in that GUID being the subject GUID */
        fileInfo.pgKnownSubject = &bogusGuid;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE ||
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN ||
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_PROVIDER_UNKNOWN,
         "Expected TRUST_E_NOSIGNATURE or TRUST_E_SUBJECT_FORM_UNKNOWN or TRUST_E_PROVIDER_UNKNOWN, got %08lx\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        if (data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE)
        {
            ok(!memcmp(&provDataSIP.gSubject, &bogusGuid, sizeof(bogusGuid)),
             "unexpected subject GUID\n");
        }
        /* Specifying a bogus GUID pointer crashes */
        if (0)
        {
            fileInfo.pgKnownSubject = (GUID *)0xdeadbeef;
            ret = funcs->pfnObjectTrust(&data);
            ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
        }
        funcs->pfnFree(data.padwTrustStepErrors);
    }
}

static const BYTE selfSignedCert[] = {
  0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43,
  0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d,
  0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x44, 0x70, 0x7a, 0x43, 0x43,
  0x41, 0x6f, 0x2b, 0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49, 0x4a,
  0x41, 0x4c, 0x59, 0x51, 0x67, 0x65, 0x66, 0x7a, 0x51, 0x41, 0x61, 0x43,
  0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33,
  0x44, 0x51, 0x45, 0x42, 0x42, 0x51, 0x55, 0x41, 0x4d, 0x47, 0x6f, 0x78,
  0x43, 0x7a, 0x41, 0x4a, 0x42, 0x67, 0x4e, 0x56, 0x0a, 0x42, 0x41, 0x59,
  0x54, 0x41, 0x6b, 0x46, 0x56, 0x4d, 0x52, 0x4d, 0x77, 0x45, 0x51, 0x59,
  0x44, 0x56, 0x51, 0x51, 0x49, 0x44, 0x41, 0x70, 0x54, 0x62, 0x32, 0x31,
  0x6c, 0x4c, 0x56, 0x4e, 0x30, 0x59, 0x58, 0x52, 0x6c, 0x4d, 0x53, 0x45,
  0x77, 0x48, 0x77, 0x59, 0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x42, 0x68,
  0x4a, 0x62, 0x6e, 0x52, 0x6c, 0x63, 0x6d, 0x35, 0x6c, 0x64, 0x43, 0x42,
  0x58, 0x0a, 0x61, 0x57, 0x52, 0x6e, 0x61, 0x58, 0x52, 0x7a, 0x49, 0x46,
  0x42, 0x30, 0x65, 0x53, 0x42, 0x4d, 0x64, 0x47, 0x51, 0x78, 0x49, 0x7a,
  0x41, 0x68, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x47, 0x6e,
  0x4e, 0x6c, 0x62, 0x47, 0x5a, 0x7a, 0x61, 0x57, 0x64, 0x75, 0x5a, 0x57,
  0x51, 0x75, 0x64, 0x47, 0x56, 0x7a, 0x64, 0x43, 0x35, 0x33, 0x61, 0x57,
  0x35, 0x6c, 0x61, 0x48, 0x45, 0x75, 0x0a, 0x62, 0x33, 0x4a, 0x6e, 0x4d,
  0x42, 0x34, 0x58, 0x44, 0x54, 0x45, 0x7a, 0x4d, 0x44, 0x59, 0x79, 0x4d,
  0x54, 0x45, 0x78, 0x4d, 0x6a, 0x55, 0x78, 0x4d, 0x46, 0x6f, 0x58, 0x44,
  0x54, 0x49, 0x7a, 0x4d, 0x44, 0x59, 0x78, 0x4f, 0x54, 0x45, 0x78, 0x4d,
  0x6a, 0x55, 0x78, 0x4d, 0x46, 0x6f, 0x77, 0x61, 0x6a, 0x45, 0x4c, 0x4d,
  0x41, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d, 0x43, 0x0a,
  0x51, 0x56, 0x55, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56,
  0x42, 0x41, 0x67, 0x4d, 0x43, 0x6c, 0x4e, 0x76, 0x62, 0x57, 0x55, 0x74,
  0x55, 0x33, 0x52, 0x68, 0x64, 0x47, 0x55, 0x78, 0x49, 0x54, 0x41, 0x66,
  0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x6f, 0x4d, 0x47, 0x45, 0x6c, 0x75,
  0x64, 0x47, 0x56, 0x79, 0x62, 0x6d, 0x56, 0x30, 0x49, 0x46, 0x64, 0x70,
  0x5a, 0x47, 0x64, 0x70, 0x0a, 0x64, 0x48, 0x4d, 0x67, 0x55, 0x48, 0x52,
  0x35, 0x49, 0x45, 0x78, 0x30, 0x5a, 0x44, 0x45, 0x6a, 0x4d, 0x43, 0x45,
  0x47, 0x41, 0x31, 0x55, 0x45, 0x41, 0x77, 0x77, 0x61, 0x63, 0x32, 0x56,
  0x73, 0x5a, 0x6e, 0x4e, 0x70, 0x5a, 0x32, 0x35, 0x6c, 0x5a, 0x43, 0x35,
  0x30, 0x5a, 0x58, 0x4e, 0x30, 0x4c, 0x6e, 0x64, 0x70, 0x62, 0x6d, 0x56,
  0x6f, 0x63, 0x53, 0x35, 0x76, 0x63, 0x6d, 0x63, 0x77, 0x0a, 0x67, 0x67,
  0x45, 0x69, 0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49,
  0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x41, 0x51, 0x55, 0x41, 0x41, 0x34,
  0x49, 0x42, 0x44, 0x77, 0x41, 0x77, 0x67, 0x67, 0x45, 0x4b, 0x41, 0x6f,
  0x49, 0x42, 0x41, 0x51, 0x44, 0x77, 0x4e, 0x6d, 0x2b, 0x46, 0x7a, 0x78,
  0x6e, 0x6b, 0x48, 0x57, 0x2f, 0x4e, 0x70, 0x37, 0x59, 0x48, 0x34, 0x4d,
  0x79, 0x45, 0x0a, 0x77, 0x4d, 0x6c, 0x49, 0x67, 0x71, 0x30, 0x66, 0x45,
  0x77, 0x70, 0x47, 0x6f, 0x41, 0x75, 0x78, 0x44, 0x64, 0x61, 0x46, 0x55,
  0x32, 0x6f, 0x70, 0x76, 0x41, 0x51, 0x56, 0x61, 0x2b, 0x41, 0x43, 0x46,
  0x38, 0x63, 0x6f, 0x38, 0x4d, 0x4a, 0x6c, 0x33, 0x78, 0x77, 0x76, 0x46,
  0x44, 0x2b, 0x67, 0x61, 0x46, 0x45, 0x7a, 0x59, 0x78, 0x53, 0x58, 0x30,
  0x43, 0x47, 0x72, 0x4a, 0x45, 0x4c, 0x63, 0x0a, 0x74, 0x34, 0x4d, 0x69,
  0x30, 0x68, 0x4b, 0x50, 0x76, 0x42, 0x70, 0x65, 0x73, 0x59, 0x6c, 0x46,
  0x4d, 0x51, 0x65, 0x6b, 0x2b, 0x63, 0x70, 0x51, 0x50, 0x33, 0x4b, 0x35,
  0x75, 0x36, 0x71, 0x58, 0x5a, 0x52, 0x49, 0x67, 0x48, 0x75, 0x59, 0x45,
  0x4c, 0x2f, 0x73, 0x55, 0x6f, 0x39, 0x32, 0x70, 0x44, 0x30, 0x7a, 0x4a,
  0x65, 0x4c, 0x47, 0x41, 0x31, 0x49, 0x30, 0x4b, 0x5a, 0x34, 0x73, 0x2f,
  0x0a, 0x51, 0x7a, 0x77, 0x61, 0x4f, 0x38, 0x62, 0x62, 0x4b, 0x6d, 0x37,
  0x42, 0x72, 0x6e, 0x56, 0x77, 0x30, 0x6e, 0x5a, 0x2f, 0x4b, 0x41, 0x5a,
  0x6a, 0x75, 0x78, 0x75, 0x6f, 0x4e, 0x33, 0x52, 0x64, 0x72, 0x69, 0x30,
  0x4a, 0x48, 0x77, 0x7a, 0x6a, 0x41, 0x55, 0x34, 0x2b, 0x71, 0x57, 0x65,
  0x55, 0x63, 0x2f, 0x64, 0x33, 0x45, 0x70, 0x4f, 0x47, 0x78, 0x69, 0x42,
  0x77, 0x5a, 0x4e, 0x61, 0x7a, 0x0a, 0x39, 0x6f, 0x4a, 0x41, 0x37, 0x54,
  0x2f, 0x51, 0x6f, 0x62, 0x75, 0x61, 0x4e, 0x53, 0x6b, 0x65, 0x55, 0x48,
  0x43, 0x61, 0x50, 0x53, 0x6a, 0x44, 0x37, 0x71, 0x7a, 0x6c, 0x43, 0x4f,
  0x52, 0x48, 0x47, 0x68, 0x75, 0x31, 0x76, 0x79, 0x79, 0x35, 0x31, 0x45,
  0x36, 0x79, 0x46, 0x43, 0x4e, 0x47, 0x66, 0x65, 0x7a, 0x71, 0x2f, 0x4d,
  0x59, 0x34, 0x4e, 0x4b, 0x68, 0x77, 0x72, 0x61, 0x59, 0x64, 0x0a, 0x62,
  0x79, 0x49, 0x2f, 0x6c, 0x42, 0x46, 0x62, 0x36, 0x35, 0x6b, 0x5a, 0x45,
  0x66, 0x49, 0x4b, 0x4b, 0x54, 0x7a, 0x79, 0x36, 0x76, 0x30, 0x44, 0x65,
  0x79, 0x50, 0x37, 0x52, 0x6b, 0x34, 0x75, 0x48, 0x44, 0x38, 0x77, 0x62,
  0x49, 0x79, 0x50, 0x32, 0x47, 0x6c, 0x42, 0x30, 0x67, 0x37, 0x2f, 0x69,
  0x79, 0x33, 0x4c, 0x61, 0x74, 0x49, 0x74, 0x49, 0x70, 0x2b, 0x49, 0x35,
  0x53, 0x50, 0x56, 0x0a, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x47, 0x6a,
  0x55, 0x44, 0x42, 0x4f, 0x4d, 0x42, 0x30, 0x47, 0x41, 0x31, 0x55, 0x64,
  0x44, 0x67, 0x51, 0x57, 0x42, 0x42, 0x53, 0x36, 0x49, 0x4c, 0x5a, 0x2f,
  0x71, 0x38, 0x66, 0x2f, 0x4b, 0x45, 0x68, 0x4b, 0x76, 0x68, 0x69, 0x2b,
  0x73, 0x6b, 0x59, 0x45, 0x31, 0x79, 0x48, 0x71, 0x39, 0x7a, 0x41, 0x66,
  0x42, 0x67, 0x4e, 0x56, 0x48, 0x53, 0x4d, 0x45, 0x0a, 0x47, 0x44, 0x41,
  0x57, 0x67, 0x42, 0x53, 0x36, 0x49, 0x4c, 0x5a, 0x2f, 0x71, 0x38, 0x66,
  0x2f, 0x4b, 0x45, 0x68, 0x4b, 0x76, 0x68, 0x69, 0x2b, 0x73, 0x6b, 0x59,
  0x45, 0x31, 0x79, 0x48, 0x71, 0x39, 0x7a, 0x41, 0x4d, 0x42, 0x67, 0x4e,
  0x56, 0x48, 0x52, 0x4d, 0x45, 0x42, 0x54, 0x41, 0x44, 0x41, 0x51, 0x48,
  0x2f, 0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62,
  0x33, 0x0a, 0x44, 0x51, 0x45, 0x42, 0x42, 0x51, 0x55, 0x41, 0x41, 0x34,
  0x49, 0x42, 0x41, 0x51, 0x41, 0x79, 0x5a, 0x59, 0x77, 0x47, 0x4b, 0x46,
  0x34, 0x34, 0x43, 0x68, 0x47, 0x51, 0x72, 0x6e, 0x74, 0x57, 0x6c, 0x38,
  0x48, 0x53, 0x4a, 0x30, 0x63, 0x69, 0x55, 0x58, 0x4d, 0x44, 0x4b, 0x32,
  0x46, 0x6c, 0x6f, 0x74, 0x47, 0x49, 0x6a, 0x30, 0x32, 0x6c, 0x4d, 0x39,
  0x38, 0x71, 0x45, 0x49, 0x65, 0x68, 0x0a, 0x56, 0x67, 0x66, 0x41, 0x34,
  0x7a, 0x69, 0x37, 0x4d, 0x45, 0x6c, 0x51, 0x61, 0x76, 0x6b, 0x52, 0x76,
  0x32, 0x54, 0x43, 0x50, 0x50, 0x55, 0x51, 0x62, 0x35, 0x51, 0x64, 0x61,
  0x6f, 0x37, 0x57, 0x78, 0x37, 0x6c, 0x66, 0x61, 0x54, 0x6f, 0x5a, 0x68,
  0x4f, 0x54, 0x2b, 0x4e, 0x52, 0x68, 0x32, 0x6b, 0x35, 0x78, 0x2b, 0x6b,
  0x6a, 0x5a, 0x46, 0x77, 0x38, 0x70, 0x45, 0x48, 0x74, 0x35, 0x51, 0x0a,
  0x69, 0x68, 0x62, 0x46, 0x4c, 0x35, 0x58, 0x2b, 0x57, 0x7a, 0x6f, 0x2b,
  0x42, 0x36, 0x36, 0x59, 0x79, 0x49, 0x76, 0x68, 0x77, 0x54, 0x63, 0x48,
  0x30, 0x46, 0x2b, 0x6e, 0x66, 0x55, 0x71, 0x66, 0x74, 0x38, 0x59, 0x74,
  0x72, 0x2f, 0x38, 0x37, 0x47, 0x45, 0x62, 0x73, 0x41, 0x48, 0x6a, 0x48,
  0x43, 0x36, 0x4c, 0x2b, 0x77, 0x6b, 0x31, 0x76, 0x4e, 0x6e, 0x64, 0x49,
  0x59, 0x47, 0x30, 0x51, 0x0a, 0x79, 0x62, 0x73, 0x7a, 0x78, 0x49, 0x72,
  0x32, 0x6d, 0x46, 0x45, 0x49, 0x4a, 0x6f, 0x69, 0x51, 0x44, 0x44, 0x67,
  0x66, 0x6c, 0x71, 0x67, 0x64, 0x76, 0x4c, 0x54, 0x32, 0x79, 0x64, 0x46,
  0x6d, 0x79, 0x33, 0x73, 0x32, 0x68, 0x49, 0x74, 0x51, 0x6c, 0x49, 0x71,
  0x4b, 0x4c, 0x42, 0x36, 0x49, 0x4a, 0x51, 0x49, 0x75, 0x69, 0x37, 0x72,
  0x37, 0x34, 0x76, 0x64, 0x72, 0x63, 0x58, 0x71, 0x58, 0x0a, 0x44, 0x7a,
  0x68, 0x6d, 0x4c, 0x66, 0x67, 0x6a, 0x67, 0x4c, 0x77, 0x33, 0x2b, 0x55,
  0x79, 0x69, 0x59, 0x74, 0x44, 0x54, 0x76, 0x63, 0x78, 0x65, 0x7a, 0x62,
  0x4c, 0x73, 0x76, 0x51, 0x6f, 0x52, 0x6b, 0x74, 0x77, 0x4b, 0x5a, 0x4c,
  0x44, 0x54, 0x42, 0x42, 0x35, 0x76, 0x59, 0x32, 0x78, 0x4b, 0x36, 0x6b,
  0x4f, 0x4f, 0x44, 0x70, 0x7a, 0x50, 0x48, 0x73, 0x4b, 0x67, 0x30, 0x42,
  0x59, 0x77, 0x0a, 0x4d, 0x6b, 0x48, 0x56, 0x56, 0x54, 0x34, 0x79, 0x2f,
  0x4d, 0x59, 0x36, 0x63, 0x63, 0x4b, 0x51, 0x2f, 0x4c, 0x56, 0x74, 0x32,
  0x66, 0x4a, 0x49, 0x74, 0x69, 0x41, 0x71, 0x49, 0x47, 0x32, 0x38, 0x64,
  0x37, 0x31, 0x53, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44,
  0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45,
  0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a
};

static void testCertTrust(SAFE_PROVIDER_FUNCTIONS *funcs, GUID *actionID)
{
    CRYPT_PROVIDER_DATA data = { 0 };
    CRYPT_PROVIDER_SIGSTATE sig_state = { 0 };
    CRYPT_PROVIDER_SGNR sgnr = { sizeof(sgnr), { 0 } };
    HRESULT ret;
    BOOL b;

    if (!CertFreeCertificateChain_p)
    {
        win_skip("CertFreeCertificateChain not found\n");
        return;
    }

    data.pSigState = &sig_state;
    data.padwTrustStepErrors =
     funcs->pfnAlloc(TRUSTERROR_MAX_STEPS * sizeof(DWORD));
    if (!data.padwTrustStepErrors)
    {
        skip("pfnAlloc failed\n");
        return;
    }
    ret = funcs->pfnCertificateTrust(&data);
    ok(ret == S_FALSE, "Expected S_FALSE, got %08lx\n", ret);
    ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_CERTPROV] ==
     TRUST_E_NOSIGNATURE, "Expected TRUST_E_NOSIGNATURE, got %08lx\n",
     data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_CERTPROV]);
    b = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
    if (b)
    {
        PCCERT_CONTEXT cert;

        /* An empty signer "succeeds," even though there's no cert */
        ret = funcs->pfnCertificateTrust(&data);
        ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
        cert = CertCreateCertificateContext(X509_ASN_ENCODING, selfSignedCert,
         sizeof(selfSignedCert));
        if (cert)
        {
            WINTRUST_DATA wintrust_data = { 0 };

            b = funcs->pfnAddCert2Chain(&data, 0, FALSE, 0, cert);
            ok(b == TRUE, "Expected TRUE, got %d\n", b);

            /* If pWintrustData isn't set, crashes attempting to access
             * pWintrustData->fdwRevocationChecks
             */
            data.pWintrustData = &wintrust_data;
            /* If psPfns isn't set, crashes attempting to access
             * psPfns->pfnCertCheckPolicy
             */
            data.psPfns = (CRYPT_PROVIDER_FUNCTIONS *)funcs;
            ret = funcs->pfnCertificateTrust(&data);
            ok(ret == S_OK, "Expected S_OK, got %08lx\n", ret);
            ok(data.csSigners == 1, "Unexpected number of signers %ld\n",
             data.csSigners);
            ok(data.pasSigners[0].pChainContext != NULL,
             "Expected a certificate chain\n");
            ok(data.pasSigners[0].csCertChain == 1,
             "Unexpected number of chain elements %ld\n",
             data.pasSigners[0].csCertChain);
            /* pasSigners and pasSigners[0].pasCertChain are guaranteed to be
             * initialized, see tests for pfnAddSgnr2Chain and pfnAddCert2Chain
             */
            ok(!data.pasSigners[0].pasCertChain[0].fTrustedRoot,
             "Didn't expect cert to be trusted\n");
            ok(data.pasSigners[0].pasCertChain[0].fSelfSigned,
             "Expected cert to be self-signed\n");
            ok(data.pasSigners[0].pasCertChain[0].dwConfidence ==
             (CERT_CONFIDENCE_SIG | CERT_CONFIDENCE_TIMENEST),
             "Expected CERT_CONFIDENCE_SIG | CERT_CONFIDENCE_TIMENEST, got %08lx\n",
             data.pasSigners[0].pasCertChain[0].dwConfidence);
            CertFreeCertificateContext(
             data.pasSigners[0].pasCertChain[0].pCert);
            CertFreeCertificateChain_p(data.pasSigners[0].pChainContext);
            CertFreeCertificateContext(cert);
        }
    }
    funcs->pfnFree(data.padwTrustStepErrors);
}

static void test_provider_funcs(void)
{
    static GUID generic_verify_v2 = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    SAFE_PROVIDER_FUNCTIONS funcs = { sizeof(SAFE_PROVIDER_FUNCTIONS), 0 };
    BOOL ret;

    ret = WintrustLoadFunctionPointers(&generic_verify_v2,
     (CRYPT_PROVIDER_FUNCTIONS *)&funcs);
    if (!ret)
        skip("WintrustLoadFunctionPointers failed\n");
    else
    {
        test_utils(&funcs);
        testInitialize(&funcs, &generic_verify_v2);
        testObjTrust(&funcs, &generic_verify_v2);
        testCertTrust(&funcs, &generic_verify_v2);
    }
}

/* minimal PE file image */
#define VA_START 0x400000
#define FILE_PE_START 0x50
#define NUM_SECTIONS 3
#define FILE_TEXT 0x200
#define RVA_TEXT 0x1000
#define RVA_BSS 0x2000
#define FILE_IDATA 0x400
#define RVA_IDATA 0x3000
#define FILE_TOTAL 0x600
#define RVA_TOTAL 0x4000
#include <pshpack1.h>
struct Imports {
    IMAGE_IMPORT_DESCRIPTOR descriptors[2];
    IMAGE_THUNK_DATA32 original_thunks[2];
    IMAGE_THUNK_DATA32 thunks[2];
    struct __IMPORT_BY_NAME {
        WORD hint;
        char funcname[0x20];
    } ibn;
    char dllname[0x10];
};
#define EXIT_PROCESS (VA_START+RVA_IDATA+FIELD_OFFSET(struct Imports, thunks))

static struct _PeImage {
    IMAGE_DOS_HEADER dos_header;
    char __alignment1[FILE_PE_START - sizeof(IMAGE_DOS_HEADER)];
    IMAGE_NT_HEADERS32 nt_headers;
    IMAGE_SECTION_HEADER sections[NUM_SECTIONS];
    char __alignment2[FILE_TEXT - FILE_PE_START - sizeof(IMAGE_NT_HEADERS32) -
        NUM_SECTIONS * sizeof(IMAGE_SECTION_HEADER)];
    unsigned char text_section[FILE_IDATA-FILE_TEXT];
    struct Imports idata_section;
    char __alignment3[FILE_TOTAL-FILE_IDATA-sizeof(struct Imports)];
} bin = {
    /* dos header */
    {IMAGE_DOS_SIGNATURE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, {0}, 0, 0, {0}, FILE_PE_START},
    /* alignment before PE header */
    {0},
    /* nt headers */
    {IMAGE_NT_SIGNATURE,
        /* basic headers - 3 sections, no symbols, EXE file */
        {IMAGE_FILE_MACHINE_I386, NUM_SECTIONS, 0, 0, 0, sizeof(IMAGE_OPTIONAL_HEADER32),
            IMAGE_FILE_32BIT_MACHINE | IMAGE_FILE_EXECUTABLE_IMAGE},
        /* optional header */
        {IMAGE_NT_OPTIONAL_HDR32_MAGIC, 4, 0, FILE_IDATA-FILE_TEXT,
            FILE_TOTAL-FILE_IDATA + FILE_IDATA-FILE_TEXT, 0x400,
            RVA_TEXT, RVA_TEXT, RVA_BSS, VA_START, 0x1000, 0x200, 4, 0, 1, 0, 4, 0, 0,
            RVA_TOTAL, FILE_TEXT, 0, IMAGE_SUBSYSTEM_WINDOWS_GUI, 0,
            0x200000, 0x1000, 0x100000, 0x1000, 0, 0x10,
            {{0, 0},
             {RVA_IDATA, sizeof(struct Imports)}
            }
        }
    },
    /* sections */
    {
        {".text", {0x100}, RVA_TEXT, FILE_IDATA-FILE_TEXT, FILE_TEXT,
            0, 0, 0, 0, IMAGE_SCN_CNT_CODE | IMAGE_SCN_MEM_EXECUTE | IMAGE_SCN_MEM_READ},
        {".bss", {0x400}, RVA_BSS, 0, 0, 0, 0, 0, 0,
            IMAGE_SCN_CNT_UNINITIALIZED_DATA | IMAGE_SCN_MEM_READ | IMAGE_SCN_MEM_WRITE},
        {".idata", {sizeof(struct Imports)}, RVA_IDATA, FILE_TOTAL-FILE_IDATA, FILE_IDATA, 0,
            0, 0, 0, IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_MEM_READ | IMAGE_SCN_MEM_WRITE}
    },
    /* alignment before first section */
    {0},
    /* .text section */
    {
        0x31, 0xC0, /* xor eax, eax */
        0xFF, 0x25, EXIT_PROCESS&0xFF, (EXIT_PROCESS>>8)&0xFF, (EXIT_PROCESS>>16)&0xFF,
            (EXIT_PROCESS>>24)&0xFF, /* jmp ExitProcess */
        0
    },
    /* .idata section */
    {
        {
            {{RVA_IDATA + FIELD_OFFSET(struct Imports, original_thunks)}, 0, 0,
            RVA_IDATA + FIELD_OFFSET(struct Imports, dllname),
            RVA_IDATA + FIELD_OFFSET(struct Imports, thunks)
            },
            {{0}, 0, 0, 0, 0}
        },
        {{{RVA_IDATA+FIELD_OFFSET(struct Imports, ibn)}}, {{0}}},
        {{{RVA_IDATA+FIELD_OFFSET(struct Imports, ibn)}}, {{0}}},
        {0,"ExitProcess"},
        "KERNEL32.DLL"
    },
    /* final alignment */
    {0}
};
#include <poppack.h>

static void test_sip_create_indirect_data(void)
{
    static GUID unknown = { 0xC689AAB8, 0x8E78, 0x11D0, { 0x8C,0x47,
     0x00,0xC0,0x4F,0xC2,0x95,0xEE } };
    static char oid_sha1[] = szOID_OIWSEC_sha1;
    BOOL ret;
    SIP_SUBJECTINFO subjinfo = { 0 };
    WCHAR temp_file[MAX_PATH];
    HANDLE file;
    DWORD count;

    if (!CryptSIPCreateIndirectData_p)
    {
        skip("Missing CryptSIPCreateIndirectData\n");
        return;
    }
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(NULL, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    subjinfo.cbSize = sizeof(subjinfo);
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    file = create_temp_file(temp_file);
    if (file == INVALID_HANDLE_VALUE)
    {
        skip("couldn't create temp file\n");
        return;
    }
    WriteFile(file, &bin, sizeof(bin), &count, NULL);
    FlushFileBuffers(file);

    subjinfo.hFile = file;
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    subjinfo.pgSubjectType = &unknown;
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    subjinfo.DigestAlgorithm.pszObjId = oid_sha1;
    count = 0xdeadbeef;
    ret = CryptSIPCreateIndirectData_p(&subjinfo, &count, NULL);
    todo_wine
    ok(ret, "CryptSIPCreateIndirectData failed: %ld\n", GetLastError());
    ok(count, "expected a positive count\n");
    if (ret)
    {
        SIP_INDIRECT_DATA *indirect = HeapAlloc(GetProcessHeap(), 0, count);

        count = 256;
        ret = CryptSIPCreateIndirectData_p(&subjinfo, &count, indirect);
        ok(ret, "CryptSIPCreateIndirectData failed: %ld\n", GetLastError());
        /* If the count is larger than needed, it's unmodified */
        ok(count == 256, "unexpected count %ld\n", count);
        ok(!strcmp(indirect->Data.pszObjId, SPC_PE_IMAGE_DATA_OBJID),
           "unexpected data oid %s\n",
           indirect->Data.pszObjId);
        ok(!strcmp(indirect->DigestAlgorithm.pszObjId, oid_sha1),
           "unexpected digest algorithm oid %s\n",
           indirect->DigestAlgorithm.pszObjId);
        ok(indirect->Digest.cbData == 20, "unexpected hash size %ld\n",
           indirect->Digest.cbData);
        if (indirect->Digest.cbData == 20)
        {
            const BYTE hash[20] = {
                0x8a,0xd5,0x45,0x53,0x3d,0x67,0xdf,0x2f,0x78,0xe0,
                0x55,0x0a,0xe0,0xd9,0x7a,0x28,0x3e,0xbf,0x45,0x2b };

            ok(!memcmp(indirect->Digest.pbData, hash, 20),
               "unexpected value\n");
        }

        HeapFree(GetProcessHeap(), 0, indirect);
    }
    CloseHandle(file);
    DeleteFileW(temp_file);
}

static void test_wintrust(void)
{
    static GUID generic_action_v2 = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    WINTRUST_DATA wtd;
    WINTRUST_FILE_INFO file;
    LONG r;
    HRESULT hr;
    WCHAR pathW[MAX_PATH];

    memset(&wtd, 0, sizeof(wtd));
    wtd.cbStruct = sizeof(wtd);
    wtd.dwUIChoice = WTD_UI_NONE;
    wtd.fdwRevocationChecks = WTD_REVOKE_WHOLECHAIN;
    wtd.dwUnionChoice = WTD_CHOICE_FILE;
    U(wtd).pFile = &file;
    wtd.dwStateAction = WTD_STATEACTION_VERIFY;
    memset(&file, 0, sizeof(file));
    file.cbStruct = sizeof(file);
    file.pcwszFilePath = pathW;
    /* Test with an empty file */
    file.hFile = create_temp_file(pathW);
    SetLastError(0xdeadbeef);
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(r == GetLastError(), "expected %08lx, got %08lx\n", GetLastError(), r);
    ok(r == TRUST_E_SUBJECT_FORM_UNKNOWN,
     "expected TRUST_E_SUBJECT_FORM_UNKNOWN, got %08lx\n", r);
    CloseHandle(file.hFile);
    DeleteFileW(pathW);
    file.hFile = NULL;
    /* Test with a known file path, which we expect not have a signature */
    getNotepadPath(pathW, MAX_PATH);
    SetLastError(0xdeadbeef);
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(r == GetLastError(), "expected %08lx, got %08lx\n", GetLastError(), r);
    ok(r == TRUST_E_NOSIGNATURE || r == CRYPT_E_FILE_ERROR,
     "expected TRUST_E_NOSIGNATURE or CRYPT_E_FILE_ERROR, got %08lx\n", r);
    wtd.dwStateAction = WTD_STATEACTION_CLOSE;
    SetLastError(0xdeadbeef);
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(GetLastError() == 0xdeadbeef, "expected 0xdeadbeef, got %08lx\n", GetLastError());
    ok(r == S_OK, "WinVerifyTrust failed: %08lx\n", r);
    wtd.dwStateAction = WTD_STATEACTION_VERIFY;
    SetLastError(0xdeadbeef);
    hr = WinVerifyTrustEx(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(hr == GetLastError(), "expected %08lx, got %08lx\n", GetLastError(), hr);
    ok(hr == TRUST_E_NOSIGNATURE || hr == CRYPT_E_FILE_ERROR,
     "expected TRUST_E_NOSIGNATURE or CRYPT_E_FILE_ERROR, got %08lx\n", hr);
    wtd.dwStateAction = WTD_STATEACTION_CLOSE;
    SetLastError(0xdeadbeef);
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(GetLastError() == 0xdeadbeef, "expected 0xdeadbeef, got %08lx\n", GetLastError());
    ok(r == S_OK, "WinVerifyTrust failed: %08lx\n", r);
}

/* Self-signed .exe, built with tcc, signed with signtool
 * (and a certificate generated on a self-signed CA).
 *
 * small.c:
 * int _start()
 * {
 *     return 0;
 * }
 *
 * tcc -nostdlib small.c
 * signtool sign /v /f codesign.pfx small.exe
 */
static const BYTE SelfSignedFile32[] =
{
    0x4D,0x5A,0x90,0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0xFF,0xFF,0x00,0x00,0xB8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x0E,0x1F,0xBA,0x0E,0x00,0xB4,0x09,0xCD,
    0x21,0xB8,0x01,0x4C,0xCD,0x21,0x54,0x68,0x69,0x73,0x20,0x70,0x72,0x6F,0x67,0x72,0x61,0x6D,0x20,0x63,0x61,0x6E,0x6E,0x6F,
    0x74,0x20,0x62,0x65,0x20,0x72,0x75,0x6E,0x20,0x69,0x6E,0x20,0x44,0x4F,0x53,0x20,0x6D,0x6F,0x64,0x65,0x2E,0x0D,0x0D,0x0A,
    0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0x45,0x00,0x00,0x4C,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0xE0,0x00,0x0F,0x03,0x0B,0x01,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x10,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,
    0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x02,0x00,0x00,
    0xE7,0x0C,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x00,0x00,
    0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x68,0x05,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2E,0x74,0x65,0x78,0x74,0x00,0x00,0x00,
    0x18,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x55,0x89,0xE5,0x81,0xEC,0x00,0x00,0x00,0x00,0x90,0xB8,0x00,0x00,0x00,0x00,0xE9,
    0x00,0x00,0x00,0x00,0xC9,0xC3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x68,0x05,0x00,0x00,0x00,0x02,0x02,0x00,
    /* Start of the signature overlay */
    0x30,0x82,0x05,0x5A,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x07,0x02,0xA0,0x82,0x05,0x4B,0x30,0x82,0x05,0x47,0x02,
    0x01,0x01,0x31,0x0B,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x30,0x4C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,
    0x82,0x37,0x02,0x01,0x04,0xA0,0x3E,0x30,0x3C,0x30,0x17,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0F,0x30,
    0x09,0x03,0x01,0x00,0xA0,0x04,0xA2,0x02,0x80,0x00,0x30,0x21,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x04,
    0x14,0xA0,0x95,0xDE,0xBD,0x1A,0xB7,0x86,0xAF,0x50,0x63,0xD8,0x8F,0x90,0xD5,0x49,0x96,0x4E,0x44,0xF0,0x71,0xA0,0x82,0x03,
    0x1D,0x30,0x82,0x03,0x19,0x30,0x82,0x02,0x01,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,0x96,0x53,0x2C,0xC9,0x23,0x56,0x8A,0x87,
    0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,
    0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,
    0x30,0x1E,0x17,0x0D,0x31,0x36,0x30,0x33,0x30,0x33,0x32,0x30,0x32,0x37,0x30,0x37,0x5A,0x17,0x0D,0x34,0x39,0x31,0x32,0x33,
    0x31,0x32,0x33,0x30,0x30,0x30,0x30,0x5A,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x43,0x6F,0x64,
    0x65,0x53,0x69,0x67,0x6E,0x54,0x65,0x73,0x74,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,
    0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xB2,0xC9,0x91,0x98,0x8C,0xDC,
    0x80,0xBC,0x16,0xBF,0xC1,0x04,0x77,0x90,0xC0,0xFD,0x8C,0xBA,0x68,0x26,0xAC,0xB7,0x20,0x68,0x41,0xED,0xC3,0x9C,0x47,0x7C,
    0x36,0xC2,0x7B,0xE1,0x5E,0xFD,0xA9,0x99,0xF4,0x29,0x36,0x86,0x93,0x40,0x55,0x53,0x65,0x79,0xBC,0x9F,0x8F,0x6E,0x2B,0x05,
    0x84,0xE1,0xFD,0xD2,0xEF,0xEA,0x89,0x8C,0xEC,0xF9,0x55,0xF0,0x2C,0xE5,0xA7,0x29,0xF9,0x7E,0x50,0xDC,0x9C,0xA1,0x23,0xA5,
    0xD9,0x78,0xA1,0xE7,0x7C,0xD7,0x04,0x4F,0x11,0xAC,0x9F,0x4A,0x47,0xA1,0x1E,0xD5,0x9E,0xE7,0x5B,0xB5,0x8C,0x9C,0x67,0x7A,
    0xD0,0xF8,0x54,0xD1,0x64,0x7F,0x39,0x48,0xB6,0xCF,0x2F,0x26,0x7D,0x7B,0x13,0x2B,0xC2,0x8F,0xA6,0x3F,0x42,0x71,0x95,0x3E,
    0x59,0x0F,0x12,0xFA,0xC2,0x70,0x89,0xB7,0xB6,0x10,0x49,0xE0,0x7D,0x4D,0xFC,0x80,0x61,0x53,0x50,0x72,0xFD,0x46,0x35,0x51,
    0x36,0xE6,0x06,0xA9,0x4C,0x0D,0x82,0x15,0xF6,0x5D,0xDE,0xD4,0xDB,0xE7,0x82,0x10,0x40,0xA1,0x47,0x68,0x88,0x0C,0x0A,0x80,
    0xD1,0xE5,0x9A,0x35,0x28,0x82,0x1F,0x0F,0x80,0x5A,0x6E,0x1D,0x22,0x22,0xB3,0xA7,0xA2,0x9E,0x82,0x2D,0xC0,0x7F,0x5A,0xD0,
    0xBA,0xB2,0xCA,0x20,0xE2,0x97,0xE9,0x72,0x41,0xB7,0xD6,0x1A,0x93,0x23,0x97,0xF0,0xA9,0x61,0xD2,0x91,0xBD,0xB6,0x6B,0x95,
    0x12,0x67,0x16,0xAC,0x0A,0xB7,0x55,0x02,0x0D,0xA5,0xAD,0x17,0x95,0x77,0xF9,0x96,0x03,0x41,0xD3,0xE1,0x61,0x68,0xBB,0x0A,
    0xB5,0xC4,0xEE,0x70,0x40,0x08,0x05,0xC4,0xF1,0x5D,0x02,0x03,0x01,0x00,0x01,0xA3,0x61,0x30,0x5F,0x30,0x13,0x06,0x03,0x55,
    0x1D,0x25,0x04,0x0C,0x30,0x0A,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x03,0x03,0x30,0x48,0x06,0x03,0x55,0x1D,0x01,0x04,
    0x41,0x30,0x3F,0x80,0x10,0x35,0x40,0x67,0x8F,0x7D,0x03,0x1B,0x76,0x52,0x62,0x2D,0xF5,0x21,0xF6,0x7C,0xBC,0xA1,0x19,0x30,
    0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,
    0x82,0x10,0xA0,0x4B,0xEB,0xAC,0xFA,0x08,0xF2,0x8B,0x47,0xD2,0xB3,0x54,0x60,0x6C,0xE6,0x29,0x30,0x0D,0x06,0x09,0x2A,0x86,
    0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x5F,0x8C,0x7F,0xDA,0x1D,0x21,0x7A,0x15,0xD8,0x20,
    0x04,0x53,0x7F,0x44,0x6D,0x7B,0x57,0xBE,0x7F,0x86,0x77,0x58,0xC4,0xD4,0x80,0xC7,0x2E,0x64,0x9B,0x44,0xC5,0x2D,0x6D,0xDB,
    0x35,0x5A,0xFE,0xA4,0xD8,0x66,0x9B,0xF7,0x6E,0xFC,0xEF,0x52,0x7B,0xC5,0x16,0xE6,0xA3,0x7D,0x59,0xB7,0x31,0x28,0xEB,0xB5,
    0x45,0xC9,0xB1,0xD1,0x08,0x67,0xC6,0x37,0xE7,0xD7,0x2A,0xE6,0x1F,0xD9,0x6A,0xE5,0x04,0xDF,0x6A,0x9D,0x91,0xFA,0x41,0xBD,
    0x2A,0x50,0xEA,0x99,0x24,0xA9,0x0F,0x2B,0x50,0x51,0x5F,0xD9,0x0B,0x89,0x1B,0xCB,0xDB,0x88,0xE8,0xEC,0x87,0xB0,0x16,0xCC,
    0x43,0xEE,0x5A,0xBD,0x57,0xE2,0x46,0xA7,0x56,0x54,0x23,0x32,0x8A,0xFB,0x25,0x51,0x39,0x38,0xE6,0x87,0xF5,0x73,0x63,0xD0,
    0x5B,0xC7,0x3F,0xFD,0x04,0x75,0x74,0x4C,0x3D,0xB5,0x31,0x22,0x7D,0xF1,0x8D,0xB4,0xE0,0xAA,0xE1,0xFF,0x8F,0xDD,0xB8,0x04,
    0x6A,0x31,0xEE,0x30,0x2D,0x6E,0x74,0x0F,0x37,0x71,0x77,0x2B,0xB8,0x9E,0x62,0x47,0x00,0x9C,0xA5,0x82,0x2B,0x9F,0x24,0x67,
    0x50,0x86,0x8B,0xC9,0x36,0x81,0xEB,0x44,0xC2,0xF1,0x91,0xA6,0x84,0x75,0x15,0x8F,0x22,0xDE,0xAC,0xB5,0x16,0xE3,0x96,0x74,
    0x72,0x2F,0x15,0xD5,0xFB,0x01,0x22,0xC4,0x24,0xEE,0x3D,0xDF,0x9E,0xA9,0x0A,0x5B,0x16,0x21,0xE8,0x4A,0x8C,0x7E,0x3A,0x9C,
    0x22,0xA0,0x49,0x60,0x97,0x1B,0x3E,0x2D,0x80,0x91,0xDB,0xF7,0x78,0x38,0x76,0x78,0x0C,0xE3,0xD4,0x27,0x77,0x69,0x96,0xE6,
    0x41,0xC7,0x2E,0xE9,0x61,0xD6,0x31,0x82,0x01,0xC4,0x30,0x82,0x01,0xC0,0x02,0x01,0x01,0x30,0x2B,0x30,0x17,0x31,0x15,0x30,
    0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,0x02,0x10,0x96,0x53,
    0x2C,0xC9,0x23,0x56,0x8A,0x87,0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,
    0x00,0xA0,0x70,0x30,0x10,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0C,0x31,0x02,0x30,0x00,0x30,0x19,0x06,
    0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x03,0x31,0x0C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,
    0x30,0x1C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0B,0x31,0x0E,0x30,0x0C,0x06,0x0A,0x2B,0x06,0x01,0x04,
    0x01,0x82,0x37,0x02,0x01,0x15,0x30,0x23,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x04,0x31,0x16,0x04,0x14,0x3D,
    0x08,0xC8,0xA3,0xEE,0x05,0x1A,0x61,0xD9,0xFE,0x1A,0x63,0xC0,0x8A,0x6E,0x9D,0xF9,0xC3,0x13,0x98,0x30,0x0D,0x06,0x09,0x2A,
    0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0x00,0x90,0xF9,0xC0,0x7F,0x1D,0x70,0x8C,0x04,0x22,0x82,
    0xB6,0x2D,0x48,0xBF,0x30,0x51,0x29,0xF8,0xE3,0x11,0x39,0xE0,0x64,0x23,0x72,0xE2,0x4C,0x09,0x9F,0x39,0xF2,0x6F,0xDD,0xB9,
    0x5A,0x3D,0xEF,0xEB,0xBE,0xEC,0x3B,0xE6,0x58,0x4C,0xC9,0x4F,0xED,0xCB,0x6E,0x9D,0x67,0x8E,0x89,0x92,0x40,0x39,0xA2,0x5F,
    0xF9,0xEF,0xD3,0xF5,0x24,0x27,0x8D,0xF7,0x3C,0x92,0x66,0x56,0xC8,0x2B,0xEA,0x04,0xA1,0x0E,0xDA,0x89,0x30,0xA7,0x01,0xD8,
    0x0B,0xF8,0xFD,0x99,0xB6,0xC0,0x38,0xB0,0x21,0x50,0x3A,0x86,0x01,0xD0,0xF3,0x86,0x72,0xE3,0x5A,0xBB,0x2A,0x6E,0xBD,0xFB,
    0x22,0xF9,0x42,0xD3,0x04,0xFE,0x8D,0xD8,0x79,0xD1,0xEE,0x61,0xC6,0x48,0x04,0x99,0x9A,0xA2,0x73,0xE5,0xFB,0x24,0x10,0xD5,
    0x6B,0x71,0x80,0x0E,0x09,0xEA,0x85,0x9A,0xBD,0xBB,0xDE,0x99,0x5D,0xA3,0x18,0x4D,0xED,0x20,0x73,0x3E,0x32,0xEF,0x2C,0xAC,
    0x5A,0x83,0x87,0x1F,0x7F,0x19,0x61,0x35,0x53,0xC1,0xAA,0x89,0x97,0xB3,0xDD,0x8D,0xA8,0x67,0x5B,0xC2,0xE2,0x09,0xB7,0xDD,
    0x6A,0xCB,0xD5,0xBF,0xD6,0x08,0xE2,0x23,0x1A,0x41,0x9D,0xD5,0x6A,0x6B,0x8D,0x3C,0x29,0x1B,0xF1,0x3F,0x4E,0x4A,0x8F,0x29,
    0x33,0xF9,0x1C,0x60,0xA0,0x92,0x7E,0x4F,0x35,0xB8,0xDD,0xEB,0xD1,0x68,0x1A,0x9D,0xA2,0xA6,0x97,0x1F,0x5F,0xC6,0x2C,0xFB,
    0xCA,0xDF,0xF7,0x95,0x33,0x95,0xD4,0x79,0x5C,0x73,0x87,0x49,0x1F,0x8C,0x6E,0xCE,0x3E,0x6D,0x3D,0x2B,0x6B,0xD7,0x66,0xE9,
    0x88,0x6F,0xF2,0x83,0xB9,0x9B,0x00,0x00
};

static const BYTE SelfSignedFile64[] =
{
    0x4D,0x5A,0x90,0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0xFF,0xFF,0x00,0x00,0xB8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x0E,0x1F,0xBA,0x0E,0x00,0xB4,0x09,0xCD,
    0x21,0xB8,0x01,0x4C,0xCD,0x21,0x54,0x68,0x69,0x73,0x20,0x70,0x72,0x6F,0x67,0x72,0x61,0x6D,0x20,0x63,0x61,0x6E,0x6E,0x6F,
    0x74,0x20,0x62,0x65,0x20,0x72,0x75,0x6E,0x20,0x69,0x6E,0x20,0x44,0x4F,0x53,0x20,0x6D,0x6F,0x64,0x65,0x2E,0x0D,0x0D,0x0A,
    0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0x45,0x00,0x00,0x64,0x86,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0xF0,0x00,0x2F,0x02,0x0B,0x02,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x10,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,
    0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x02,0x00,0x00,
    0x02,0xB9,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x20,0x00,0x00,0x0C,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x68,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2E,0x74,0x65,0x78,0x74,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x10,0x00,0x00,
    0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x60,
    0x2E,0x70,0x64,0x61,0x74,0x61,0x00,0x00,0x0C,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x55,0x48,0x89,0xE5,0x48,0x81,0xEC,0x00,0x00,0x00,0x00,0xB8,0x00,0x00,0x00,0x00,
    0xE9,0x00,0x00,0x00,0x00,0xC9,0xC3,0x00,0x01,0x04,0x02,0x05,0x04,0x03,0x01,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0B,0x10,0x00,0x00,0x17,0x10,0x00,0x00,
    0x18,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    /* Start of the signature overlay */
    0x68,0x05,0x00,0x00,0x00,0x02,0x02,0x00,0x30,0x82,0x05,0x5A,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x07,0x02,0xA0,
    0x82,0x05,0x4B,0x30,0x82,0x05,0x47,0x02,0x01,0x01,0x31,0x0B,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x30,
    0x4C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0xA0,0x3E,0x30,0x3C,0x30,0x17,0x06,0x0A,0x2B,0x06,0x01,
    0x04,0x01,0x82,0x37,0x02,0x01,0x0F,0x30,0x09,0x03,0x01,0x00,0xA0,0x04,0xA2,0x02,0x80,0x00,0x30,0x21,0x30,0x09,0x06,0x05,
    0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x04,0x14,0xCA,0x7C,0x10,0xFB,0x5A,0x96,0x6D,0x69,0xEF,0x26,0x30,0x1A,0xE9,0xC7,0x22,
    0x19,0xEB,0x6E,0x17,0x07,0xA0,0x82,0x03,0x1D,0x30,0x82,0x03,0x19,0x30,0x82,0x02,0x01,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,
    0x96,0x53,0x2C,0xC9,0x23,0x56,0x8A,0x87,0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,
    0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,
    0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,0x30,0x1E,0x17,0x0D,0x31,0x36,0x30,0x33,0x30,0x33,0x32,0x30,0x32,0x37,0x30,0x37,
    0x5A,0x17,0x0D,0x34,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x30,0x30,0x30,0x30,0x5A,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,
    0x55,0x04,0x03,0x13,0x0C,0x43,0x6F,0x64,0x65,0x53,0x69,0x67,0x6E,0x54,0x65,0x73,0x74,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,
    0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,
    0x01,0x00,0xB2,0xC9,0x91,0x98,0x8C,0xDC,0x80,0xBC,0x16,0xBF,0xC1,0x04,0x77,0x90,0xC0,0xFD,0x8C,0xBA,0x68,0x26,0xAC,0xB7,
    0x20,0x68,0x41,0xED,0xC3,0x9C,0x47,0x7C,0x36,0xC2,0x7B,0xE1,0x5E,0xFD,0xA9,0x99,0xF4,0x29,0x36,0x86,0x93,0x40,0x55,0x53,
    0x65,0x79,0xBC,0x9F,0x8F,0x6E,0x2B,0x05,0x84,0xE1,0xFD,0xD2,0xEF,0xEA,0x89,0x8C,0xEC,0xF9,0x55,0xF0,0x2C,0xE5,0xA7,0x29,
    0xF9,0x7E,0x50,0xDC,0x9C,0xA1,0x23,0xA5,0xD9,0x78,0xA1,0xE7,0x7C,0xD7,0x04,0x4F,0x11,0xAC,0x9F,0x4A,0x47,0xA1,0x1E,0xD5,
    0x9E,0xE7,0x5B,0xB5,0x8C,0x9C,0x67,0x7A,0xD0,0xF8,0x54,0xD1,0x64,0x7F,0x39,0x48,0xB6,0xCF,0x2F,0x26,0x7D,0x7B,0x13,0x2B,
    0xC2,0x8F,0xA6,0x3F,0x42,0x71,0x95,0x3E,0x59,0x0F,0x12,0xFA,0xC2,0x70,0x89,0xB7,0xB6,0x10,0x49,0xE0,0x7D,0x4D,0xFC,0x80,
    0x61,0x53,0x50,0x72,0xFD,0x46,0x35,0x51,0x36,0xE6,0x06,0xA9,0x4C,0x0D,0x82,0x15,0xF6,0x5D,0xDE,0xD4,0xDB,0xE7,0x82,0x10,
    0x40,0xA1,0x47,0x68,0x88,0x0C,0x0A,0x80,0xD1,0xE5,0x9A,0x35,0x28,0x82,0x1F,0x0F,0x80,0x5A,0x6E,0x1D,0x22,0x22,0xB3,0xA7,
    0xA2,0x9E,0x82,0x2D,0xC0,0x7F,0x5A,0xD0,0xBA,0xB2,0xCA,0x20,0xE2,0x97,0xE9,0x72,0x41,0xB7,0xD6,0x1A,0x93,0x23,0x97,0xF0,
    0xA9,0x61,0xD2,0x91,0xBD,0xB6,0x6B,0x95,0x12,0x67,0x16,0xAC,0x0A,0xB7,0x55,0x02,0x0D,0xA5,0xAD,0x17,0x95,0x77,0xF9,0x96,
    0x03,0x41,0xD3,0xE1,0x61,0x68,0xBB,0x0A,0xB5,0xC4,0xEE,0x70,0x40,0x08,0x05,0xC4,0xF1,0x5D,0x02,0x03,0x01,0x00,0x01,0xA3,
    0x61,0x30,0x5F,0x30,0x13,0x06,0x03,0x55,0x1D,0x25,0x04,0x0C,0x30,0x0A,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x03,0x03,
    0x30,0x48,0x06,0x03,0x55,0x1D,0x01,0x04,0x41,0x30,0x3F,0x80,0x10,0x35,0x40,0x67,0x8F,0x7D,0x03,0x1B,0x76,0x52,0x62,0x2D,
    0xF5,0x21,0xF6,0x7C,0xBC,0xA1,0x19,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,
    0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,0x82,0x10,0xA0,0x4B,0xEB,0xAC,0xFA,0x08,0xF2,0x8B,0x47,0xD2,0xB3,0x54,0x60,0x6C,
    0xE6,0x29,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x5F,0x8C,
    0x7F,0xDA,0x1D,0x21,0x7A,0x15,0xD8,0x20,0x04,0x53,0x7F,0x44,0x6D,0x7B,0x57,0xBE,0x7F,0x86,0x77,0x58,0xC4,0xD4,0x80,0xC7,
    0x2E,0x64,0x9B,0x44,0xC5,0x2D,0x6D,0xDB,0x35,0x5A,0xFE,0xA4,0xD8,0x66,0x9B,0xF7,0x6E,0xFC,0xEF,0x52,0x7B,0xC5,0x16,0xE6,
    0xA3,0x7D,0x59,0xB7,0x31,0x28,0xEB,0xB5,0x45,0xC9,0xB1,0xD1,0x08,0x67,0xC6,0x37,0xE7,0xD7,0x2A,0xE6,0x1F,0xD9,0x6A,0xE5,
    0x04,0xDF,0x6A,0x9D,0x91,0xFA,0x41,0xBD,0x2A,0x50,0xEA,0x99,0x24,0xA9,0x0F,0x2B,0x50,0x51,0x5F,0xD9,0x0B,0x89,0x1B,0xCB,
    0xDB,0x88,0xE8,0xEC,0x87,0xB0,0x16,0xCC,0x43,0xEE,0x5A,0xBD,0x57,0xE2,0x46,0xA7,0x56,0x54,0x23,0x32,0x8A,0xFB,0x25,0x51,
    0x39,0x38,0xE6,0x87,0xF5,0x73,0x63,0xD0,0x5B,0xC7,0x3F,0xFD,0x04,0x75,0x74,0x4C,0x3D,0xB5,0x31,0x22,0x7D,0xF1,0x8D,0xB4,
    0xE0,0xAA,0xE1,0xFF,0x8F,0xDD,0xB8,0x04,0x6A,0x31,0xEE,0x30,0x2D,0x6E,0x74,0x0F,0x37,0x71,0x77,0x2B,0xB8,0x9E,0x62,0x47,
    0x00,0x9C,0xA5,0x82,0x2B,0x9F,0x24,0x67,0x50,0x86,0x8B,0xC9,0x36,0x81,0xEB,0x44,0xC2,0xF1,0x91,0xA6,0x84,0x75,0x15,0x8F,
    0x22,0xDE,0xAC,0xB5,0x16,0xE3,0x96,0x74,0x72,0x2F,0x15,0xD5,0xFB,0x01,0x22,0xC4,0x24,0xEE,0x3D,0xDF,0x9E,0xA9,0x0A,0x5B,
    0x16,0x21,0xE8,0x4A,0x8C,0x7E,0x3A,0x9C,0x22,0xA0,0x49,0x60,0x97,0x1B,0x3E,0x2D,0x80,0x91,0xDB,0xF7,0x78,0x38,0x76,0x78,
    0x0C,0xE3,0xD4,0x27,0x77,0x69,0x96,0xE6,0x41,0xC7,0x2E,0xE9,0x61,0xD6,0x31,0x82,0x01,0xC4,0x30,0x82,0x01,0xC0,0x02,0x01,
    0x01,0x30,0x2B,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,
    0x52,0x6F,0x6F,0x74,0x02,0x10,0x96,0x53,0x2C,0xC9,0x23,0x56,0x8A,0x87,0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x09,
    0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0xA0,0x70,0x30,0x10,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,
    0x0C,0x31,0x02,0x30,0x00,0x30,0x19,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x03,0x31,0x0C,0x06,0x0A,0x2B,0x06,
    0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0x30,0x1C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0B,0x31,0x0E,
    0x30,0x0C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x15,0x30,0x23,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,
    0x01,0x09,0x04,0x31,0x16,0x04,0x14,0x0C,0xEC,0x76,0xF2,0x3F,0xE4,0x6F,0xEB,0xFF,0x00,0xDA,0x95,0xE7,0x8B,0x64,0xBC,0x55,
    0xBA,0xF0,0xEA,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0x00,0x05,0x22,
    0xD1,0xB3,0x85,0x09,0x46,0x99,0x77,0x69,0xC8,0xD2,0x0C,0xFC,0x8D,0xF4,0x01,0xD4,0x5B,0xF0,0xB4,0x13,0x63,0xAF,0x24,0x0E,
    0x6C,0x1B,0x14,0xCF,0xA8,0x9A,0xEC,0x7E,0xF2,0x60,0xED,0x6C,0x39,0x4D,0x7A,0x73,0x9C,0x9F,0x24,0x46,0xE2,0xEA,0xFA,0x70,
    0xB4,0xAC,0xFC,0x38,0x90,0xF2,0x4F,0x70,0xCC,0x00,0xD1,0x2B,0xB6,0xFB,0xCD,0x7F,0xFC,0xCB,0x35,0xA9,0xA6,0x76,0x37,0xD6,
    0x08,0x82,0x99,0x4C,0x47,0xD7,0x4E,0xB5,0xDE,0xCA,0x4E,0xED,0x71,0x48,0xD4,0x84,0xE1,0x30,0x10,0x33,0x7F,0x84,0xEE,0x2F,
    0x44,0x99,0xE4,0x26,0x27,0xB5,0xB8,0xC1,0xA1,0x40,0x6B,0x87,0x04,0x95,0xC3,0xF0,0xFF,0x25,0x97,0xFD,0xDB,0x9C,0x67,0x80,
    0x39,0x97,0x72,0x75,0x07,0x92,0xA5,0x08,0x19,0x5B,0xD3,0xC9,0x5E,0xC4,0x7B,0xA9,0x04,0x02,0x63,0xCC,0xC5,0x92,0xF6,0xE9,
    0xD6,0xB0,0xA8,0xF9,0xD0,0x9F,0x3F,0xBC,0x86,0x77,0x1E,0x12,0x9A,0x9A,0x9B,0x05,0x77,0x39,0x42,0x01,0xB7,0x23,0xF0,0x78,
    0x4F,0x52,0x6D,0x1B,0x9F,0xBA,0x29,0xEC,0x90,0xA9,0x1E,0x1E,0x5C,0xA9,0x28,0xA0,0x0B,0x09,0xDC,0x99,0x82,0xE3,0x34,0xBB,
    0x5C,0x66,0x8E,0x54,0x95,0x4B,0x65,0x95,0xCD,0x87,0x72,0x74,0xCD,0x3B,0x5C,0x72,0xBB,0x61,0x6A,0x98,0x44,0x9C,0xB0,0x2A,
    0xE7,0xB0,0xA6,0x2B,0xDA,0x47,0x5C,0x75,0x36,0xB5,0x90,0x8E,0x82,0x47,0xCD,0x3F,0x4B,0xD0,0xFB,0x8E,0x17,0x6B,0x40,0x57,
    0x9C,0x68,0x1A,0x5D,0x92,0xCD,0xD0,0x5F,0x02,0xA1,0x2C,0xD9,0x56,0x20,0x00,0x00
};

/* Self-signed 32 bit .exe, built with mingw-gcc, stripped, signed with signtool
 * (certificates generated with a self-signed CA).
 *
 * small.c:
 * int _start()
 * {
 *     return 0;
 * }
 *
 * i686-w64-mingw32-gcc -s -nodefaultlibs -fno-PIC ./small.c -o sign_3certs.exe
 * strip -R .idata -R .rdata -R .edata -R .eh_fram ./sign_3certs.exe
 * signtool.exe sign /v /f cert1.pfx /fd SHA256 /t http://timestamp.digicert.com sign_3certs.exe
 * signtool.exe sign /v /f cert2.pfx /as /fd SHA256 sign_3certs.exe
 * signtool.exe sign /v /f cert3.pfx /as /fd SHA256 sign_3certs.exe */

static const BYTE self_signed_3certs[] =
{
    0x4d,0x5a,0x90,0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0xff,0xff,0x00,0x00,0xb8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x0e,0x1f,0xba,0x0e,0x00,0xb4,0x09,0xcd,
    0x21,0xb8,0x01,0x4c,0xcd,0x21,0x54,0x68,0x69,0x73,0x20,0x70,0x72,0x6f,0x67,0x72,0x61,0x6d,0x20,0x63,0x61,0x6e,0x6e,0x6f,
    0x74,0x20,0x62,0x65,0x20,0x72,0x75,0x6e,0x20,0x69,0x6e,0x20,0x44,0x4f,0x53,0x20,0x6d,0x6f,0x64,0x65,0x2e,0x0d,0x0d,0x0a,
    0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0x45,0x00,0x00,0x4c,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0xe0,0x00,0x0e,0x03,0x0b,0x01,0x02,0x25,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x10,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,
    0x04,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x02,0x00,0x00,
    0x76,0x3e,0x00,0x00,0x03,0x00,0x40,0x01,0x00,0x00,0x20,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x00,0x00,
    0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x14,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x48,0x26,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2e,0x74,0x65,0x78,0x74,0x00,0x00,0x00,
    0x1c,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x55,0x89,0xe5,0xb8,0x00,0x00,0x00,0x00,0x5d,0xc3,0x90,0x90,0xff,0xff,0xff,0xff,
    0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x26,0x00,0x00,0x00,0x02,0x02,0x00,
    0x30,0x82,0x26,0x36,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x07,0x02,0xa0,0x82,0x26,0x27,0x30,0x82,0x26,0x23,0x02,
    0x01,0x01,0x31,0x0f,0x30,0x0d,0x06,0x09,0x60,0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0x30,0x5c,0x06,0x0a,0x2b,
    0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0xa0,0x4e,0x30,0x4c,0x30,0x17,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,
    0x02,0x01,0x0f,0x30,0x09,0x03,0x01,0x00,0xa0,0x04,0xa2,0x02,0x80,0x00,0x30,0x31,0x30,0x0d,0x06,0x09,0x60,0x86,0x48,0x01,
    0x65,0x03,0x04,0x02,0x01,0x05,0x00,0x04,0x20,0xdd,0x8b,0xd7,0x29,0x3b,0xae,0x16,0xec,0xbb,0x81,0x80,0x55,0x15,0xd8,0x87,
    0xa5,0x3e,0xeb,0x0b,0x74,0x59,0xb6,0x56,0xf1,0x0b,0x2e,0xe1,0xb4,0x42,0x4d,0x8b,0x18,0xa0,0x82,0x16,0x0c,0x30,0x82,0x03,
    0x01,0x30,0x82,0x01,0xe9,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0xd1,0x73,0x97,0xaa,0xa7,0x3a,0x31,0xa2,0x44,0xc0,0x4b,0x40,
    0x69,0x40,0x4b,0xfa,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x12,0x31,0x10,0x30,
    0x0e,0x06,0x03,0x55,0x04,0x03,0x13,0x07,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x32,0x32,0x30,0x39,0x33,
    0x30,0x31,0x37,0x31,0x39,0x33,0x32,0x5a,0x17,0x0d,0x33,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,
    0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x63,0x65,0x72,0x74,0x31,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,
    0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,
    0x01,0x00,0xca,0x9c,0xd9,0xd4,0x25,0xb6,0x45,0x61,0x22,0x8d,0xdf,0xe9,0x11,0x0f,0xa1,0x7e,0x45,0xc5,0x0b,0xd0,0x42,0xfc,
    0x1f,0x3e,0xce,0x20,0xfc,0x1b,0x37,0xe4,0x0d,0x06,0x83,0x1c,0x3a,0x71,0x0f,0x75,0xf5,0xe5,0x06,0x33,0x01,0x77,0xda,0xc5,
    0xe9,0x2e,0xe3,0x37,0x1e,0x51,0x6e,0x08,0xe2,0x02,0xa1,0x8c,0x11,0xc6,0xfc,0x43,0xa2,0xf5,0x7d,0x74,0x5d,0x5a,0xcc,0x85,
    0x27,0x38,0xd4,0xfa,0xad,0xd7,0xf9,0x77,0xe4,0xef,0xdd,0xb0,0xb1,0x3e,0xdc,0xf5,0x5d,0x7e,0x62,0xdf,0x16,0x01,0x88,0xcd,
    0xb0,0xfa,0x06,0x24,0xd7,0xce,0xdc,0xe2,0x27,0xab,0xc3,0x0e,0x44,0x59,0x39,0x38,0xae,0x0a,0x5a,0xbd,0x5c,0xfd,0x11,0xed,
    0x5e,0xb8,0xd3,0x09,0x9c,0x84,0x80,0x6f,0x38,0xdf,0xd2,0xed,0x12,0x33,0xc9,0x66,0x3e,0x77,0x95,0x40,0xca,0xbb,0x63,0xd8,
    0x44,0x62,0x1d,0x60,0xc1,0x0d,0x92,0x18,0x68,0x4c,0xc7,0x26,0x83,0x5b,0x38,0x45,0xda,0x8d,0xe6,0x11,0xd0,0x08,0x79,0x0c,
    0x13,0xb8,0xe0,0xab,0xf5,0x78,0xe2,0x45,0xfd,0x42,0x7f,0x33,0xab,0x6d,0x53,0x10,0xa3,0x02,0x3c,0xd3,0x6f,0xaf,0x50,0x2f,
    0x20,0xfc,0x92,0xd1,0xab,0x68,0xe8,0x00,0xa0,0x1c,0x4b,0x6f,0x02,0x5a,0xf4,0x1a,0xf1,0x06,0x79,0xa1,0x34,0x8d,0x04,0x5c,
    0x0d,0xfe,0x2d,0x3c,0x53,0xb6,0xae,0x80,0x7d,0x98,0xb9,0x02,0x60,0x15,0x2c,0xb2,0xe5,0xc7,0x9b,0xcf,0x78,0x53,0x37,0xd9,
    0xbf,0x84,0x04,0xb0,0x61,0x1c,0xea,0x24,0x7b,0xf7,0xcd,0x71,0x45,0x1a,0x00,0x22,0x21,0xa9,0x02,0x03,0x01,0x00,0x01,0xa3,
    0x55,0x30,0x53,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x43,0x06,0x03,0x55,0x1d,0x01,
    0x04,0x3c,0x30,0x3a,0x80,0x10,0x88,0x17,0xf7,0x38,0x65,0x8b,0x78,0x78,0xf6,0x77,0xe3,0x25,0x47,0x54,0x33,0x4c,0xa1,0x14,
    0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x03,0x13,0x07,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x82,0x10,0x2b,0x59,
    0xb4,0xc7,0xe2,0xce,0x08,0x97,0x46,0x48,0x32,0x17,0x0f,0x97,0xc5,0x08,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
    0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x9d,0x05,0x0e,0xc5,0xa0,0x5e,0x47,0x18,0x31,0x60,0xf2,0x1b,0x37,0xa4,
    0x89,0xf7,0x05,0x3e,0xea,0xc2,0x00,0x9f,0xcb,0xdd,0x28,0xba,0xc9,0x1f,0xfa,0x7a,0x9b,0x24,0x3d,0xb6,0x47,0x80,0xc1,0xa6,
    0x67,0x4d,0x48,0x3d,0xe0,0x0b,0x32,0x6a,0xa7,0x93,0xf3,0x40,0x20,0x8a,0xff,0x0f,0x9a,0xe2,0x00,0x95,0xa3,0xb3,0x57,0xc7,
    0x11,0xe1,0x28,0xc5,0x63,0x01,0xdf,0x4a,0xd2,0x37,0xb2,0x53,0x09,0x5c,0x4e,0x50,0x4e,0x14,0xb8,0x3e,0xb4,0x52,0xfe,0xa5,
    0x5d,0x14,0x3f,0x07,0x4f,0xda,0x9a,0xb9,0xbe,0x40,0xc5,0x3b,0x90,0x54,0x03,0x2e,0x79,0x0e,0x9b,0xf7,0xa9,0x74,0xeb,0x7c,
    0x6b,0x71,0x12,0xf2,0xce,0x9f,0xc0,0x3e,0x8a,0x09,0xa4,0x91,0x91,0x93,0x64,0x11,0xcc,0x96,0x7b,0xf9,0xac,0x65,0x6b,0xc3,
    0x02,0x1d,0xf8,0x0c,0x82,0x72,0x04,0x19,0x05,0x06,0x33,0x44,0x48,0x4f,0x34,0x13,0x04,0x1e,0x6c,0x11,0xc0,0x7b,0x63,0x32,
    0x1e,0xb3,0x4f,0x79,0xfe,0x9d,0xe6,0x3a,0xbe,0x8e,0xa7,0x5f,0x67,0x1d,0xae,0xad,0x58,0x0e,0x53,0xb8,0x15,0xe3,0x85,0x6e,
    0x91,0xfe,0x2d,0x81,0x84,0xb9,0xc3,0x23,0x13,0xa0,0x3f,0x72,0xb7,0xb3,0x26,0xda,0x08,0xcf,0x10,0x65,0x1e,0xd5,0x3b,0xf4,
    0x8f,0x18,0xe0,0xab,0xe7,0x5e,0xfc,0x62,0x9e,0x7e,0x54,0xf9,0x35,0x5a,0xf8,0xfa,0x1f,0x10,0x6f,0x63,0x3d,0xa2,0xe9,0x8a,
    0xd6,0x49,0xc0,0x40,0x0b,0xa1,0x5e,0x83,0xb0,0x01,0xb6,0x03,0x66,0xa5,0x8a,0xb4,0x29,0x06,0xea,0x27,0x0c,0x28,0x88,0xf3,
    0x38,0x5e,0x30,0x82,0x05,0x8d,0x30,0x82,0x04,0x75,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x0e,0x9b,0x18,0x8e,0xf9,0xd0,0x2d,
    0xe7,0xef,0xdb,0x50,0xe2,0x08,0x40,0x18,0x5a,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0c,0x05,0x00,
    0x30,0x65,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x0a,
    0x13,0x0c,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x49,0x6e,0x63,0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x0b,0x13,
    0x10,0x77,0x77,0x77,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x31,0x24,0x30,0x22,0x06,0x03,0x55,
    0x04,0x03,0x13,0x1b,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x41,0x73,0x73,0x75,0x72,0x65,0x64,0x20,0x49,0x44,0x20,
    0x52,0x6f,0x6f,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x32,0x32,0x30,0x38,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,
    0x17,0x0d,0x33,0x31,0x31,0x31,0x30,0x39,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x62,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,
    0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x0a,0x13,0x0c,0x44,0x69,0x67,0x69,0x43,0x65,0x72,
    0x74,0x20,0x49,0x6e,0x63,0x31,0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x0b,0x13,0x10,0x77,0x77,0x77,0x2e,0x64,0x69,0x67,0x69,
    0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x31,0x21,0x30,0x1f,0x06,0x03,0x55,0x04,0x03,0x13,0x18,0x44,0x69,0x67,0x69,0x43,
    0x65,0x72,0x74,0x20,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x20,0x52,0x6f,0x6f,0x74,0x20,0x47,0x34,0x30,0x82,0x02,0x22,0x30,
    0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,0x02,
    0x82,0x02,0x01,0x00,0xbf,0xe6,0x90,0x73,0x68,0xde,0xbb,0xe4,0x5d,0x4a,0x3c,0x30,0x22,0x30,0x69,0x33,0xec,0xc2,0xa7,0x25,
    0x2e,0xc9,0x21,0x3d,0xf2,0x8a,0xd8,0x59,0xc2,0xe1,0x29,0xa7,0x3d,0x58,0xab,0x76,0x9a,0xcd,0xae,0x7b,0x1b,0x84,0x0d,0xc4,
    0x30,0x1f,0xf3,0x1b,0xa4,0x38,0x16,0xeb,0x56,0xc6,0x97,0x6d,0x1d,0xab,0xb2,0x79,0xf2,0xca,0x11,0xd2,0xe4,0x5f,0xd6,0x05,
    0x3c,0x52,0x0f,0x52,0x1f,0xc6,0x9e,0x15,0xa5,0x7e,0xbe,0x9f,0xa9,0x57,0x16,0x59,0x55,0x72,0xaf,0x68,0x93,0x70,0xc2,0xb2,
    0xba,0x75,0x99,0x6a,0x73,0x32,0x94,0xd1,0x10,0x44,0x10,0x2e,0xdf,0x82,0xf3,0x07,0x84,0xe6,0x74,0x3b,0x6d,0x71,0xe2,0x2d,
    0x0c,0x1b,0xee,0x20,0xd5,0xc9,0x20,0x1d,0x63,0x29,0x2d,0xce,0xec,0x5e,0x4e,0xc8,0x93,0xf8,0x21,0x61,0x9b,0x34,0xeb,0x05,
    0xc6,0x5e,0xec,0x5b,0x1a,0xbc,0xeb,0xc9,0xcf,0xcd,0xac,0x34,0x40,0x5f,0xb1,0x7a,0x66,0xee,0x77,0xc8,0x48,0xa8,0x66,0x57,
    0x57,0x9f,0x54,0x58,0x8e,0x0c,0x2b,0xb7,0x4f,0xa7,0x30,0xd9,0x56,0xee,0xca,0x7b,0x5d,0xe3,0xad,0xc9,0x4f,0x5e,0xe5,0x35,
    0xe7,0x31,0xcb,0xda,0x93,0x5e,0xdc,0x8e,0x8f,0x80,0xda,0xb6,0x91,0x98,0x40,0x90,0x79,0xc3,0x78,0xc7,0xb6,0xb1,0xc4,0xb5,
    0x6a,0x18,0x38,0x03,0x10,0x8d,0xd8,0xd4,0x37,0xa4,0x2e,0x05,0x7d,0x88,0xf5,0x82,0x3e,0x10,0x91,0x70,0xab,0x55,0x82,0x41,
    0x32,0xd7,0xdb,0x04,0x73,0x2a,0x6e,0x91,0x01,0x7c,0x21,0x4c,0xd4,0xbc,0xae,0x1b,0x03,0x75,0x5d,0x78,0x66,0xd9,0x3a,0x31,
    0x44,0x9a,0x33,0x40,0xbf,0x08,0xd7,0x5a,0x49,0xa4,0xc2,0xe6,0xa9,0xa0,0x67,0xdd,0xa4,0x27,0xbc,0xa1,0x4f,0x39,0xb5,0x11,
    0x58,0x17,0xf7,0x24,0x5c,0x46,0x8f,0x64,0xf7,0xc1,0x69,0x88,0x76,0x98,0x76,0x3d,0x59,0x5d,0x42,0x76,0x87,0x89,0x97,0x69,
    0x7a,0x48,0xf0,0xe0,0xa2,0x12,0x1b,0x66,0x9a,0x74,0xca,0xde,0x4b,0x1e,0xe7,0x0e,0x63,0xae,0xe6,0xd4,0xef,0x92,0x92,0x3a,
    0x9e,0x3d,0xdc,0x00,0xe4,0x45,0x25,0x89,0xb6,0x9a,0x44,0x19,0x2b,0x7e,0xc0,0x94,0xb4,0xd2,0x61,0x6d,0xeb,0x33,0xd9,0xc5,
    0xdf,0x4b,0x04,0x00,0xcc,0x7d,0x1c,0x95,0xc3,0x8f,0xf7,0x21,0xb2,0xb2,0x11,0xb7,0xbb,0x7f,0xf2,0xd5,0x8c,0x70,0x2c,0x41,
    0x60,0xaa,0xb1,0x63,0x18,0x44,0x95,0x1a,0x76,0x62,0x7e,0xf6,0x80,0xb0,0xfb,0xe8,0x64,0xa6,0x33,0xd1,0x89,0x07,0xe1,0xbd,
    0xb7,0xe6,0x43,0xa4,0x18,0xb8,0xa6,0x77,0x01,0xe1,0x0f,0x94,0x0c,0x21,0x1d,0xb2,0x54,0x29,0x25,0x89,0x6c,0xe5,0x0e,0x52,
    0x51,0x47,0x74,0xbe,0x26,0xac,0xb6,0x41,0x75,0xde,0x7a,0xac,0x5f,0x8d,0x3f,0xc9,0xbc,0xd3,0x41,0x11,0x12,0x5b,0xe5,0x10,
    0x50,0xeb,0x31,0xc5,0xca,0x72,0x16,0x22,0x09,0xdf,0x7c,0x4c,0x75,0x3f,0x63,0xec,0x21,0x5f,0xc4,0x20,0x51,0x6b,0x6f,0xb1,
    0xab,0x86,0x8b,0x4f,0xc2,0xd6,0x45,0x5f,0x9d,0x20,0xfc,0xa1,0x1e,0xc5,0xc0,0x8f,0xa2,0xb1,0x7e,0x0a,0x26,0x99,0xf5,0xe4,
    0x69,0x2f,0x98,0x1d,0x2d,0xf5,0xd9,0xa9,0xb2,0x1d,0xe5,0x1b,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x3a,0x30,0x82,0x01,
    0x36,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,0x55,0x1d,
    0x0e,0x04,0x16,0x04,0x14,0xec,0xd7,0xe3,0x82,0xd2,0x71,0x5d,0x64,0x4c,0xdf,0x2e,0x67,0x3f,0xe7,0xba,0x98,0xae,0x1c,0x0f,
    0x4f,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x45,0xeb,0xa2,0xaf,0xf4,0x92,0xcb,0x82,0x31,0x2d,
    0x51,0x8b,0xa7,0xa7,0x21,0x9d,0xf3,0x6d,0xc8,0x0f,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,
    0x01,0x86,0x30,0x79,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x6d,0x30,0x6b,0x30,0x24,0x06,0x08,0x2b,0x06,
    0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x18,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x64,0x69,0x67,0x69,
    0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x30,0x43,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x37,0x68,0x74,
    0x74,0x70,0x3a,0x2f,0x2f,0x63,0x61,0x63,0x65,0x72,0x74,0x73,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,
    0x6d,0x2f,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x41,0x73,0x73,0x75,0x72,0x65,0x64,0x49,0x44,0x52,0x6f,0x6f,0x74,0x43,
    0x41,0x2e,0x63,0x72,0x74,0x30,0x45,0x06,0x03,0x55,0x1d,0x1f,0x04,0x3e,0x30,0x3c,0x30,0x3a,0xa0,0x38,0xa0,0x36,0x86,0x34,
    0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x33,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,
    0x2f,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x41,0x73,0x73,0x75,0x72,0x65,0x64,0x49,0x44,0x52,0x6f,0x6f,0x74,0x43,0x41,
    0x2e,0x63,0x72,0x6c,0x30,0x11,0x06,0x03,0x55,0x1d,0x20,0x04,0x0a,0x30,0x08,0x30,0x06,0x06,0x04,0x55,0x1d,0x20,0x00,0x30,
    0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0c,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x70,0xa0,0xbf,0x43,0x5c,
    0x55,0xe7,0x38,0x5f,0xa0,0xa3,0x74,0x1b,0x3d,0xb6,0x16,0xd7,0xf7,0xbf,0x57,0x07,0xbd,0x9a,0xac,0xa1,0x87,0x2c,0xec,0x85,
    0x5e,0xa9,0x1a,0xbb,0x22,0xf8,0x87,0x1a,0x69,0x54,0x22,0xed,0xa4,0x88,0x77,0x6d,0xbd,0x1a,0x14,0xf4,0x13,0x4a,0x7a,0x2f,
    0x2d,0xb7,0x38,0xef,0xf4,0xff,0x80,0xb9,0xf8,0xa1,0xf7,0xf2,0x72,0xde,0x24,0xbc,0x52,0x03,0xc8,0x4e,0xd0,0x2a,0xde,0xfa,
    0x2d,0x56,0xcf,0xf9,0xf4,0xf7,0xac,0x30,0x7a,0x9a,0x8b,0xb2,0x5e,0xd4,0xcf,0xd1,0x43,0x44,0x9b,0x43,0x21,0xeb,0x96,0x72,
    0xa1,0x48,0xb4,0x99,0xcb,0x9d,0x4f,0xa7,0x06,0x03,0x13,0x77,0x27,0x44,0xd4,0xe7,0x7f,0xe8,0x59,0xa8,0xf0,0xbf,0x2f,0x0b,
    0xa6,0xe9,0xf2,0x34,0x3c,0xec,0xf7,0x03,0xc7,0x87,0xa8,0xd2,0x4c,0x40,0x19,0x35,0x46,0x6a,0x69,0x54,0xb0,0xb8,0xa1,0x56,
    0x8e,0xec,0xa4,0xd5,0x3d,0xe8,0xb1,0xdc,0xfd,0x1c,0xd8,0xf4,0x77,0x5a,0x5c,0x54,0x8c,0x6f,0xef,0xa1,0x50,0x3d,0xfc,0x76,
    0x09,0x68,0x84,0x9f,0x6f,0xca,0xdb,0x20,0x8d,0x35,0x60,0x1c,0x02,0x03,0xcb,0x20,0xb0,0xac,0x58,0xa0,0x0e,0x40,0x63,0xc5,
    0x98,0x22,0xc1,0xb2,0x59,0xf5,0x55,0x6b,0xcf,0x27,0xab,0x6c,0x76,0xce,0x6f,0x23,0x2d,0xf4,0x7e,0x71,0x6a,0x23,0x6b,0x22,
    0xff,0x12,0xb8,0x54,0x2d,0x27,0x7e,0xd8,0x3a,0xd9,0xf0,0xb6,0x87,0x96,0xfd,0x5b,0xd1,0x5c,0xac,0x18,0xc3,0x4d,0x9f,0x73,
    0xb7,0x01,0xa9,0x9f,0x57,0xaa,0x5e,0x28,0xe2,0xb9,0x94,0x30,0x82,0x06,0xae,0x30,0x82,0x04,0x96,0xa0,0x03,0x02,0x01,0x02,
    0x02,0x10,0x07,0x36,0x37,0xb7,0x24,0x54,0x7c,0xd8,0x47,0xac,0xfd,0x28,0x66,0x2a,0x5e,0x5b,0x30,0x0d,0x06,0x09,0x2a,0x86,
    0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x62,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,
    0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x0a,0x13,0x0c,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x49,0x6e,0x63,0x31,
    0x19,0x30,0x17,0x06,0x03,0x55,0x04,0x0b,0x13,0x10,0x77,0x77,0x77,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,
    0x6f,0x6d,0x31,0x21,0x30,0x1f,0x06,0x03,0x55,0x04,0x03,0x13,0x18,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x54,0x72,
    0x75,0x73,0x74,0x65,0x64,0x20,0x52,0x6f,0x6f,0x74,0x20,0x47,0x34,0x30,0x1e,0x17,0x0d,0x32,0x32,0x30,0x33,0x32,0x33,0x30,
    0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x33,0x37,0x30,0x33,0x32,0x32,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x63,0x31,
    0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x44,
    0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x3b,0x30,0x39,0x06,0x03,0x55,0x04,0x03,0x13,0x32,
    0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x20,0x47,0x34,0x20,0x52,0x53,0x41,0x34,
    0x30,0x39,0x36,0x20,0x53,0x48,0x41,0x32,0x35,0x36,0x20,0x54,0x69,0x6d,0x65,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x20,
    0x43,0x41,0x30,0x82,0x02,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x02,
    0x0f,0x00,0x30,0x82,0x02,0x0a,0x02,0x82,0x02,0x01,0x00,0xc6,0x86,0x35,0x06,0x49,0xb3,0xc1,0x3d,0x72,0x49,0x51,0x55,0xc7,
    0x25,0x03,0xc4,0xf2,0x91,0x37,0xa9,0x97,0x51,0xa1,0xd6,0xd2,0x83,0xd1,0x9e,0x4c,0xa2,0x6d,0xa0,0xb0,0xcc,0x83,0xf9,0x5a,
    0xf6,0x11,0xa1,0x44,0x15,0x42,0x5f,0xa4,0x88,0xf3,0x68,0xfa,0x7d,0xf3,0x9c,0x89,0x0b,0x7f,0x9d,0x1f,0x9e,0x0f,0x33,0x1f,
    0x50,0x13,0x0b,0x26,0x73,0x96,0x6d,0xf8,0x57,0xa8,0x02,0x7d,0xfd,0x43,0xb4,0x84,0xda,0x11,0xf1,0x73,0xb1,0xb3,0xee,0x2b,
    0x80,0x84,0x8a,0x22,0x18,0xdf,0xeb,0xda,0x3d,0xc4,0x17,0x7f,0xab,0x19,0x2b,0x3e,0x42,0xdc,0x67,0x8e,0xea,0x51,0x3d,0xf0,
    0xd6,0x56,0xd4,0xe7,0x28,0x2d,0xeb,0xd3,0xb1,0xb5,0x75,0xe7,0x1f,0x06,0x65,0x8d,0x94,0x29,0xd3,0xd9,0xec,0x69,0xdf,0xd9,
    0x90,0x87,0x46,0x00,0x7b,0xdb,0x44,0x41,0x89,0xdc,0x7c,0x6a,0x57,0x7a,0xf0,0x37,0x79,0x9f,0x5d,0xac,0xcb,0xe8,0x84,0x64,
    0xb4,0x52,0xf2,0x76,0x47,0xf7,0x61,0x83,0x19,0xdd,0x5f,0xb4,0x54,0x0b,0x21,0x68,0x6e,0x37,0x21,0xbb,0x40,0xac,0x5f,0xb2,
    0xde,0x4a,0x7d,0xce,0xf5,0x39,0x12,0x67,0xef,0x0e,0xa5,0x63,0x6c,0xe4,0xa6,0xc5,0x1d,0xcd,0x36,0x0d,0x5c,0xd5,0xe6,0x1b,
    0xa8,0xc1,0x64,0x74,0x40,0xa7,0xc0,0x72,0xc5,0xba,0x4e,0x1f,0xb1,0xb5,0x58,0x4d,0x79,0xfe,0xd7,0x8f,0x73,0x93,0xac,0x2c,
    0x39,0xe2,0xa5,0x48,0xd6,0xf0,0xb0,0x31,0x13,0xa9,0x57,0x29,0x96,0x27,0x2e,0xf5,0x87,0xa6,0x8f,0x4e,0x76,0x15,0x55,0x26,
    0x70,0x98,0x26,0x7f,0xa0,0x1a,0x47,0x20,0x43,0xe3,0x43,0x63,0x80,0x7b,0x75,0x6e,0x27,0x25,0x90,0x98,0x3a,0x38,0x11,0xb3,
    0xf6,0xf6,0x9e,0xe6,0x3b,0x5b,0xec,0x81,0xde,0x22,0x14,0xd9,0x82,0x2a,0xc7,0x92,0xbf,0xa0,0xde,0xe3,0x3e,0xa2,0x73,0xfa,
    0xe7,0x1f,0x5a,0x6c,0x94,0xf2,0x52,0x95,0x11,0x2b,0x58,0x74,0x40,0x28,0xab,0x73,0x43,0xce,0xdf,0x4a,0xa1,0x1c,0x6b,0x38,
    0xc5,0x29,0xf3,0xca,0xaa,0x96,0x73,0x42,0x68,0x9f,0xb6,0x46,0xb3,0x9d,0x3a,0xa3,0xd5,0x03,0xe0,0xbf,0xf0,0xa2,0x3c,0xca,
    0x42,0xdc,0x18,0x48,0x7f,0x14,0x34,0xcf,0xd2,0x4c,0xab,0xef,0x9b,0x3d,0xfe,0x0e,0xb8,0x64,0x2a,0xfa,0x75,0x28,0x24,0x41,
    0xed,0x42,0xbf,0x05,0x9c,0x66,0x49,0x52,0x50,0xf4,0x51,0xf3,0x36,0x49,0x4d,0x8b,0x20,0xd2,0x2c,0x57,0x35,0x79,0x2b,0xa8,
    0xf3,0x45,0x60,0xbc,0x23,0x8d,0x58,0xf7,0xdc,0x61,0xde,0x93,0xfe,0x39,0xc0,0xf9,0xb2,0x30,0xa5,0x4c,0xd7,0xe9,0x98,0x4a,
    0x58,0x3e,0xd3,0x03,0x88,0xfe,0xb3,0x8f,0xd3,0x5e,0x4b,0x76,0x12,0x51,0x93,0xc9,0x8c,0x0c,0x3b,0x5b,0x8a,0x22,0xa8,0xc1,
    0x26,0x08,0xf9,0x14,0x10,0x12,0x03,0x7d,0x5f,0x23,0xbb,0x64,0xe3,0x63,0xe0,0xa6,0xe1,0x3e,0xf6,0xc2,0x74,0xb2,0x3f,0x1e,
    0x09,0x76,0xec,0xab,0x5d,0x46,0x75,0xe2,0x60,0xa3,0x58,0x09,0x01,0x28,0x00,0x0e,0x84,0x54,0xee,0xce,0xe9,0x5d,0xc8,0x5e,
    0x30,0x12,0xbd,0x46,0x9e,0xb5,0xd3,0x76,0xb9,0xd2,0x0e,0x6b,0x99,0x0c,0xd2,0x33,0xb4,0xcd,0xb1,0x02,0x03,0x01,0x00,0x01,
    0xa3,0x82,0x01,0x5d,0x30,0x82,0x01,0x59,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,
    0xff,0x02,0x01,0x00,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0xba,0x16,0xd9,0x6d,0x4d,0x85,0x2f,0x73,0x29,
    0x76,0x9a,0x2f,0x75,0x8c,0x6a,0x20,0x8f,0x9e,0xc8,0x6f,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,
    0xec,0xd7,0xe3,0x82,0xd2,0x71,0x5d,0x64,0x4c,0xdf,0x2e,0x67,0x3f,0xe7,0xba,0x98,0xae,0x1c,0x0f,0x4f,0x30,0x0e,0x06,0x03,
    0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x01,0x86,0x30,0x13,0x06,0x03,0x55,0x1d,0x25,0x04,0x0c,0x30,0x0a,0x06,
    0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x08,0x30,0x77,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x6b,0x30,
    0x69,0x30,0x24,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x18,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,
    0x73,0x70,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x30,0x41,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,
    0x07,0x30,0x02,0x86,0x35,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x61,0x63,0x65,0x72,0x74,0x73,0x2e,0x64,0x69,0x67,0x69,
    0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x54,0x72,0x75,0x73,0x74,0x65,0x64,
    0x52,0x6f,0x6f,0x74,0x47,0x34,0x2e,0x63,0x72,0x74,0x30,0x43,0x06,0x03,0x55,0x1d,0x1f,0x04,0x3c,0x30,0x3a,0x30,0x38,0xa0,
    0x36,0xa0,0x34,0x86,0x32,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x33,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,
    0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x52,0x6f,0x6f,
    0x74,0x47,0x34,0x2e,0x63,0x72,0x6c,0x30,0x20,0x06,0x03,0x55,0x1d,0x20,0x04,0x19,0x30,0x17,0x30,0x08,0x06,0x06,0x67,0x81,
    0x0c,0x01,0x04,0x02,0x30,0x0b,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xfd,0x6c,0x07,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
    0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x02,0x01,0x00,0x7d,0x59,0x8e,0xc0,0x93,0xb6,0x6f,0x98,0xa9,0x44,0x22,
    0x01,0x7e,0x66,0xd6,0xd8,0x21,0x42,0xe1,0xb0,0x18,0x2e,0x10,0x4d,0x13,0xcf,0x30,0x53,0xce,0xbf,0x18,0xfb,0xc7,0x50,0x5d,
    0xe2,0x4b,0x29,0xfb,0x70,0x8a,0x0d,0xaa,0x29,0x69,0xfc,0x69,0xc1,0xcf,0x1d,0x07,0xe9,0x3e,0x60,0xc8,0xd8,0x0b,0xe5,0x5c,
    0x5b,0xd7,0x6d,0x87,0xfa,0x84,0x20,0x25,0x34,0x31,0x67,0xcd,0xb6,0x12,0x96,0x6f,0xc4,0x50,0x4c,0x62,0x1d,0x0c,0x08,0x82,
    0xa8,0x16,0xbd,0xa9,0x56,0xcf,0x15,0x73,0x8d,0x01,0x22,0x25,0xce,0x95,0x69,0x3f,0x47,0x77,0xfb,0x72,0x74,0x14,0xd7,0xff,
    0xab,0x4f,0x8a,0x2c,0x7a,0xab,0x85,0xcd,0x43,0x5f,0xed,0x60,0xb6,0xaa,0x4f,0x91,0x66,0x9e,0x2c,0x9e,0xe0,0x8a,0xac,0xe5,
    0xfd,0x8c,0xbc,0x64,0x26,0x87,0x6c,0x92,0xbd,0x9d,0x7c,0xd0,0x70,0x0a,0x7c,0xef,0xa8,0xbc,0x75,0x4f,0xba,0x5a,0xf7,0xa9,
    0x10,0xb2,0x5d,0xe9,0xff,0x28,0x54,0x89,0xf0,0xd5,0x8a,0x71,0x76,0x65,0xda,0xcc,0xf0,0x72,0xa3,0x23,0xfa,0xc0,0x27,0x82,
    0x44,0xae,0x99,0x27,0x1b,0xab,0x24,0x1e,0x26,0xc1,0xb7,0xde,0x2a,0xeb,0xf6,0x9e,0xb1,0x79,0x99,0x81,0xa3,0x56,0x86,0xab,
    0x0a,0x45,0xc9,0xdf,0xc4,0x8d,0xa0,0xe7,0x98,0xfb,0xfb,0xa6,0x9d,0x72,0xaf,0xc4,0xc7,0xc1,0xc1,0x6a,0x71,0xd9,0xc6,0x13,
    0x80,0x09,0xc4,0xb6,0x9f,0xcd,0x87,0x87,0x24,0xbb,0x4f,0xa3,0x49,0xb9,0x77,0x66,0x91,0xf1,0x72,0x9c,0xe9,0x4b,0x02,0x52,
    0xa7,0x37,0x7e,0x93,0x53,0xac,0x3b,0x1d,0x08,0x49,0x0f,0x94,0xcd,0x39,0x7a,0xdd,0xff,0x25,0x63,0x99,0x27,0x2c,0x3d,0x3f,
    0x6b,0xa7,0xf1,0x66,0xc3,0x41,0xcd,0x4f,0xb6,0x40,0x9b,0x21,0x21,0x40,0xd0,0xb7,0x13,0x24,0xcd,0xdc,0x1d,0x78,0x3a,0xe4,
    0x9e,0xad,0xe5,0x34,0x71,0x92,0xd7,0x26,0x6b,0xe4,0x38,0x73,0xab,0xa6,0x01,0x4f,0xbd,0x3f,0x3b,0x78,0xad,0x4c,0xad,0xfb,
    0xc4,0x95,0x7b,0xed,0x0a,0x5f,0x33,0x39,0x87,0x41,0x78,0x7a,0x38,0xe9,0x9c,0xe1,0xdd,0x23,0xfd,0x1d,0x28,0xd3,0xc7,0xf9,
    0xe8,0xf1,0x98,0x5f,0xfb,0x2b,0xd8,0x7e,0xf2,0x46,0x9d,0x75,0x2c,0x1e,0x27,0x2c,0x26,0xdb,0x6f,0x15,0x7b,0x1e,0x19,0x8b,
    0x36,0xb8,0x93,0xd4,0xe6,0xf2,0x17,0x99,0x59,0xca,0x70,0xf0,0x37,0xbf,0x98,0x00,0xdf,0x20,0x16,0x4f,0x27,0xfb,0x60,0x67,
    0x16,0xa1,0x66,0xba,0xdd,0x55,0xc0,0x3a,0x29,0x86,0xb0,0x98,0xa0,0x2b,0xed,0x95,0x41,0xb7,0x3a,0xd5,0x15,0x98,0x31,0xb4,
    0x62,0x09,0x0f,0x0a,0xbd,0x81,0xd9,0x13,0xfe,0xbf,0xa4,0xd1,0xf3,0x57,0xd9,0xbc,0x04,0xfa,0x82,0xde,0x32,0xdf,0x04,0x89,
    0xf0,0x00,0xcd,0x5d,0xc2,0xf9,0xd0,0x23,0x7f,0x00,0x0b,0xe4,0x76,0x02,0x26,0xd9,0xf0,0x65,0x76,0x42,0xa6,0x29,0x87,0x09,
    0x47,0x2b,0xe6,0x7f,0x1a,0xa4,0x85,0x0f,0xfc,0x98,0x96,0xf6,0x55,0x54,0x2b,0x1f,0x80,0xfa,0xc0,0xf2,0x0e,0x2b,0xe5,0xd6,
    0xfb,0xa9,0x2f,0x44,0x15,0x4a,0xe7,0x13,0x0e,0x1d,0xdb,0x37,0x38,0x1a,0xa1,0x2b,0xf6,0xed,0xd6,0x7c,0xfc,0x30,0x82,0x06,
    0xc0,0x30,0x82,0x04,0xa8,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x0c,0x4d,0x69,0x72,0x4b,0x94,0xfa,0x3c,0x2a,0x4a,0x3d,0x29,
    0x07,0x80,0x3d,0x5a,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x63,0x31,0x0b,0x30,
    0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x44,0x69,0x67,
    0x69,0x43,0x65,0x72,0x74,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x3b,0x30,0x39,0x06,0x03,0x55,0x04,0x03,0x13,0x32,0x44,0x69,
    0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x20,0x47,0x34,0x20,0x52,0x53,0x41,0x34,0x30,0x39,
    0x36,0x20,0x53,0x48,0x41,0x32,0x35,0x36,0x20,0x54,0x69,0x6d,0x65,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x20,0x43,0x41,
    0x30,0x1e,0x17,0x0d,0x32,0x32,0x30,0x39,0x32,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x33,0x33,0x31,0x31,0x32,
    0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x46,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,
    0x11,0x30,0x0f,0x06,0x03,0x55,0x04,0x0a,0x13,0x08,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x31,0x24,0x30,0x22,0x06,0x03,
    0x55,0x04,0x03,0x13,0x1b,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,0x20,0x54,0x69,0x6d,0x65,0x73,0x74,0x61,0x6d,0x70,0x20,
    0x32,0x30,0x32,0x32,0x20,0x2d,0x20,0x32,0x30,0x82,0x02,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
    0x01,0x05,0x00,0x03,0x82,0x02,0x0f,0x00,0x30,0x82,0x02,0x0a,0x02,0x82,0x02,0x01,0x00,0xcf,0xec,0xa5,0x26,0x3a,0xc6,0xa9,
    0xf2,0x6b,0xbb,0x8d,0xc1,0x0d,0x9a,0xdb,0xa1,0xe8,0x14,0x85,0x74,0x33,0x1a,0x26,0xac,0xd0,0x1c,0x55,0x1e,0x1e,0x36,0x6d,
    0xbc,0x92,0x55,0x0c,0x61,0xf4,0x9d,0x09,0x77,0x3d,0x15,0x96,0x08,0x2f,0x6b,0x64,0xa4,0xfd,0x06,0x83,0x16,0xd7,0x91,0x92,
    0x38,0x1c,0x31,0x02,0x96,0xfb,0x72,0xb1,0x97,0x3a,0x55,0xaf,0x33,0xec,0x61,0x8a,0xe9,0xa6,0x28,0xdb,0x90,0x63,0x5c,0xbd,
    0x89,0x53,0xe0,0x3a,0x2d,0x8c,0x87,0x42,0xae,0x26,0xa4,0xe4,0xbb,0x78,0x78,0xb9,0x7a,0x16,0xe1,0x56,0xc6,0xc0,0xba,0x64,
    0x53,0xbb,0x2a,0x16,0xe7,0x50,0x48,0xbb,0x88,0x69,0x0c,0x88,0xc6,0xf1,0xbe,0xe0,0x2f,0x7d,0x3b,0xb1,0xca,0x53,0x8d,0x40,
    0x83,0x1e,0xe7,0xcb,0x72,0x49,0x28,0x1e,0x4c,0x80,0x1e,0x85,0x56,0xe7,0x85,0xed,0xf2,0x61,0xbc,0xaa,0x3a,0x07,0x7d,0xf6,
    0xab,0x6e,0xe5,0x66,0xdd,0xe2,0x5c,0xf5,0x2f,0xed,0x8d,0xd4,0x4d,0x95,0x84,0x68,0xe3,0x80,0xcb,0x6a,0x79,0xd1,0xd2,0x10,
    0x91,0x46,0x29,0xeb,0x3e,0x26,0xf2,0xb4,0x8c,0xcd,0x4c,0xb9,0x66,0xc8,0xbb,0xaa,0x50,0x38,0x0d,0xe5,0x8c,0x94,0x5d,0x19,
    0x5a,0xbf,0xf5,0x7b,0x40,0x6e,0x6f,0x16,0xa8,0x9a,0x9c,0x95,0x47,0x86,0x85,0x79,0x3e,0x0c,0x5e,0x66,0x8c,0x1a,0x0a,0x24,
    0xbe,0x9c,0xaa,0xd2,0x9c,0xb6,0xf7,0x4f,0x6e,0x78,0xc4,0x28,0x3f,0xa3,0x1c,0x0f,0x50,0x06,0x37,0xba,0x08,0xd9,0x35,0xa6,
    0xb5,0x1e,0xda,0x78,0x58,0x1d,0x39,0xe8,0xf8,0x4c,0x91,0x10,0x96,0x7e,0x4d,0xe1,0xdd,0xc2,0xad,0xa5,0x7e,0xf8,0x2d,0x1b,
    0x1f,0xec,0x2b,0x46,0x18,0xa3,0x19,0xf6,0x39,0xf7,0xf5,0xc1,0x4f,0x71,0x2e,0x89,0x03,0x11,0xa2,0x4b,0xbb,0x98,0xbf,0xfa,
    0x4f,0xe4,0x7b,0x36,0xef,0x06,0x44,0xe4,0x55,0xff,0x36,0xea,0xe5,0x7c,0x31,0xe7,0xf3,0xc2,0x52,0xc4,0xe6,0x16,0x7b,0x5a,
    0x7e,0xa5,0x25,0x73,0xdb,0xc0,0x6a,0x99,0x21,0x2d,0x63,0xe5,0x59,0xf5,0x4d,0x2f,0x90,0x1f,0x27,0xb7,0xd2,0xab,0x14,0xe5,
    0x38,0x66,0x87,0x51,0x08,0x6b,0xfb,0x53,0x43,0x39,0xd0,0x64,0xfa,0x56,0xcf,0xe0,0xf4,0x0a,0xe6,0x14,0x6d,0x64,0x78,0xbb,
    0x98,0xfd,0x94,0xc3,0x73,0x21,0xf3,0x2f,0xc2,0x2e,0x20,0xd7,0x81,0xac,0xd3,0xf1,0x07,0xd4,0xe1,0xbd,0xd9,0x5d,0x4b,0x6e,
    0x31,0x94,0x29,0x8b,0xe6,0x41,0xa4,0x65,0x94,0xc0,0x58,0xe5,0xe5,0x2e,0x29,0x90,0xa6,0xb7,0x61,0x64,0xfa,0xd9,0x20,0x6c,
    0x18,0x51,0x60,0xba,0xa6,0x81,0x0f,0x09,0x25,0x53,0xf1,0xbf,0x3b,0xe9,0xab,0x07,0x0e,0x6a,0x07,0x39,0x62,0x19,0xc9,0xd6,
    0x85,0x7f,0x13,0xd9,0x8d,0x79,0xcf,0x62,0xc5,0xec,0xe1,0x7b,0xb9,0xcc,0x67,0x13,0x07,0x9a,0xc1,0x78,0xed,0xc6,0x88,0xc8,
    0xb0,0x6e,0x32,0x79,0xc7,0x0b,0x59,0x83,0x8d,0xc6,0xee,0xf5,0x2c,0x7c,0x7b,0x8e,0xcb,0x64,0x89,0xf1,0xb1,0xc4,0xb8,0xe7,
    0x53,0x5e,0x5f,0x55,0xd2,0x7d,0x19,0x29,0x59,0x03,0x4e,0xfa,0x5d,0xea,0x45,0x73,0x1c,0x84,0x7e,0xd7,0xce,0xe2,0xd4,0x3a,
    0x77,0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x8b,0x30,0x82,0x01,0x87,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,
    0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x16,0x06,0x03,
    0x55,0x1d,0x25,0x01,0x01,0xff,0x04,0x0c,0x30,0x0a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x08,0x30,0x20,0x06,0x03,
    0x55,0x1d,0x20,0x04,0x19,0x30,0x17,0x30,0x08,0x06,0x06,0x67,0x81,0x0c,0x01,0x04,0x02,0x30,0x0b,0x06,0x09,0x60,0x86,0x48,
    0x01,0x86,0xfd,0x6c,0x07,0x01,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0xba,0x16,0xd9,0x6d,0x4d,
    0x85,0x2f,0x73,0x29,0x76,0x9a,0x2f,0x75,0x8c,0x6a,0x20,0x8f,0x9e,0xc8,0x6f,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,
    0x04,0x14,0x62,0x8a,0xde,0xd0,0x61,0xfc,0x8f,0x31,0x14,0xed,0x97,0x0b,0xcd,0x3d,0x2a,0x94,0x14,0xdf,0x52,0x9c,0x30,0x5a,
    0x06,0x03,0x55,0x1d,0x1f,0x04,0x53,0x30,0x51,0x30,0x4f,0xa0,0x4d,0xa0,0x4b,0x86,0x49,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,
    0x63,0x72,0x6c,0x33,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x44,0x69,0x67,0x69,0x43,0x65,
    0x72,0x74,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x47,0x34,0x52,0x53,0x41,0x34,0x30,0x39,0x36,0x53,0x48,0x41,0x32,0x35,0x36,
    0x54,0x69,0x6d,0x65,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x43,0x41,0x2e,0x63,0x72,0x6c,0x30,0x81,0x90,0x06,0x08,0x2b,
    0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x81,0x83,0x30,0x81,0x80,0x30,0x24,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,
    0x01,0x86,0x18,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,
    0x63,0x6f,0x6d,0x30,0x58,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x4c,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,
    0x63,0x61,0x63,0x65,0x72,0x74,0x73,0x2e,0x64,0x69,0x67,0x69,0x63,0x65,0x72,0x74,0x2e,0x63,0x6f,0x6d,0x2f,0x44,0x69,0x67,
    0x69,0x43,0x65,0x72,0x74,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x47,0x34,0x52,0x53,0x41,0x34,0x30,0x39,0x36,0x53,0x48,0x41,
    0x32,0x35,0x36,0x54,0x69,0x6d,0x65,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x43,0x41,0x2e,0x63,0x72,0x74,0x30,0x0d,0x06,
    0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x02,0x01,0x00,0x55,0xaa,0x2a,0x1a,0xf3,0x46,0xf3,
    0x78,0x57,0x37,0x30,0xfc,0x75,0xe3,0x4f,0xd6,0x85,0x23,0xf1,0xfc,0xf9,0x95,0x39,0x9b,0x25,0xe6,0xf7,0x72,0x8a,0x98,0xc3,
    0x77,0xd4,0x64,0xfc,0x15,0xfb,0x36,0xc2,0x49,0x51,0x2c,0x78,0x88,0x63,0x55,0x09,0x46,0x39,0x00,0xfc,0x69,0xd4,0xca,0x9b,
    0x29,0xfb,0xa3,0x3f,0xc0,0xc9,0x00,0x9b,0x13,0x1d,0xb0,0x98,0x89,0xdc,0x78,0xf2,0xcd,0x7c,0x85,0xcd,0x53,0x9d,0xaf,0x62,
    0xe2,0x61,0x66,0xa3,0x14,0x2a,0x45,0x87,0x4a,0x98,0x42,0x2b,0x50,0xfc,0x1b,0xb5,0x9e,0x08,0x30,0x09,0xfa,0xe4,0x2d,0xd7,
    0x09,0x89,0x79,0xf9,0x09,0xe6,0x88,0xce,0x7d,0x1b,0xb8,0x6a,0xa2,0x9b,0xc1,0x53,0x60,0x09,0xe8,0xa3,0xb8,0x9d,0xd7,0xad,
    0x1f,0x1c,0xb8,0xec,0x98,0x41,0xf0,0xf6,0x0e,0x80,0xfb,0xe4,0xff,0xdf,0x9d,0x10,0xa7,0xeb,0x00,0xba,0x5f,0x4a,0x8f,0x1a,
    0x3a,0x52,0xb4,0xea,0xbf,0x09,0x49,0x15,0x35,0x36,0x59,0x9a,0x0f,0x54,0xd2,0xb2,0x1b,0x7f,0x7e,0x5e,0x09,0xad,0x76,0x54,
    0x8a,0x74,0x6d,0xca,0xd2,0x05,0x67,0x2b,0x76,0xeb,0xff,0x98,0xb2,0x26,0x95,0x38,0x19,0x88,0x44,0x14,0xe5,0x0a,0x59,0xa2,
    0x6b,0xe7,0x22,0x3e,0x44,0x21,0xd2,0x3f,0x1c,0xc0,0x9b,0xed,0x7c,0x48,0xb2,0xd8,0x92,0x0c,0x91,0x4f,0x3c,0x66,0x94,0xaf,
    0x5d,0x02,0x53,0xeb,0x9e,0xe2,0x9e,0xe4,0xd3,0x1f,0x86,0x01,0x64,0x9c,0x00,0xc2,0xe9,0x5a,0x74,0x75,0x0d,0x3d,0xe1,0x79,
    0x88,0xbf,0x1c,0x01,0x97,0xc9,0x19,0x23,0x80,0xd7,0x36,0x5a,0x5f,0x96,0x16,0xb1,0x63,0x0c,0xc6,0x46,0x40,0x3b,0xce,0x5d,
    0x35,0xd4,0x59,0x3e,0x43,0x9a,0x18,0xae,0xc3,0xc9,0xcb,0xc3,0xfb,0x9b,0x13,0x5f,0x6a,0xb5,0xc7,0xe0,0xf3,0x05,0xc3,0x59,
    0xdf,0x27,0x62,0x2b,0xde,0x41,0xc9,0x53,0xb9,0xff,0x34,0x10,0x67,0xf6,0x26,0x32,0x98,0x7b,0xfe,0x5c,0x42,0x94,0x81,0x94,
    0x82,0x9d,0xac,0x0a,0x8b,0xc6,0x4b,0x15,0x4a,0xd3,0x98,0x90,0x45,0x60,0x33,0x80,0xe0,0x23,0xde,0xf8,0x03,0xa4,0xf6,0x45,
    0x47,0xe5,0xce,0xb8,0x03,0x42,0x47,0xe8,0x41,0x36,0x71,0x77,0xad,0xfd,0xa2,0xe8,0x97,0x74,0x4e,0x2e,0xda,0x1e,0x1d,0x8c,
    0x5a,0xc8,0x1e,0x9a,0xd5,0xc2,0xf0,0xc6,0x22,0xa8,0x4f,0x9b,0xbd,0xd8,0x1c,0x9a,0x51,0xc4,0x2f,0x9a,0xf6,0x5f,0xa7,0x27,
    0x97,0xba,0x96,0x2e,0x85,0x57,0xc0,0x60,0xe7,0x78,0x56,0x7f,0x6a,0xef,0xc2,0x95,0x9a,0x4b,0x11,0x02,0xc8,0x82,0x9c,0xc9,
    0x1a,0x05,0x7c,0xba,0x71,0xb5,0x4e,0x7a,0x99,0x6c,0xf4,0xe8,0x9e,0xd4,0x5a,0x98,0xc8,0x9f,0xbf,0x8d,0xbb,0x18,0x5c,0x43,
    0xf5,0xd0,0x2a,0xe8,0xe2,0x62,0xee,0x78,0x04,0xdb,0xbd,0xd1,0xfb,0x5b,0x0a,0xa8,0x70,0x7e,0xf0,0x97,0x84,0x78,0xe3,0x08,
    0x03,0x5d,0x47,0x2c,0x63,0xa8,0x25,0x38,0x97,0x01,0xd2,0x3f,0x3a,0xda,0xe5,0xe5,0xf6,0xe6,0x9b,0xdc,0x7e,0x2c,0xcc,0xff,
    0x17,0x4c,0x4d,0x00,0xa2,0xd8,0xd6,0x01,0x0e,0xb8,0x8b,0xee,0xe6,0xe0,0x72,0x55,0x89,0x2c,0x27,0x19,0x61,0xf6,0x77,0x01,
    0x8c,0x31,0x82,0x0f,0x9d,0x30,0x82,0x0f,0x99,0x02,0x01,0x01,0x30,0x26,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,
    0x03,0x13,0x07,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x02,0x10,0xd1,0x73,0x97,0xaa,0xa7,0x3a,0x31,0xa2,0x44,0xc0,0x4b,0x40,
    0x69,0x40,0x4b,0xfa,0x30,0x0d,0x06,0x09,0x60,0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0xa0,0x5e,0x30,0x10,0x06,
    0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0c,0x31,0x02,0x30,0x00,0x30,0x19,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
    0x0d,0x01,0x09,0x03,0x31,0x0c,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0x30,0x2f,0x06,0x09,0x2a,0x86,
    0x48,0x86,0xf7,0x0d,0x01,0x09,0x04,0x31,0x22,0x04,0x20,0xcb,0xa1,0x2f,0x7b,0x0f,0x6a,0x14,0x06,0x26,0x77,0x7b,0xee,0xd1,
    0x90,0x34,0x1c,0xa7,0xcd,0x94,0x65,0xe0,0xe4,0x24,0x12,0x0c,0x4e,0xa2,0x89,0xeb,0x2d,0xe5,0xbb,0x30,0x0d,0x06,0x09,0x2a,
    0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0x00,0x73,0xb7,0xdf,0xb4,0xd2,0x60,0x31,0x2d,0x14,0x63,
    0x71,0x87,0x5b,0xd9,0xee,0x66,0x85,0x88,0xbe,0x1e,0xc3,0x5e,0x30,0x25,0x62,0x05,0x66,0x6e,0x9b,0x51,0xaa,0x5b,0xee,0xed,
    0xe4,0x4c,0x83,0x19,0x7e,0x7b,0x78,0xe0,0x04,0xb5,0xdb,0xb3,0x48,0xba,0xa3,0xa4,0x54,0xb2,0x3e,0xe4,0x99,0xf1,0xb2,0x6e,
    0xb7,0x7a,0x44,0x76,0x11,0xb4,0x51,0xae,0xe3,0xec,0xc2,0x43,0x65,0xf5,0x95,0x1c,0x57,0x9e,0x4d,0x49,0x4d,0x4c,0xee,0x09,
    0xdd,0xdb,0xab,0xf3,0x14,0x89,0xec,0x6e,0x94,0xd2,0xac,0xd8,0xed,0xe3,0xbb,0x8a,0xf0,0x05,0x9d,0x1d,0xbe,0x8b,0x7f,0x34,
    0x63,0xe7,0x87,0x04,0x25,0x5a,0xff,0xc9,0xca,0xa9,0xab,0xd5,0xf2,0x16,0xf6,0x26,0x5d,0xf9,0xf9,0xbf,0xb4,0xce,0x86,0x55,
    0xd4,0x95,0x91,0x1e,0x12,0x25,0x36,0x43,0x37,0x6e,0x93,0x14,0xf0,0x86,0xad,0xc6,0x7d,0x07,0x86,0xe5,0x18,0x4f,0x3d,0xe3,
    0x92,0x67,0x7b,0x74,0xbf,0xa9,0x71,0x5a,0x49,0xcf,0xf5,0x60,0xf4,0x09,0x65,0x38,0xfd,0x13,0xe2,0x03,0x8a,0x84,0x17,0xc7,
    0x83,0x7a,0xd5,0x42,0x95,0x47,0xd8,0x9d,0x76,0x52,0xb5,0xbc,0x11,0x63,0x78,0x53,0x66,0x0e,0x95,0xc8,0xd4,0xfa,0x6e,0x3d,
    0x3b,0x1b,0x56,0xf2,0x98,0xc3,0x4c,0xc8,0xdc,0x1f,0x7e,0xa3,0x8b,0x1a,0x3b,0xa6,0x4d,0xed,0x70,0x75,0x55,0xcd,0x0c,0xf9,
    0x4e,0xfb,0xa6,0x84,0xcb,0xbc,0xf5,0x28,0x89,0x2e,0x30,0x07,0x8f,0x1d,0x8d,0x10,0x03,0x20,0xe9,0xb0,0x56,0x53,0x9d,0xf9,
    0x43,0xfb,0xba,0xb7,0x25,0x5d,0xa1,0x82,0x0d,0xe8,0x30,0x82,0x03,0x1c,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,
    0x06,0x31,0x82,0x03,0x0d,0x30,0x82,0x03,0x09,0x02,0x01,0x01,0x30,0x77,0x30,0x63,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,
    0x06,0x13,0x02,0x55,0x53,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x44,0x69,0x67,0x69,0x43,0x65,0x72,0x74,
    0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x3b,0x30,0x39,0x06,0x03,0x55,0x04,0x03,0x13,0x32,0x44,0x69,0x67,0x69,0x43,0x65,0x72,
    0x74,0x20,0x54,0x72,0x75,0x73,0x74,0x65,0x64,0x20,0x47,0x34,0x20,0x52,0x53,0x41,0x34,0x30,0x39,0x36,0x20,0x53,0x48,0x41,
    0x32,0x35,0x36,0x20,0x54,0x69,0x6d,0x65,0x53,0x74,0x61,0x6d,0x70,0x69,0x6e,0x67,0x20,0x43,0x41,0x02,0x10,0x0c,0x4d,0x69,
    0x72,0x4b,0x94,0xfa,0x3c,0x2a,0x4a,0x3d,0x29,0x07,0x80,0x3d,0x5a,0x30,0x0d,0x06,0x09,0x60,0x86,0x48,0x01,0x65,0x03,0x04,
    0x02,0x01,0x05,0x00,0xa0,0x69,0x30,0x18,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x03,0x31,0x0b,0x06,0x09,0x2a,
    0x86,0x48,0x86,0xf7,0x0d,0x01,0x07,0x01,0x30,0x1c,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x05,0x31,0x0f,0x17,
    0x0d,0x32,0x32,0x30,0x39,0x33,0x30,0x31,0x37,0x35,0x33,0x34,0x33,0x5a,0x30,0x2f,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
    0x01,0x09,0x04,0x31,0x22,0x04,0x20,0xbf,0x5c,0x03,0xc6,0x50,0xb0,0xde,0xd1,0x96,0x6d,0x74,0x64,0xa4,0xda,0x0f,0x51,0x71,
    0x0f,0x5a,0x87,0x97,0x78,0x2e,0x17,0x99,0xc6,0xa2,0x7b,0xa7,0x9b,0x75,0xeb,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
    0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x02,0x00,0x0e,0xb2,0xbf,0xf0,0xf2,0x35,0x55,0xc7,0x9d,0x49,0xce,0x0a,0xac,0x3a,
    0x6a,0xd8,0x85,0xa0,0x2b,0x81,0x97,0x85,0xc2,0x16,0x4d,0x6d,0x77,0x72,0xdf,0x2e,0x0a,0x6b,0x2e,0xbd,0xd6,0xb4,0x66,0x22,
    0x7b,0x43,0x1b,0x7e,0x7a,0x1b,0xdb,0x72,0xfb,0x1d,0xd5,0xe9,0x84,0x01,0xe1,0x64,0x15,0x05,0x0d,0xb5,0x85,0x1d,0x93,0xf0,
    0xcf,0x72,0x77,0x07,0x30,0x82,0xa9,0x6e,0x9c,0x5d,0xc9,0x39,0xda,0x19,0x9b,0xca,0x34,0x05,0xf0,0xe4,0xd7,0x02,0xbe,0x8a,
    0x5f,0x74,0x39,0xe6,0xe9,0xb4,0xdf,0x00,0x4a,0xeb,0xb4,0x0d,0xf6,0xb2,0x5b,0x7f,0x10,0xb1,0xef,0x05,0x53,0xfc,0x74,0x41,
    0x5b,0x83,0xeb,0xf9,0x37,0x9f,0x03,0xc2,0x1a,0x98,0x13,0xc9,0x6c,0x1c,0xd8,0x82,0xd8,0xd7,0xbf,0x13,0x07,0x20,0xf9,0xf2,
    0xea,0x96,0x46,0xb7,0x2d,0x51,0x2f,0xc3,0x40,0x12,0x3f,0x35,0xeb,0x88,0xfa,0x3c,0x66,0x30,0xd9,0x16,0xe2,0x4b,0x05,0x14,
    0x11,0x07,0x9a,0x64,0xc5,0x92,0x01,0xec,0xdb,0x5d,0x01,0x91,0x08,0x4f,0x1e,0xec,0x12,0xdd,0x7d,0xa0,0x32,0x5a,0x84,0x2a,
    0x7c,0xb9,0x25,0x34,0x57,0x12,0x0c,0x86,0x2d,0xcd,0x01,0x26,0x85,0x17,0x8c,0x7d,0x07,0x3f,0x2c,0x08,0xb6,0xac,0xc3,0xca,
    0x72,0x17,0xfe,0x4c,0xef,0xbb,0x69,0x20,0x11,0x33,0x63,0xf8,0xca,0x0a,0xe1,0x8c,0xf5,0x31,0x4e,0x83,0x87,0xfb,0xa9,0xd2,
    0x29,0x4b,0x82,0x84,0x27,0xfa,0x01,0x31,0x48,0xc1,0x25,0xe5,0x51,0x49,0xf5,0x52,0x60,0xdb,0x64,0x08,0x36,0xd2,0x2f,0x4e,
    0x7a,0x5d,0x92,0xee,0x06,0xec,0xe4,0x3b,0x5d,0xdc,0xc3,0x49,0xa1,0x8e,0x96,0xa6,0x17,0x00,0x2d,0x0d,0xee,0x6e,0x88,0x16,
    0x1e,0xea,0x9c,0x9b,0x55,0x9d,0xea,0x4a,0xa5,0xbb,0x54,0x33,0xaa,0x72,0x8c,0x5f,0xb7,0xe6,0x22,0xe9,0x1d,0xca,0xbc,0x6d,
    0xdf,0x7b,0x68,0xb5,0x71,0x3b,0x9f,0xfd,0x2b,0x53,0x35,0xa6,0xbb,0x78,0xf2,0x5a,0x41,0x69,0x52,0x5c,0x47,0xdc,0xf0,0x3e,
    0x06,0xef,0x60,0xb6,0xd3,0xb1,0x30,0xda,0x41,0xee,0x5a,0x2e,0x8e,0x24,0x0e,0x1e,0xcd,0x6c,0xa6,0x2e,0x9b,0x3c,0x77,0x02,
    0x21,0x86,0x16,0x68,0xfc,0xd6,0x64,0x84,0x46,0x1e,0xd1,0xe6,0x86,0x6f,0xee,0x5f,0x5c,0xca,0xb6,0xc9,0x9b,0xb8,0x03,0x46,
    0x1b,0x67,0x52,0x48,0x48,0x1c,0x07,0xad,0xbd,0x3c,0x2f,0x1b,0xd1,0xe3,0xb5,0xba,0x21,0x20,0x3b,0x6d,0xfb,0x30,0xa6,0x6d,
    0x2d,0xd7,0xb0,0x67,0xa4,0x19,0x0f,0x38,0xff,0xb0,0xad,0xf0,0xc3,0xb3,0xd7,0x7c,0x92,0xa0,0xe8,0x9c,0x22,0x60,0x0f,0x88,
    0x08,0xa7,0xf0,0xfa,0x90,0x45,0x2c,0x26,0xa6,0x88,0x25,0x24,0x7b,0x69,0x23,0x0b,0x20,0x04,0x89,0xd4,0x66,0x89,0x7d,0xb6,
    0x50,0x0e,0xb3,0xfe,0xf9,0xd0,0x91,0x2b,0x1d,0x17,0x62,0x6d,0xad,0x8e,0xf9,0x52,0x63,0x8f,0xe9,0x93,0xe6,0xea,0xb8,0xe6,
    0xce,0x0c,0x69,0xc9,0x47,0xd7,0x7e,0xb4,0x0d,0x49,0xbe,0xcb,0x05,0xe4,0xac,0x0b,0xd4,0xa1,0x6b,0x66,0x0f,0xff,0xe7,0x01,
    0x5a,0x91,0x05,0xa8,0xc0,0x2b,0xda,0xe9,0x5c,0x18,0xfb,0xe3,0x4d,0x60,0x0f,0x34,0x19,0x68,0x30,0x82,0x0a,0xc4,0x06,0x0a,
    0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x04,0x01,0x31,0x82,0x0a,0xb4,0x30,0x82,0x05,0x56,0x06,0x09,0x2a,0x86,0x48,0x86,
    0xf7,0x0d,0x01,0x07,0x02,0xa0,0x82,0x05,0x47,0x30,0x82,0x05,0x43,0x02,0x01,0x01,0x31,0x0f,0x30,0x0d,0x06,0x09,0x60,0x86,
    0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0x30,0x5c,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0xa0,
    0x4e,0x30,0x4c,0x30,0x17,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0f,0x30,0x09,0x03,0x01,0x00,0xa0,0x04,
    0xa2,0x02,0x80,0x00,0x30,0x31,0x30,0x0d,0x06,0x09,0x60,0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0x04,0x20,0xdd,
    0x8b,0xd7,0x29,0x3b,0xae,0x16,0xec,0xbb,0x81,0x80,0x55,0x15,0xd8,0x87,0xa5,0x3e,0xeb,0x0b,0x74,0x59,0xb6,0x56,0xf1,0x0b,
    0x2e,0xe1,0xb4,0x42,0x4d,0x8b,0x18,0xa0,0x82,0x03,0x05,0x30,0x82,0x03,0x01,0x30,0x82,0x01,0xe9,0xa0,0x03,0x02,0x01,0x02,
    0x02,0x10,0x8d,0x01,0xec,0xa9,0x68,0x41,0x93,0x8f,0x40,0x42,0x93,0x4a,0x72,0x6b,0x03,0xcc,0x30,0x0d,0x06,0x09,0x2a,0x86,
    0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x03,0x13,0x07,0x54,0x65,
    0x73,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x32,0x32,0x30,0x39,0x33,0x30,0x31,0x37,0x32,0x34,0x31,0x39,0x5a,0x17,0x0d,
    0x33,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,
    0x13,0x05,0x63,0x65,0x72,0x74,0x33,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,
    0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xf4,0x21,0x90,0xec,0x4a,0x19,0xca,0xa7,
    0x1f,0x25,0x80,0x0b,0x92,0x21,0x2d,0x53,0xb4,0x25,0xa3,0x37,0x40,0xae,0xce,0xf1,0x46,0xf6,0xda,0x21,0xd3,0x54,0x8e,0x02,
    0x95,0xfe,0x37,0x3a,0xff,0xb9,0x36,0x88,0x70,0xf2,0x33,0x0c,0x5c,0x60,0x36,0xbf,0x8d,0x51,0xef,0x18,0x5d,0x47,0x68,0xbc,
    0xda,0xd8,0xf8,0x94,0xaf,0xcf,0xa6,0x48,0x61,0x78,0x04,0xcd,0x73,0x03,0xa0,0x6c,0xdb,0x24,0x94,0x20,0x54,0xbb,0x19,0xf2,
    0xb6,0xf9,0x0d,0xf4,0x31,0xbd,0x79,0xb5,0x98,0x12,0xc4,0x62,0x06,0xfb,0x16,0x08,0x51,0xa2,0xbf,0x12,0xcf,0x88,0xf4,0x4c,
    0x26,0xec,0x80,0xc7,0xa7,0xeb,0x89,0x24,0x6a,0xe4,0x1e,0x3a,0x1d,0x4a,0x7c,0x42,0xaa,0x03,0x5e,0x47,0xde,0x21,0xc4,0x06,
    0x86,0x57,0x0d,0xce,0xe0,0x27,0xf2,0x43,0x56,0x2a,0x60,0x1b,0x18,0xf5,0x03,0x91,0xa4,0xda,0x86,0x1c,0xcb,0x2c,0x2d,0x21,
    0x54,0xc5,0xd8,0xe2,0xe2,0x02,0x19,0xb0,0x73,0xe0,0xf8,0x95,0x5c,0x4f,0x1d,0x8f,0xd2,0xe0,0x4a,0x7c,0x20,0x9e,0x29,0xec,
    0x05,0x34,0xb0,0x4c,0x1a,0xaf,0x2a,0x01,0xf1,0x0f,0x26,0xdc,0x38,0x68,0x85,0xbd,0xc0,0x73,0xec,0x10,0x17,0x3a,0xd8,0x6d,
    0x33,0x3b,0xab,0xd3,0x79,0x61,0x2c,0xac,0x11,0xe1,0x09,0x5e,0xe0,0x7b,0xd3,0xc0,0xb8,0xf8,0xb5,0x9a,0x20,0x68,0xa7,0xf0,
    0x66,0x9b,0xc2,0xe8,0x83,0x14,0x7d,0x3c,0x7f,0x7a,0x85,0x95,0xc9,0x05,0x74,0xf4,0xee,0xa9,0x83,0x6a,0x55,0xa5,0x78,0xa8,
    0xa8,0xb4,0x6d,0xb1,0xab,0x49,0x50,0xd1,0x02,0x03,0x01,0x00,0x01,0xa3,0x55,0x30,0x53,0x30,0x0c,0x06,0x03,0x55,0x1d,0x13,
    0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x43,0x06,0x03,0x55,0x1d,0x01,0x04,0x3c,0x30,0x3a,0x80,0x10,0x88,0x17,0xf7,0x38,
    0x65,0x8b,0x78,0x78,0xf6,0x77,0xe3,0x25,0x47,0x54,0x33,0x4c,0xa1,0x14,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,
    0x03,0x13,0x07,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x82,0x10,0x2b,0x59,0xb4,0xc7,0xe2,0xce,0x08,0x97,0x46,0x48,0x32,0x17,
    0x0f,0x97,0xc5,0x08,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x01,0x01,0x00,
    0x8e,0xe0,0xc9,0x58,0x24,0x1b,0x16,0x6a,0x8a,0x4f,0x67,0xac,0xd7,0x75,0x62,0x53,0x94,0xfc,0xeb,0x8b,0x36,0x2a,0x9f,0x9c,
    0x8b,0x8f,0x60,0x42,0xdd,0x37,0x13,0x10,0x5f,0x5a,0x52,0xc8,0xee,0x51,0x92,0x18,0xaf,0x84,0x18,0x5f,0x27,0x69,0xf4,0xde,
    0x22,0x4b,0x9c,0xaa,0x18,0x9e,0xde,0x04,0xc0,0xc4,0xfd,0x74,0x08,0x25,0x43,0xbf,0x00,0x1d,0xc2,0xd6,0xb2,0x4e,0xa4,0x4a,
    0x73,0xa1,0xff,0x71,0x3d,0xa5,0xf1,0x21,0xcf,0x4d,0xb4,0x5c,0x55,0x54,0x6f,0x94,0x50,0x21,0xbb,0x85,0xcb,0x54,0xeb,0x07,
    0xaf,0x74,0x62,0x21,0xf5,0x89,0x43,0xcb,0x10,0x62,0xd6,0xbe,0xc0,0x3a,0xb3,0x6b,0x9f,0x80,0xde,0xe0,0xc0,0x6e,0x8a,0x0a,
    0xe7,0x1f,0x08,0x9b,0x89,0x38,0xc2,0x30,0xfa,0xd9,0xc2,0x8c,0xf7,0xbd,0xbd,0xd4,0x6b,0x99,0xbd,0x5f,0x0e,0xb1,0x76,0xd6,
    0x5b,0x1f,0x1a,0xd7,0x27,0x5d,0x5b,0x19,0x1c,0x6d,0x5a,0x91,0x81,0x06,0x83,0x82,0x6d,0xaf,0x48,0x70,0x72,0x8b,0x7c,0x8e,
    0x57,0xcd,0x35,0x5d,0x7d,0x96,0xe5,0x2d,0x31,0xd2,0xa9,0xf8,0xad,0x9d,0x13,0xb4,0x89,0x75,0x7e,0xbc,0x39,0x11,0x27,0x9d,
    0xc3,0x7b,0xf1,0x40,0x2a,0x23,0xed,0x50,0xee,0x10,0x1b,0x97,0x13,0x71,0x1c,0x5a,0x7f,0x06,0xab,0x9b,0x51,0x2b,0x1f,0x85,
    0x27,0x99,0x11,0x18,0x62,0x97,0xcb,0x08,0x24,0x6d,0xf5,0x08,0x56,0xb5,0x36,0x00,0xf3,0x6b,0x3d,0xa9,0x13,0xd3,0xcf,0x0c,
    0x94,0x6f,0x9e,0x05,0x62,0xeb,0x9c,0x52,0x9d,0x0c,0x15,0x81,0x4d,0x7d,0xe3,0x2f,0x31,0x82,0x01,0xc4,0x30,0x82,0x01,0xc0,
    0x02,0x01,0x01,0x30,0x26,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x03,0x13,0x07,0x54,0x65,0x73,0x74,0x20,0x43,
    0x41,0x02,0x10,0x8d,0x01,0xec,0xa9,0x68,0x41,0x93,0x8f,0x40,0x42,0x93,0x4a,0x72,0x6b,0x03,0xcc,0x30,0x0d,0x06,0x09,0x60,
    0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0xa0,0x71,0x30,0x10,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,
    0x01,0x0c,0x31,0x02,0x30,0x00,0x30,0x11,0x06,0x0a,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x19,0x04,0x31,0x03,0x02,0x01,
    0x02,0x30,0x19,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x03,0x31,0x0c,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,
    0x37,0x02,0x01,0x04,0x30,0x2f,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x04,0x31,0x22,0x04,0x20,0xcb,0xa1,0x2f,
    0x7b,0x0f,0x6a,0x14,0x06,0x26,0x77,0x7b,0xee,0xd1,0x90,0x34,0x1c,0xa7,0xcd,0x94,0x65,0xe0,0xe4,0x24,0x12,0x0c,0x4e,0xa2,
    0x89,0xeb,0x2d,0xe5,0xbb,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0x00,
    0x68,0xfc,0x34,0xac,0x24,0xe1,0x93,0x00,0xec,0x8d,0x29,0xe1,0x32,0x03,0xe6,0x8a,0xd6,0x39,0x79,0x07,0x85,0x97,0xf4,0xb8,
    0x0c,0xf1,0x24,0xa0,0x16,0x09,0x0c,0xf8,0x02,0x40,0x94,0xdb,0x19,0x7a,0x6f,0x91,0xee,0x24,0x36,0x77,0x32,0x02,0xb5,0xd1,
    0xf9,0xe0,0xa2,0xdd,0xe0,0xe1,0x14,0x30,0x11,0xe1,0x25,0x02,0x0a,0x7c,0x10,0xdb,0xbd,0xb0,0x4e,0xbb,0x36,0xef,0x87,0xf0,
    0x17,0x49,0x77,0x45,0xa0,0x9e,0x47,0x70,0x1a,0xe2,0x87,0x39,0x41,0x24,0x1c,0xe0,0x09,0xb0,0xe0,0xfa,0xc5,0xf3,0xba,0xba,
    0x03,0x65,0x64,0xf9,0xa8,0x7d,0xe5,0x0e,0x84,0xc8,0xd1,0xe2,0xf5,0x44,0xa4,0x6f,0x33,0xac,0xbb,0x15,0x3b,0x0a,0x1a,0x04,
    0x6e,0xc2,0x54,0xa7,0x78,0x77,0x7d,0x32,0x21,0x4d,0x0c,0x3f,0x7b,0x0a,0x61,0x18,0x58,0xdb,0x59,0x02,0x3f,0xcf,0xb2,0xd0,
    0x5c,0xa5,0xea,0x96,0xd4,0x5c,0xd2,0x09,0xd3,0x18,0x61,0x73,0x6e,0x9f,0xdf,0xcb,0x17,0x4f,0xd1,0xc0,0xa2,0x2d,0x8b,0xf5,
    0x46,0xdf,0xf8,0xb8,0x4f,0x47,0x98,0xf4,0x44,0xa6,0xa1,0x5b,0xcb,0xfa,0xc1,0x31,0x4e,0xc4,0x03,0xea,0x06,0x1b,0x9b,0x94,
    0xa6,0xc8,0x1c,0x7a,0x69,0x3b,0x8d,0x8d,0x83,0x20,0x56,0x18,0xf1,0xe0,0xd2,0xfb,0xbc,0xaf,0xf7,0xdc,0x17,0x3b,0xcd,0xac,
    0x2b,0x07,0x86,0xc6,0x7f,0x25,0xc3,0xa2,0x6c,0x7c,0x49,0xa9,0xc1,0xe2,0x5e,0x40,0x05,0xfb,0x2f,0xab,0xd5,0x98,0x3a,0x69,
    0xbb,0x83,0x1c,0xbd,0xde,0x55,0xc0,0x74,0x71,0x8d,0xdb,0xc7,0x95,0xf4,0xf5,0xca,0x30,0x82,0x05,0x56,0x06,0x09,0x2a,0x86,
    0x48,0x86,0xf7,0x0d,0x01,0x07,0x02,0xa0,0x82,0x05,0x47,0x30,0x82,0x05,0x43,0x02,0x01,0x01,0x31,0x0f,0x30,0x0d,0x06,0x09,
    0x60,0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0x30,0x5c,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,
    0x04,0xa0,0x4e,0x30,0x4c,0x30,0x17,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0f,0x30,0x09,0x03,0x01,0x00,
    0xa0,0x04,0xa2,0x02,0x80,0x00,0x30,0x31,0x30,0x0d,0x06,0x09,0x60,0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0x04,
    0x20,0xdd,0x8b,0xd7,0x29,0x3b,0xae,0x16,0xec,0xbb,0x81,0x80,0x55,0x15,0xd8,0x87,0xa5,0x3e,0xeb,0x0b,0x74,0x59,0xb6,0x56,
    0xf1,0x0b,0x2e,0xe1,0xb4,0x42,0x4d,0x8b,0x18,0xa0,0x82,0x03,0x05,0x30,0x82,0x03,0x01,0x30,0x82,0x01,0xe9,0xa0,0x03,0x02,
    0x01,0x02,0x02,0x10,0xae,0xfb,0x3e,0x08,0x15,0xa4,0xe3,0xa7,0x4d,0x91,0x6a,0x85,0x68,0x5b,0x58,0xa1,0x30,0x0d,0x06,0x09,
    0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x03,0x13,0x07,
    0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x32,0x32,0x30,0x39,0x33,0x30,0x31,0x37,0x32,0x33,0x30,0x33,0x5a,
    0x17,0x0d,0x33,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,
    0x04,0x03,0x13,0x05,0x63,0x65,0x72,0x74,0x32,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
    0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xca,0x01,0x2f,0x98,0x69,0x71,
    0x64,0x63,0x56,0x38,0xd7,0xc4,0xad,0x64,0x02,0x34,0xd4,0x31,0xc2,0x27,0x5d,0xd7,0x8f,0x72,0xd0,0x70,0x95,0xd9,0x75,0x65,
    0x2e,0x8c,0x5b,0x76,0xcd,0x54,0x3f,0xd9,0x0a,0xcc,0x3f,0x03,0x8f,0x74,0x2b,0x8c,0x3d,0x3d,0x4c,0xd3,0xaa,0x3c,0x97,0xf1,
    0x44,0x46,0x57,0x92,0xa9,0xdd,0xd9,0xf0,0xc7,0x8b,0x39,0xf5,0x8d,0x28,0x41,0x18,0xaf,0xca,0x99,0xd1,0xf1,0xe4,0xab,0x93,
    0x0a,0xb6,0xd4,0xad,0x2b,0x9f,0x60,0x27,0x4c,0xf2,0xc9,0x14,0xde,0xf2,0xc6,0xbe,0x82,0x14,0x83,0x65,0x13,0x9f,0x9c,0x8d,
    0xfa,0xac,0x95,0x12,0x00,0xd0,0xa4,0x36,0x4d,0xf0,0x8f,0xfc,0x1a,0x43,0x47,0xc3,0xff,0xce,0x1b,0x24,0xd6,0xcf,0x63,0xd1,
    0x41,0x23,0xb8,0x62,0x5f,0x31,0x4e,0x30,0x3f,0x63,0x64,0xff,0x72,0xb5,0x9d,0xe5,0xaa,0x22,0xbc,0x1d,0xb3,0x23,0xc9,0x16,
    0x49,0x10,0xed,0x51,0x02,0xd2,0x90,0xc6,0x86,0x47,0x40,0x7e,0xf1,0xcf,0xc1,0x17,0xa0,0x72,0xaf,0x40,0xb1,0x23,0x3d,0x5a,
    0xa1,0xf9,0xed,0xc8,0xb6,0x66,0xa7,0x94,0x39,0x09,0x03,0x6d,0x16,0x4e,0xc4,0x2a,0x4b,0x1f,0x5b,0x22,0x39,0xf7,0x60,0x1c,
    0x71,0x65,0x4c,0x11,0x29,0x59,0x96,0x5e,0x9e,0xfe,0xaf,0x23,0xd1,0xe3,0x2c,0xce,0xd2,0x31,0x8c,0x80,0x29,0x6c,0x82,0x99,
    0xe8,0x68,0xbd,0x7e,0x66,0xaa,0x35,0x0c,0xae,0x61,0xde,0x59,0x7d,0x5b,0x16,0x09,0x07,0x52,0x6a,0x14,0x26,0x3c,0x48,0x3e,
    0x03,0xdb,0xd4,0x8a,0xea,0x0e,0x46,0x1a,0x24,0xbd,0x02,0x03,0x01,0x00,0x01,0xa3,0x55,0x30,0x53,0x30,0x0c,0x06,0x03,0x55,
    0x1d,0x13,0x01,0x01,0xff,0x04,0x02,0x30,0x00,0x30,0x43,0x06,0x03,0x55,0x1d,0x01,0x04,0x3c,0x30,0x3a,0x80,0x10,0x88,0x17,
    0xf7,0x38,0x65,0x8b,0x78,0x78,0xf6,0x77,0xe3,0x25,0x47,0x54,0x33,0x4c,0xa1,0x14,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,
    0x55,0x04,0x03,0x13,0x07,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x82,0x10,0x2b,0x59,0xb4,0xc7,0xe2,0xce,0x08,0x97,0x46,0x48,
    0x32,0x17,0x0f,0x97,0xc5,0x08,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x01,
    0x01,0x00,0xd1,0x0d,0xdd,0x83,0x0c,0xd8,0xf9,0x0c,0x71,0xe3,0x2f,0x7c,0xc9,0xd7,0x8e,0x33,0x27,0xb6,0x6b,0x34,0x3c,0x41,
    0xf0,0x13,0x03,0xd6,0x5a,0xe2,0x55,0x12,0x42,0x06,0x20,0x03,0xb1,0x74,0xc7,0xc0,0x08,0x00,0x21,0xbe,0x90,0xe7,0xfd,0xac,
    0xe0,0x67,0x42,0xe7,0x53,0x86,0xcf,0x53,0x55,0x40,0xf1,0xbc,0xfc,0x87,0xab,0x67,0xb6,0x09,0xe1,0xf1,0xa2,0xce,0xf6,0xbf,
    0xe6,0x1d,0x43,0x4f,0x41,0xf0,0xf5,0xc0,0xfa,0xc5,0xd2,0x14,0x2d,0xd9,0x23,0x8e,0x9c,0xeb,0x68,0xff,0x3c,0x5f,0x18,0xca,
    0x4b,0x09,0xad,0xcd,0xbd,0x23,0x62,0x33,0x4e,0x02,0x10,0xf9,0xe3,0x68,0x6f,0x22,0xb0,0x86,0x0b,0x5a,0xbe,0xd3,0xee,0x8a,
    0x0b,0x4c,0x92,0x9e,0x06,0x31,0x1f,0x95,0x4f,0xbf,0x27,0x7f,0x1f,0xcd,0xcc,0x9c,0x70,0xa1,0x51,0x07,0x7a,0x09,0x36,0x3f,
    0x0a,0x2f,0x16,0x77,0x26,0x9b,0xb4,0xc9,0x1e,0x86,0xe3,0xb3,0xb7,0xc3,0xcc,0xf1,0x44,0x6e,0x2e,0xf4,0xc9,0x5b,0x23,0x08,
    0x0a,0xc0,0xdb,0xc1,0x1a,0x37,0xb3,0xb1,0x91,0xce,0x24,0x26,0x56,0x7f,0x26,0x37,0x88,0xa0,0x02,0x37,0x6e,0x9c,0xca,0xc1,
    0x8c,0x19,0x99,0xca,0x6c,0x9a,0x98,0x75,0x89,0xfc,0x6d,0x92,0xfc,0xb5,0x12,0x5b,0x29,0xb1,0x88,0x68,0x3b,0xef,0xf0,0xc0,
    0x8f,0x82,0x5e,0x33,0xf9,0x67,0x6b,0xe8,0x60,0x1b,0x14,0xec,0x9c,0xdf,0x21,0x38,0xbb,0x0d,0x3f,0xd9,0xbc,0xd2,0x01,0x2a,
    0x92,0x0c,0xc2,0x97,0x2e,0x12,0x22,0x54,0x76,0xeb,0x80,0x51,0x99,0x9d,0x0f,0x26,0x12,0xb7,0x31,0x82,0x01,0xc4,0x30,0x82,
    0x01,0xc0,0x02,0x01,0x01,0x30,0x26,0x30,0x12,0x31,0x10,0x30,0x0e,0x06,0x03,0x55,0x04,0x03,0x13,0x07,0x54,0x65,0x73,0x74,
    0x20,0x43,0x41,0x02,0x10,0xae,0xfb,0x3e,0x08,0x15,0xa4,0xe3,0xa7,0x4d,0x91,0x6a,0x85,0x68,0x5b,0x58,0xa1,0x30,0x0d,0x06,
    0x09,0x60,0x86,0x48,0x01,0x65,0x03,0x04,0x02,0x01,0x05,0x00,0xa0,0x71,0x30,0x10,0x06,0x0a,0x2b,0x06,0x01,0x04,0x01,0x82,
    0x37,0x02,0x01,0x0c,0x31,0x02,0x30,0x00,0x30,0x11,0x06,0x0a,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x19,0x04,0x31,0x03,
    0x02,0x01,0x01,0x30,0x19,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x03,0x31,0x0c,0x06,0x0a,0x2b,0x06,0x01,0x04,
    0x01,0x82,0x37,0x02,0x01,0x04,0x30,0x2f,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x09,0x04,0x31,0x22,0x04,0x20,0xcb,
    0xa1,0x2f,0x7b,0x0f,0x6a,0x14,0x06,0x26,0x77,0x7b,0xee,0xd1,0x90,0x34,0x1c,0xa7,0xcd,0x94,0x65,0xe0,0xe4,0x24,0x12,0x0c,
    0x4e,0xa2,0x89,0xeb,0x2d,0xe5,0xbb,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,
    0x01,0x00,0x26,0x76,0xb3,0xf2,0xc9,0xb1,0x73,0x13,0xb5,0xd2,0xc5,0xb7,0x01,0x5c,0xc6,0x94,0x38,0x9f,0xc7,0x57,0x56,0x95,
    0xb0,0xf4,0x6d,0xc2,0xd4,0x6a,0xf1,0x4d,0x09,0xa1,0x51,0xa6,0x91,0xf0,0x0e,0x84,0xc0,0x2c,0x74,0xa3,0x97,0x1f,0x41,0xe0,
    0x4a,0xfa,0x1a,0x78,0xa9,0xd5,0x3c,0x85,0x29,0x2b,0xaf,0xbb,0xc3,0x61,0x0d,0x50,0x20,0x20,0xf5,0x80,0x0d,0x6a,0x15,0x4b,
    0x38,0x6c,0x55,0xd9,0xf9,0xd0,0x44,0x22,0x46,0x98,0xe6,0x07,0xd4,0xba,0x3d,0x9d,0x50,0xa7,0x8e,0x1f,0xa8,0x82,0x25,0x7e,
    0x39,0xda,0xe1,0x49,0xc7,0x24,0x3f,0x31,0xfb,0x4b,0xba,0x75,0xdb,0x10,0x0a,0xbe,0xc5,0xad,0x3e,0x30,0x16,0x9b,0x15,0xbb,
    0xc0,0x59,0xf2,0xf5,0x4f,0xf5,0x56,0xc6,0x28,0xd0,0x1e,0x7d,0x8f,0x2e,0x2b,0xb6,0x76,0x94,0x52,0x87,0x99,0xa3,0x66,0x3d,
    0x94,0x0d,0x73,0xb0,0xd5,0xd4,0x76,0x5b,0x69,0x95,0x0a,0x16,0x4f,0x5c,0xf4,0x95,0x5b,0x42,0x45,0x04,0x5c,0x53,0xb7,0x1a,
    0x61,0x6c,0x82,0xdc,0x95,0x94,0x38,0x64,0x34,0x01,0x98,0x2e,0xf8,0xcf,0xf8,0x66,0xae,0xba,0xf8,0x70,0x9e,0x9e,0xde,0xa2,
    0x7f,0x56,0x8d,0xd9,0x6a,0x7b,0x41,0x02,0x46,0x05,0x5c,0xba,0xed,0x43,0x98,0x56,0x39,0x52,0xc0,0x0b,0x3c,0xe1,0x7d,0x1b,
    0xf5,0xac,0x03,0x5b,0xbb,0x7a,0x65,0x80,0x4b,0xcb,0xb7,0x51,0xa7,0x19,0x8a,0x38,0x75,0x76,0x75,0xc2,0x1f,0x12,0xc4,0x68,
    0x96,0xe0,0x89,0x9f,0x37,0x3d,0xab,0xfd,0x6b,0x03,0xb3,0xa1,0x51,0xf8,0x69,0x17,0xea,0xff,0x00,0x00,0x00,0x00,0x00,0x00,
};

static void call_winverify(WCHAR *pathW, LONG *status, BOOL hash_only)
{
    static GUID WVTPolicyGUID = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    WINTRUST_FILE_INFO file_info = { sizeof(file_info), 0 };
    WINTRUST_DATA data = { sizeof(data), 0 };
    LONG ret;

    file_info.pcwszFilePath     = pathW;

    data.dwUIChoice             = WTD_UI_NONE;
    data.fdwRevocationChecks    = WTD_REVOKE_NONE;
    data.dwUnionChoice          = WTD_CHOICE_FILE;
    data.pFile                  = &file_info;
    data.dwStateAction          = WTD_STATEACTION_VERIFY;
    data.dwProvFlags            = hash_only ? WTD_HASH_ONLY_FLAG : 0;
    *status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);

    data.dwStateAction = WTD_STATEACTION_CLOSE;
    ret = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    ok(ret == S_OK, "WinVerifyTrust failed: %08lx\n", ret);
}

static void test_wintrust_digest(void)
{
    static const BYTE Dummy[] = { 0x11,0x22,0x33,0x44 };
    static const struct
    {
        struct { const BYTE *data; DWORD length; } blocks[5];
        struct { LONG status; BOOL todo; } t1;
        struct { LONG status; BOOL todo; } t2;
    }
    tests[] =
    {
        /* 32-bit tests */
        {
            {{ SelfSignedFile32, sizeof(SelfSignedFile32) }},
            { CERT_E_CHAINING, TRUE }, { S_OK, FALSE }
        },
        {
            {{ SelfSignedFile32, sizeof(SelfSignedFile32) },
             { Dummy, sizeof(Dummy) }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ Dummy, sizeof(Dummy) },
             { SelfSignedFile32 + sizeof(Dummy), sizeof(SelfSignedFile32) - sizeof(Dummy) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile32, 19 },
             { Dummy, sizeof(Dummy) },
             { SelfSignedFile32 + 19 + sizeof(Dummy), sizeof(SelfSignedFile32) - 19 - sizeof(Dummy) }},
            { TRUST_E_BAD_DIGEST, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile32, sizeof(IMAGE_DOS_HEADER) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, TRUE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ SelfSignedFile32, sizeof(IMAGE_DOS_HEADER) + sizeof(IMAGE_NT_HEADERS32) * 2 }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },

        /* 64-bit tests */
        {
            {{ SelfSignedFile64, sizeof(SelfSignedFile64) }},
            { CERT_E_CHAINING, TRUE }, { S_OK, FALSE }
        },
        {
            {{ SelfSignedFile64, sizeof(SelfSignedFile64) },
             { Dummy, sizeof(Dummy) }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ Dummy, sizeof(Dummy) },
             { SelfSignedFile64 + sizeof(Dummy), sizeof(SelfSignedFile64) - sizeof(Dummy) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile64, 19 },
             { Dummy, sizeof(Dummy) },
             { SelfSignedFile64 + 19 + sizeof(Dummy), sizeof(SelfSignedFile64) - 19 - sizeof(Dummy) }},
            { TRUST_E_BAD_DIGEST, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile64, sizeof(IMAGE_DOS_HEADER) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, TRUE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ SelfSignedFile64, sizeof(IMAGE_DOS_HEADER) + sizeof(IMAGE_NT_HEADERS64) * 2 }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
    };
    WCHAR pathW[MAX_PATH];
    DWORD written;
    HANDLE file;
    LONG status;
    BOOL ret;
    int i, j;

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        file = create_temp_file(pathW);
        ok(file != INVALID_HANDLE_VALUE, "failed to create temporary file\n");

        for (j = 0; tests[i].blocks[j].data; j++)
        {
            ret = WriteFile(file, tests[i].blocks[j].data, tests[i].blocks[j].length, &written, NULL);
            ok(ret && written == tests[i].blocks[j].length, "WriteFile failed with %lu\n", GetLastError());
        }

        CloseHandle(file);

        call_winverify(pathW, &status, FALSE);
        todo_wine_if(tests[i].t1.todo)
        ok(status == tests[i].t1.status, "test %d/1: expected %08lx, got %08lx\n", i, tests[i].t1.status, status);

        call_winverify(pathW, &status, TRUE);
        todo_wine_if(tests[i].t2.todo)
        ok(status == tests[i].t2.status, "test %d/2: expected %08lx, got %08lx\n", i, tests[i].t2.status, status);

        DeleteFileW(pathW);
    }
}

static void test_get_known_usages(void)
{
    BOOL ret;
    PCCRYPT_OID_INFO *usages;

    if (!pWTHelperGetKnownUsages)
    {
        skip("missing WTHelperGetKnownUsages\n");
        return;
    }
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(0, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(1, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(0, &usages);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
    /* A value of 1 for the first parameter seems to imply the value is
     * allocated
     */
    SetLastError(0xdeadbeef);
    usages = NULL;
    ret = pWTHelperGetKnownUsages(1, &usages);
    ok(ret, "WTHelperGetKnownUsages failed: %ld\n", GetLastError());
    ok(usages != NULL, "expected a pointer\n");
    if (ret && usages)
    {
        PCCRYPT_OID_INFO *ptr;

        /* The returned usages are an array of PCCRYPT_OID_INFOs, terminated with a
         * NULL pointer.
         */
        for (ptr = usages; *ptr; ptr++)
        {
            ok((*ptr)->cbSize == sizeof(CRYPT_OID_INFO) ||
             (*ptr)->cbSize == (sizeof(CRYPT_OID_INFO) + 2 * sizeof(LPCWSTR)), /* Vista */
             "unexpected size %ld\n", (*ptr)->cbSize);
            /* Each returned usage is in the CRYPT_ENHKEY_USAGE_OID_GROUP_ID group */
            ok((*ptr)->dwGroupId == CRYPT_ENHKEY_USAGE_OID_GROUP_ID,
             "expected group CRYPT_ENHKEY_USAGE_OID_GROUP_ID, got %ld\n",
             (*ptr)->dwGroupId);
        }
    }
    /* A value of 2 for the second parameter seems to imply the value is freed
     */
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(2, &usages);
    ok(ret, "WTHelperGetKnownUsages failed: %ld\n", GetLastError());
    ok(usages == NULL, "expected pointer to be cleared\n");
    SetLastError(0xdeadbeef);
    usages = NULL;
    ret = pWTHelperGetKnownUsages(2, &usages);
    ok(ret, "WTHelperGetKnownUsages failed: %ld\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(2, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %ld\n", GetLastError());
}

static void test_multiple_signatures(void)
{
    static const BYTE serials[][16] =
    {
        { 0xfa, 0x4b, 0x40, 0x69, 0x40, 0x4b, 0xc0, 0x44, 0xa2, 0x31, 0x3a, 0xa7, 0xaa, 0x97, 0x73, 0xd1, },
        { 0xcc, 0x03, 0x6b, 0x72, 0x4a, 0x93, 0x42, 0x40, 0x8f, 0x93, 0x41, 0x68, 0xa9, 0xec, 0x01, 0x8d, },
        { 0xa1, 0x58, 0x5b, 0x68, 0x85, 0x6a, 0x91, 0x4d, 0xa7, 0xe3, 0xa4, 0x15, 0x08, 0x3e, 0xfb, 0xae, },
    };
    static GUID WVTPolicyGUID = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    WINTRUST_SIGNATURE_SETTINGS settings = { sizeof(settings) };
    WINTRUST_FILE_INFO file_info = { sizeof(file_info) };
    WINTRUST_DATA data = { sizeof(data) };
    CRYPT_PROVIDER_DATA *prov;
    WCHAR pathW[MAX_PATH];
    CERT_INFO *cert_info;
    unsigned int i;
    BYTE buf[4096];
    DWORD written;
    LONG status;
    HANDLE file;
    DWORD size;
    BOOL bret;

    file = create_temp_file(pathW);
    ok(file != INVALID_HANDLE_VALUE, "Failed to create temporary file.\n");
    bret = WriteFile(file, self_signed_3certs, sizeof(self_signed_3certs), &written, NULL);
    ok(bret, "Failed, err %lu.\n", GetLastError());
    CloseHandle(file);

    file_info.pcwszFilePath = pathW;
    data.dwUIChoice = WTD_UI_NONE;
    data.fdwRevocationChecks = WTD_REVOKE_NONE;
    data.dwUnionChoice = WTD_CHOICE_FILE;
    data.pFile = &file_info;
    data.dwStateAction = WTD_STATEACTION_VERIFY;
    data.dwProvFlags = 0;
    data.pSignatureSettings = &settings;

    settings.cSecondarySigs = 0xcccccccc;
    settings.dwVerifiedSigIndex = 0xcccccccc;
    status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    todo_wine ok(status == CERT_E_UNTRUSTEDROOT || status == CERT_E_CHAINING, "Failed, ret %#lx\n", status);
    ok(settings.cSecondarySigs == 0xcccccccc, "Got %lu.\n", settings.cSecondarySigs);
    todo_wine ok(settings.dwVerifiedSigIndex == 2, "Got %lu.\n", settings.dwVerifiedSigIndex);

    prov = (CRYPT_PROVIDER_DATA *)data.hWVTStateData;
    ok(prov->cbStruct == sizeof(*prov), "Got size %lu.\n", prov->cbStruct);
    ok(prov->csSigners == 1, "Got %lu.\n", prov->csSigners);
    ok(prov->pSigSettings == &settings, "Got %p, expected %p.\n", prov->pSigSettings, &settings);
    ok(!!prov->pSigState, "Got %p, expected %p.\n", prov->pSigSettings, &settings);
    if (prov->cbStruct == sizeof(*prov) && prov->pSigState)
    {
        ok(prov->pSigState->cbStruct == sizeof(*prov->pSigState)
                || broken(prov->pSigState->cbStruct == offsetof(CRYPT_PROVIDER_SIGSTATE, iAttemptCount)) /* Win7 */,
                "Got %lu.\n", prov->pSigState->cbStruct);
        ok(prov->pSigState->fSupportMultiSig, "Got %d.\n", prov->pSigState->fSupportMultiSig);
        ok(prov->pSigState->dwCryptoPolicySupport == (WSS_SIGTRUST_SUPPORT | WSS_OBJTRUST_SUPPORT
                | WSS_CERTTRUST_SUPPORT), "Got %#lx.\n", prov->pSigState->dwCryptoPolicySupport);
        ok(prov->pSigState->cSecondarySigs == 2, "Got %lu.\n", prov->pSigState->cSecondarySigs);

        size = sizeof(buf);
        bret = CryptMsgGetParam(prov->pSigState->hPrimarySig, CMSG_SIGNER_CERT_INFO_PARAM, 0, buf, &size);
        ok(bret, "Failed, err %#lx.\n", GetLastError());
        cert_info = (CERT_INFO *)buf;
        ok(cert_info->SerialNumber.cbData == sizeof(serials[0]), "Got %lu.\n", cert_info->SerialNumber.cbData);
        ok(!memcmp(cert_info->SerialNumber.pbData, serials[0], sizeof(serials[0])), "Data does not match.\n");
        for (i = 0; i < prov->pSigState->cSecondarySigs; ++i)
        {
            bret = CryptMsgGetParam(prov->pSigState->rhSecondarySigs[i], CMSG_SIGNER_CERT_INFO_PARAM, 0, buf, &size);
            ok(bret, "Failed, err %#lx.\n", GetLastError());
            ok(cert_info->SerialNumber.cbData == sizeof(serials[0]), "Got %lu.\n", cert_info->SerialNumber.cbData);
            ok(!memcmp(cert_info->SerialNumber.pbData, serials[i + 1], sizeof(serials[0])), "Data does not match.\n");
        }
    }

    data.dwStateAction = WTD_STATEACTION_CLOSE;
    status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    ok(status == S_OK, "Failed, ret %#lx\n", status);

    data.dwStateAction = WTD_STATEACTION_VERIFY;
    settings.dwFlags = WSS_GET_SECONDARY_SIG_COUNT;
    settings.cSecondarySigs = 0xcccccccc;
    settings.dwVerifiedSigIndex = 0xcccccccc;
    status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    todo_wine ok(status == CERT_E_UNTRUSTEDROOT || status == CERT_E_CHAINING, "Failed, ret %#lx\n", status);
    ok(settings.cSecondarySigs == 2, "Got %lu.\n", settings.cSecondarySigs);
    todo_wine ok(settings.dwVerifiedSigIndex == 2, "Got %lu.\n", settings.dwVerifiedSigIndex);

    data.dwStateAction = WTD_STATEACTION_CLOSE;
    status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    ok(status == S_OK, "Failed, ret %#lx\n", status);

    data.dwStateAction = WTD_STATEACTION_VERIFY;
    settings.dwFlags = WSS_VERIFY_SPECIFIC | WSS_GET_SECONDARY_SIG_COUNT;
    settings.cSecondarySigs = 0xcccccccc;
    settings.dwVerifiedSigIndex = 0xcccccccc;
    settings.dwIndex = 1;
    status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    todo_wine ok(status == CERT_E_UNTRUSTEDROOT || status == CERT_E_CHAINING, "Failed, ret %#lx\n", status);
    ok(settings.cSecondarySigs == 2, "Got %lu.\n", settings.cSecondarySigs);
    todo_wine ok(settings.dwVerifiedSigIndex == 1, "Got %lu.\n", settings.dwVerifiedSigIndex);
    settings.dwIndex = 0;

    data.dwStateAction = WTD_STATEACTION_CLOSE;
    status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    ok(status == S_OK, "Failed, ret %#lx\n", status);

    DeleteFileW(pathW);
}

START_TEST(softpub)
{
    InitFunctionPtrs();
    test_provider_funcs();
    test_sip_create_indirect_data();
    test_wintrust();
    test_wintrust_digest();
    test_get_known_usages();
    test_multiple_signatures();
}
