/* md5.c - Functions to compute MD5 message digest of files or memory blocks
 * according to the definition of MD5 in RFC 1321 from April 1992.
 *
 * Thanks to Ulrich Drepper for the md5sum example code
 *
 * Copyright (C) 2002 by the Widelands Development Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

// a MD5 checksum is 16 bytes long. We use buffers to store it

#include "md5.h"

const unsigned char ChkSum::fillbuf[64] = { 0x80, 0 /* , 0, 0, ...  */ };

/**
 * Default Constructor
 *
 * Args: none
 * Ret:	none
 */
ChkSum::ChkSum(void) {
		  can_handle_data=1;
		  nread=0;

		  ctx.A = 0x67452301;
		  ctx.B = 0xefcdab89;
		  ctx.C = 0x98badcfe;
		  ctx.D = 0x10325476;

		  ctx.total[0] = ctx.total[1] = 0;
		  ctx.buflen = 0;
}

/**
 * Default destructor
 *
 * Args: none
 * Ret: none
 */
ChkSum::~ChkSum(void) {
		  can_handle_data=0;
}

/**
 * void pass_data(void* data, int length);
 *
 * This function cares for new data. It takes it,
 * buffers it, and when the buffer is full calculates
 * one MD5 block.
 *
 * Args: 	data 		data to compute chksum for
 * 			length	len of data
 */
void ChkSum::pass_data(const void* data, uint length) {
		  if(can_handle_data==0) return; // this checksum is finished! don't touch it

		  if(length < BLOCKSIZE-nread) {
					 // Simply add
					 memcpy(buf+nread, data, length);
					 nread+=length;
					 return;
		  } else {


					 // Buffers is full
					 memcpy(buf+nread, data, BLOCKSIZE-nread);
					 // Handle buffer
					 md5_process_block(buf, BLOCKSIZE, &ctx);
					 data=(((char*) data)+BLOCKSIZE-nread);
                length-=BLOCKSIZE-nread;
                nread=0;
                pass_data(data, length);
/*
                memcpy(buf, (const char*) (data)+(BLOCKSIZE-nread), length-(BLOCKSIZE-nread));
					 nread=length-(BLOCKSIZE-nread);
*/		  }

}


/** void finish_chksum(void)
 *
 * This function finishes the checksum. after this, calls
 * to pass_data get ignored
 *
 * Args:	none
 */
void ChkSum::finish_chksum(void) {
		  can_handle_data=0;

		  if(nread>0) {
					 md5_process_bytes(buf, nread, &ctx);
		  }

		  md5_finish_ctx(&ctx, sum);
}

/** Overloaded == operator
 *
 * Args: sum1,sum2 	sums to check to
 * returns: true if match, fales otherwise
 */
bool operator==(ChkSum& sum1, ChkSum& sum2) {
		  ulong* buf1=sum1.get_chksum();
		  ulong* buf2=sum2.get_chksum();

		  if(!buf1 || !buf2) return false;

		  if(buf1[0] == buf2[0])
					 if(buf1[1] == buf2[1])
								if(buf1[2] == buf2[2])
										  if(buf1[3] == buf2[3])
													 return true;
		  return false;
}

/** Opverloaded == operator
 *
 * Args: buf	buf to check to
 * 		sum1	sum to check to
 * returns: true if match, false otherwise
 */
bool operator==(ChkSum& sum1, const void* buf) {
		  ulong* buf1=sum1.get_chksum();
		  ulong* buf2=(ulong*) buf;

		  if(!buf1 || !buf2) return false;

		  if(buf1[0] == buf2[0])
					 if(buf1[1] == buf2[1])
								if(buf1[2] == buf2[2])
										  if(buf1[3] == buf2[3])
													 return true;
		  return false;
}

std::ostream& operator<<(std::ostream& o, ChkSum& sum) {
		  uchar	*buf=(uchar*) sum.get_chksum();
		  if(!buf) {
					 o << "(inval)";
		  } else {
					 char buf1[33];
					 sprintf(buf1, "%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
									buf[0], buf[1], buf[2], buf[3], buf[4], buf[5], buf[6], buf[7],
									buf[8], buf[9], buf[10], buf[11], buf[12], buf[13], buf[14], buf[15]);
					 o << buf1;
		  }
		  return o;
}

/********************************************************************
 *
 * Down here: private functions originally from Ulrich Drepper
 *
 * From GNU textutils. md5.c
 *******************************************************************/

/* Process the remaining bytes in the internal buffer and the usual
   prolog according to the standard and write the result to RESBUF.

   IMPORTANT: On some systems it is required that RESBUF is correctly
   aligned for a 32 bits value.  */
void * ChkSum::md5_finish_ctx (md5_ctx* ctx, void* resbuf)
{
  /* Take yet unprocessed bytes into account.  */
  ulong bytes = ctx->buflen;
  ulong pad;

  /* Now count remaining bytes.  */
  ctx->total[0] += bytes;
  if (ctx->total[0] < bytes)
    ++ctx->total[1];

  pad = bytes >= 56 ? 64 + 56 - bytes : 56 - bytes;
  memcpy (&ctx->buffer[bytes], fillbuf, pad);

  /* Put the 64-bit file length in *bits* at the end of the buffer.  */
  *(ulong *) &ctx->buffer[bytes + pad] = (ctx->total[0] << 3);
  *(ulong *) &ctx->buffer[bytes + pad + 4] = ((ctx->total[1] << 3) |
							(ctx->total[0] >> 29));

  /* Process last bytes.  */
  md5_process_block (ctx->buffer, bytes + pad + 8, ctx);


  ((ulong *) resbuf)[0] = (ctx->A);
  ((ulong *) resbuf)[1] = (ctx->B);
  ((ulong *) resbuf)[2] = (ctx->C);
  ((ulong *) resbuf)[3] = (ctx->D);


  return resbuf;
}

/* Processes some bytes in the internal buffer */
void ChkSum::md5_process_bytes (const void* buffer, ulong len, struct md5_ctx* ctx)
{
  /* When we already have some bits in our internal buffer concatenate
     both inputs first.  */
  if (ctx->buflen != 0)
    {
      ulong left_over = ctx->buflen;
      ulong add = 128 - left_over > len ? len : 128 - left_over;

      memcpy (&ctx->buffer[left_over], buffer, add);
      ctx->buflen += add;

      if (left_over + add > 64)
	{
	  md5_process_block (ctx->buffer, (left_over + add) & ~63, ctx);
	  /* The regions in the following copy operation cannot overlap.  */
	  memcpy (ctx->buffer, &ctx->buffer[(left_over + add) & ~63],
		  (left_over + add) & 63);
	  ctx->buflen = (left_over + add) & 63;
	}

      buffer = (const char *) buffer + add;
      len -= add;
    }

  /* Process available complete blocks.  */
  if (len > 64)
    {
      md5_process_block (buffer, len & ~63, ctx);
      buffer = (const char *) buffer + (len & ~63);
      len &= 63;
    }

  /* Move remaining bytes in internal buffer.  */
  if (len > 0)
    {
      memcpy (ctx->buffer, buffer, len);
      ctx->buflen = len;
    }
}


/* These are the four functions used in the four steps of the MD5 algorithm
   and defined in the RFC 1321.  The first function is a little bit optimized
   (as found in Colin Plumbs public domain implementation).  */
/* #define FF(b, c, d) ((b & c) | (~b & d)) */
#define FF(b, c, d) (d ^ (b & (c ^ d)))
#define FG(b, c, d) FF (d, b, c)
#define FH(b, c, d) (b ^ c ^ d)
#define FI(b, c, d) (c ^ (b | ~d))

/* Process LEN bytes of BUFFER, accumulating context into CTX.
   It is assumed that LEN % 64 == 0.  */

void
ChkSum::md5_process_block (const void* buffer, ulong len, md5_ctx* ctx)
{
  ulong correct_words[16];
  const ulong *words = (ulong*) buffer;
  ulong nwords = len / sizeof (ulong);
  const ulong *endp = words + nwords;
  ulong A = ctx->A;
  ulong B = ctx->B;
  ulong C = ctx->C;
  ulong D = ctx->D;

  /* First increment the byte count.  RFC 1321 specifies the possible
     length of the file up to 2^64 bits.  Here we only compute the
     number of bytes.  Do a double word increment.  */
  ctx->total[0] += len;
  if (ctx->total[0] < len)
    ++ctx->total[1];

  /* Process all bytes in the buffer with 64 bytes in each round of
     the loop.  */
  while (words < endp)
    {
      ulong *cwp = correct_words;
      ulong A_save = A;
      ulong B_save = B;
      ulong C_save = C;
      ulong D_save = D;

      /* First round: using the given function, the context and a constant
	 the next context is computed.  Because the algorithms processing
	 unit is a 32-bit word and it is determined to work on words in
	 little endian byte order we perhaps have to change the byte order
	 before the computation.  To reduce the work for the next steps
	 we store the swapped words in the array CORRECT_WORDS.  */

#define OP(a, b, c, d, s, T)						\
      do								\
        {								\
	  a += FF (b, c, d) + (*cwp++ = (*words)) + T;		\
	  ++words;							\
	  CYCLIC (a, s);						\
	  a += b;							\
        }								\
      while (0)

      /* It is unfortunate that C does not provide an operator for
	 cyclic rotation.  Hope the C compiler is smart enough.  */
#define CYCLIC(w, s) (w = (w << s) | (w >> (32 - s)))

      /* Before we start, one word to the strange constants.
	 They are defined in RFC 1321 as

	 T[i] = (int) (4294967296.0 * fabs (sin (i))), i=1..64
       */

      /* Round 1.  */
      OP (A, B, C, D,  7, 0xd76aa478);
      OP (D, A, B, C, 12, 0xe8c7b756);
      OP (C, D, A, B, 17, 0x242070db);
      OP (B, C, D, A, 22, 0xc1bdceee);
      OP (A, B, C, D,  7, 0xf57c0faf);
      OP (D, A, B, C, 12, 0x4787c62a);
      OP (C, D, A, B, 17, 0xa8304613);
      OP (B, C, D, A, 22, 0xfd469501);
      OP (A, B, C, D,  7, 0x698098d8);
      OP (D, A, B, C, 12, 0x8b44f7af);
      OP (C, D, A, B, 17, 0xffff5bb1);
      OP (B, C, D, A, 22, 0x895cd7be);
      OP (A, B, C, D,  7, 0x6b901122);
      OP (D, A, B, C, 12, 0xfd987193);
      OP (C, D, A, B, 17, 0xa679438e);
      OP (B, C, D, A, 22, 0x49b40821);

      /* For the second to fourth round we have the possibly swapped words
	 in CORRECT_WORDS.  Redefine the macro to take an additional first
	 argument specifying the function to use.  */
#undef OP
#define OP(f, a, b, c, d, k, s, T)					\
      do 								\
	{								\
	  a += f (b, c, d) + correct_words[k] + T;			\
	  CYCLIC (a, s);						\
	  a += b;							\
	}								\
      while (0)

      /* Round 2.  */
      OP (FG, A, B, C, D,  1,  5, 0xf61e2562);
      OP (FG, D, A, B, C,  6,  9, 0xc040b340);
      OP (FG, C, D, A, B, 11, 14, 0x265e5a51);
      OP (FG, B, C, D, A,  0, 20, 0xe9b6c7aa);
      OP (FG, A, B, C, D,  5,  5, 0xd62f105d);
      OP (FG, D, A, B, C, 10,  9, 0x02441453);
      OP (FG, C, D, A, B, 15, 14, 0xd8a1e681);
      OP (FG, B, C, D, A,  4, 20, 0xe7d3fbc8);
      OP (FG, A, B, C, D,  9,  5, 0x21e1cde6);
      OP (FG, D, A, B, C, 14,  9, 0xc33707d6);
      OP (FG, C, D, A, B,  3, 14, 0xf4d50d87);
      OP (FG, B, C, D, A,  8, 20, 0x455a14ed);
      OP (FG, A, B, C, D, 13,  5, 0xa9e3e905);
      OP (FG, D, A, B, C,  2,  9, 0xfcefa3f8);
      OP (FG, C, D, A, B,  7, 14, 0x676f02d9);
      OP (FG, B, C, D, A, 12, 20, 0x8d2a4c8a);

      /* Round 3.  */
      OP (FH, A, B, C, D,  5,  4, 0xfffa3942);
      OP (FH, D, A, B, C,  8, 11, 0x8771f681);
      OP (FH, C, D, A, B, 11, 16, 0x6d9d6122);
      OP (FH, B, C, D, A, 14, 23, 0xfde5380c);
      OP (FH, A, B, C, D,  1,  4, 0xa4beea44);
      OP (FH, D, A, B, C,  4, 11, 0x4bdecfa9);
      OP (FH, C, D, A, B,  7, 16, 0xf6bb4b60);
      OP (FH, B, C, D, A, 10, 23, 0xbebfbc70);
      OP (FH, A, B, C, D, 13,  4, 0x289b7ec6);
      OP (FH, D, A, B, C,  0, 11, 0xeaa127fa);
      OP (FH, C, D, A, B,  3, 16, 0xd4ef3085);
      OP (FH, B, C, D, A,  6, 23, 0x04881d05);
      OP (FH, A, B, C, D,  9,  4, 0xd9d4d039);
      OP (FH, D, A, B, C, 12, 11, 0xe6db99e5);
      OP (FH, C, D, A, B, 15, 16, 0x1fa27cf8);
      OP (FH, B, C, D, A,  2, 23, 0xc4ac5665);

      /* Round 4.  */
      OP (FI, A, B, C, D,  0,  6, 0xf4292244);
      OP (FI, D, A, B, C,  7, 10, 0x432aff97);
      OP (FI, C, D, A, B, 14, 15, 0xab9423a7);
      OP (FI, B, C, D, A,  5, 21, 0xfc93a039);
      OP (FI, A, B, C, D, 12,  6, 0x655b59c3);
      OP (FI, D, A, B, C,  3, 10, 0x8f0ccc92);
      OP (FI, C, D, A, B, 10, 15, 0xffeff47d);
      OP (FI, B, C, D, A,  1, 21, 0x85845dd1);
      OP (FI, A, B, C, D,  8,  6, 0x6fa87e4f);
      OP (FI, D, A, B, C, 15, 10, 0xfe2ce6e0);
      OP (FI, C, D, A, B,  6, 15, 0xa3014314);
      OP (FI, B, C, D, A, 13, 21, 0x4e0811a1);
      OP (FI, A, B, C, D,  4,  6, 0xf7537e82);
      OP (FI, D, A, B, C, 11, 10, 0xbd3af235);
      OP (FI, C, D, A, B,  2, 15, 0x2ad7d2bb);
      OP (FI, B, C, D, A,  9, 21, 0xeb86d391);

      /* Add the starting values of the context.  */
      A += A_save;
      B += B_save;
      C += C_save;
      D += D_save;
    }

  /* Put checksum in context given as argument.  */
  ctx->A = A;
  ctx->B = B;
  ctx->C = C;
  ctx->D = D;
}
