# ui-lib.pl
# Common functions for generating HTML for Webmin user interface elements

####################### table generation functions

# ui_table_start(heading, [tabletags], [cols])
# A table with a heading and table inside
sub ui_table_start
{
return &theme_ui_table_start(@_) if (defined(&theme_ui_table_start));
local ($heading, $tabletags, $cols) = @_;
local $rv;
$rv .= "<table border $tabletags>\n";
$rv .= "<tr $tb> <td><b>$heading</b></td> </tr>\n" if (defined($heading));
$rv .= "<tr $cb> <td><table width=100%>\n";
$main::ui_table_cols = $cols || 4;
$main::ui_table_pos = 0;
return $rv;
}

# ui_table_end()
# The end of a table started by ui_table_start
sub ui_table_end
{
return &theme_ui_table_end(@_) if (defined(&theme_ui_table_end));
return "</table></td></tr></table>\n";
}

# ui_columns_start(&headings, [width-percent], [noborder], [&tdtags])
# Returns HTML for a multi-column table, with the given headings
sub ui_columns_start
{
return &theme_ui_columns_start(@_) if (defined(&theme_ui_columns_start));
local ($heads, $width, $noborder, $tdtags) = @_;
local $rv;
$rv .= "<table".($noborder ? "" : " border").
		(defined($width) ? " width=$width%" : "").">\n";
$rv .= "<tr $tb>\n";
local $i;
for($i=0; $i<@$heads; $i++) {
	$rv .= "<td ".$tdtags->[$i]."><b>".
	       ($heads->[$i] eq "" ? "<br>" : $heads->[$i])."</b></td>\n";
	}
$rv .= "</tr>\n";
return $rv;
}

# ui_columns_row(&columns, &tdtags)
# Returns HTML for a row in a multi-column table
sub ui_columns_row
{
return &theme_ui_columns_row(@_) if (defined(&theme_ui_columns_row));
local ($cols, $tdtags) = @_;
local $rv;
$rv .= "<tr $cb>\n";
local $i;
for($i=0; $i<@$cols; $i++) {
	$rv .= "<td ".$tdtags->[$i].">".
	       ($cols->[$i] eq "" ? "<br>" : $cols->[$i])."</td>\n";
	}
$rv .= "</tr>\n";
return $rv;
}

# ui_columns_end()
# Returns HTML to end a table started by ui_columns_start
sub ui_columns_end
{
return &theme_ui_columns_end(@_) if (defined(&theme_ui_columns_end));
return "</table>\n";
}

####################### form generation functions

# ui_form_start(script, method)
# Returns HTML for a form that submits to some script
sub ui_form_start
{
return &theme_ui_form_start(@_) if (defined(&theme_ui_form_start));
local ($script, $method) = @_;
local $rv;
$rv .= "<form action='$script' ".
		($method eq "post" ? "method=post" :
		 $method eq "form-data" ?
			"method=post enctype=multipart/form-data" :
			"method=get").">\n";
return $rv;
}

# ui_form_end([&buttons], [width])
# Returns HTML for the end of a form, optionally with a row of submit buttons
sub ui_form_end
{
return &theme_ui_form_end(@_) if (defined(&theme_ui_form_end));
local ($buttons, $width) = @_;
local $rv;
if ($buttons) {
	$rv .= "<table".($width ? " width=$width" : "")."><tr>\n";
	local $b;
	foreach $b (@$buttons) {
		if (ref($b)) {
			$rv .= "<td".(!$width ? "" :
				      $b eq $buttons->[0] ? " align=left" :
				      $b eq $buttons->[@$buttons-1] ?
					" align=right" : " align=center").">".
			       &ui_submit($b->[1], $b->[0]).
			       ($b->[2] ? " ".$b->[2] : "")."</td>\n";
			}
		elsif ($b) {
			$rv .= "<td>$b</td>\n";
			}
		else {
			$rv .= "<td>&nbsp;&nbsp;</td>\n";
			}
		}
	$rv .= "</tr></table>\n";
	}
$rv .= "</form>\n";
return $rv;
}

# ui_textbox(name, value, size, [disabled?])
# Returns HTML for a text input
sub ui_textbox
{
return &theme_ui_textbox(@_) if (defined(&theme_ui_textbox));
local ($name, $value, $size, $dis) = @_;
return "<input name=\"".&quote_escape($name)."\" ".
       "value=\"".&quote_escape($value)."\" ".
       "size=$size ".($dis ? "disabled=true" : "").">";
}

# ui_upload(name, size, [disabled?])
# Returns HTML for a file upload input
sub ui_upload
{
return &theme_ui_upload(@_) if (defined(&theme_ui_upload));
local ($name, $size, $dis) = @_;
return "<input type=file name=\"".&quote_escape($name)."\" ".
       "size=$size ".($dis ? "disabled=true" : "").">";
}

# ui_password(name, value, size, [disabled?])
# Returns HTML for a password text input
sub ui_password
{
return &theme_ui_password(@_) if (defined(&theme_ui_password));
local ($name, $value, $size, $dis) = @_;
return "<input type=password name=\"".&quote_escape($name)."\" ".
       "value=\"".&quote_escape($value)."\" ".
       "size=$size ".($dis ? "disabled=true" : "").">";
}

# ui_hidden(name, value)
# Returns HTML for a hidden field
sub ui_hidden
{
return &theme_ui_hidden(@_) if (defined(&theme_ui_hidden));
local ($name, $value) = @_;
return "<input type=hidden name=\"".&quote_escape($name)."\" ".
       "value=\"".&quote_escape($value)."\">";
}

# ui_select(name, value|&values, &options, [size], [multiple],
#	    [add-if-missing], [disabled?])
# Returns HTML for a drop-down menu or multiple selection list
sub ui_select
{
return &theme_ui_select(@_) if (defined(&theme_ui_select));
local ($name, $value, $opts, $size, $multiple, $missing, $dis) = @_;
local $rv;
$rv .= "<select name=\"".&quote_escape($name)."\"".
       ($size ? " size=$size" : "").
       ($multiple ? " multiple" : "").
       ($dis ? " disabled=true" : "").">\n";
local ($o, %opt, $s);
local %sel = ref($value) ? ( map { $_, 1 } @$value ) : ( $value, 1 );
foreach $o (@$opts) {
	$rv .= "<option value=\"".&quote_escape($o->[0])."\"".
	       ($sel{$o->[0]} ? " selected" : "").">".
	       ($o->[1] || $o->[0])."\n";
	$opt{$o->[0]}++;
	}
foreach $s (keys %sel) {
	if (!$opt{$s} && $missing) {
		$rv .= "<option value=\"".&quote_escape($s)."\"".
		       "selected>".($s eq "" ? "&nbsp;" : $s)."\n";
		}
	}
$rv .= "</select>\n";
return $rv;
}

# ui_radio(name, value, &options, [disabled?])
# Returns HTML for a series of radio buttons
sub ui_radio
{
return &theme_ui_radio(@_) if (defined(&theme_ui_radio));
local ($name, $value, $opts, $dis) = @_;
local $rv;
local $o;
foreach $o (@$opts) {
	$rv .= "<input type=radio name=\"".&quote_escape($name)."\" ".
               "value=\"".&quote_escape($o->[0])."\"".
	       ($o->[0] eq $value ? " checked" : "").
	       ($dis ? " disabled=true" : "")." $o->[2]> ".
	       ($o->[1] || $o->[0])."\n";
	}
return $rv;
}

# ui_yesno_radio(name, value, [yes], [no], [disabled?])
# Like ui_yesno, but always displays just two inputs (yes and no)
sub ui_yesno_radio
{
local ($name, $value, $yes, $no, $dis) = @_;
$yes = 1 if (!defined($yes));
$no = 0 if (!defined($no));
return &theme_ui_yesno_radio(@_) if (defined(&theme_ui_yesno_radio));
return &ui_radio($name, $value, [ [ $yes, $text{'yes'} ],
				  [ $no, $text{'no'} ] ], $dis);
}

# ui_checkbox(name, value, label, selected?, [tags], [disabled?])
# Returns HTML for a single checkbox
sub ui_checkbox
{
return &theme_ui_checkbox(@_) if (defined(&theme_ui_checkbox));
local ($name, $value, $label, $sel, $tags, $dis) = @_;
return "<input type=checkbox name=\"".&quote_escape($name)."\" ".
       "value=\"".&quote_escape($value)."\" ".
       ($sel ? " checked" : "").($dis ? " disabled=true" : "").
       " $tags> $label\n";
}

# ui_oneradio(name, value, label, selected?, [tags], [disabled?])
# Returns HTML for a single radio button
sub ui_oneradio
{
return &theme_ui_oneradio(@_) if (defined(&theme_ui_oneradio));
local ($name, $value, $label, $sel, $tags, $dis) = @_;
return "<input type=radio name=\"".&quote_escape($name)."\" ".
       "value=\"".&quote_escape($value)."\" ".
       ($sel ? " checked" : "").($dis ? " disabled=true" : "").
       " $tags> $label\n";
}

# ui_textarea(name, value, rows, cols, [wrap], [disabled?])
# Returns HTML for a multi-line text input
sub ui_textarea
{
return &theme_ui_textarea(@_) if (defined(&theme_ui_textarea));
local ($name, $value, $rows, $cols, $wrap, $dis) = @_;
return "<textarea name=\"".&quote_escape($name)."\" ".
       "rows=$rows cols=$cols".($wrap ? " wrap=$wrap" : "").
       ($dis ? " disabled=true" : "").">".
       &html_escape($value).
       "</textarea>";
}

# ui_user_textbox(name, value, [form], [disabled?])
# Returns HTML for a Unix user input
sub ui_user_textbox
{
return &theme_ui_user_textbox(@_) if (defined(&theme_ui_user_textbox));
return &ui_textbox($_[0], $_[1], 13, $_[3])." ".
       &user_chooser_button($_[0], 1, $_[2]);
}

# ui_group_textbox(name, value, [form], [disabled?])
# Returns HTML for a Unix group input
sub ui_group_textbox
{
return &theme_ui_group_textbox(@_) if (defined(&theme_ui_group_textbox));
return &ui_textbox($_[0], $_[1], 13, $_[3])." ".
       &group_chooser_button($_[0], 1, $_[2]);
}

# ui_opt_textbox(name, value, size, option1, [option2], [disabled?],
#		 [&extra-fields])
# Returns HTML for a text field that is optional
sub ui_opt_textbox
{
return &theme_ui_opt_textbox(@_) if (defined(&theme_ui_opt_textbox));
local ($name, $value, $size, $opt1, $opt2, $dis, $extra) = @_;
local $dis1 = &js_disable_inputs([ $name, @$extra ], [ ]);
local $dis2 = &js_disable_inputs([ ], [ $name, @$extra ]);
local $rv;
$rv .= "<input type=radio name=\"".&quote_escape($name."_def")."\" ".
       "value=1 ".($value ne '' ? "" : "checked").
       ($dis ? " disabled=true" : "")." onClick='$dis1'> ".$opt1."\n";
$rv .= "<input type=radio name=\"".&quote_escape($name."_def")."\" ".
       "value=0 ".($value ne '' ? "checked" : "").
       ($dis ? " disabled=true" : "")." onClick='$dis2'> ".$opt2."\n";
$rv .= "<input name=\"".&quote_escape($name)."\" ".
       "size=$size value=\"".&quote_escape($value)."\" ".
       ($value eq "" || $dis ? "disabled=true" : "").">\n";
return $rv;
}

# ui_submit(label, [name], [disabled?])
# Returns HTML for a form submit button
sub ui_submit
{
return &theme_ui_submit(@_) if (defined(&theme_ui_submit));
local ($label, $name, $dis) = @_;
return "<input type=submit".
       ($name ne '' ? " name=\"".&quote_escape($name)."\"" : "").
       " value=\"".&quote_escape($label).
       ($dis ? " disabled=true" : "")."\">\n";
			
}

# ui_reset(label, [disabled?])
# Returns HTML for a form reset button
sub ui_reset
{
return &theme_ui_reset(@_) if (defined(&theme_ui_reset));
local ($label, $dis) = @_;
return "<input type=submit value=\"".&quote_escape($label).
       ($dis ? " disabled=true" : "")."\">\n";
			
}

# ui_date_input(day, month, year, day-name, month-name, year-name, [disabled?])
# Returns HTML for a date-selection field
sub ui_date_input
{
local ($day, $month, $year, $dayname, $monthname, $yearname, $dis) = @_;
local $rv;
$rv .= &ui_textbox($dayname, $day, 3, $dis);
$rv .= "/";
$rv .= &ui_select($monthname, $month,
		  [ map { [ $_, $text{"smonth_$_"} ] } (1 .. 12) ],
		  1, 0, 0, $dis);
$rv .= "/";
$rv .= &ui_textbox($yearname, $year, 5, $dis);
return $rv;
}

# ui_table_row(label, value, [cols], [&td-tags])
# Returns HTML for a row in a table started by ui_table_start, with a 1-column
# label and 1+ column value.
sub ui_table_row
{
return &theme_ui_table_row(@_) if (defined(&theme_ui_table_row));
local ($label, $value, $cols) = @_;
$cols ||= 1;
local $rv;
$rv .= "<tr>\n" if ($main::ui_table_pos%$main::ui_table_cols == 0);
$rv .= "<td valign=top $_[3]->[0]><b>$label</b></td>\n" if (defined($label));
$rv .= "<td valign=top colspan=$cols $_[3]->[1]>$value</td>\n";
$main::ui_table_pos += $cols+(defined($label) ? 1 : 0);
$rv .= "</tr>\n" if ($main::ui_table_pos%$main::ui_table_cols == 0);
return $rv;
}

# ui_table_hr()
sub ui_table_hr
{
return &theme_ui_table_hr(@_) if (defined(&theme_ui_table_hr));
$ui_table_pos = 0;
return "<tr> <td colspan=$main::ui_table_cols><hr></td> </tr>\n";
}

# ui_buttons_start()
sub ui_buttons_start
{
return &theme_ui_buttons_start(@_) if (defined(&theme_ui_buttons_start));
return "<table width=100%>\n";
}

# ui_buttons_end()
sub ui_buttons_end
{
return &theme_(@_) if (defined(&theme_));
return "</table>\n";
}

# ui_buttons_row(script, button-label, description, [hiddens], [after-submit],
#		 [before-submit]) 
sub ui_buttons_row
{
return &theme_ui_buttons_row(@_) if (defined(&theme_ui_buttons_row));
local ($script, $label, $desc, $hiddens, $after, $before) = @_;
return "<form action=$script>\n".
       $hiddens.
       "<tr> <td nowrap width=20%>".($before ? $before." " : "").
       &ui_submit($label).($after ? " ".$after : "")."</td>\n".
       "<td valign=top width=80%>$desc</td> </tr>\n".
       "</form>\n";
}

# ui_buttons_hr()
sub ui_buttons_hr
{
return &theme_ui_buttons_hr(@_) if (defined(&theme_ui_buttons_hr));
return "<tr> <td colspan=2><hr></td> </tr>\n";
}

####################### header and footer functions

# ui_post_header([subtext])
# Returns HTML to appear directly after a standard header() call
sub ui_post_header
{
return &theme_ui_post_header(@_) if (defined(&theme_ui_post_header));
local ($text) = @_;
local $rv;
$rv .= "<center><font size=+1>$text</font></center>\n" if (defined($text));
$rv .= "<hr>\n";
return $rv;
}

# ui_pre_footer()
# Returns HTML to appear directly before a standard footer() call
sub ui_pre_footer
{
return &theme_ui_pre_footer(@_) if (defined(&theme_ui_pre_footer));
return "<hr>\n";
}

# ui_print_header(subtext, args...)
# Print HTML for a header with the post-header line. The args are the same
# as those passed to header()
sub ui_print_header
{
&load_theme_library();
return &theme_ui_print_header(@_) if (defined(&theme_ui_print_header));
local ($text, @args) = @_;
&header(@args);
print &ui_post_header($text);
}

# ui_print_unbuffered_header(subtext, args...)
# Like ui_print_header, but ensures that output for this page is not buffered
# or contained in a table.
sub ui_print_unbuffered_header
{
&load_theme_library();
return &theme_ui_print_unbuffered_header(@_) if (defined(&theme_ui_print_unbuffered_header));
$| = 1;
$theme_no_table = 1;
&ui_print_header(@_);
}

# ui_print_footer(args...)
# Print HTML for a footer with the pre-footer line. Args are the same as those
# passed to footer()
sub ui_print_footer
{
return &theme_ui_print_footer(@_) if (defined(&theme_ui_print_footer));
local @args = @_;
print &ui_pre_footer();
&footer(@args);
}

# ui_config_link(text, &subs)
# Returns HTML for a module config link. The first non-null sub will be
# replaced with the appropriate URL.
sub ui_config_link
{
return &theme_ui_config_link(@_) if (defined(&theme_ui_config_link));
local ($text, $subs) = @_;
local @subs = map { $_ || "../config.cgi?$module_name" }
		  ($subs ? @$subs : ( undef ));
return "<p>".&text($text, @subs)."<p>\n";
}

# ui_print_endpage(text)
# Prints HTML for an error message followed by a page footer with a link to
# /, then exits. Good for main page error messages.
sub ui_print_endpage
{
return &theme_ui_print_endpage(@_) if (defined(&theme_ui_print_endpage));
local ($text) = @_;
print $text,"<p>\n";
&ui_print_footer("/", $text{'index'});
exit;
}

# ui_subheading(text, ...)
# Returns HTML for a section heading
sub ui_subheading
{
return &theme_ui_subheading(@_) if (defined(&theme_ui_subheading));
return "<h3>",join("", @_),"</h3>\n";
}

####################### javascript functions

# js_disable_input(&disable-inputs, &enable-inputs, [tag])
# Returns Javascript to disable some form elements and enable others
sub js_disable_inputs
{
local $rv;
local $f;
foreach $f (@{$_[0]}) {
	$rv .= "form.$f.disabled = true; ";
	}
foreach $f (@{$_[1]}) {
	$rv .= "form.$f.disabled = false; ";
	}
return $_[2] ? "$_[2]='$rv'" : $rv;
}

# js_checkbox_disable(name, &checked-disable, &checked-enable, [tag])
sub js_checkbox_disable
{
local $rv;
local $f;
foreach $f (@{$_[1]}) {
	$rv .= "form.$f.disabled = $_[0].checked; ";
	}
foreach $f (@{$_[2]}) {
	$rv .= "form.$f.disabled = !$_[0].checked; ";
	}
return $_[3] ? "$_[3]='$rv'" : $rv;
}

1;

