/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "rowfilters.h"

#include "attributes/datetimehelper.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "tablemodels.h"
#include "tablemodelshelper.h"

IssueRowFilter::IssueRowFilter( QObject* parent ) : AbstractRowFilter( parent ),
    m_searchColumn( Column_Name )
{
}

IssueRowFilter::~IssueRowFilter()
{
}

void IssueRowFilter::setConditions( const QList<ColumnCondition>& conditions )
{
    m_conditions = conditions;

    emit conditionsChanged();
} 

void IssueRowFilter::setQuickSearch( int column, const QString& text )
{
    m_searchColumn = column;
    m_searchText = text;

    emit conditionsChanged();
}

bool IssueRowFilter::filterRow( int id )
{
    const IssueRow* row = dataManager->issues()->find( id );
    if ( !row )
        return false;

    for ( int i = 0; i < m_conditions.count(); i++ ) {
        QString value = extractValue( row, m_conditions.at( i ).column() );
        if ( !m_conditions.at( i ).match( value ) )
            return false;
    }

    if ( !m_searchText.isEmpty() ) {
        QString value = extractValue( row, m_searchColumn );
        if ( !value.contains( m_searchText, Qt::CaseInsensitive ) )
            return false;
    }

    return true;
}

QString IssueRowFilter::extractValue( const IssueRow* row, int column )
{
    if ( column > Column_UserDefined ) {
        int attributeId = column - Column_UserDefined;
        const ValueRow* value = dataManager->values()->find( attributeId, row->issueId() );
        return value ? value->value() : QString();
    }

    switch ( column ) {
        case Column_Name:
            return row->name();
        case Column_ID:
            return QString::number( row->issueId() );
        case Column_CreatedDate:
            return DateTimeHelper::formatDateTime( row->createdDate() );
        case Column_CreatedBy:
            return TableModelsHelper::userName( row->createdUser() );
        case Column_ModifiedDate:
            return DateTimeHelper::formatDateTime( row->modifiedDate() );
        case Column_ModifiedBy:
            return TableModelsHelper::userName( row->modifiedUser() );
        default:
            return QString();
    }
}
