/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "filterdialog.h"

#include <QLayout>
#include <QLabel>
#include <QComboBox>
#include <QGroupBox>
#include <QPushButton>
#include <QLineEdit>
#include <QDialogButtonBox>
#include <QScrollArea>

#include "attributes/abstractvalueeditor.h"
#include "models/columnconditionhelper.h"
#include "models/tablemodelshelper.h"
#include "dialogs/savefilterdialog.h"
#include "widgets/gradientwidget.h"
#include "iconloader.h"

FilterDialog::FilterDialog( int projectId, QWidget* parent ) : QDialog( parent ),
    m_projectId( projectId ),
    m_filterMode( false )
{
    setWindowTitle( tr( "Filter Settings" ) );

    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* promptWidget = new GradientWidget( this );
    topLayout->addWidget( promptWidget );

    QHBoxLayout* promptLayout = new QHBoxLayout( promptWidget );
    promptLayout->setSpacing( 10 );

    QLabel* promptPixmap = new QLabel( promptWidget );
    promptPixmap->setPixmap( IconLoader::pixmap( "filter", 22 ) );
    promptLayout->addWidget( promptPixmap, 0 );

    m_promptLabel = new QLabel( promptWidget );
    m_promptLabel->setWordWrap( true );
    m_promptLabel->setMinimumWidth( 250 );
    promptLayout->addWidget( m_promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );
    topLayout->addLayout( mainLayout );

    QHBoxLayout* contentsLayout = new QHBoxLayout();
    mainLayout->addLayout( contentsLayout );

    QGroupBox* group = new QGroupBox( tr( "Filter Conditions" ), this );
    QVBoxLayout* layout = new QVBoxLayout( group );
    layout->setMargin( 0 );

    QScrollArea* scroll = new QScrollArea( group );
    scroll->setWidgetResizable( true );
    scroll->setFrameStyle( QFrame::NoFrame );

    m_gridPanel = new QWidget( scroll );

    m_gridLayout = new QGridLayout( m_gridPanel );
    m_gridLayout->setAlignment( Qt::AlignTop );
    m_gridLayout->setColumnStretch( 2, 1 );

    scroll->setWidget( m_gridPanel );

    layout->addWidget( scroll );
    contentsLayout->addWidget( group );

    QVBoxLayout* buttonsLayout = new QVBoxLayout();
    contentsLayout->addLayout( buttonsLayout );

    buttonsLayout->addStretch( 1 );

    m_moreButton = new QPushButton( tr( "&More" ), this );
    buttonsLayout->addWidget( m_moreButton );

    m_fewerButton = new QPushButton( tr( "&Fewer" ), this );
    buttonsLayout->addWidget( m_fewerButton );

    buttonsLayout->addSpacing( 10 );

    m_clearButton = new QPushButton( tr( "Cl&ear" ), this );
    buttonsLayout->addWidget( m_clearButton );

    buttonsLayout->addStretch( 1 );

    connect( m_moreButton, SIGNAL( clicked() ), this, SLOT( addRow() ) );
    connect( m_fewerButton, SIGNAL( clicked() ), this, SLOT( removeRow() ) );
    connect( m_clearButton, SIGNAL( clicked() ), this, SLOT( clear() ) );

    mainLayout->addSpacing( 10 );

    m_buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Apply | QDialogButtonBox::Cancel,
        Qt::Horizontal, this );
    mainLayout->addWidget( m_buttonBox );

    m_buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    m_buttonBox->button( QDialogButtonBox::Apply )->setText( tr( "&Apply" ) );
    m_buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( m_buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( m_buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );
    connect( m_buttonBox->button( QDialogButtonBox::Apply ), SIGNAL( clicked() ), this, SIGNAL( settingsApplied() ) );

    m_fewerButton->setEnabled( false );

    resize( 600, 250 );
}

FilterDialog::~FilterDialog()
{
}

void FilterDialog::setPrompt( const QString& text )
{
    m_promptLabel->setText( text );

    m_promptLabel->setFixedHeight( m_promptLabel->heightForWidth( m_promptLabel->minimumWidth() ) );
}

void FilterDialog::setAvailableColumns( const QList<int>& columns )
{
    QMap<QString, int> map;
    for ( int i = 0; i < columns.count(); i++ ) {
        int column = columns.at( i );
        map.insert( TableModelsHelper::columnName( column ).toLower(), column );
    }

    m_columns.clear();

    for ( QMap<QString, int>::iterator it = map.begin(); it != map.end(); ++it )
        m_columns.append( it.value() );
}

void FilterDialog::setConditions( const QList<ColumnCondition>& conditions )
{
    clearAllRows();

    if ( conditions.count() > 0 ) {
        for ( int i = 0; i < conditions.count(); i++ )
            createRow( conditions[ i ] );

        m_fewerButton->setEnabled( conditions.count() > 1 );
    } else {
        addRow();
    }
}

QList<ColumnCondition> FilterDialog::conditions() const
{
    QList<ColumnCondition> conditions;

    for ( int i = 0; i < m_rows.count(); i++ ) {
        const FilterRow& row = m_rows[ i ];

        ColumnCondition condition;
        condition.setColumn( m_columns[ row.m_columnCombo->currentIndex() ] );
        condition.setCondition( row.m_conditions[ row.m_conditionCombo->currentIndex() ] );
        if ( row.m_editor )
            condition.setOperand( row.m_editor->value() );

        conditions.append( condition );
    }

    return conditions;
}

void FilterDialog::setFilterMode( bool mode )
{
    m_filterMode = mode;

    m_buttonBox->button( QDialogButtonBox::Apply )->setVisible( !mode );
}

void FilterDialog::setFilterName( const QString& name )
{
    m_filterName = name;
}

void FilterDialog::setExistingFilters( const QStringList& filters )
{
    m_existingFilters = filters;
}

void FilterDialog::accept()
{
    if ( m_filterMode ) {
        if ( m_filterName.isEmpty() ) {
            SaveFilterDialog dialog( this );
            dialog.setExistingFilters( m_existingFilters );

            if ( dialog.exec() != QDialog::Accepted )
                return;

            m_filterName = dialog.filterName();
        }
    } else {
        emit settingsApplied();
    }

    QDialog::accept();
}

void FilterDialog::addRow()
{
    ColumnCondition condition;
    condition.setColumn( -1 );
    condition.setCondition( (ColumnCondition::Condition)-1 );

    createRow( condition );

    m_fewerButton->setEnabled( m_rows.count() > 1 );
}

void FilterDialog::removeRow()
{
    int index = m_rows.count() - 1;
    if ( index <= 0 )
        return;

    delete m_rows[ index ].m_columnCombo;
    delete m_rows[ index ].m_conditionCombo;
    delete m_rows[ index ].m_editor;

    m_rows.pop_back();

    m_fewerButton->setEnabled( m_rows.count() > 1 );
}

void FilterDialog::clear()
{
    clearAllRows();

    addRow();

    m_fewerButton->setEnabled( false );
}

void FilterDialog::columnActivated( int index )
{
    for ( int i = 0; i < m_rows.count(); i++ ) {
        if ( sender() == m_rows[ i ].m_columnCombo ) {
            selectColumn( m_rows[ i ], m_columns[ index ] );
            selectCondition( m_rows[ i ], i, (ColumnCondition::Condition)-1 );
            break;
        }
    }
}

void FilterDialog::conditionActivated( int index )
{
    for ( int i = 0; i < m_rows.count(); i++ ) {
        if ( sender() == m_rows[ i ].m_conditionCombo ) {
            selectCondition( m_rows[ i ], i, m_rows[ i ].m_conditions[ index ] );
            break;
        }
    }
}

void FilterDialog::clearAllRows()
{
    for ( int i = 0; i < m_rows.count(); i++ ) {
        delete m_rows[ i ].m_columnCombo;
        delete m_rows[ i ].m_conditionCombo;
        delete m_rows[ i ].m_editor;
    }

    m_rows.clear();
}

void FilterDialog::createRow( const ColumnCondition& condition )
{
    int index = m_rows.count();

    QComboBox* columnCombo = new QComboBox( m_gridPanel );
    columnCombo->setMinimumWidth( 120 );
    m_gridLayout->addWidget( columnCombo, index, 0 );

    QComboBox* conditionCombo = new QComboBox( m_gridPanel );
    conditionCombo->setMinimumWidth( 170 );
    m_gridLayout->addWidget( conditionCombo, index, 1 );

    FilterRow row;
    row.m_columnCombo = columnCombo;
    row.m_conditionCombo = conditionCombo;
    row.m_editor = NULL;

    for ( int i = 0; i < m_columns.count(); i++ )
        columnCombo->addItem( TableModelsHelper::columnName( m_columns[ i ] ) );

    connect( columnCombo, SIGNAL( activated( int ) ), this, SLOT( columnActivated( int ) ) );
    connect( conditionCombo, SIGNAL( activated( int ) ), this, SLOT( conditionActivated( int ) ) );

    if ( isVisible() ) {
        columnCombo->show();
        conditionCombo->show();
    }

    selectColumn( row, condition.column() );
    selectCondition( row, index, condition.condition() );

    if ( row.m_editor )
        row.m_editor->setValue( condition.operand() );

    m_rows.append( row );
}

void FilterDialog::selectColumn( FilterRow& row, int column )
{
    int item = -1;
    for ( int i = 0; i < m_columns.count(); i++ ) {
        if ( m_columns[ i ] != -1 && m_columns[ i ] == column ) {
            item = i;
            break;
        }
    }

    if ( item == -1 ) {
        item = 0;
        column = m_columns[ 0 ];
    }

    row.m_columnCombo->setCurrentIndex( item );

    row.m_definitionInfo = ColumnConditionHelper::columnDefinition( column );
    row.m_conditions = ColumnConditionHelper::availableConditions( row.m_definitionInfo );

    row.m_conditionCombo->clear();

    for ( int i = 0; i < row.m_conditions.count(); i++ ) {
        QString name = ColumnCondition::name( row.m_conditions[ i ] );
        row.m_conditionCombo->addItem( name );
    }

    delete row.m_editor;
    row.m_editor = NULL;
}

void FilterDialog::selectCondition( FilterRow& row, int index, ColumnCondition::Condition condition )
{
    int item = -1;
    for ( int i = 0; i < row.m_conditions.count(); i++ ) {
        if ( row.m_conditions[ i ] == condition ) {
            item = i;
            break;
        }
    }

    if ( item == -1 ) {
        item = 0;
        condition = row.m_conditions[ 0 ];
    }

    row.m_conditionCombo->setCurrentIndex( item );

    QString value;
    if ( row.m_editor )
        value = row.m_editor->value();

    delete row.m_editor;

    row.m_editor = ColumnConditionHelper::createValueEditor( row.m_definitionInfo, condition,
        m_projectId, this, m_gridPanel );

    if ( row.m_editor ) {
        row.m_editor->setValue( value );

        if ( isVisible() )
            row.m_editor->widget()->show();

        m_gridLayout->addWidget( row.m_editor->widget(), index, 2 );
    }
}
