/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "connectioninfodialog.h"

#include <QLayout>
#include <QGroupBox>
#include <QPushButton>
#include <QDialogButtonBox>

#if defined( HAVE_OPENSSL )
#include <QSslCipher>
#endif

#include "sslcertificatesdialog.h"
#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "widgets/propertypanel.h"
#include "connectionmanager.h"

ConnectionInfoDialog::ConnectionInfoDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Connection Details" ) );

    QVBoxLayout* layout = new QVBoxLayout( this );

    QGroupBox* serverBox = new QGroupBox( tr( "Server Information" ), this );
    QVBoxLayout* serverLayout = new QVBoxLayout( serverBox );
    layout->addWidget( serverBox );

    m_serverPanel = new PropertyPanel( serverBox );
    m_serverPanel->setInnerMargin( 0 );

    m_serverPanel->addProperty( "url", tr( "URL:" ) );
    m_serverPanel->addProperty( "encrypted", tr( "Encrypted:" ) );
    m_serverPanel->addProperty( "name", tr( "Name:" ) );
    m_serverPanel->addProperty( "version", tr( "Version:" ) );
    m_serverPanel->addProperty( "notify", tr( "Notifications:" ) );

    serverLayout->addWidget( m_serverPanel );

    QHBoxLayout* buttonLayout = new QHBoxLayout();
    buttonLayout->addStretch();

    m_certificatesButton = new QPushButton( tr( "&View Certificate" ), serverBox );
    buttonLayout->addWidget( m_certificatesButton );

    serverLayout->addLayout( buttonLayout );

    connect( m_certificatesButton, SIGNAL( clicked() ), this, SLOT( viewCertificates() ) );

    QGroupBox* userBox = new QGroupBox( tr( "User Information" ), this );
    QVBoxLayout* userLayout = new QVBoxLayout( userBox );
    layout->addWidget( userBox );

    m_userPanel = new PropertyPanel( serverBox );
    m_userPanel->setInnerMargin( 0 );

    m_userPanel->addProperty( "login", tr( "Login:" ) );
    m_userPanel->addProperty( "name", tr( "Name:" ) );
    m_userPanel->addProperty( "access", tr( "Access:" ) );

    userLayout->addWidget( m_userPanel );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok, Qt::Horizontal, this );
    buttonBox->setCenterButtons( true );
    layout->addWidget( buttonBox );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );

    updateInformation();

    setFixedSize( sizeHint().expandedTo( QSize( 300, 100 ) ) );
}

ConnectionInfoDialog::~ConnectionInfoDialog()
{
}

void ConnectionInfoDialog::updateInformation()
{
    m_serverPanel->setValue( "url", commandManager->serverUrl().toString() );

    m_certificatesButton->setEnabled( commandManager->connectionMode() == CommandManager::HttpsMode );

#if defined( HAVE_OPENSSL )
    QSslCipher cipher = commandManager->sessionCipher();
    if ( !cipher.isNull() )
        m_serverPanel->setValue( "encrypted", tr( "Yes (%1, %2-bit)" )
            .arg( cipher.authenticationMethod() ).arg( cipher.usedBits() ) );
    else
#endif
        m_serverPanel->setValue( "encrypted", tr( "No" ) );

    m_serverPanel->setValue( "name", connectionManager->connectionInfo()->serverName() );

    QString version = connectionManager->connectionInfo()->serverVersion();
    m_serverPanel->setValue( "version", version.isEmpty() ? tr( "0.8.2 or older" ) : version );

    bool hasNotify = connectionManager->connectionInfo()->checkFeature( "notify" );
    m_serverPanel->setValue( "notify", hasNotify ? tr( "Enabled" ) : tr( "Not Available" ));

    int userId = connectionManager->connectionInfo()->userId();
    const UserRow* user = dataManager->users()->find( userId );

    m_userPanel->setValue( "login", user ? user->login() : QString() );
    m_userPanel->setValue( "name", user ? user->name() : QString() );

    Access access = connectionManager->connectionInfo()->access();
    m_userPanel->setValue( "access", ( access == AdminAccess ) ? tr( "Administrator" ) : tr( "Regular" ) );
}

void ConnectionInfoDialog::viewCertificates()
{
#if defined( HAVE_OPENSSL )
    SslCertificatesDialog dialog( this );
    dialog.setCertificates( commandManager->certificateChain() );
    dialog.exec();
#endif
}
