/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "commandmanager.h"

#include <QHttp>
#include <QRegExp>

#if defined( HAVE_OPENSSL )
#include <QSslSocket>
#include <QSslCipher>
#endif

#include "command.h"
#include "abstractbatch.h"
#include "formdatamessage.h"
#include "cookiejar.h"

CommandManager* commandManager = NULL;

CommandManager::CommandManager() :
#if defined( HAVE_OPENSSL )
    m_sslSocket( NULL ),
#endif
    m_mode( InvalidMode ),
    m_currentRequest( 0 ),
    m_currentBatch( NULL ),
    m_currentCommand( NULL ),
    m_error( NoError )
{
    m_http = new QHttp();

    connect( m_http, SIGNAL( dataSendProgress( int , int ) ),
        this, SLOT( dataSendProgress( int , int ) ) );
    connect( m_http, SIGNAL( dataReadProgress( int , int ) ),
        this, SLOT( dataReadProgress( int , int ) ) );

    connect( m_http, SIGNAL( readyRead( const QHttpResponseHeader& ) ),
        this, SLOT( readyRead( const QHttpResponseHeader& ) ) );

    connect( m_http, SIGNAL( requestFinished( int, bool ) ),
        this, SLOT( requestFinished( int, bool ) ) );

#if defined( HAVE_OPENSSL )
    connect( m_http, SIGNAL( sslErrors( const QList<QSslError>& ) ),
        this, SLOT( handleSslErrors( const QList<QSslError>& ) ) );
#endif

    m_cookieJar = new CookieJar();
}

CommandManager::~CommandManager()
{
    delete m_http;
    m_http = NULL;

    while ( !m_batches.isEmpty() ) {
        AbstractBatch* batch = m_batches.takeFirst();
        delete batch;
    }

#if defined( HAVE_OPENSSL )
    delete m_sslSocket;
    m_sslSocket = NULL;
#endif

    delete m_cookieJar;
    m_cookieJar = NULL;
}

void CommandManager::setServerUrl( const QUrl& url )
{
    m_url = url;
    m_cookieJar->clear();

    sendSetHostRequest();
}

void CommandManager::execute( AbstractBatch* batch )
{
    int pos = 0;
    for ( int i = 0; i < m_batches.count(); i++ ) {
        if ( m_batches.at( i )->priority() < batch->priority() )
            break;
        pos++;
    }
    m_batches.insert( pos, batch );

    checkPendingCommand();
}

void CommandManager::abort( AbstractBatch* batch )
{
    if ( batch == m_currentBatch ) {
        m_http->abort();
    } else {
        setError( Aborted );
        m_batches.removeAt( m_batches.indexOf( batch ) );
        batch->setCompleted( false );
        delete batch;
    }
}

void CommandManager::abortAll()
{
    m_http->abort();

    setError( Aborted );

    while ( !m_batches.isEmpty() ) {
        AbstractBatch* batch = m_batches.takeFirst();
        batch->setCompleted( false );
        delete batch;
    }
}

QString CommandManager::errorMessage( const QString& whatFailed )
{
    QString message;

    switch ( m_error ) {
        case Aborted:
            message = tr( "Operation aborted" );
            break;
        case InvalidUrl:
            message = tr( "Invalid server URL" );
            break;
        case UnknownUrlScheme:
            message = tr( "Unsupported URL scheme" );
            break;
        case ConnectionError:
            message = tr( "Connection failed" );
            break;
        case InvalidServer:
            message = tr( "Not a WebIssues server" );
            break;
        case InvalidVersion:
            message = tr( "Unsupported server version" );
            break;
        case WebIssuesError:
            message = whatFailed;
            break;
        case InvalidResponse:
            message = tr( "Invalid server response" );
            break;
        case HttpError:
            message = tr( "Server error" );
            break;
        default:
            break;
    }

    if ( m_error == CommandManager::WebIssuesError || m_error == CommandManager::HttpError )
        message += QString( " (%1 %2)" ).arg( m_errorCode ).arg( m_errorString );

    return message;
}

void CommandManager::checkPendingCommand()
{
    if ( m_currentBatch )
        return;

    while ( !m_batches.isEmpty() ) {
        AbstractBatch* batch = m_batches.first();

        Command* command = batch->fetchNext();
        if ( command ) {
            m_currentBatch = batch;
            m_currentCommand = command;
            sendCommandRequest( command );
            break;
        }

        m_batches.removeFirst();
        batch->setCompleted( true );
        delete batch;

        if ( m_currentBatch )
            break;
    }
}

void CommandManager::sendSetHostRequest()
{
    if ( !m_url.isValid() || m_url.scheme().isEmpty() || m_url.host().isEmpty() ) {
        m_http->setHost( QString(), 0 );
        m_mode = InvalidMode;
        return;
    }

    QString scheme = m_url.scheme().toLower();
    QString host = m_url.host();
    int port = m_url.port();

    if ( scheme == QLatin1String( "http" ) ) {
        m_http->setSocket( NULL );
        m_http->setHost( host, ( port < 0 ) ? 80 : port );
        m_mode = HttpMode;
        return;
    }

#if defined( HAVE_OPENSSL )
    if ( scheme == QLatin1String( "https" ) && QSslSocket::supportsSsl() ) {
        if ( !m_sslSocket )
            m_sslSocket = new QSslSocket();
        m_http->setSocket( m_sslSocket );
        m_http->setHost( host, QHttp::ConnectionModeHttps, ( port < 0 ) ? 443 : port );
        m_mode = HttpsMode;
        return;
    }
#endif

    m_http->setHost( QString(), 0 );
    m_mode = UnknownMode;
}

void CommandManager::sendCommandRequest( Command* command )
{
    QHttpRequestHeader header;

    QString path = m_url.path();
    if ( path.isEmpty() )
        path = "/";

    header.setRequest( "POST", path );
    header.setValue( "Host", m_url.host() );

    m_cookieJar->insertCookies( header );
    
    QString commandLine = command->keyword();

    for ( int i = 0; i < command->args().count(); i++ ) {
        commandLine += QLatin1Char( ' ' );
        QVariant arg = command->args().at( i );
        if ( arg.type() == QVariant::String )
            commandLine += quoteString( arg.toString() );
        else
            commandLine += arg.toString();
    }

    FormDataMessage message;

    message.addField( "command", commandLine.toUtf8() );

    if ( !command->attachment().isEmpty() )
        message.addAttachment( "file", "file", command->attachment() );

    message.finish();

    header.setContentType( message.contentType() );

    m_currentRequest = m_http->request( header, message.body() );
}

void CommandManager::dataSendProgress( int done, int total )
{
    if ( m_currentCommand )
        m_currentCommand->setSendProgress( done, total );
}

void CommandManager::dataReadProgress( int done, int total )
{
    if ( m_currentCommand )
        m_currentCommand->setReadProgress( done, total );
}

void CommandManager::readyRead( const QHttpResponseHeader& response )
{
    if ( m_currentCommand && m_currentCommand->acceptBinaryResponse()
        && response.contentType() == QLatin1String( "application/octet-stream" ) ) {
        int length;
        char buffer[ 8192 ];
        while ( ( length = m_http->read( buffer, 8192 ) ) > 0 )
            m_currentCommand->setBinaryBlock( buffer, length );
    }
}

void CommandManager::requestFinished( int id, bool error )
{
    if ( !m_currentBatch || id != m_currentRequest )
        return;

    m_currentRequest = 0;

    QHttpResponseHeader response = m_http->lastResponse();

    if ( !error && ( response.statusCode() == 301 || response.statusCode() == 302 ) ) {
        m_url = m_url.resolved( response.value( "Location" ) );
        m_cookieJar->clear();
        sendSetHostRequest();
        sendCommandRequest( m_currentCommand );
        return;
    }

    bool successful = handleCommandResponse( response );

    if ( !successful ) {
        m_batches.removeAt( m_batches.indexOf( m_currentBatch ) );
        m_currentBatch->setCompleted( false );
        delete m_currentBatch;
    }

    m_currentCommand->deleteLater();

    m_currentBatch = NULL;
    m_currentCommand = NULL;

    QMetaObject::invokeMethod( this, "checkPendingCommand", Qt::QueuedConnection );
}

bool CommandManager::handleCommandResponse( const QHttpResponseHeader& response )
{
    if ( m_http->error() != QHttp::NoError ) {
        if ( m_mode == InvalidMode )
            setError( InvalidUrl );
        else if ( m_mode == UnknownMode )
            setError( UnknownUrlScheme );
        else if ( m_http->error() == QHttp::Aborted )
            setError( Aborted );
        else
            setError( ConnectionError, m_http->error(), m_http->errorString() );
        return false;
    }

    if ( !response.isValid() ) {
        setError( InvalidResponse );
        return false;
    }

    if ( response.statusCode() != 200 ) {
        setError( HttpError, response.statusCode(), response.reasonPhrase() );
        return false;
    }

    m_cookieJar->extractCookies( response );

    m_protocolVersion = response.value( "X-WebIssues-Version" );
    m_serverVersion = response.value( "X-WebIssues-Server" );

    if ( m_protocolVersion.isEmpty() ) {
        setError( InvalidServer );
        return false;
    }

    if ( m_protocolVersion != QLatin1String( "0.8" ) ) {
        setError( InvalidVersion, 0, m_protocolVersion );
        return false;
    }

    if ( response.contentType() == QLatin1String( "text/plain" ) ) {
        QByteArray body = m_http->readAll();

        QString string = QString::fromUtf8( body.data(), body.size() );

        Reply reply;
        if ( !parseReply( string, reply ) ) {
            setError( InvalidResponse );
            return false;
        }

        return handleCommandReply( reply );
    }

    if ( response.contentType() == QLatin1String( "application/octet-stream" ) ) {
        if ( !m_currentCommand->acceptBinaryResponse() ) {
            setError( InvalidResponse );
            return false;
        }

        setError( NoError );
        return true;
    }

    setError( InvalidResponse );
    return false;
}

bool CommandManager::handleCommandReply( const Reply& reply )
{
    bool isNull = false;

    if ( reply.lines().count() == 1 ) {
        ReplyLine line = reply.lines().at( 0 );
        QString signature = makeSignature( line );

        if ( signature == QLatin1String( "ERROR is" ) ) {
            setError( WebIssuesError, line.argInt( 0 ), line.argString( 1 ) );
            return false;
        }

        if ( signature == QLatin1String( "NULL" ) )
            isNull = true;
    }

    bool isValid = isNull ? m_currentCommand->acceptNullReply() : validateReply( reply );

    if ( !isValid ) {
        setError( InvalidResponse );
        return false;
    }

    if ( !isNull )
        m_currentCommand->setCommandReply( reply );
    else if ( m_currentCommand->reportNullReply() )
        m_currentCommand->setCommandReply( Reply() );

    setError( NoError );
    return true;
}

bool CommandManager::parseReply( const QString& string, Reply& reply )
{
    QStringList lines = string.split( "\r\n", QString::SkipEmptyParts );

    QRegExp lineRegExp( "([A-Z]+)(?: ('(?:\\\\['\\\\n]|[^'\\\\])*'|-?[0-9]+))*" );
    QRegExp argumentRegExp( "('(?:\\\\['\\\\n]|[^'\\\\])*'|-?[0-9]+)" );

    for ( QStringList::iterator it = lines.begin(); it != lines.end(); ++it ) {
        if ( !lineRegExp.exactMatch( *it ) )
            return false;

        ReplyLine line;
        line.setKeyword( lineRegExp.cap( 1 ) );

        int pos = 0;
        while ( ( pos = argumentRegExp.indexIn( *it, pos ) ) >= 0 ) {
            QString argument = argumentRegExp.cap( 0 );
            if ( argument[ 0 ] == QLatin1Char( '\'' ) )
                line.addArg( unquoteString( argument ) );
            else
                line.addArg( argument.toInt() );
            pos += argumentRegExp.matchedLength();
        }

        reply.addLine( line );
    }

    return true;
}

bool CommandManager::validateReply( const Reply& reply )
{
    int line = 0;
    int rule = 0;

    while ( line < reply.lines().count() && rule < m_currentCommand->rules().count() ) {
        ReplyRule replyRule = m_currentCommand->rules().at( rule );
        if ( makeSignature( reply.lines().at( line ) ) == replyRule.signature() ) {
            line++;
            if ( replyRule.multiplicity() == ReplyRule::One )
                rule++;
        } else {
            if ( replyRule.multiplicity() == ReplyRule::One )
                return false;
            rule++;
        }
    }

    while ( rule < m_currentCommand->rules().count() ) {
        if ( m_currentCommand->rules().at( rule ).multiplicity() == ReplyRule::One )
            return false;
        rule++;
    }

    if ( line < reply.lines().count() )
        return false;

    return true;
}

QString CommandManager::makeSignature( const ReplyLine& line )
{
    if ( line.args().isEmpty() )
        return line.keyword();

    QString signature = line.keyword() + ' ';

    for ( int i = 0; i < line.args().count(); i++ ) {
        switch ( line.args().at( i ).type() ) {
            case QVariant::Int:
                signature += QLatin1Char( 'i' );
                break;
            case QVariant::String:
                signature += QLatin1Char( 's' );
                break;
            default:
                signature += QLatin1Char( '?' );
                break;
        }
    }

    return signature;
}

QString CommandManager::quoteString( const QString& string )
{
    QString result = "\'";
    int length = string.length();
    for ( int i = 0; i < length; i++ ) {
        QChar ch = string[ i ];
        if  ( ch == QLatin1Char( '\\' ) || ch == QLatin1Char( '\'' ) || ch == QLatin1Char( '\n' ) ) {
            result += QLatin1Char( '\\' );
            if ( ch == QLatin1Char( '\n' ) )
                ch = QLatin1Char( 'n' );
        }
        result += ch;
    }
    result += QLatin1Char( '\'' );
    return result;
}

QString CommandManager::unquoteString( const QString& string )
{
    QString result;
    int length = string.length();
    for ( int i = 1; i < length - 1; i++ ) {
        QChar ch = string[ i ];
        if ( ch == QLatin1Char( '\\' ) ) {
            ch = string[ ++i ];
            if ( ch == QLatin1Char( 'n' ) )
                ch = QLatin1Char( '\n' );
        }
        result += ch;
    }
    return result;
}

void CommandManager::setError( Error error, int code /*= 0*/, const QString& string /*= QString()*/ )
{
    m_error = error;
    m_errorCode = code;
    m_errorString = string;
}

#if defined( HAVE_OPENSSL )

QList<QSslCertificate> CommandManager::certificateChain()
{
    if ( m_mode != HttpsMode )
        return QList<QSslCertificate>();
    return m_sslSocket->peerCertificateChain();
}

QSslCipher CommandManager::sessionCipher()
{
    if ( m_mode != HttpsMode )
        return QSslCipher();
    return m_sslSocket->sessionCipher();
}

void CommandManager::setAcceptedDigests( const QList<QByteArray>& digests )
{
    m_acceptedDigests = digests;
}

void CommandManager::ignoreSslErrors()
{
    QSslCertificate certificate = m_sslSocket->peerCertificate();
    if ( !certificate.isNull() ) {
        QByteArray digest = certificate.digest( QCryptographicHash::Sha1 );
        if ( !m_acceptedDigests.contains( digest ) )
            m_acceptedDigests.append( digest );
    }

    m_http->ignoreSslErrors();
}

void CommandManager::handleSslErrors( const QList<QSslError>& errors )
{
    QSslCertificate certificate = m_sslSocket->peerCertificate();
    if ( certificate.isNull() )
        return;

    QByteArray digest = certificate.digest( QCryptographicHash::Sha1 );

    if ( m_acceptedDigests.contains( digest ) )
        m_http->ignoreSslErrors();
    else
        emit sslErrors( errors );
}

#endif // defined( HAVE_OPENSSL )
