/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "definitioninfo.h"

#include "definitionbuilder.h"
#include "definitionparser.h"

DefinitionInfo::DefinitionInfo() :
    m_type( Invalid ),
    m_required( false )
{
}

DefinitionInfo::~DefinitionInfo()
{
}

void DefinitionInfo::setMetadata( const QString& key, const QVariant& value )
{
    m_metadata.insert( key, value );
}

QVariant DefinitionInfo::metadata( const QString& key ) const
{
    return m_metadata.value( key );
}

QString DefinitionInfo::toString() const
{
    if ( !isValid() )
        return QString();

    DefinitionBuilder builder;

    switch ( m_type ) {
        case Text:
            builder.setType( "TEXT" );
            break;
        case Enum:
            builder.setType( "ENUM" );
            break;
        case Numeric:
            builder.setType( "NUMERIC" );
            break;
        case DateTime:
            builder.setType( "DATETIME" );
            break;
        case User:
            builder.setType( "USER" );
            break;
        default:
            return QString();
    };

    if ( m_required )
        builder.addMetadata( "required", 1 );
    if ( !m_default.isEmpty() )
        builder.addMetadata( "default", m_default );

    for ( QMap<QString, QVariant>::const_iterator it = m_metadata.constBegin(); it != m_metadata.constEnd(); ++it )
        builder.addMetadata( it.key(), it.value() );

    return builder.result();
}

DefinitionInfo DefinitionInfo::fromString( const QString& text )
{
    DefinitionInfo info;

    DefinitionParser parser;

    if ( !parser.parse( text ) )
        return info;

    if ( parser.type() == QLatin1String( "TEXT" ) )
        info.m_type = Text;
    else if ( parser.type() == QLatin1String( "ENUM" ) )
        info.m_type = Enum;
    else if ( parser.type() == QLatin1String( "NUMERIC" ) )
        info.m_type = Numeric;
    else if ( parser.type() == QLatin1String( "DATETIME" ) )
        info.m_type = DateTime;
    else if ( parser.type() == QLatin1String( "USER" ) )
        info.m_type = User;
    else
        return info;

    QMap<QString, QVariant> metadata = parser.metadata();

    QVariant value = metadata.value( "default" );
    if ( value.isValid() ) {
        info.m_default = value.toString();
        metadata.remove( "default" );
    }

    value = metadata.value( "required" );
    if ( value.isValid() ) {
        info.m_required = value.toBool();
        metadata.remove( "required" );
    }

    info.m_metadata = metadata;

    return info;
}
