/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This code is partially based on qprintpreviewwidget.cpp and
* qpaintengine_preview.cpp from Qt 4.4, licensed unsed the GPL license.
**************************************************************************/

#include "previewwidget.h"

#if ( QT_VERSION < 0x040400 )

#include <QPaintEngine>
#include <QPrintEngine>
#include <QPainter>
#include <QPicture>
#include <QGraphicsScene>
#include <QGraphicsItem>
#include <QScrollBar>
#include <QStyleOption>

#include <math.h>

class PreviewPaintEngine : public QPaintEngine, public QPrintEngine
{
public:
    PreviewPaintEngine( QPrintEngine* printProxy, QPaintEngine* paintProxy );
    ~PreviewPaintEngine();

public:
    QList<const QPicture*> detachPages();

public: // QPaintEngine overrides
    bool begin( QPaintDevice* device );
    bool end();

    void updateState( const QPaintEngineState& state );

    void drawPath( const QPainterPath& path );
    void drawPolygon( const QPointF* points, int pointCount, PolygonDrawMode mode );
    void drawTextItem( const QPointF& point, const QTextItem& item );

    void drawPixmap( const QRectF& rect, const QPixmap& pixmap, const QRectF& sourceRect );
    void drawTiledPixmap( const QRectF& rect, const QPixmap& pixmap, const QPointF& point );

    QPaintEngine::Type type() const { return Picture; }

public: // QPrintEngine overrides
    bool newPage();
    bool abort();

    void setProperty( PrintEnginePropertyKey key, const QVariant& value );
    QVariant property( PrintEnginePropertyKey key ) const;

    int metric( QPaintDevice::PaintDeviceMetric id ) const;

    QPrinter::PrinterState printerState() const { return m_state; }

private:
    QList<const QPicture*> m_pages;

    QPaintEngine* m_engine;
    QPainter* m_painter;

    QPrinter::PrinterState m_state;

    QPrintEngine* m_printProxy;
    QPaintEngine* m_paintProxy;
};

PreviewPaintEngine::PreviewPaintEngine( QPrintEngine* printProxy, QPaintEngine* paintProxy ) :
    QPaintEngine( AllFeatures ),
    m_engine( NULL ),
    m_painter( NULL ),
    m_state( QPrinter::Idle ),
    m_printProxy( printProxy ),
    m_paintProxy( paintProxy )
{
}

PreviewPaintEngine::~PreviewPaintEngine()
{
    qDeleteAll( m_pages );
}

QList<const QPicture*> PreviewPaintEngine::detachPages()
{
    QList<const QPicture*> temp = m_pages;
    m_pages.clear();
    return temp;
}

bool PreviewPaintEngine::begin( QPaintDevice* device )
{
    qDeleteAll( m_pages );
    m_pages.clear();

    QPicture* page = new QPicture();
    m_pages.append( page );

    m_painter = new QPainter( page );
    m_engine = m_painter->paintEngine();

    m_state = QPrinter::Active;

    return true;
}

bool PreviewPaintEngine::end()
{
    delete m_painter;
    m_painter = NULL;
    m_engine = NULL;

    m_state = QPrinter::Idle;

    return true;
}

void PreviewPaintEngine::updateState( const QPaintEngineState& state )
{
    m_engine->updateState( state );
}

void PreviewPaintEngine::drawPath( const QPainterPath& path )
{
    m_engine->drawPath( path );
}

void PreviewPaintEngine::drawPolygon( const QPointF* points, int pointCount, PolygonDrawMode mode )
{
    m_engine->drawPolygon( points, pointCount, mode );
}

void PreviewPaintEngine::drawTextItem( const QPointF& point, const QTextItem& item )
{
    m_engine->drawTextItem( point, item );
}

void PreviewPaintEngine::drawPixmap( const QRectF& rect, const QPixmap& pixmap, const QRectF& sourceRect )
{
    m_engine->drawPixmap( rect, pixmap, sourceRect );
}

void PreviewPaintEngine::drawTiledPixmap( const QRectF& rect, const QPixmap& pixmap, const QPointF& point )
{
    m_engine->drawTiledPixmap( rect, pixmap, point );
}

bool PreviewPaintEngine::newPage()
{
    QPicture* page = new QPicture();
    m_pages.append( page );

    QPainter* painter = new QPainter( page );
    QPaintEngine* engine = painter->paintEngine();

    // TODO: copy state from old painter to new painter

    delete m_painter;
    m_painter = painter;
    m_engine = engine;

    return true;
}

bool PreviewPaintEngine::abort()
{
    end();

    qDeleteAll( m_pages );
    m_state = QPrinter::Aborted;

    return true;
}

void PreviewPaintEngine::setProperty( PrintEnginePropertyKey key, const QVariant& value )
{
    m_printProxy->setProperty( key, value );
}

QVariant PreviewPaintEngine::property( PrintEnginePropertyKey key ) const
{
    return m_printProxy->property( key );
}

int PreviewPaintEngine::metric( QPaintDevice::PaintDeviceMetric id ) const
{
    return m_printProxy->metric( id );
}

PreviewPrinter::PreviewPrinter() : QPrinter( QPrinter::HighResolution )
{
    m_printer = new QPrinter( QPrinter::HighResolution );
    m_engine = new PreviewPaintEngine( m_printer->printEngine(), m_printer->paintEngine() );
    setPreviewMode( false );
}

PreviewPrinter::~PreviewPrinter()
{
    delete m_printer;
    delete m_engine;
}

void PreviewPrinter::setPreviewMode( bool preview )
{
    if ( preview )
        setEngines( m_engine, m_engine );
    else
        setEngines( m_printer->printEngine(), m_printer->paintEngine() );
}

class PreviewPageItem : public QGraphicsItem
{
public:
    PreviewPageItem( int pageNumber, const QPicture* picture, QSize paperSize, QRect pageRect ) :
        m_pageNumber( pageNumber ),
        m_picture( picture ),
        m_paperSize( paperSize ),
        m_pageRect( pageRect )
    {
        qreal border = qMax( paperSize.height(), paperSize.width() ) / 25.0;
        m_boundingRect = QRectF( QPointF( -border, -border ), QSizeF( paperSize ) + QSizeF( 2 * border, 2 * border ) );
    }

public:
    QRectF boundingRect() const { return m_boundingRect; }
    int pageNumber() const { return m_pageNumber; }

public: // overrides
    void paint( QPainter* painter, const QStyleOptionGraphicsItem* item, QWidget* widget );

private:
    int m_pageNumber;
    const QPicture* m_picture;
    QSize m_paperSize;
    QRect m_pageRect;
    QRectF m_boundingRect;
};

void PreviewPageItem::paint( QPainter* painter, const QStyleOptionGraphicsItem* option, QWidget* widget )
{
    QRectF paperRect( 0,0, m_paperSize.width(), m_paperSize.height() );

    painter->setClipRect( paperRect & option->exposedRect );
    painter->fillRect( paperRect, Qt::white );

    if ( m_picture )
        painter->drawPicture( m_pageRect.topLeft(), *m_picture );

    painter->setClipRect( option->exposedRect );

    qreal width = paperRect.width() / 100;

    QRectF right( paperRect.topRight() + QPointF(0, width),
                    paperRect.bottomRight() + QPointF( width, 0 ) );
    QLinearGradient gradRight( right.topLeft(), right.topRight() );
    gradRight.setColorAt( 0.0, QColor( 0, 0, 0, 255 ) );
    gradRight.setColorAt( 1.0, QColor( 0, 0, 0, 0 ) );
    painter->fillRect( right, gradRight );

    QRectF bottom( paperRect.bottomLeft() + QPointF( width, 0 ),
                    paperRect.bottomRight() + QPointF( 0, width ) );
    QLinearGradient gradBottom( bottom.topLeft(), bottom.bottomLeft() );
    gradBottom.setColorAt( 0.0, QColor( 0, 0, 0, 255 ) );
    gradBottom.setColorAt( 1.0, QColor( 0, 0, 0, 0 ) );
    painter->fillRect( bottom, gradBottom );

    QRectF corner( paperRect.bottomRight(), paperRect.bottomRight() + QPointF( width, width ) );
    QRadialGradient gradCorner( corner.topLeft(), width, corner.topLeft() );
    gradCorner.setColorAt( 0.0, QColor( 0, 0, 0, 255 ) );
    gradCorner.setColorAt( 1.0, QColor( 0, 0, 0, 0 ) );
    painter->fillRect( corner, gradCorner );
}

PreviewWidget::PreviewWidget( PreviewPrinter* printer, QWidget* parent ) : QGraphicsView( parent ),
    m_printer( printer ),
    m_viewMode( SinglePageView ),
    m_zoomMode( FitInView ),
    m_currentPage( 1 ),
    m_initialized( false ),
    m_fitting( true )
{
    setInteractive( false );
    setDragMode( QGraphicsView::ScrollHandDrag );
    setViewportUpdateMode( QGraphicsView::SmartViewportUpdate );
    connect( verticalScrollBar(), SIGNAL( valueChanged(int) ), this, SLOT( updateCurrentPage() ) );

    m_scene = new QGraphicsScene( this );
    m_scene->setBackgroundBrush( Qt::gray );
    setScene( m_scene );
}

PreviewWidget::~PreviewWidget()
{
}

QPrinter::Orientation PreviewWidget::orientation() const
{
    return m_printer->orientation();
}

int PreviewWidget::numPages() const
{
    return m_pages.count();
}

void PreviewWidget::print()
{
    emit paintRequested( m_printer );
}

void PreviewWidget::zoomIn( qreal zoom /*= 1.1*/ )
{
    m_fitting = false;
    m_zoomMode = CustomZoom;
    m_zoomFactor *= zoom;
    scale( zoom, zoom );
}

void PreviewWidget::zoomOut( qreal zoom /*= 1.1*/ )
{
    zoomIn( 1.0 / zoom );
}

void PreviewWidget::setZoomFactor( qreal zoomFactor )
{
    m_fitting = false;
    m_zoomMode = CustomZoom;
    m_zoomFactor = zoomFactor;

    qreal zoom = zoomFactor * (qreal)logicalDpiY() / (qreal)m_printer->logicalDpiY();

    resetTransform();
    scale( zoom, zoom );
}

void PreviewWidget::setOrientation( QPrinter::Orientation orientation )
{
    m_printer->setOrientation( orientation );
    generatePreview();
}

void PreviewWidget::setViewMode( ViewMode viewMode )
{
    m_viewMode = viewMode;
    layoutPages();

    if ( viewMode == AllPagesView ) {
        QGraphicsView::fitInView( m_scene->itemsBoundingRect(), Qt::KeepAspectRatio );
        m_fitting = false;
        m_zoomMode = CustomZoom;
        m_zoomFactor = transform().m11() * (qreal)m_printer->logicalDpiY() / (qreal)logicalDpiY();
        emit previewChanged();
    } else {
        m_fitting = true;
        fit();
    }
}

void PreviewWidget::setZoomMode( ZoomMode zoomMode )
{
    m_zoomMode = zoomMode;

    if ( zoomMode == FitInView || zoomMode == FitToWidth ) {
        m_fitting = true;
        fit( true );
    } else {
        m_fitting = false;
    }
}

void PreviewWidget::setCurrentPage( int pageNumber )
{
    if ( pageNumber < 1 || pageNumber > m_pages.count() )
        return;

    int lastPage = m_currentPage;
    m_currentPage = pageNumber;

    if ( lastPage != m_currentPage && lastPage > 0 && lastPage <= m_pages.count() ) {
        if ( m_zoomMode != FitInView ) {
            QPointF point = transform().map( m_pages.at( m_currentPage - 1 )->pos() );
            verticalScrollBar()->setValue( (int)point.y() - 10 );
            horizontalScrollBar()->setValue( (int)point.x() - 10 );
        } else {
            centerOn( m_pages.at( m_currentPage - 1 ) );
        }
    }
}

void PreviewWidget::fitToWidth()
{
    setZoomMode( FitToWidth );
}

void PreviewWidget::fitInView()
{
    setZoomMode( FitInView );
}

void PreviewWidget::setLandscapeOrientation()
{
    setOrientation( QPrinter::Landscape );
}

void PreviewWidget::setPortraitOrientation()
{
    setOrientation( QPrinter::Portrait );
}

void PreviewWidget::setSinglePageViewMode()
{
    setViewMode( SinglePageView );
}

void PreviewWidget::setFacingPagesViewMode()
{
    setViewMode( FacingPagesView );
}

void PreviewWidget::setAllPagesViewMode()
{
    setViewMode( AllPagesView );
}

void PreviewWidget::updatePreview()
{
    m_initialized = true;
    generatePreview();
    updateGeometry();
}

void PreviewWidget::setVisible( bool visible )
{
    if ( visible && !m_initialized )
        updatePreview();
    QGraphicsView::setVisible( visible );
}

void PreviewWidget::resizeEvent( QResizeEvent* e )
{
    QGraphicsView::resizeEvent( e );
    fit();
}

void PreviewWidget::showEvent( QShowEvent* e )
{
    QGraphicsView::showEvent( e );
    fit();
}

void PreviewWidget::fit( bool doFitting /*= false*/ )
{
    if ( m_currentPage < 1 || m_currentPage > m_pages.count() )
        return;

    if ( !doFitting && !m_fitting )
        return;

    if ( doFitting && m_fitting) {
        QRect viewRect = viewport()->rect();
        if ( m_zoomMode == FitInView ) {
            QList<QGraphicsItem*> containedItems = items( viewRect, Qt::ContainsItemBoundingRect );
            foreach( QGraphicsItem* item, containedItems ) {
                PreviewPageItem* page = static_cast<PreviewPageItem*>( item );
                if ( page->pageNumber() == m_currentPage )
                    return;
            }
        }

        m_currentPage = calcCurrentPage();
    }

    QRectF target = m_pages.at( m_currentPage - 1 )->sceneBoundingRect();

    if ( m_viewMode == FacingPagesView ) {
        if ( m_currentPage % 2 )
            target.setLeft( target.left() - target.width() );
        else
            target.setRight( target.right() + target.width() );
    } else if ( m_viewMode == AllPagesView ) {
        target = m_scene->itemsBoundingRect();
    }

    if ( m_zoomMode == FitToWidth ) {
        QTransform transform;
        qreal scale = viewport()->width() / target.width();
        transform.scale( scale, scale );
        setTransform( transform );
        if ( doFitting && m_fitting ) {
            QRectF viewSceneRect = viewportTransform().mapRect( viewport()->rect() );
            viewSceneRect.moveTop( target.top() );
            ensureVisible( viewSceneRect );
        }
    } else {
        QGraphicsView::fitInView( target, Qt::KeepAspectRatio );
        if ( m_zoomMode == FitInView ) {
            int step = qRound( matrix().mapRect( target ).height() );
            verticalScrollBar()->setSingleStep( step );
            verticalScrollBar()->setPageStep( step );
        }
    }

    m_zoomFactor = transform().m11() * (qreal)m_printer->logicalDpiY() / (qreal)logicalDpiY();

    emit previewChanged();
}

void PreviewWidget::updateCurrentPage()
{
    if ( m_viewMode == AllPagesView )
        return;

    int page = calcCurrentPage();
    if ( page != m_currentPage ) {
        m_currentPage = page;
        emit previewChanged();
    }
}

void PreviewWidget::populateScene()
{
    for ( int i = 0; i < m_pages.count(); i++ )
        m_scene->removeItem( m_pages.at( i ) );

    qDeleteAll( m_pages );
    m_pages.clear();

    QSize paperSize = m_printer->paperRect().size();
    QRect pageRect = m_printer->pageRect();

    for ( int i = 0; i < m_pictures.count(); i++ ) {
        PreviewPageItem* item = new PreviewPageItem( i + 1, m_pictures.at( i ), paperSize, pageRect );
        m_pages.append( item );
        m_scene->addItem( item );
    }
}

void PreviewWidget::layoutPages()
{
    int numPages = m_pages.count();
    if ( numPages < 1 )
        return;

    int numPagePlaces = numPages;
    int columns = 1;
    if ( m_viewMode == AllPagesView ) {
        if ( m_printer->orientation() == QPrinter::Portrait)
            columns = static_cast<int>( ceil( sqrt( (float)numPages ) ) );
        else
            columns = static_cast<int>( floor( sqrt( (float)numPages ) ) );
        columns += columns % 2;
    } else if ( m_viewMode == FacingPagesView ) {
        columns = 2;
        numPagePlaces += 1;
    }
    int rows = static_cast<int>( ceil( (double)numPagePlaces / columns ) );

    qreal itemWidth = m_pages.first()->boundingRect().width();
    qreal itemHeight = m_pages.first()->boundingRect().height();

    int pageNum = 1;
    for ( int i = 0; i < rows && pageNum <= numPages; i++ ) {
        for ( int j = 0; j < columns && pageNum <= numPages; j++ ) {
            if ( !i && !j && m_viewMode == FacingPagesView )
                continue;
            m_pages.at( pageNum - 1 )->setPos( QPointF( j * itemWidth, i * itemHeight ) );
            pageNum++;
        }
    }

    m_scene->setSceneRect( m_scene->itemsBoundingRect() );
}

void PreviewWidget::generatePreview()
{
    m_printer->setPreviewMode( true );

    emit paintRequested( m_printer );

    m_printer->setPreviewMode( false );

    m_pictures = m_printer->previewEngine()->detachPages();

    populateScene();
    layoutPages();

    m_currentPage = qBound( 1, m_currentPage, m_pages.count() );

    if ( m_fitting )
        fit();

    emit previewChanged();
}

int PreviewWidget::calcCurrentPage()
{
    int maxArea = 0;
    int newPage = m_currentPage;

    QRect viewRect = viewport()->rect();
    QList<QGraphicsItem*> items = QGraphicsView::items( viewRect );

    for ( int i = 0; i < items.count(); i++ ) {
        PreviewPageItem* page = static_cast<PreviewPageItem*>( items.at( i ) );
        QRect overlap = mapFromScene( page->sceneBoundingRect() ).boundingRect() & viewRect;
        int area = overlap.width() * overlap.height();
        if ( area > maxArea ) {
            maxArea = area;
            newPage = page->pageNumber();
        } else if ( area == maxArea && page->pageNumber() < newPage ) {
            newPage = page->pageNumber();
        }
    }
    return newPage;
}

#else // ( QT_VERSION < 0x040400 )

PreviewPrinter::PreviewPrinter() : QPrinter( QPrinter::HighResolution )
{
}

PreviewPrinter::~PreviewPrinter()
{
}

PreviewWidget::PreviewWidget( PreviewPrinter* printer, QWidget* parent ) : QPrintPreviewWidget( printer, parent )
{
}

PreviewWidget::~PreviewWidget()
{
}

#endif // ( QT_VERSION < 0x040400 )
