/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "usersview.h"

#include <QTreeView>
#include <QAction>
#include <QMenu>

#include "models/tablemodels.h"
#include "models/treeviewsettings.h"
#include "models/treeviewhelper.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/updatebatch.h"
#include "dialogs/userdialogs.h"
#include "xmlui/builder.h"
#include "connectionmanager.h"
#include "iconloader.h"

UsersView::UsersView( QObject* parent, QWidget* parentWidget ) : View( parent )
{
    m_systemAdmin = connectionManager->connectionInfo()->access() == AdminAccess;

    QAction* action;

    action = new QAction( IconLoader::icon( "file-reload" ), tr( "&Update Users" ), this );
    action->setShortcut( QKeySequence::Refresh );
    connect( action, SIGNAL( triggered() ), this, SLOT( updateUsers() ) );
    setAction( "updateUsers", action );

    if ( m_systemAdmin ) {
        action = new QAction( IconLoader::icon( "user-new" ), tr( "&Add User..." ), this );
        action->setShortcut( QKeySequence::New );
        connect( action, SIGNAL( triggered() ), this, SLOT( addUser() ) );
        setAction( "addUser", action );

        action = new QAction( IconLoader::icon( "edit-rename" ), tr( "&Rename..." ), this );
        action->setShortcut( tr( "F2" ) );
        connect( action, SIGNAL( triggered() ), this, SLOT( editRename() ) );
        setAction( "editRename", action );

        action = new QAction( IconLoader::icon( "edit-access" ), tr( "&Change Access..." ), this );
        connect( action, SIGNAL( triggered() ), this, SLOT( changeAccess() ) );
        setAction( "changeAccess", action );

        action = new QAction( IconLoader::icon( "edit-password" ), tr( "Change &Password..." ), this );
        connect( action, SIGNAL( triggered() ), this, SLOT( changePassword() ) );
        setAction( "changePassword", action );
    }

    setTitle( "menuMain", tr( "&Users" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );

    loadXmlUiFile( ":/resources/usersview.xml" );

    m_list = new QTreeView( parentWidget );
    m_list->setSortingEnabled( true );
    m_list->setRootIsDecorated( false );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( contextMenu( const QPoint& ) ) );
    connect( m_list, SIGNAL( activated( const QModelIndex& ) ),
        this, SLOT( activated( const QModelIndex& ) ) );

    setMainWidget( m_list );

    setViewerSizeHint( QSize( 400, 400 ) );

    setAccess( NormalAccess );
}

UsersView::~UsersView()
{
    TreeViewSettings settings;
    settings.openUsersList();

    settings.saveColumnWidths( TreeViewHelper::readColumnWidths( m_list ) );
}

void UsersView::initialUpdate()
{
    m_model = new RDB::TableItemModel( this );
    m_model->setRootTableModel( new UsersTableModel( m_model ), dataManager->users()->index() );

    TreeViewSettings settings;
    settings.openUsersList();

    m_model->setColumns( settings.loadColumns() );

    m_list->setModel( m_model );

    m_list->sortByColumn( 0, Qt::AscendingOrder );

    TreeViewHelper::applyColumnWidths( m_list, settings.loadColumnWidths() );

    setCaption( tr( "User Accounts" ) );

    connect( m_list->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
        this, SLOT( updateActions() ) );
    connect( m_model, SIGNAL( layoutChanged() ), this, SLOT( updateActions() ) );

    updateActions();
}

void UsersView::updateActions()
{
    m_selectedUserId = 0;

    QModelIndex index = selectedIndex();
    if ( index.isValid() )
        m_selectedUserId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();

    if ( m_systemAdmin ) {
        action( "editRename" )->setEnabled( m_selectedUserId != 0 );
        action( "changeAccess" )->setEnabled( m_selectedUserId != 0 && m_selectedUserId != 1 );
        action( "changePassword" )->setEnabled( m_selectedUserId != 0 );
    }
}

void UsersView::updateUsers()
{
    if ( !isUpdating() ) {
        UpdateBatch* batch = new UpdateBatch();
        batch->updateUsers();

        executeUpdate( batch );
    }
}

void UsersView::addUser()
{
    if ( m_systemAdmin ) {
        AddUserDialog dialog( mainWidget() );
        dialog.exec();
    }
}

void UsersView::editRename()
{
    if ( m_selectedUserId != 0 && m_systemAdmin ) {
        RenameUserDialog dialog( m_selectedUserId, mainWidget() );
        dialog.exec();
    }
}

void UsersView::changeAccess()
{
    if ( m_selectedUserId != 0 && m_selectedUserId != 1 && m_systemAdmin ) {
        ChangeUserAccessDialog dialog( m_selectedUserId, mainWidget() );
        dialog.exec();
    }
}

void UsersView::changePassword()
{
    if ( m_selectedUserId != 0 && m_systemAdmin ) {
        SetPasswordDialog dialog( m_selectedUserId, mainWidget() );
        dialog.exec();
    }
}

void UsersView::contextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );

    QString menuName;
    if ( index.isValid() )
        menuName = "contextUser";
    else
        menuName = "contextNull";

    QMenu* menu = builder()->contextMenu( menuName );
    if ( menu )
        menu->exec( m_list->viewport()->mapToGlobal( pos ) );
}

void UsersView::activated( const QModelIndex& index )
{
    if ( index.isValid() && m_systemAdmin ) {
        int userId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();

        if ( userId != 1 ) {
            ChangeUserAccessDialog dialog( userId, mainWidget() );
            dialog.exec();
        }
    }
}

QModelIndex UsersView::selectedIndex()
{
    if ( !m_list->selectionModel() )
        return QModelIndex();

    QModelIndexList selection = m_list->selectionModel()->selectedRows();
    if ( selection.isEmpty() )
        return QModelIndex();

    return selection.at( 0 );
}
