/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef TREEVIEWSETTINGS_H
#define TREEVIEWSETTINGS_H

#include <QObject>
#include <QMap>

class QSettings;

class DefinitionInfo;

/**
* Class for saving and restoring tree view settings.
*/
class TreeViewSettings : public QObject
{
    Q_OBJECT
public:
    /**
    * Default constructor.
    */
    TreeViewSettings();

    /**
    * Destructor.
    */
    ~TreeViewSettings();

public:
    /**
    * Read settings for the users list.
    */
    void openUsersList();

    /**
    * Read settings for the members list.
    */
    void openMembersList();

    /**
    * Read settings for the types tree.
    */
    void openTypesTree();

    /**
    * Read settings for the projects tree.
    */
    void openProjectsTree();

    /**
    * Read settings for the issues list.
    *
    * @param typeId Type of issues displayed in the list.
    */
    void openIssuesList( int typeId );

    /**
    * Read settings for the attachments list.
    */
    void openAttachmentsList();

    /**
    * Return a list of columns in the view.
    */
    QList<int> loadColumns();

    /**
    * Set the list of columns for the view.
    */
    void saveColumns( const QList<int>& columns );

    /**
    * Return the default list of columns for the view.
    */
    QList<int> defaultColumns() { return m_columns; }

    /**
    * Return all available columns for the view.
    */
    QList<int> availableColumns();

    /**
    * Return columns which cannot be removed from the view.
    */
    QList<int> fixedColumns();

    /**
    * Return column widths in the view.
    */
    QMap<int, int> loadColumnWidths();

    /**
    * Set column widths for the view.
    */
    void saveColumnWidths( const QMap<int, int>& widths );

    /**
    * Return identifiers of expanded nodes.
    */
    QList<int> loadExpandedNodes();

    /**
    * Set expanded nodes in the view.
    */
    void saveExpandedNodes( const QList<int>& nodes );

private:
    QStringList serializeColumns( const QList<int>& columns );
    QList<int> deserializeColumns( const QStringList& list );

    int defaultColumnWidth( int column );
    int defaultColumnWidth( const DefinitionInfo& info );

private:
    QString m_group;
    int m_typeId;

    QList<int> m_columns;

    bool m_longName;
};

#endif
