/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef HTMLREPORTWRITER_H
#define HTMLREPORTWRITER_H

#include <QStringList>

/**
* Class for writing reports in HTML format.
*
* This class is responsible for adding HTML markup and writing the reports
* into an HTML document.
*/
class HtmlReportWriter
{
public:
    /**
    * Default constructor.
    */
    HtmlReportWriter();

    /**
    * Destructor.
    */
    ~HtmlReportWriter();

public:
    /**
    * Write HTML document header.
    */
    void begin( const QString& generator, const QString& title );
    /**
    * End the document.
    */
    void end();

    /**
    * Write a h2 tag with an anchor.
    */
    void writeHeader2( int id, const QString& text );
    /**
    * Write a h3 tag.
    */
    void writeHeader3( const QString& text );
    /**
    * Write a paragraph.
    */
    void writeParagraph( const QString& text );

    /**
    * Begin a summary table.
    */
    void beginSummary();
    /**
    * Write a single row of the summary table.
    */
    void writeSummaryItem( const QString& item, const QString& value );
    /**
    * Finish the summary table.
    */
    void endSummary();

    /**
    * Begin a multi-column table.
    */
    void beginTable();
    /**
    * Write the header of a multi-column table.
    */
    void writeTableHeader( const QStringList& cells );
    /**
    * Write a single row of a multi-column table.
    */
    void writeTableRow( int id, const QStringList& cells );
    /**
    * Finish the multi-column table.
    */
    void endTable();

    /**
    * Write document footer.
    */
    void writeFooter( const QString& text );

    /**
    * Insert raw HTML markup into the document.
    */
    void writeRawHtml( const QString& text );

    /**
    * Add HTML markup to an attribute name or user name.
    *
    * Names are displayed with bold font.
    */
    QString name( const QString& text );

    /**
    * Add HTML markup to a date.
    *
    * Dates are displayed with bold font.
    */
    QString date( const QString& text );

    /**
    * Add HTML markup to an item.
    *
    * Items are displayed without additional formatting.
    */
    QString item( const QString& text );

    /**
    * Add HTML markup to an issue name or attribute value.
    *
    * URLs and issue numbers are converted to links.
    */
    QString value( const QString& text );

    /**
    * Add HTML markup to a title or header.
    *
    * Titles are displayed without additional formatting.
    */
    QString titleName( const QString& text );

    /**
    * Return the content of the generated report.
    */
    QString result() const;

private:
    QStringList m_lines;

    int m_row;
};

#endif
