/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "folderwatchprovider.h"

#include "data/datamanager.h"
#include "data/datarows.h"
#include "rdb/utilities.h"
#include "viewmanager.h"
#include "folderwatchwriter.h"
#include "columnconditionsettings.h"
#include "rowfilters.h"

FolderWatchProvider::FolderWatchProvider()
{
    m_filter = new IssueRowFilter( this );
}

FolderWatchProvider::~FolderWatchProvider()
{
}

void FolderWatchProvider::updateAll()
{
    m_map.clear();

    RDB::IndexConstIterator<TypeRow> it( dataManager->types()->index() );
    while ( it.next() )
        updateFolders( it.key() );
}

void FolderWatchProvider::updateFolders( int typeId )
{
    ColumnConditionSettings settings;
    settings.openIssueFilters( typeId );

    RDB::IndexConstIterator<FolderRow> it( dataManager->folders()->index() );
    while ( it.next() ) {
        if ( it.get()->typeId() == typeId )
            calculateWatches( it.key(), &settings );
    }
}

void FolderWatchProvider::updateFolder( int folderId )
{
    const FolderRow* folder = dataManager->folders()->find( folderId );
    if ( folder != NULL ) {
        ColumnConditionSettings settings;
        settings.openIssueFilters( folder->typeId() );

        calculateWatches( folderId, &settings );
    }
}

void FolderWatchProvider::calculateWatches( int folderId, ColumnConditionSettings* settings )
{
    WatchState folderState;

    QStringList watches = settings->loadWatches( folderId );

    if ( watches.isEmpty() ) {
        folderState = NotWatched;

        m_map.remove( folderId );
    } else {
        folderState = WatchNormal;

        WatchList list;

        for ( int i = 0; i < watches.count(); i++ ) {
            Watch watch;
            watch.m_filter = watches[ i ];
            watch.m_total = 0;
            watch.m_new = 0;
            watch.m_updated = 0;

            if ( !watches[ i ].isEmpty() ) {
                QList<ColumnCondition> conditions = settings->loadFilter( watches[ i ] );
                m_filter->setConditions( conditions );
            } else {
                m_filter->setConditions( QList<ColumnCondition>() );
            }

            RDB::ForeignConstIterator<IssueRow> it( dataManager->issues()->parentIndex(), folderId );
            while ( it.next() ) {
                int issueId = it.get()->issueId();
                if ( m_filter->filterRow( issueId ) ) {
                    watch.m_total++;
                    WatchState issueState = dataManager->issueWatchState( issueId, true );
                    if ( issueState == WatchUpdated ) {
                        watch.m_updated++;
                        if ( folderState < WatchUpdated )
                            folderState = WatchUpdated;
                    } else if ( issueState == WatchNew ) {
                        watch.m_new++;
                        if ( folderState < WatchNew )
                            folderState = WatchNew;
                    }
                }
            }

            list.append( watch );
        }

        m_map.insert( folderId, list );
    }

    if ( dataManager->folderWatchState( folderId ) != folderState ) {
        dataManager->setFolderWatchState( folderId, folderState );

        viewManager->postViewEvent( NULL, ViewEvent::UpdateWatches, folderId );
    }
}

bool FolderWatchProvider::isEmpty()
{
    return m_map.isEmpty();
}

void FolderWatchProvider::write( FolderWatchWriter* writer )
{
    writer->begin();

    RDB::IndexConstIterator<ProjectRow> itp( dataManager->projects()->index() );
    QList<const ProjectRow*> projects = localeAwareSortRows( itp, &ProjectRow::name );

    for ( int i = 0; i < projects.count(); i++ ) {
        int projectId = projects[ i ]->projectId();

        RDB::ForeignConstIterator<FolderRow> itf( dataManager->folders()->parentIndex(), projectId );
        QList<const FolderRow*> folders = localeAwareSortRows( itf, &FolderRow::name );

        bool header = false;

        for ( int j = 0; j < folders.count(); j++ ) {
            int folderId = folders[ j ]->folderId();

            WatchList watches = m_map.value( folderId );
            if ( !watches.isEmpty() ) {
                if ( !header ) {
                    writer->writeProjectHeader( writer->name( projects[ i ]->name() ) );
                    header = true;
                }

                writer->writeFolderHeader( writer->name( folders[ j ]->name() ) );

                qSort( watches.begin(), watches.end(), Watch::compare );

                for ( int k = 0; k < watches.count(); k++ ) {
                    QString linkCell;
                    QString watch = watches[ k ].m_filter;
                    if ( watch.isEmpty() )
                        linkCell = writer->linkCell( folderId, QString(), tr( "(All Issues)" ) );
                    else
                        linkCell = writer->linkCell( folderId, watch, watch );

                    QString totalCell = writer->watchCell( tr( "total" ), WatchNormal, watches[ k ].m_total );
                    QString newCell = writer->watchCell( tr( "new" ), WatchNew, watches[ k ].m_new );
                    QString updatedCell = writer->watchCell( tr( "modified" ), WatchUpdated, watches[ k ].m_updated );

                    writer->writeWatchRow( linkCell, totalCell, newCell, updatedCell );
                }
            }
        }
    }

    writer->end();
}

bool FolderWatchProvider::Watch::compare( const Watch& w1, const Watch& w2 )
{
    return QString::localeAwareCompare( w1.m_filter, w2.m_filter ) < 0;
}
