/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "columncondition.h"

#include <QApplication>

#include "attributes/datetimehelper.h"
#include "data/connectioninfo.h"
#include "tablemodelshelper.h"
#include "connectionmanager.h"

ColumnCondition::ColumnCondition()
{
}

ColumnCondition::~ColumnCondition()
{
}

bool ColumnCondition::match( const QString& value ) const
{
    switch ( m_condition ) {
        case String_Equals:
        case String_NotEquals:
        case String_Contains:
        case String_NotContains:
        case String_StartsWith:
        case String_NotStartsWith:
        case String_EndsWith:
        case String_NotEndsWith:
            return matchString( value );

        case Numeric_Equals:
        case Numeric_NotEquals:
        case Numeric_GreaterThan:
        case Numeric_GreaterOrEqual:
        case Numeric_LessThan:
        case Numeric_LessOrEqual:
            return matchNumeric( value );

        case DateTime_Equals:
        case DateTime_NotEquals:
        case DateTime_Before:
        case DateTime_After:
            return matchDateTime( value );

        case Age_OlderThan:
        case Age_NewerThan:
            return matchAge( value );

        case Check_IsEmpty:
            return value.isEmpty();
        case Check_IsNotEmpty:
            return !value.isEmpty();

        case Check_IsMe:
            return checkIsMe( value );
        case Check_IsNotMe:
            return !checkIsMe( value );

        default:
            return false;
    }
}

bool ColumnCondition::matchString( const QString& value ) const
{
    switch ( m_condition ) {
        case String_Equals:
            return value.localeAwareCompare( m_operand ) == 0;
        case String_NotEquals:
            return value.localeAwareCompare( m_operand ) != 0;
        case String_Contains:
            return value.contains( m_operand, Qt::CaseInsensitive ) > 0;
        case String_NotContains:
            return !value.contains( m_operand, Qt::CaseInsensitive );
        case String_StartsWith:
            return value.startsWith( m_operand, Qt::CaseInsensitive );
        case String_NotStartsWith:
            return !value.startsWith( m_operand, Qt::CaseInsensitive );
        case String_EndsWith:
            return value.endsWith( m_operand, Qt::CaseInsensitive );
        case String_NotEndsWith:
            return !value.endsWith( m_operand, Qt::CaseInsensitive );
        default:
            return false;
    }
}

bool ColumnCondition::matchNumeric( const QString& value ) const
{
    bool ok;
    double number = value.toDouble( &ok );
    if ( !ok )
        return false;

    double operand = m_operand.toDouble();

    switch ( m_condition ) {
        case Numeric_Equals:
            return number == operand;
        case Numeric_NotEquals:
            return number != operand;
        case Numeric_GreaterThan:
            return number > operand;
        case Numeric_GreaterOrEqual:
            return number >= operand;
        case Numeric_LessThan:
            return number < operand;
        case Numeric_LessOrEqual:
            return number <= operand;
        default:
            return false;
    }
}

bool ColumnCondition::matchDateTime( const QString& value ) const
{
    QDateTime dateTime = DateTimeHelper::parseDateTime( value );
    if ( dateTime.isNull() )
        return false;

    QDateTime operand = DateTimeHelper::parseDateTime( m_operand );

    switch ( m_condition ) {
        case DateTime_Equals:
            return dateTime.date() == operand.date();
        case DateTime_NotEquals:
            return dateTime.date() != operand.date();
        case DateTime_Before:
            return dateTime < operand;
        case DateTime_After:
            return dateTime >= operand;
        default:
            return false;
    }
}

bool ColumnCondition::matchAge( const QString& value ) const
{
    QDateTime dateTime = DateTimeHelper::parseDateTime( value );
    if ( dateTime.isNull() )
        return false;

    int secsTo = dateTime.secsTo( QDateTime::currentDateTime() );
    int operand = m_operand.toInt();

    switch ( m_condition ) {
        case Age_OlderThan:
            return secsTo > operand;
        case Age_NewerThan:
            return secsTo <= operand;
        default:
            return false;
    }
}

bool ColumnCondition::checkIsMe( const QString& value ) const
{
    int userId = connectionManager->connectionInfo()->userId();
    QString name = TableModelsHelper::userName( userId );

    return value.localeAwareCompare( name ) == 0;
}

QString ColumnCondition::name( Condition condition )
{
    switch ( condition ) {
        case String_Equals:
        case Numeric_Equals:
        case DateTime_Equals:
            return tr( "is equal to" );
        case String_NotEquals:
        case Numeric_NotEquals:
        case DateTime_NotEquals:
            return tr( "is not equal to" );
        case String_Contains:
            return tr( "contains" );
        case String_NotContains:
            return tr( "does not contain" );
        case String_StartsWith:
            return tr( "starts with" );
        case String_NotStartsWith:
            return tr( "does not start with" );
        case String_EndsWith:
            return tr( "ends with" );
        case String_NotEndsWith:
            return tr( "does not end with" );
        case Numeric_GreaterThan:
            return tr( "is greater than" );
        case Numeric_GreaterOrEqual:
            return tr( "is greater than or equal to" );
        case Numeric_LessThan:
            return tr( "is less than" );
        case Numeric_LessOrEqual:
            return tr( "is less than or equal to" );
        case DateTime_Before:
            return tr( "is before" );
        case DateTime_After:
            return tr( "is after" );
        case Age_OlderThan:
            return tr( "is older than" );
        case Age_NewerThan:
            return tr( "is newer than" );
        case Check_IsEmpty:
            return tr( "is empty" );
        case Check_IsNotEmpty:
            return tr( "is not empty" );
        case Check_IsMe:
            return tr( "is me" );
        case Check_IsNotMe:
            return tr( "is not me" );
        default:
            return QString();
    }
}

QString ColumnCondition::tr( const char* text )
{
    return qApp->translate( "ColumnCondition", text );
}
