/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef DATAROWS_H
#define DATAROWS_H

#include <QString>
#include <QDateTime>

/**
* Access level for user or member.
*/
enum Access
{
    /**
    * Data is unavailable and the access level is not known.
    */
    UnknownAccess = -1,
    /**
    * User has no access to the system or project.
    */
    NoAccess = 0,
    /**
    * User has normal access to the system or project.
    */
    NormalAccess = 1,
    /**
    * User is a system or project administrator.
    */
    AdminAccess = 2
};

/**
* A row of the users table.
*/
class UserRow
{
public:
    /**
    * Constructor.
    */
    UserRow( int userId, const QString& login, const QString& name, Access access );

    /**
    * Destructor.
    */
    ~UserRow();

public:
    /**
    * Return the identifier of the user.
    */
    int userId() const { return m_userId; }

    /**
    * Return the login of the user.
    */
    const QString& login() const { return m_login; }

    /**
    * Return the display name of the user.
    */
    const QString& name() const { return m_name; }

    /**
    * Return the global access level of the user.
    */
    Access access() const { return m_access; }

private:
    int m_userId;
    QString m_login;
    QString m_name;
    Access m_access;
};

/**
* A row of the members table.
*/
class MemberRow
{
public:
    /**
    * Constructor.
    */
    MemberRow( int userId, int projectId, Access access );

    /**
    * Destructor.
    */
    ~MemberRow();

public:
    /**
    * Return the idenifier of the user.
    */
    int userId() const { return m_userId; }

    /**
    * Return the identifier of the project.
    */
    int projectId() const { return m_projectId; }

    /**
    * Return the user's access level to the project.
    */
    Access access() const { return m_access; }

private:
    int m_userId;
    int m_projectId;
    Access m_access;
};

/**
* A row of the issue types table.
*/
class TypeRow
{
public:
    /**
    * Constructor.
    */
    TypeRow( int typeId, const QString& name );

    /**
    * Destructor.
    */
    ~TypeRow();

public:
    /**
    * Return the identifier of the issue type.
    */
    int typeId() const { return m_typeId; }

    /**
    * Return the name of the issue type.
    */
    const QString& name() const { return m_name; }

private:
    int m_typeId;
    QString m_name;
};

/**
* A row of the attribute definitions table.
*/
class AttributeRow
{
public:
    /**
    * Constructor.
    */
    AttributeRow( int attributeId, int typeId, const QString& name, const QString& definition );

    /**
    * Destructor.
    */
    ~AttributeRow();

public:
    /**
    * Return the identifier of the attribute.
    */
    int attributeId() const { return m_attributeId; }

    /**
    * Return the identifier of the issue type.
    */
    int typeId() const { return m_typeId; }

    /**
    * Return the name of the attribute.
    */
    const QString& name() const { return m_name; }

    /**
    * Return the definition of the attribute.
    */
    const QString& definition() const { return m_definition; }

private:
    int m_attributeId;
    int m_typeId;
    QString m_name;
    QString m_definition;
};

/**
* A row of the projects table.
*/
class ProjectRow
{
public:
    /**
    * Constructor.
    */
    ProjectRow( int projectId, const QString& name );

    /**
    * Destructor.
    */
    ~ProjectRow();

public:
    /**
    * Return the identifier of the project.
    */
    int projectId() const { return m_projectId; }

    /**
    * Return the name of the project.
    */
    const QString& name() const { return m_name; }

private:
    int m_projectId;
    QString m_name;
};

/**
* A row of the folders table.
*/
class FolderRow
{
public:
    /**
    * Constructor.
    */
    FolderRow( int folderId, int projectId, const QString& name, int typeId, int stamp );

    /**
    * Destructor.
    */
    ~FolderRow();

public:
    /**
    * Return the identifier of the folder.
    */
    int folderId() const { return m_folderId; }

    /**
    * Return the identifier of the folder's project.
    */
    int projectId() const { return m_projectId; }

    /**
    * Return the folder name.
    */
    const QString& name() const { return m_name; }

    /**
    * Return the identifier of the issue type of the folder.
    */
    int typeId() const { return m_typeId; }

    /**
    * Return the stamp of the last modification in the folder.
    */
    int stamp() const { return m_stamp; }

private:
    int m_folderId;
    int m_projectId;
    QString m_name;
    int m_typeId;
    int m_stamp;
};

/**
* A row of the issues table.
*/
class IssueRow
{
public:
    /**
    * Constructor.
    */
    IssueRow( int issueId, int folderId, const QString& name, int stamp, const QDateTime& createdDate,
        int createdUser, const QDateTime& modifiedDate, int modifiedUser );

    /**
    * Destructor.
    */
    ~IssueRow();

public:
    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Return the identifier of the issue's folder.
    */
    int folderId() const { return m_folderId; }

    /**
    * Return the issue name.
    */
    const QString& name() const { return m_name; }

    /**
    * Return the stamp of the last modification of the issue.
    */
    int stamp() const { return m_stamp; }

    /**
    * Return the creation date.
    */
    const QDateTime& createdDate() const { return m_createdDate; }

    /**
    * Return the identifier of the user who created the issue.
    */
    int createdUser() const { return m_createdUser; }

    /**
    * Return the last modification date.
    */
    const QDateTime& modifiedDate() const { return m_modifiedDate; }

    /**
    * Return the identifier of the user who last modified the issue.
    */
    int modifiedUser() const { return m_modifiedUser; }

private:
    int m_issueId;
    int m_folderId;
    QString m_name;
    int m_stamp;
    QDateTime m_createdDate;
    int m_createdUser;
    QDateTime m_modifiedDate;
    int m_modifiedUser;
};

/**
* A row of the attribute values table.
*/
class ValueRow
{
public:
    /**
    * Constructor.
    */
    ValueRow( int attributeId, int issueId, const QString& value );

    /**
    * Destructor.
    */
    ~ValueRow();

public:
    /**
    * Return the identifier of the attribute.
    */
    int attributeId() const { return m_attributeId; }

    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Return the attribute value.
    */
    const QString& value() const { return m_value; }

private:
    int m_attributeId;
    int m_issueId;
    QString m_value;
};

/**
* A row of the comments table.
*/
class CommentRow
{
public:
    /**
    * Constructor.
    */
    CommentRow( int commentId, int issueId, const QDateTime& createdDate, int createdUser,
        const QString& text );

    /**
    * Destructor.
    */
    ~CommentRow();

public:
    /**
    * Return the identifier of the comment.
    */
    int commentId() const { return m_commentId; }

    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Return the creation date.
    */
    const QDateTime& createdDate() const { return m_createdDate; }

    /**
    * Return the identifier of the user who created the comment.
    */
    int createdUser() const { return m_createdUser; }

    /**
    * Return the text of the comment.
    */
    const QString& text() const { return m_text; }

private:
    int m_commentId;
    int m_issueId;
    QDateTime m_createdDate;
    int m_createdUser;
    QString m_text;
};

/**
* A row of the attachments table.
*/
class AttachmentRow
{
public:
    /**
    * Constructor.
    */
    AttachmentRow( int attachmentId, int issueId, const QString& name, const QDateTime& createdDate,
        int createdUser, int size, const QString& description );

    /**
    * Destructor.
    */
    ~AttachmentRow();

public:
    /**
    * Return the identifier of the attachment.
    */
    int attachmentId() const { return m_attachmentId; }

    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Return the name of the attachment.
    */
    const QString& name() const { return m_name; }

    /**
    * Return the creation date.
    */
    const QDateTime& createdDate() const { return m_createdDate; }

    /**
    * Return the identifier of the user who created the attachment.
    */
    int createdUser() const { return m_createdUser; }

    /**
    * Return the size of the attachment.
    */
    int size() const { return m_size; }

    /**
    * Return the description of the attachment.
    */
    const QString& description() const { return m_description; }

private:
    int m_attachmentId;
    int m_issueId;
    QString m_name;
    QDateTime m_createdDate;
    int m_createdUser;
    int m_size;
    QString m_description;
};

/**
* A row of the changes table.
*/
class ChangeRow
{
public:
    /**
    * Constructor.
    */
    ChangeRow( int changeId, int issueId, const QDateTime& modifiedDate, int modifiedUser,
        int attributeId, const QString& oldValue, const QString& newValue );

    /**
    * Destructor.
    */
    ~ChangeRow();

public:
    /**
    * Return the identifier of the change.
    */
    int changeId() const { return m_changeId; }

    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Return the date of the change.
    */
    const QDateTime& modifiedDate() const { return m_modifiedDate; }

    /**
    * Return the identifier of the user who made the change.
    */
    int modifiedUser() const { return m_modifiedUser; }

    /**
    * Return the identifier of the changed attribute ( 0 if the issue name was changed ).
    */
    int attributeId() const { return m_attributeId; }

    /**
    * Return the old value of the changed attribute.
    */
    const QString& oldValue() const { return m_oldValue; }

    /**
    * Return the new value of the changed attribute.
    */
    const QString& newValue() const { return m_newValue; }

private:
    int m_changeId;
    int m_issueId;
    QDateTime m_modifiedDate;
    int m_modifiedUser;
    int m_attributeId;
    QString m_oldValue;
    QString m_newValue;
};

#endif
