/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef CONFIGDATA_H
#define CONFIGDATA_H

#include <QObject>
#include <QNetworkProxy>

class QSettings;

/**
* Action performed on an attachment.
*/
enum AttachmentAction
{
    ActionAsk,
    ActionOpen,
    ActionSaveAs
};

/**
* Restore option settings.
*/
enum RestoreOption
{
    RestoreNever,
    RestoreAuto,
    RestoreAlways
};

/**
* Class storing the configuration of the application.
*
* The instance of this class is available using the configData
* global variable. It is created and owned by the Application object.
*/
class ConfigData : public QObject
{
    Q_OBJECT
public:
    /**
    * Constructor.
    */
    ConfigData();

    /**
    * Destructor.
    */
    ~ConfigData();

public:
    /**
    * Return the application settings.
    */
    QSettings* settings() const { return m_settings; }

    /**
    * Load configuration data from the settings file.
    */
    void loadSettings();

    /**
    * Save configuration data in the settings file.
    *
    * The settingsChanged() signal is emitted.
    */
    void saveSettings();

    /**
    * Dock main window is system tray.
    */
    bool dockWindow() const { return m_dockWindow; }

    /**
    * Set the "Dock main window is system tray" flag.
    */
    void setDockWindow( bool value ) { m_dockWindow = value; }

    /**
    * Automatically show main window at startup.
    */
    RestoreOption showAtStartup() const { return m_showAtStartup; }

    /**
    * Set the "Automatically show main window at startup" option.
    */
    void setShowAtStartup( RestoreOption value ) { m_showAtStartup = value; }

    /**
    * Automatically restore connection at startup.
    */
    RestoreOption reconnectAtStartup() const { return m_reconnectAtStartup; }

    /**
    * Set the "Automatically restore connection at startup" option.
    */
    void setReconnectAtStartup( RestoreOption value ) { m_reconnectAtStartup = value; }

    /**
    * Proxy used for Internet connections.
    */
    const QNetworkProxy& networkProxy() const { return m_proxy; }

    /**
    * Set the proxy used for Internet connections.
    */
    void setNetworkProxy( const QNetworkProxy& value ) { m_proxy = value; }

    /**
    * Interval of periodic updates (in minutes).
    */
    int updateInterval() const { return m_updateInterval; }

    /**
    * Set the "Interval of periodic updates" parameter.
    */
    void setUpdateInterval( int value ) { m_updateInterval = value; }

    /**
    * Default action for attachments.
    */
    AttachmentAction defaultAttachmentAction() const { return m_defaultAttachmentAction; }

    /**
    * Set the "Default action for attachments" parameter.
    */
    void setDefaultAttachmentAction( AttachmentAction value ) { m_defaultAttachmentAction = value; }

    /**
    * Size of attachments cache (in megabytes).
    */
    int attachmentsCacheSize() const { return m_attachmentsCacheSize; }

    /**
    * Set the "Size of attachments cache" parameter.
    */
    void setAttachmentsCacheSize( int value ) { m_attachmentsCacheSize = value; }

    /**
    * Warn when adding attachment larger than (in megabytes)
    */
    int warnAttachmentSize() const { return m_warnAttachmentSize; }

    /**
    * Set the "Warn when adding attachment larger than" parameter.
    */
    void setWarnAttachmentSize( int value ) { m_warnAttachmentSize = value; }

#if defined( Q_WS_WIN )

    /**
    * Check if the autostart registry key exists.
    */
    bool checkAutoStart();

    /**
    * Create or remove the autostart registry key.
    */
    void setAutoStart( bool start );

#endif

signals:
    /**
    * Emitted when new settings are saved.
    */
    void settingsChanged();

private:
    QSettings* m_settings;

    bool m_dockWindow;
    RestoreOption m_showAtStartup;
    RestoreOption m_reconnectAtStartup;

    QNetworkProxy m_proxy;

    int m_updateInterval;

    AttachmentAction m_defaultAttachmentAction;
    int m_attachmentsCacheSize;
    int m_warnAttachmentSize;
};

/**
* Global pointer used to access the ConfigData.
*/
extern ConfigData* configData;

#endif
