/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "definitionparser.h"

#include <QStringList>

DefinitionParser::DefinitionParser()
{
    QString string( "\"(?:\\\\[\"\\\\]|[^\"\\\\])*\"" );
    QString attribute( "([-a-z]+)=(-?\\d+|%1|\\{%2(?:,%3)*\\})" );

    m_string.setPattern( string );
    m_attribute.setPattern( attribute.arg( string, string, string ) );

    m_types.insert( "TEXT", DefinitionInfo::Text );
    m_types.insert( "ENUM", DefinitionInfo::Enum );
    m_types.insert( "NUMERIC", DefinitionInfo::Numeric );
    m_types.insert( "DATETIME", DefinitionInfo::DateTime );
    m_types.insert( "USER", DefinitionInfo::User );
}

DefinitionParser::~DefinitionParser()
{
}

DefinitionInfo DefinitionParser::parse( const QString& definition )
{
    QString type, attrs;

    int index = definition.indexOf( ' ' );
    if ( index == -1 ) {
        type = definition;
    } else {
        type = definition.left( index );
        attrs = definition.mid( index + 1 );
    }

    DefinitionInfo info;

    info.setType( parseType( type ) );

    if ( !info.isValid() )
        return info;

    int pos = 0;
    while ( ( pos = m_attribute.indexIn( attrs, pos ) ) != -1 ) {
        QString key = m_attribute.cap( 1 );
        QString value = m_attribute.cap( 2 );

        if ( value[ 0 ] == QLatin1Char( '\"' ) )
            info.setMetadata( key, parseString( value ) );
        else if ( value[ 0 ] == QLatin1Char( '{' ) )
            info.setMetadata( key, parseStringArray( value ) );
        else
            info.setMetadata( key, value.toInt() );

        pos += m_attribute.matchedLength();
    }

    QVariant value = info.metadata( "default" );
    if ( value.isValid() ) {
        info.setDefaultValue( value.toString() );
        info.setMetadata( "default", QVariant() );
    }

    value = info.metadata( "required" );
    if ( value.isValid() ) {
        info.setRequired( value.toBool() );
        info.setMetadata( "required", QVariant() );
    }

    return info;
}

DefinitionInfo::Type DefinitionParser::parseType( const QString& value )
{
    QMap<QString, DefinitionInfo::Type>::iterator it = m_types.find( value );
    if ( it == m_types.end() )
        return DefinitionInfo::Invalid;
    return *it;
}

QStringList DefinitionParser::parseStringArray( const QString& value )
{
    QStringList result;
    int pos = 0;
    while ( ( pos = m_string.indexIn( value, pos ) ) != -1 ) {
        result.append( parseString( m_string.cap( 0 ) ) );
        pos += m_string.matchedLength();
    }
    return result;
}

QString DefinitionParser::parseString( const QString& value )
{
    QString result;
    int len = value.length() - 1;
    for ( int i = 1; i < len; i++ ) {
        QChar ch = value[ i ];
        if ( ch == QLatin1Char( '\\' ) )
            ch = value[ ++i ];
        result += ch;
    }
    return result;
}
