/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "commentview.h"

#include <QTextEdit>
#include <QAction>
#include <QMenu>
#include <QApplication>
#include <QClipboard>
#include <QMessageBox>

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/issuebatch.h"
#include "data/updateevent.h"
#include "xmlui/builder.h"
#include "viewmanager.h"
#include "iconloader.h"

using namespace WebIssues;

CommentView::CommentView( QObject* parent, QWidget* parentWidget ) : View( parent ),
    m_sending( false )
{
    QAction* action;

    action = new QAction( IconLoader::icon( "comment-send" ), tr( "&Send Comment" ), this );
    action->setShortcut( tr( "Ctrl+Return" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( sendComment() ) );
    setAction( "sendComment", action );

    m_edit = new QTextEdit( parentWidget );
    m_edit->setAcceptRichText( false );
    m_edit->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_edit, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( contextMenu( const QPoint& ) ) );

    m_edit->setHtml( "<pre></pre>" );
    QFont fixedFont = m_edit->currentCharFormat().font();
    m_edit->clear();
    m_edit->setFont( fixedFont );

    action = new QAction( IconLoader::icon( "edit-undo" ), tr( "&Undo" ), this );
    action->setShortcut( QKeySequence::Undo );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( undo() ) );
    connect( m_edit, SIGNAL( undoAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editUndo", action );

    action = new QAction( IconLoader::icon( "edit-redo" ), tr( "&Redo" ), this );
    action->setShortcut( QKeySequence::Redo );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( redo() ) );
    connect( m_edit, SIGNAL( redoAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editRedo", action );

    action = new QAction( IconLoader::icon( "edit-cut" ), tr( "Cu&t" ), this );
    action->setShortcut( QKeySequence::Cut );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( cut() ) );
    connect( m_edit, SIGNAL( copyAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editCut", action );

    action = new QAction( IconLoader::icon( "edit-copy" ), tr( "&Copy" ), this );
    action->setShortcut( QKeySequence::Copy );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( copy() ) );
    connect( m_edit, SIGNAL( copyAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editCopy", action );

    action = new QAction( IconLoader::icon( "edit-paste" ), tr( "&Paste" ), this );
    action->setShortcut( QKeySequence::Paste );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( paste() ) );
    setAction( "editPaste", action );

    action = new QAction( tr( "Select &All" ), this );
    action->setShortcut( QKeySequence::SelectAll );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( selectAll() ) );
    setAction( "editSelectAll", action );

    setTitle( "menuMain", tr( "&Comment" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );

    loadXmlFile( ":/resources/commentview.xml" );

    connect( m_edit, SIGNAL( textChanged() ), this, SLOT( updateActions() ) );

    connect( QApplication::clipboard(), SIGNAL( dataChanged() ), this, SLOT( updateClipboard() ) );

    setMainWidget( m_edit );

    setViewerSizeHint( QSize( 600, 400 ) );

    setAccess( NormalAccess );
}

CommentView::~CommentView()
{
}

void CommentView::initialUpdate()
{
    setCaption( tr( "Add Comment" ) );

    updateActions();
    updateClipboard();
}

void CommentView::updateActions()
{
    action( "sendComment" )->setEnabled( !m_edit->document()->isEmpty() );
    action( "editUndo" )->setEnabled( m_edit->document()->isUndoAvailable() );
    action( "editRedo" )->setEnabled( m_edit->document()->isRedoAvailable() );
    action( "editCut" )->setEnabled( m_edit->textCursor().hasSelection() );
    action( "editCopy" )->setEnabled( m_edit->textCursor().hasSelection() );
    action( "editSelectAll" )->setEnabled( !m_edit->document()->isEmpty() );
}

void CommentView::updateClipboard()
{
    action( "editPaste" )->setEnabled( !QApplication::clipboard()->text().isEmpty() );
}

void CommentView::sendComment()
{
    if ( !m_sending ) {
        showBusy( tr( "Sending comment..." ) );

        IssueBatch* batch = new IssueBatch( id() );
        batch->addComment( m_edit->toPlainText() );

        connect( batch, SIGNAL( completed( bool ) ), this, SLOT( sendCompleted( bool ) ) );

        commandManager->execute( batch );

        m_sending = true;
    }
}

void CommentView::sendCompleted( bool successful )
{
    if ( successful )
        viewManager->closeView( this );
    else
        showWarning( commandManager->errorMessage( tr( "Sending failed" ) ) );

    m_sending = false;
}

bool CommentView::queryClose()
{
    if ( !m_edit->document()->isEmpty() ) {
        if ( QMessageBox::warning( mainWidget(), tr( "Warning" ),
            tr( "The comment has not been sent. Are you sure you want to close it?" ),
            QMessageBox::Yes | QMessageBox::No ) == QMessageBox::No )
            return false;
    }

    return true;
}

void CommentView::updateEvent( UpdateEvent* e )
{
}

void CommentView::contextMenu( const QPoint& pos )
{
    QMenu* menu = builder()->contextMenu( "contextEdit" );
    if ( menu )
        menu->exec( m_edit->mapToGlobal( pos ) );
}

#include "commentview.moc"
