/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "treeviewsettings.h"

#include <QSettings>
#include <QStringList>

#include "attributes/definitionparser.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "data/rdb/tableiterators.h"
#include "connectionmanager.h"

using namespace WebIssues;

TreeViewSettings::TreeViewSettings() :
    m_typeId( 0 ),
    m_longName( false )
{
}

TreeViewSettings::~TreeViewSettings()
{
}

void TreeViewSettings::openUsersList()
{
    m_group = "UsersList";
    m_typeId = 0;

    m_columns.clear();
    m_columns.append( Column_Name );
    m_columns.append( Column_Login );
    m_columns.append( Column_Access );

    m_longName = false;
}

void TreeViewSettings::openMembersList()
{
    m_group = "MembersList";
    m_typeId = 0;

    m_columns.clear();
    m_columns.append( Column_Name );
    m_columns.append( Column_Access );

    m_longName = false;
}

void TreeViewSettings::openTypesTree()
{
    m_group = "TypesTree";
    m_typeId = 0;

    m_columns.clear();
    m_columns.append( Column_Name );
    m_columns.append( Column_Type );
    m_columns.append( Column_DefaultValue );
    m_columns.append( Column_Required );
    m_columns.append( Column_Details );

    m_longName = false;
}

void TreeViewSettings::openProjectsTree()
{
    m_group = "ProjectsTree";
    m_typeId = 0;

    m_columns.clear();
    m_columns.append( Column_Name );
    m_columns.append( Column_Type );

    m_longName = false;
}

void TreeViewSettings::openIssuesList( int typeId )
{
    m_group = "IssuesList";
    m_typeId = typeId;

    m_columns.clear();
    m_columns.append( Column_ID );
    m_columns.append( Column_Name );
    m_columns.append( Column_ModifiedDate );
    m_columns.append( Column_ModifiedBy );

    m_longName = true;
}

void TreeViewSettings::openAttachmentsList()
{
    m_group = "AttachmentsList";
    m_typeId = 0;

    m_columns.clear();
    m_columns.append( Column_ID );
    m_columns.append( Column_Name );
    m_columns.append( Column_Size );
    m_columns.append( Column_CreatedDate );
    m_columns.append( Column_CreatedBy );
    m_columns.append( Column_Description );

    m_longName = false;
}

QList<Column> TreeViewSettings::loadColumns()
{
    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Columns" ).arg( m_group ) );

    QStringList list = settings->value( QString( "Type%1" ).arg( m_typeId ) ).toStringList();

    settings->endGroup();

    if ( list.isEmpty() )
        return m_columns;

    return deserializeColumns( list );
}

void TreeViewSettings::saveColumns( const QList<Column>& columns )
{
    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Columns" ).arg( m_group ) );

    settings->setValue( QString( "Type%1" ).arg( m_typeId ), serializeColumns( columns ) );

    settings->endGroup();
}

QList<Column> TreeViewSettings::availableColumns()
{
    QList<Column> columns = m_columns;

    if ( m_group == "IssuesList" ) {
        columns.append( Column_CreatedDate );
        columns.append( Column_CreatedBy );
    }

    if ( m_typeId != 0 ) {
        RDB::ForeignConstIterator<AttributeRow> it( dataManager->attributes()->parentIndex(), m_typeId );
        while ( it.next() )
            columns.append( (Column)( Column_UserDefined + it.get()->attributeId() ) );
    }

    return columns;
}

QMap<Column, int> TreeViewSettings::loadColumnWidths()
{
    QList<Column> usedColumns = loadColumns();
    QMap<Column, int> widths;

    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Widths" ).arg( m_group ) );

    for ( int i = 0; i < usedColumns.count(); i++ ) {
        Column column = usedColumns.at( i );
        int width = settings->value( QString( "Column%1" ).arg( (int)column ), defaultColumnWidth( column ) ).toInt();
        widths.insert( column, width );
    }

    settings->endGroup();

    return widths;
}

void TreeViewSettings::saveColumnWidths( const QMap<Column, int>& widths )
{
    QList<Column> usedColumns = widths.keys();

    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Widths" ).arg( m_group ) );

    for ( int i = 0; i < usedColumns.count(); i++ ) {
        Column column = usedColumns.at( i );
        int width = widths.value( column );
        settings->setValue( QString( "Column%1" ).arg( (int)column ), width );
    }

    settings->endGroup();
}

QList<int> TreeViewSettings::loadExpandedNodes()
{
    QList<int> nodes;

    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Nodes" ).arg( m_group ) );

    QStringList list = settings->value( "Expanded" ).toStringList();

    for ( int i = 0; i < list.count(); i++ )
        nodes.append( list.at( i ).toInt() );

    settings->endGroup();

    return nodes;
}

void TreeViewSettings::saveExpandedNodes( const QList<int>& nodes )
{
    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Nodes" ).arg( m_group ) );

    QStringList list;
    for ( int i = 0; i < nodes.count(); i++ )
        list.append( QString::number( nodes.at( i ) ) );

    settings->setValue( "Expanded", list );

    settings->endGroup();
}

QStringList TreeViewSettings::serializeColumns( const QList<Column>& columns )
{
    QStringList list;

    for ( int i = 0; i < columns.count(); i++ )
        list.append( QString::number( columns.at( i ) ) );

    return list;
}

QList<Column> TreeViewSettings::deserializeColumns( const QStringList& list )
{
    QList<Column> columns;

    for ( int i = 0; i < list.count(); i++ ) {
        int column = list.at( i ).toInt();
        if ( column > Column_UserDefined ) {
            int attributeId = column - Column_UserDefined;
            if ( !dataManager->attributes()->find( attributeId ) )
                continue;
        }
        columns.append( (Column)column );
    }

    if ( columns.count() == 0 )
        columns.append( Column_Name );

    return columns;
}

int TreeViewSettings::defaultColumnWidth( Column column )
{
    switch ( column ) {
        case Column_Name:
            return m_longName ? 300 : 150;
        case Column_ID:
            return 50;
        case Column_CreatedDate:
        case Column_ModifiedDate:
            return 110;
        case Column_CreatedBy:
        case Column_ModifiedBy:
            return 150;
        case Column_Login:
            return 100;
        case Column_Size:
            return 70;
        case Column_Description:
            return 300;
        case Column_Access:
            return 150;
        case Column_Required:
            return 60;
        case Column_Details:
            return 300;
        case Column_DefaultValue:
        case Column_OldValue:
        case Column_NewValue:
            return 150;
        case Column_Type:
            return 150;
        default:
            if ( column > Column_UserDefined ) {
                int attributeId = column - Column_UserDefined;
                const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
                if ( attribute ) {
                    DefinitionParser parser;
                    DefinitionInfo info = parser.parse( attribute->definition() );
                    return defaultColumnWidth( info );
                }
            }
            return 150;
    }
}

int TreeViewSettings::defaultColumnWidth( const DefinitionInfo& info )
{
    switch ( info.type() ) {
        case DefinitionInfo::Text:
            return 150;
        case DefinitionInfo::Enum:
            return 150;
        case DefinitionInfo::Numeric:
            return 100;
        case DefinitionInfo::DateTime:
            return 110;
        case DefinitionInfo::User:
            return 150;
        default:
            return 150;
    }
}

#include "treeviewsettings.moc"
