/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef USERDIALOGS_H
#define USERDIALOGS_H

#include "commanddialog.h"

class QLineEdit;
class QButtonGroup;
class QListWidget;

namespace WebIssues
{

/**
* Base class for dialogs for executing commands related to users.
*/
class UsersDialog : public CommandDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    UsersDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~UsersDialog();

protected:
    /**
    * Verify that the login is valid.
    *
    * A warning is displayed if the name is empty or a user
    * with this login already exists.
    *
    * @param login The login to check.
    *
    * @return @c true if the login is valid, @c false otherwise.
    */
    bool checkUserLogin( const QString& login );

    /**
    * Verify that the user name is valid.
    *
    * A warning is displayed if the name is empty or a user
    * with this name already exists.
    *
    * @param name The name to check.
    *
    * @return @c true if the name is valid, @c false otherwise.
    */
    bool checkUserName( const QString& name );

    /**
    * Verify that the password is valid.
    *
    * A warning is displayed if the password is empty or the two
    * passwords are different.
    *
    * @param password The password to check.
    *
    * @param password2 The re-typed password to check.
    *
    * @return @c true if the password is valid, @c false otherwise.
    */
    bool checkPassword( const QString& password, const QString& password2 );
};

/**
* Dialog for executing the <tt>ADD USER</tt> command.
*/
class AddUserDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    AddUserDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~AddUserDialog();

public: // overrides
    void accept();

private:
    QLineEdit* m_loginEdit;
    QLineEdit* m_nameEdit;
    QLineEdit* m_passwordEdit;
    QLineEdit* m_passwordEdit2;
};

/**
* Dialog for executing the <tt>SET PASSWORD</tt> command.
*/
class SetPasswordDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param userId Identifier of the user.
    *
    * @param parent The parent widget.
    */
    SetPasswordDialog( int userId, QWidget* parent );

    /**
    * Destructor.
    */
    ~SetPasswordDialog();

public: // overrides
    void accept();

private:
    int m_userId;

    QLineEdit* m_passwordEdit;
    QLineEdit* m_passwordEdit2;
};

/**
* Dialog for executing the <tt>GRANT USER</tt> command.
*/
class ChangeUserAccessDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param userId Identifier of the user.
    *
    * @param parent The parent widget.
    */
    ChangeUserAccessDialog( int userId, QWidget* parent );

    /**
    * Destructor.
    */
    ~ChangeUserAccessDialog();

public: // overrides
    void accept();

private:
    int m_userId;
    int m_oldAccess;

    QButtonGroup* m_accessGroup;
};

/**
* Dialog for executing the <tt>RENAME USER</tt> command.
*/
class RenameUserDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param userId Identifier of the user to rename.
    *
    * @param parent The parent widget.
    */
    RenameUserDialog( int userId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RenameUserDialog();

public: // overrides
    void accept();

private:
    int m_userId;
    QString m_oldName;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>GRANT MEMBER</tt> command.
*
* This dialog allows to select new members of the project.
*/
class AddMemberDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param projectId Identifier of the project.
    *
    * @param parent The parent widget.
    */
    AddMemberDialog( int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~AddMemberDialog();

public: // overrides
    void accept();

private:
    int m_projectId;

    QListWidget* m_list;
    QButtonGroup* m_accessGroup;
};

/**
* Dialog for executing the <tt>GRANT MEMBER</tt> command.
*
* This dialog allows to change the access level for existing members.
*/
class ChangeMemberAccessDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param users Identifiers of the users.
    *
    * @param projectId Identifier of the project.
    *
    * @param parent The parent widget.
    */
    ChangeMemberAccessDialog( const QList<int>& users, int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~ChangeMemberAccessDialog();

public: // overrides
    void accept();

private:
    QList<int> m_users;
    int m_projectId;

    QButtonGroup* m_accessGroup;
};

/**
* Dialog for executing the <tt>GRANT MEMBER</tt> command.
*
* This dialog allows to remove project membership from existing members.
*/
class RemoveMemberDialog : public UsersDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param users Identifiers of the users.
    *
    * @param projectId Identifier of the project.
    *
    * @param parent The parent widget.
    */
    RemoveMemberDialog( const QList<int>& users, int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RemoveMemberDialog();

public: // overrides
    void accept();

private:
    QList<int> m_users;
    int m_projectId;
};

}

#endif
