/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "typedialogs.h"

#include <QLayout>
#include <QLabel>
#include <QGroupBox>
#include <QCheckBox>
#include <QRadioButton>
#include <QButtonGroup>
#include <QLineEdit>
#include <QComboBox>

#include "attributes/abstractvalueeditor.h"
#include "attributes/valueeditorfactory.h"
#include "attributes/definitionhelper.h"
#include "attributes/definitionbuilder.h"
#include "attributes/definitionparser.h"
#include "attributes/metadatadialogbase.h"
#include "attributes/metadatadialogfactory.h"
#include "data/rdb/tableiterators.h"
#include "data/datamanager.h"
#include "data/typesbatch.h"
#include "widgets/popeditbox.h"
#include "iconloader.h"

using namespace WebIssues;

TypesDialog::TypesDialog( QWidget* parent ) : CommandDialog( parent )
{
}

TypesDialog::~TypesDialog()
{
}

bool TypesDialog::checkTypeName( const QString& name )
{
    if ( name.isEmpty() ) {
        showWarning( tr( "Type name cannot be empty" ) );
        return false;
    }

    RDB::TableConstIterator<TypeRow> it( dataManager->types() );
    while ( it.next() ) {
        if ( it.get()->name() == name ) {
            showWarning( tr( "A type with this name already exists" ) );
            return false;
        }
    }

    return true;
}

bool TypesDialog::checkAttributeName( int typeId, const QString& name )
{
    if ( name.isEmpty() ) {
        showWarning( tr( "Attribute name cannot be empty" ) );
        return false;
    }

    RDB::ForeignConstIterator<AttributeRow> it( dataManager->attributes()->parentIndex(), typeId );
    while ( it.next() ) {
        if ( it.get()->name() == name ) {
            showWarning( tr( "An attribute with this name already exists" ) );
            return false;
        }
    }

    return true;
}

AddTypeDialog::AddTypeDialog( QWidget* parent ) : TypesDialog( parent )
{
    setWindowTitle( tr( "Add Type" ) );
    setPrompt( tr( "Create a new type:" ) );
    setPromptPixmap( IconLoader::pixmap( "type-new", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( tr( "New Type" ), this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

AddTypeDialog::~AddTypeDialog()
{
}

void AddTypeDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( !checkTypeName( name ) )
        return;

    TypesBatch* batch = new TypesBatch();
    batch->addType( name );

    executeBatch( batch );
}

RenameTypeDialog::RenameTypeDialog( int typeId, QWidget* parent ) : TypesDialog( parent ),
    m_typeId( typeId )
{
    const TypeRow* type = dataManager->types()->find( typeId );
    m_oldName = type ? type->name() : QString();

    setWindowTitle( tr( "Rename Type" ) );
    setPrompt( tr( "Enter the new name of type <b>%1</b>:" ).arg( m_oldName ) );
    setPromptPixmap( IconLoader::pixmap( "edit-rename", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( m_oldName, this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

RenameTypeDialog::~RenameTypeDialog()
{
}

void RenameTypeDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( name == m_oldName ) {
        QDialog::accept();
        return;
    }

    if ( !checkTypeName( name ) )
        return;

    TypesBatch* batch = new TypesBatch();
    batch->renameType( m_typeId, name );

    executeBatch( batch );
}

DeleteTypeDialog::DeleteTypeDialog( int typeId, QWidget* parent ) : TypesDialog( parent ),
    m_typeId( typeId )
{
    const TypeRow* type = dataManager->types()->find( typeId );
    QString name = type ? type->name() : QString();

    setWindowTitle( tr( "Delete Type" ) );
    setPrompt( tr( "Are you sure you want to delete type <b>%1</b> and all its attributes?" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "edit-delete", 22 ) );

    showInfo( tr( "Confirm deletion" ) );

    setContentLayout( NULL, true );
}

DeleteTypeDialog::~DeleteTypeDialog()
{
}

void DeleteTypeDialog::accept()
{
    TypesBatch* batch = new TypesBatch();
    batch->deleteType( m_typeId );

    executeBatch( batch );
}

AttributeDialog::AttributeDialog( QWidget* parent ) : TypesDialog( parent ),
    m_nameEdit( NULL ),
    m_typeCombo( NULL ),
    m_detailsBox( NULL ),
    m_requiredCheck( NULL ),
    m_valueGroup( NULL ),
    m_valueLayout( NULL ),
    m_editor( NULL )
{
}

AttributeDialog::~AttributeDialog()
{
}

void AttributeDialog::createWidgets( bool withName )
{
    int offset = withName ? 1 : 0;

    QGridLayout* layout = new QGridLayout();
    layout->setColumnStretch( 1, 1 );

    if ( withName ) {
        QLabel* nameLabel = new QLabel( tr( "&Name:" ), this );
        layout->addWidget( nameLabel, 0, 0 );

        m_nameEdit = new QLineEdit( tr( "New Attribute" ), this );
        m_nameEdit->setMaxLength( 40 );
        layout->addWidget( m_nameEdit, 0, 1 );

        nameLabel->setBuddy( m_nameEdit );
    }

    QLabel* typeLabel = new QLabel( tr( "&Type:" ), this );
    layout->addWidget( typeLabel, 0 + offset, 0 );

    m_typeCombo = new QComboBox( this );
    layout->addWidget( m_typeCombo, 0 + offset, 1 );

    typeLabel->setBuddy( m_typeCombo );

    for ( int i = DefinitionInfo::First; i <= DefinitionInfo::Last; i++ )
        m_typeCombo->addItem( DefinitionHelper::typeName( (DefinitionInfo::Type)i ) );

    QLabel* detailsLabel = new QLabel( tr( "&Details:" ), this );
    layout->addWidget( detailsLabel, 1 + offset, 0 );

    m_detailsBox = new PopEditBox( this );
    layout->addWidget( m_detailsBox, 1 + offset, 1 );

    detailsLabel->setBuddy( m_detailsBox );

    m_requiredCheck = new QCheckBox( tr( "&Attribute is required" ), this );
    layout->addWidget( m_requiredCheck, 2 + offset, 0, 1, 2 );

    m_valueButton1 = new QRadioButton( tr( "No d&efault value" ), this );
    layout->addWidget( m_valueButton1, 3 + offset, 0, 1, 2 );

    m_valueLayout = new QHBoxLayout();
    layout->addLayout( m_valueLayout, 4 + offset, 0, 1, 2 );

    m_valueButton2 = new QRadioButton( tr( "Default &value:" ), this );
    m_valueLayout->addWidget( m_valueButton2, 0  );

    m_valueGroup = new QButtonGroup( this );
    m_valueGroup->addButton( m_valueButton1, 0 );
    m_valueGroup->addButton( m_valueButton2, 1 );

    connect( m_typeCombo, SIGNAL( activated( int ) ), this, SLOT( typeActivated( int ) ) );
    connect( m_detailsBox, SIGNAL( buttonClicked() ), this, SLOT( detailsClicked() ) );
    connect( m_requiredCheck, SIGNAL( toggled( bool ) ), this, SLOT( requiredToggled( bool ) ) );

    m_info = DefinitionInfo();
    m_info.setType( DefinitionInfo::First );

    updateWidgets();

    setContentLayout( layout, true );

    if ( withName )
        m_nameEdit->setFocus();
    else
        m_typeCombo->setFocus();
}

void AttributeDialog::setDefinitionInfo( DefinitionInfo& info )
{
    m_info = info;

    m_typeCombo->setCurrentIndex( m_info.type() - DefinitionInfo::First );

    updateWidgets();
}

DefinitionInfo AttributeDialog::definitionInfo()
{
    DefinitionInfo info = m_info;

    if ( m_requiredCheck->isChecked() )
        info.setRequired( true );

    if ( m_valueGroup->checkedId() == 1 ) {
        QString value = m_editor->value();
        if ( value.isEmpty() ) {
            showWarning( tr( "No default value entered" ) );
            return DefinitionInfo();
        }
        info.setDefaultValue( value );
    }

    return info;
}

QString AttributeDialog::attributeName() const
{
    return m_nameEdit->text();
}

void AttributeDialog::typeActivated( int index )
{
    m_info = DefinitionInfo();
    m_info.setType( (DefinitionInfo::Type)( index + DefinitionInfo::First ) );

    updateWidgets();
}

void AttributeDialog::detailsClicked()
{
    MetadataDialogBase* dialog = MetadataDialogFactory::createMetadataDialog( m_info.type(), this );
    dialog->setDefinitionInfo( m_info );

    if ( dialog->exec() == QDialog::Accepted ) {
        m_info = dialog->definitionInfo();
        updateMetadata();
    }

    delete dialog;
}

void AttributeDialog::requiredToggled( bool on )
{
    m_valueButton1->setEnabled( !on );

    if ( on ) {
        m_valueGroup->button( 1 )->setChecked( true );
        m_editor->widget()->setEnabled( true );
    }
}

void AttributeDialog::updateWidgets()
{
    m_detailsBox->setText( DefinitionHelper::metadataDetails( m_info ) );

    delete m_editor;

    m_editor = ValueEditorFactory::createValueEditor( m_info, 0, this, this );
    m_valueLayout->addWidget( m_editor->widget(), 1 );

    connect( m_valueButton2, SIGNAL( toggled( bool ) ), m_editor->widget(), SLOT( setEnabled( bool ) ) );
    connect( m_valueButton2, SIGNAL( clicked() ), m_editor->widget(), SLOT( setFocus() ) );

    m_requiredCheck->setChecked( m_info.required() );

    QString value = m_info.defaultValue();

    if ( value.isEmpty() ) {
        m_valueGroup->button( 0 )->setChecked( true );
        m_editor->widget()->setEnabled( false );
    } else {
        m_valueGroup->button( 1 )->setChecked( true );
        m_editor->setValue( value );
    }

    if ( isVisible() )
        m_editor->widget()->show();
}

void AttributeDialog::updateMetadata()
{
    m_detailsBox->setText( DefinitionHelper::metadataDetails( m_info ) );

    QString value = m_editor->value();

    delete m_editor;

    m_editor = ValueEditorFactory::createValueEditor( m_info, 0, this, this );
    m_editor->setValue( value );

    m_valueLayout->addWidget( m_editor->widget(), 1 );

    connect( m_valueButton2, SIGNAL( toggled( bool ) ), m_editor->widget(), SLOT( setEnabled( bool ) ) );
    connect( m_valueButton2, SIGNAL( clicked() ), m_editor->widget(), SLOT( setFocus() ) );

    if ( m_valueGroup->checkedId() == 0 )
        m_editor->widget()->setEnabled( false );

    m_editor->widget()->show();
}

AddAttributeDialog::AddAttributeDialog( int typeId, QWidget* parent ) : AttributeDialog( parent ),
    m_typeId( typeId )
{
    const TypeRow* type = dataManager->types()->find( typeId );
    QString name = type ? type->name() : QString();

    setWindowTitle( tr( "Add Attribute" ) );
    setPrompt( tr( "Create a new attribute in type <b>%1</b>:" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "attribute-new", 22 ) );

    createWidgets( true );
}

AddAttributeDialog::~AddAttributeDialog()
{
}

void AddAttributeDialog::accept()
{
    QString name = attributeName();
    if ( !checkAttributeName( m_typeId, name ) )
        return;

    DefinitionInfo info = definitionInfo();
    if ( !info.isValid() )
        return;

    DefinitionBuilder builder;
    QString definition = builder.build( info );

    TypesBatch* batch = new TypesBatch();
    batch->addAttribute( m_typeId, name, definition );

    executeBatch( batch );
}

ModifyAttributeDialog::ModifyAttributeDialog( int attributeId, QWidget* parent ) : AttributeDialog( parent ),
    m_attributeId( attributeId )
{
    const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
    QString name = attribute ? attribute->name() : QString();
    m_oldDefinition = attribute ? attribute->definition() : QString();

    setWindowTitle( tr( "Modify Attribute" ) );
    setPrompt( tr( "Modify attribute <b>%1</b>:" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "edit-modify", 22 ) );

    createWidgets( false );

    DefinitionParser parser;
    DefinitionInfo info = parser.parse( m_oldDefinition );

    setDefinitionInfo( info );
}

ModifyAttributeDialog::~ModifyAttributeDialog()
{
}

void ModifyAttributeDialog::accept()
{
    DefinitionInfo info = definitionInfo();
    if ( !info.isValid() )
        return;

    DefinitionBuilder builder;
    QString definition = builder.build( info );

    TypesBatch* batch = new TypesBatch();
    batch->modifyAttribute( m_attributeId, definition );

    executeBatch( batch );
}

RenameAttributeDialog::RenameAttributeDialog( int attributeId, QWidget* parent ) : TypesDialog( parent ),
    m_attributeId( attributeId )
{
    const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
    m_oldName = attribute ? attribute->name() : QString();

    setWindowTitle( tr( "Rename Attribute" ) );
    setPrompt( tr( "Enter the new name of attribute <b>%1</b>:" ).arg( m_oldName ) );
    setPromptPixmap( IconLoader::pixmap( "edit-rename", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( m_oldName, this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

RenameAttributeDialog::~RenameAttributeDialog()
{
}

void RenameAttributeDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( name == m_oldName ) {
        QDialog::accept();
        return;
    }

    const AttributeRow* attribute = dataManager->attributes()->find( m_attributeId );
    int typeId = attribute ? attribute->typeId() : 0;

    if ( !checkAttributeName( typeId, name ) )
        return;

    TypesBatch* batch = new TypesBatch();
    batch->renameAttribute( m_attributeId, name );

    executeBatch( batch );
}

DeleteAttributeDialog::DeleteAttributeDialog( int attributeId, QWidget* parent ) : TypesDialog( parent ),
    m_attributeId( attributeId )
{
    const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
    QString name = attribute ? attribute->name() : QString();

    setWindowTitle( tr( "Delete Attribute" ) );
    setPrompt( tr( "Are you sure you want to delete attribute <b>%1</b> and all its values?" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "edit-delete", 22 ) );

    showInfo( tr( "Confirm deletion" ) );

    setContentLayout( NULL, true );
}

DeleteAttributeDialog::~DeleteAttributeDialog()
{
}

void DeleteAttributeDialog::accept()
{
    TypesBatch* batch = new TypesBatch();
    batch->deleteAttribute( m_attributeId );

    executeBatch( batch );
}

#include "typedialogs.moc"
