/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "settingsdialog.h"

#include <QLayout>
#include <QLabel>
#include <QTabWidget>
#include <QCheckBox>
#include <QComboBox>
#include <QSpinBox>
#include <QDialogButtonBox>
#include <QPushButton>

#include "configdata.h"
#include "iconloader.h"

using namespace WebIssues;

SettingsDialog::SettingsDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Settings" ) );

    QVBoxLayout* mainLayout = new QVBoxLayout( this );

    QTabWidget* tabWidget = new QTabWidget( this );

    QWidget* generalTab = new QWidget( tabWidget );
    QGridLayout* generalLayout = new QGridLayout( generalTab );

    m_dockCheckBox = new QCheckBox( tr( "&Dock main window in system tray" ), generalTab );
    generalLayout->addWidget( m_dockCheckBox, 0, 0, 1, 2 );

    connect( m_dockCheckBox, SIGNAL( stateChanged( int ) ), this, SLOT( dockChanged() ) );

    QFrame* separator = new QFrame( generalTab );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    generalLayout->addWidget( separator, 1, 0, 1, 2 );

    QLabel* showLabel = new QLabel( tr( "&Show main window at startup:" ), generalTab );
    generalLayout->addWidget( showLabel, 2, 0 );

    m_showComboBox = new QComboBox( generalTab );
    m_showComboBox->addItem( tr( "Never" ) );
    m_showComboBox->addItem( tr( "Automatically" ) );
    m_showComboBox->addItem( tr( "Always" ) );
    generalLayout->addWidget( m_showComboBox, 2, 1 );

    showLabel->setBuddy( m_showComboBox );

    QLabel* reconnectLabel = new QLabel( tr( "&Restore last connection at startup:" ), generalTab );
    generalLayout->addWidget( reconnectLabel, 3, 0 );

    m_reconnectComboBox = new QComboBox( generalTab );
    m_reconnectComboBox->addItem( tr( "Never" ) );
    m_reconnectComboBox->addItem( tr( "Automatically" ) );
    m_reconnectComboBox->addItem( tr( "Always" ) );
    generalLayout->addWidget( m_reconnectComboBox, 3, 1 );

    reconnectLabel->setBuddy( m_reconnectComboBox );

#if defined( Q_WS_WIN )
    QFrame* separator2 = new QFrame( generalTab );
    separator2->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    generalLayout->addWidget( separator2, 4, 0, 1, 2 );

    m_autoStartCheckBox = new QCheckBox( tr( "Start &WebIssues when I start Windows" ), generalTab );
    generalLayout->addWidget( m_autoStartCheckBox, 5, 0, 1, 2 );
#endif

    generalLayout->setRowStretch( generalLayout->rowCount(), 1 );
    generalLayout->setColumnStretch( 0, 1 );

    tabWidget->addTab( generalTab, IconLoader::icon( "configure" ), tr( "General" ) );

    QWidget* advancedTab = new QWidget( tabWidget );
    QGridLayout* advancedLayout = new QGridLayout( advancedTab );

    QLabel* intervalLabel = new QLabel( tr( "I&nterval of periodic data updates:" ), advancedTab );
    advancedLayout->addWidget( intervalLabel, 0, 0 );

    m_intervalSpin = new QSpinBox( advancedTab );
    m_intervalSpin->setRange( 1, 120 );
    m_intervalSpin->setSuffix( tr( " min" ) );
    advancedLayout->addWidget( m_intervalSpin, 0, 1 );

    intervalLabel->setBuddy( m_intervalSpin );

    QFrame* separator3 = new QFrame( advancedTab );
    separator3->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    advancedLayout->addWidget( separator3, 1, 0, 1, 2 );

    QLabel* actionLabel = new QLabel( tr( "&Default action for attachments:" ), advancedTab );
    advancedLayout->addWidget( actionLabel, 2, 0 );

    m_actionComboBox = new QComboBox( advancedTab );
    m_actionComboBox->addItem( tr( "Ask what to do" ) );
    m_actionComboBox->addItem( tr( "Open" ) );
    m_actionComboBox->addItem( tr( "Save As..." ) );
    advancedLayout->addWidget( m_actionComboBox, 2, 1 );

    actionLabel->setBuddy( m_actionComboBox );

    QLabel* cacheLabel = new QLabel( tr( "&Maximum size of attachments cache:" ), advancedTab );
    advancedLayout->addWidget( cacheLabel, 3, 0 );

    m_cacheSpin = new QSpinBox( advancedTab );
    m_cacheSpin->setRange( 1, 200 );
    m_cacheSpin->setSuffix( tr( " MB" ) );
    advancedLayout->addWidget( m_cacheSpin, 3, 1 );

    cacheLabel->setBuddy( m_cacheSpin );

    QLabel* warnLabel = new QLabel( tr( "&Warn when adding attachments larger than:" ), advancedTab );
    advancedLayout->addWidget( warnLabel, 4, 0 );

    m_warnSpin = new QSpinBox( advancedTab );
    m_warnSpin->setRange( 0, 50 );
    m_warnSpin->setSuffix( tr( " MB" ) );
    m_warnSpin->setSpecialValueText( tr( "Off" ) );
    advancedLayout->addWidget( m_warnSpin, 4, 1 );

    warnLabel->setBuddy( m_warnSpin );

    advancedLayout->setRowStretch( 5, 1 );
    advancedLayout->setColumnStretch( 0, 1 );

    tabWidget->addTab( advancedTab, IconLoader::icon( "gear" ), tr( "Advanced" ) );

    mainLayout->addWidget( tabWidget );

    mainLayout->addSpacing( 10 );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Apply | QDialogButtonBox::Cancel,
        Qt::Horizontal, this );
    mainLayout->addWidget( buttonBox );

    buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    buttonBox->button( QDialogButtonBox::Apply )->setText( tr( "&Apply" ) );
    buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );
    connect( buttonBox->button( QDialogButtonBox::Apply ), SIGNAL( clicked() ), this, SLOT( apply() ) );

    m_dockCheckBox->setChecked( configData->dockWindow() );
    m_reconnectComboBox->setCurrentIndex( configData->reconnectAtStartup() );

    m_intervalSpin->setValue( configData->updateInterval() );
    m_actionComboBox->setCurrentIndex( configData->defaultAttachmentAction() );
    m_cacheSpin->setValue( configData->attachmentsCacheSize() );
    m_warnSpin->setValue( configData->warnAttachmentSize() );

#if defined( Q_WS_WIN )
    m_autoStartCheckBox->setChecked( configData->checkAutoStart() );
#endif

    dockChanged();
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::accept()
{
    apply();

    QDialog::accept();
}

void SettingsDialog::apply()
{
    configData->setDockWindow( m_dockCheckBox->isChecked() );
    configData->setShowAtStartup( (RestoreOption)m_showComboBox->currentIndex() );
    configData->setReconnectAtStartup( (RestoreOption)m_reconnectComboBox->currentIndex() );

    configData->setUpdateInterval( m_intervalSpin->value() );
    configData->setDefaultAttachmentAction( (AttachmentAction)m_actionComboBox->currentIndex() );
    configData->setAttachmentsCacheSize( m_cacheSpin->value() );
    configData->setWarnAttachmentSize( m_warnSpin->value() );

#if defined( Q_WS_WIN )
    configData->setAutoStart( m_autoStartCheckBox->isChecked() );
#endif

    configData->saveSettings();
}

void SettingsDialog::dockChanged()
{
    if ( m_dockCheckBox->isChecked() ) {
        m_showComboBox->setCurrentIndex( configData->showAtStartup() );
        m_showComboBox->setEnabled( true );
    } else {
        m_showComboBox->setCurrentIndex( 2 );
        m_showComboBox->setEnabled( false );
    }
}

#include "settingsdialog.moc"
