/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Bookmarks Sync.
 *
 * The Initial Developer of the Original Code is Mozilla.
 * Portions created by the Initial Developer are Copyright (C) 2007
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *  Dan Mills <thunder@mozilla.com>
 *  Chris Beard <cbeard@mozilla.com>
 *  Dan Mosedale <dmose@mozilla.org>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

if (typeof(Cc) == "undefined")
  var Cc = Components.classes;
if (typeof(Ci) == "undefined")
  var Ci = Components.interfaces;
if (typeof Cu == "undefined")
  var Cu = Components.utils;
if (typeof Cr == "undefined")
  var Cr = Components.results;

function WeaveWindow() {
  let obs = [["weave:service:ready", "onReady"],
    ["weave:service:sync:start", "onActivityStart"],
    ["weave:service:sync:finish", "onSyncFinish"],
    ["weave:service:sync:error", "onSyncError"],
    ["weave:service:sync:delayed", "onSyncDelay"],
    ["weave:service:setup-complete", "onLoginFinish"],
    ["weave:service:start-over", "onStartOver"],
    ["weave:service:login:start", "onActivityStart"],
    ["weave:service:login:finish", "onLoginFinish"],
    ["weave:service:login:error", "onLoginError"],
    ["weave:service:logout:finish", "onLogout"]];

  if (window.location.href == getBrowserURL()) {
    obs.push(["weave:notification:added", "onNotificationAdded"],
             ["weave:notification:removed", "onNotificationRemoved"]);
  }

  // Add the observers now and remove them on unload
  let weaveWin = this;
  let addRem = function(add) obs.forEach(function([topic, func])
    Weave.Svc.Obs[add ? "add" : "remove"](topic, weaveWin[func], weaveWin));
  addRem(true);
  window.addEventListener("unload", function() addRem(false), false);

  if (Weave.Svc.Prefs.get("lastversion") == "firstrun") {
    setTimeout(this.openSetup, 500);
    Weave.Svc.Prefs.set("lastversion", Weave.WEAVE_VERSION);

  } else if (Weave.Svc.Prefs.get("lastversion") != Weave.WEAVE_VERSION) {
    setTimeout(function() window.openUILinkIn(Weave.Service.updatedURL, "tab"), 500);
    Weave.Svc.Prefs.set("lastversion", Weave.WEAVE_VERSION);
  }

  // Only show the activity log for dev-channel releases
  if (Weave.WEAVE_CHANNEL == "dev")
    document.getElementById("sync-openlogitem").hidden = false;

  this.updateUI();
  // makes sure notifications are set up right for new windows
  this.onNotificationRemoved();
}
WeaveWindow.prototype = {
  onReady: function onReady() {
    Weave.Svc.Obs.remove("weave:service:ready", arguments.callee);

    // Wait a bit before accessing window.gBrowser. If we do it too
    // early, it will trigger the toolbar constructors too early.
    Weave.Utils.delay(function() {
       // Add one second delay for each busy tab in every window
      let enum = Weave.Svc.WinMediator.getEnumerator("navigator:browser");
      let wait = 3;
      while (enum.hasMoreElements()) {
        Array.forEach(enum.getNext().gBrowser.mTabs, function(tab) {
          wait += tab.hasAttribute("busy");
        });
      }
      Weave.Service.delayedAutoConnect(wait);
    }, 2000);
  },

  get _isTopBrowserWindow() {
    // TODO: This code is mostly just a workaround that ensures that only one
    // browser window ever performs any actions that are meant to only
    // be performed once in response to a weave event.  Ideally, such code
    // should not be handled by browser windows, but instead by e.g. actual
    // singleton services.
    var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                       .getService(Components.interfaces.nsIWindowMediator);
    var win = wm.getMostRecentWindow("navigator:browser");
    return (win == window);
  },

  get _stringBundle() {
    let stringBundle = document.getElementById("weaveStringBundle");
    this.__defineGetter__("_stringBundle",
                          function() { return stringBundle; });
    return this._stringBundle;
  },

  _needsSetup: function() {
    return Weave.Status.service == Weave.CLIENT_NOT_CONFIGURED ||
           Weave.Svc.Prefs.get("firstSync", "") == "notReady";
  },

  updateUI: function updateUI() {
    document.getElementById("sync-menu").hidden = this._needsSetup();
    document.getElementById("sync-setup").hidden = !this._needsSetup();
    if (window.location.href == getBrowserURL()) {
      let showLabel = !Weave.Service.isLoggedIn && !this._needsSetup();
      let button = document.getElementById("sync-status-button");
      button.setAttribute("class", showLabel ? "statusbarpanel-iconic-text" 
                                             : "statusbarpanel-iconic");
      button.image = "chrome://weave/skin/firefox/sync-16x16.png";

      if (!Weave.Service.isLoggedIn) {
        // If we're not set up yet, the tooltip should read "Set up
        // sync" or similar, but we threw the string away in 1.4 :(
        button.removeAttribute("tooltiptext");
      }
    }
  },

  handleSyncButton: function WeaveWin_handleSyncButton() {
    if (Weave.Service.isLoggedIn)
      this.doSync();
    else if (this._needsSetup())
      this.openSetup();
    else
      this.doLogin();
  },

  onStartOver: function onStartOver() {
    this.updateUI();
  },

  onActivityStart: function WeaveWin_onActivityStart() {
    if (window.location.href == getBrowserURL()) {
      document.getElementById("sync-status-button").image = 
        "chrome://weave/skin/firefox/sync-throbber-16x16-active.apng";
    }
  },

  onLoginError: function WeaveWin_onLoginError() {
    // if login fails, any other notifications are essentially moot
    Weave.Notifications.removeAll();

    // if we haven't set up the client, don't show errors
    if (this._needsSetup()) {
      this.updateUI();
      return;
    }

    let title = this._stringBundle.getString("error.login.title");
    let reason = Weave.Utils.getErrorString(Weave.Status.login);
    let description =
      this._stringBundle.getFormattedString("error.login.description", [reason]);
    let buttons = [];
    buttons.push(new Weave.NotificationButton(
      this._stringBundle.getString("error.login.prefs.label"),
      this._stringBundle.getString("error.login.prefs.accesskey"),
      function() { gWeaveWin.openPrefs(); return true; }
    ));

    let notification = new Weave.Notification(title, description, null,
                                              Weave.Notifications.PRIORITY_WARNING, buttons);
    Weave.Notifications.replaceTitle(notification);
    this.updateUI();
  },

  onLoginFinish: function WeaveWin_onLoginFinish() {
    // Clear out any login failure notifications
    let title = this._stringBundle.getString("error.login.title");
    Weave.Notifications.removeAll(title);

    this.updateUI();
    this._updateLastSyncItem();
  },

  onLogout: function WeaveWin_onLogout() {
    this.updateUI();
  },

  _onSyncEnd: function WeaveWin__onSyncEnd(status) {
    let title = this._stringBundle.getString("error.sync.title");
    if (!status) {
      let error = Weave.Utils.getErrorString(Weave.Status.sync);
      let description = this._stringBundle
                            .getFormattedString("error.sync.description", [error]);

      let priority = Weave.Notifications.PRIORITY_WARNING;
      let buttons = [];

      // Check if the client is outdated in some way
      let outdated = Weave.Status.sync == Weave.VERSION_OUT_OF_DATE;
      for (let [engine, reason] in Iterator(Weave.Status.engines))
        outdated = outdated || reason == Weave.VERSION_OUT_OF_DATE;

      if (outdated) {
        description = Weave.Str.sync.get("error.sync.needUpdate.description");
        let label = Weave.Str.sync.get("error.sync.needUpdate.label");
        let accesskey = Weave.Str.sync.get("error.sync.needUpdate.accesskey");
        buttons.push(new Weave.NotificationButton(label, accesskey, function() {
          let theEM = Weave.Svc.WinMediator.getMostRecentWindow("Extension:Manager");
          if (theEM != null) {
            theEM.focus();
            theEM.showView("extensions");
          }
          else {
            const EMURL = "chrome://mozapps/content/extensions/extensions.xul";
            const EMFEATURES = "chrome,menubar,extra-chrome,toolbar,dialog=no,resizable";
            window.openDialog(EMURL, "", EMFEATURES, "extensions");
          }
          return true;
        }));
      }
      else if (!Weave.Status.enforceBackoff) {
        priority = Weave.Notifications.PRIORITY_INFO;
        buttons.push(new Weave.NotificationButton(
          this._stringBundle.getString("error.sync.tryAgainButton.label"),
          this._stringBundle.getString("error.sync.tryAgainButton.accesskey"),
          function() { gWeaveWin.doSync(); return true; }
        ));
      }

      let notification =
        new Weave.Notification(title, description, null, priority, buttons);
      Weave.Notifications.replaceTitle(notification);
    }
    // Clear out sync failures on a successful sync
    else
      Weave.Notifications.removeAll(title);

    if (this._wasDelayed && Weave.Status.sync != Weave.NO_SYNC_NODE_FOUND) {
      title = this._stringBundle.getString("error.sync.no_node_found.title");
      Weave.Notifications.removeAll(title);
      this._wasDelayed = false;
    }

    this.updateUI();
    this._updateLastSyncItem();
  },

  onSyncFinish: function WeaveWin_onSyncFinish(subject, data) {
    this._onSyncEnd(true);
  },

  onSyncError: function WeaveWin_onSyncError(subject, data) {
    this._onSyncEnd(false);
  },

  onSyncDelay: function WeaveWin_onSyncDelay(subject, data) {
    // basically, we want to just inform users that stuff is going to take a while
    let title = this._stringBundle.getString("error.sync.no_node_found.title");
    let description = this._stringBundle.getString("error.sync.no_node_found");
    let notification = new Weave.Notification(title, description, null, Weave.Notifications.PRIORITY_INFO);
    Weave.Notifications.replaceTitle(notification);
    this._wasDelayed = true;
  },

  openPrefs: function openPrefs() {
    let pane = "paneWeaveServices";
    switch (Weave.Svc.AppInfo.ID) {
      case Weave.FIREFOX_ID:
        openPreferences(pane);
        break;

      case Weave.SEAMONKEY_ID:
        goPreferences(pane);
        break;
    }
  },

  openSetup: function openSetup() {
    var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                       .getService(Components.interfaces.nsIWindowMediator);
    var win = wm.getMostRecentWindow("Weave:AccountSetup");
    if (win)
      win.focus();
    else {
      window.openDialog("chrome://weave/content/firefox/setup.xul",
                        "weaveSetup", "centerscreen,chrome,resizable=no");
    }
  },

  doLogin: function WeaveWin_doLogout(event) {
    setTimeout(function() Weave.Service.login(), 0);
  },

  doLogout: function WeaveWin_doLogout(event) {
    setTimeout(function() Weave.Service.logout(), 0);
  },

  doSync: function WeaveWin_doSync(event) {
    setTimeout(function() Weave.Service.sync(), 0);
  },

  doOpenActivityLog: function WeaveWin_doOpenActivityLog(event) {
    window.openUILinkIn("about:sync-log", "tab");
  },

  doPopup: function WeaveWin_doPopup(event) {
    this._updateLastSyncItem();

    let loginItem = document.getElementById("sync-loginitem");
    let logoutItem = document.getElementById("sync-logoutitem");
    let syncItem = document.getElementById("sync-syncnowitem");

    // Don't allow "login" to be selected in some cases
    let offline = Weave.Svc.IO.offline;
    let locked = Weave.Service.locked;
    let noUser = Weave.Service.username == "";
    let notReady = offline || locked || noUser;
    loginItem.setAttribute("disabled", notReady);
    logoutItem.setAttribute("disabled", notReady);

    // Don't allow "sync now" to be selected in some cases
    let loggedIn = Weave.Service.isLoggedIn;
    let noNode = Weave.Status.sync == Weave.NO_SYNC_NODE_FOUND;
    let disableSync = notReady || !loggedIn || noNode;
    syncItem.setAttribute("disabled", disableSync);

    // Only show one of login/logout
    loginItem.setAttribute("hidden", loggedIn);
    logoutItem.setAttribute("hidden", !loggedIn);
  },

  onNotificationAdded: function WeaveWin_onNotificationAdded() {
    let button = document.getElementById("sync-notifications-button");
    if (!button)
      return;

    button.hidden = false;    
    let notifications = Weave.Notifications.notifications;
    let priority = 0;
    for (let i = 0; i < notifications.length; i++) {
      if (notifications[i].priority > priority) {
        priority = notifications[i].priority;
        title = notifications[i].title;
      }
    }

    let image = priority >= Weave.Notifications.PRIORITY_WARNING ?
                "chrome://global/skin/icons/warning-16.png" :
                "chrome://global/skin/icons/information-16.png";
    button.setAttribute("image", image);
    button.setAttribute("label", title);
  },

  onNotificationRemoved: function WeaveWin_onNotificationRemoved() {
    if (Weave.Notifications.notifications.length == 0) {
      let button = document.getElementById("sync-notifications-button");
      if (button)
        button.hidden = true;
    }
    else
      // Display remaining notifications (if any).
      this.onNotificationAdded();
  },

  _updateLastSyncItem: function WeaveWin__updateLastSyncItem() {
    let lastSync = Weave.Svc.Prefs.get("lastSync");
    if (!lastSync)
      return;

    let lastSyncItem = document.getElementById("sync-lastsyncitem");
    if (!lastSyncItem)
      return;

    // Show the day-of-week and time (HH:MM) of last sync
    let lastSyncDate = new Date(lastSync).toLocaleFormat("%a %H:%M");
    let lastSyncLabel =
      this._stringBundle.getFormattedString("lastSync.label", [lastSyncDate]);
    lastSyncItem.setAttribute("label", lastSyncLabel);
    lastSyncItem.setAttribute("hidden", "false");
    document.getElementById("sync-lastsyncsep").hidden = false;

    if (window.location.href == getBrowserURL())
      document.getElementById("sync-status-button")
              .setAttribute("tooltiptext", lastSyncLabel);
  }
};

let gWeaveWin;

window.addEventListener("load", function(e) { gWeaveWin = new WeaveWindow(); }, false);
