#ifndef SET_OBJECT_H
#define SET_OBJECT_H

// Copyright (c) 1994-1996 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY 
// FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT 
// PARTIES PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
// PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE 
// PROGRAM IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME 
// THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 

// IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING 
// WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR 
// REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR 
// DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL 
// DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THE PROGRAM 
// (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED 
// INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE OF 
// THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS), EVEN IF SUCH HOLDER 
// OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. 

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped.h"
#include <iostream>
using std::ostream;

/** The SetObject Class.

    This class is used for encapsulating events in different event
    sets.
 */
template <class Element>
class SetObject {
public:
   /**@name Public Class Methods of SetObject. */
   //@{

   /** Constructor.

       @param currentTime Current time  of event.
       @param currentElement Element to be encapsulated.
   */
   SetObject( const VTime& currentTime, 
	      const Element *currentElement ):
     processed(false), 
     time(currentTime.clone()), 
     element(currentElement){}

   /** Constructor.

       @param currentTime Current time  of event.
   */
  SetObject(const VTime &currentTime ):
    processed(false), time(currentTime.clone()), element(NULL){}
  
  SetObject( const SetObject &toCopy ):
    processed(toCopy.processed), 
    time(toCopy.time->clone()), 
    element(toCopy.element){}

  
   /// Destructor.
   ~SetObject(){
     delete time;
     time = 0;
   }

   /** Get time of element.

       @return Time of element.
   */
   const VTime &getTime() const {
     ASSERT( time != 0 );
     return *time;
   }

   /** Get a pointer to the element.

       @return Pointer to the element.
   */
  const Element *getElement() const {
    return element;
  }

   /// Overloaded operator =
   SetObject<Element> &operator=( const SetObject &rhs ) const {
     processed = rhs.processed;
     time = rhs.time->clone();
     element = rhs.element;
     return *this;
   }

   /// Overloaded operator <
   bool operator<( const SetObject &rhs ) const {
     return getTime() < rhs.getTime();
   }
  
   /// Overloaded operator ==
   bool operator==( const SetObject &rhs ) const {
      return getTime() == rhs.getTime();
   }
  
   /// Overloaded operator >
   bool operator>( const SetObject &rhs ) const {
      return getTime() > rhs.getTime();
   }
  
   void setProcessed() {
      processed = true;
   }
   
   bool processed;
   
   //@} // End of Public Class Methods of SetObject.

protected:

private:
   /**@name Protected Class Attributes of SetObject. */
   //@{

   /// Timestamp of element.
  const VTime *time;

  /// Pointer to the element.
  const Element *element;

   //@} // End of Protected Class Attributes of SetObject.
};

/// Overloaded operator <<
template <class Element> 
inline std::ostream &
operator<<( std::ostream& os,  
	    const SetObject<Element> &a ){
  os << a.getTime();
  return os;
}
#endif
