#ifndef HTTPEX_H
#define HTTPEX_H

#include <QtCore>
#include <QtNetwork>

#define DEFAULT_TIMEOUT 30000
#define DEFAULT_TRIES 5

namespace HttpEx
{
  class Base;

  class Interface : public QObject
  {
    Q_OBJECT

    friend class Base;

  protected slots:
    virtual void internalAbort() = 0;

  public:
    Interface(QObject *parent = 0) : QObject(parent) { }
    virtual ~Interface() { }

    virtual qint64 bytesAvailable() const = 0;
    virtual void clearPendingRequests() = 0;
    virtual int close() = 0;
    virtual QIODevice *currentDestinationDevice() const = 0;
    virtual int currentId() const = 0;
    virtual QHttpRequestHeader currentRequest() const = 0;
    virtual QIODevice *currentSourceDevice() const = 0;
    virtual QHttp::Error error() const = 0;
    virtual QString errorString() const = 0;
    virtual int get(const QString &path, QIODevice *to = 0) = 0;
    virtual bool hasPendingRequests() const = 0;
    virtual int head(const QString &path) = 0;
    virtual QHttpResponseHeader lastResponse() const = 0;
    virtual int post(const QString &path, QIODevice *data, QIODevice *to = 0) = 0;
    virtual int post(const QString &path, const QByteArray &data, QIODevice *to = 0) = 0;
    virtual qint64 read(char *data, qint64 maxlen) = 0;
    virtual QByteArray readAll() = 0;
    virtual int request(const QHttpRequestHeader &header, QIODevice *data = 0, QIODevice *to = 0) = 0;
    virtual int request(const QHttpRequestHeader &header, const QByteArray &data, QIODevice *to = 0) = 0;
    virtual int setHost(const QString &hostName, quint16 port = 80) = 0;
    virtual int setHost(const QString &hostName, QHttp::ConnectionMode mode, quint16 port = 0) = 0;
    virtual int setProxy(const QString &host, int port, const QString &username = QString(), const QString &password = QString()) = 0;
    virtual int setProxy(const QNetworkProxy &proxy) = 0;
    virtual int setSocket(QTcpSocket *socket) = 0;
    virtual int setUser(const QString &userName, const QString &password = QString()) = 0;
    virtual QHttp::State state() const = 0;

  public slots:
    virtual void abort() = 0;

  signals:
    void authenticationRequired(const QString &hostname, quint16 port, QAuthenticator *authenticator);
    void dataReadProgress(int done, int total);
    void dataSendProgress(int done, int total);
    void done(bool error);
    void proxyAuthenticationRequired(const QNetworkProxy &proxy, QAuthenticator *authenticator);
    void readyRead(const QHttpResponseHeader &resp);
    void requestFinished(int id, bool error);
    void requestStarted(int id);
    void responseHeaderReceived(const QHttpResponseHeader &resp);
    void stateChanged(int state);
  };

  class Core : public Interface
  {
    Q_OBJECT

    QHttp *_http;

  protected slots:
    void internalAbort() { abort(); }

  public:
    Core(QHttp *http, QObject *parent = 0);
    virtual ~Core() { }

    qint64 bytesAvailable() const { return _http->bytesAvailable(); }
    void clearPendingRequests() { _http->clearPendingRequests(); }
    int close() { return _http->close(); }
    QIODevice *currentDestinationDevice() const { return _http->currentDestinationDevice(); }
    int currentId() const { return _http->currentId(); }
    QHttpRequestHeader currentRequest() const { return _http->currentRequest(); }
    QIODevice *currentSourceDevice() const { return _http->currentSourceDevice(); }
    QHttp::Error error() const { return _http->error(); }
    QString errorString() const { return _http->errorString(); }
    int get(const QString &path, QIODevice *to = 0) { return _http->get(path,to); }
    bool hasPendingRequests() const { return _http->hasPendingRequests(); }
    int head(const QString &path) { return _http->head(path); }
    QHttpResponseHeader lastResponse() const { return _http->lastResponse(); }
    int post(const QString &path, QIODevice *data, QIODevice *to = 0) { return _http->post(path,data,to); }
    int post(const QString &path, const QByteArray &data, QIODevice *to = 0) { return _http->post(path,data,to); }
    qint64 read(char *data, qint64 maxlen) { return _http->read(data,maxlen); }
    QByteArray readAll() { return _http->readAll(); }
    int request(const QHttpRequestHeader &header, QIODevice *data = 0, QIODevice *to = 0) { return _http->request(header,data,to); }
    int request(const QHttpRequestHeader &header, const QByteArray &data, QIODevice *to = 0) { return _http->request(header,data,to); }
    int setHost(const QString &hostName, quint16 port = 80) { return _http->setHost(hostName,port); }
    int setHost(const QString &hostName, QHttp::ConnectionMode mode, quint16 port = 0) { return _http->setHost(hostName,mode,port); }
    int setProxy(const QString &host, int port, const QString &username = QString(), const QString &password = QString()) { return _http->setProxy(host,port,username,password); }
    int setProxy(const QNetworkProxy &proxy) { return _http->setProxy(proxy); }
    int setSocket(QTcpSocket *socket) { return _http->setSocket(socket); }
    int setUser(const QString &userName, const QString &password = QString()) { return _http->setUser(userName,password); }
    QHttp::State state() const { return _http->state(); }

  public slots:
    void abort() { _http->abort(); }
  };

  class Base : public Interface
  {
    Q_OBJECT

    Interface *_interface;

  protected slots:
    void internalAbort() { _interface->internalAbort(); }

  public:
    Base(Interface *interface);
    virtual ~Base() { }

    qint64 bytesAvailable() const { return _interface->bytesAvailable(); }
    void clearPendingRequests() { _interface->clearPendingRequests(); }
    int close() { return _interface->close(); }
    QIODevice *currentDestinationDevice() const { return _interface->currentDestinationDevice(); }
    int currentId() const { return _interface->currentId(); }
    QHttpRequestHeader currentRequest() const { return _interface->currentRequest(); }
    QIODevice *currentSourceDevice() const { return _interface->currentSourceDevice(); }
    QHttp::Error error() const { return _interface->error(); }
    QString errorString() const { return _interface->errorString(); }
    int get(const QString &path, QIODevice *to = 0) { return _interface->get(path,to); }
    bool hasPendingRequests() const { return _interface->hasPendingRequests(); }
    int head(const QString &path) { return _interface->head(path); }
    QHttpResponseHeader lastResponse() const { return _interface->lastResponse(); }
    int post(const QString &path, QIODevice *data, QIODevice *to = 0) { return _interface->post(path,data,to); }
    int post(const QString &path, const QByteArray &data, QIODevice *to = 0) { return _interface->post(path,data,to); }
    qint64 read(char *data, qint64 maxlen) { return _interface->read(data,maxlen); }
    QByteArray readAll() { return _interface->readAll(); }
    int request(const QHttpRequestHeader &header, QIODevice *data = 0, QIODevice *to = 0) { return _interface->request(header,data,to); }
    int request(const QHttpRequestHeader &header, const QByteArray &data, QIODevice *to = 0) { return _interface->request(header,data,to); }
    int setHost(const QString &hostName, quint16 port = 80) { return _interface->setHost(hostName,port); }
    int setHost(const QString &hostName, QHttp::ConnectionMode mode, quint16 port = 0) { return _interface->setHost(hostName,mode,port); }
    int setProxy(const QString &host, int port, const QString &username = QString(), const QString &password = QString()) { return _interface->setProxy(host,port,username,password); }
    int setProxy(const QNetworkProxy &proxy) { return _interface->setProxy(proxy); }
    int setSocket(QTcpSocket *socket) { return _interface->setSocket(socket); }
    int setUser(const QString &userName, const QString &password = QString()) { return _interface->setUser(userName,password); }
    QHttp::State state() const { return _interface->state(); }

  public slots:
    void abort() { _interface->abort(); }
  };

  class Log : public Base
  {
    Q_OBJECT

    QMap<QHttp::ConnectionMode, QString> connectionModes;
    QMap<QHttp::Error, QString> errors;
    QMap<QHttp::State, QString> states;

  private slots:
    void authenticationRequiredSlot(const QString &hostname, quint16 port, QAuthenticator *authenticator);
    void dataReadProgressSlot(int done, int total);
    void dataSendProgressSlot(int done, int total);
    void doneSlot(bool error);
    void proxyAuthenticationRequiredSlot(const QNetworkProxy &proxy, QAuthenticator *authenticator);
    void readyReadSlot(const QHttpResponseHeader &resp);
    void requestFinishedSlot(int id, bool error);
    void requestStartedSlot(int id);
    void responseHeaderReceivedSlot(const QHttpResponseHeader &resp);
    void stateChangedSlot(int state);

  protected slots:
    void internalAbort();

  public:
    Log(Interface *interface);
    virtual ~Log() { }

    void clearPendingRequests();
    int close();
    int get(const QString &path, QIODevice *to = 0);
    int head(const QString &path);
    int post(const QString &path, QIODevice *data, QIODevice *to = 0);
    int post(const QString &path, const QByteArray &data, QIODevice *to = 0);
    int request(const QHttpRequestHeader &header, QIODevice *data = 0, QIODevice *to = 0);
    int request(const QHttpRequestHeader &header, const QByteArray &data, QIODevice *to = 0);
    int setHost(const QString &hostName, quint16 port = 80);
    int setHost(const QString &hostName, QHttp::ConnectionMode mode, quint16 port = 0);
    int setProxy(const QString &host, int port, const QString &username = QString(), const QString &password = QString());
    int setProxy(const QNetworkProxy &proxy);
    int setSocket(QTcpSocket *socket);
    int setUser(const QString &userName, const QString &password = QString());

  public slots:
    void abort();
  };

  class Timeout : public Base
  {
    Q_OBJECT

    QTimer *timer;

  private slots:
    void evaluateState(int state);
    void processTimeout();

  public:
    Timeout(Interface *interface);
    virtual ~Timeout() { }

    void setTimeout(int timeout);
    int timeout() const;
  };

  class Retry : public Base
  {
    Q_OBJECT

    enum States { Idle, Init, StoreRequest, Downloading, Closed };

    States _state;
    QByteArray _data;
    int _tries;
    int _count;
    int _total;
    int _current;
    int _nowAt;
    int _currentId;
    bool _aborted;
    bool _recoverable;
    QString lastHostName;
    quint16 lastPort;
    QHttp::ConnectionMode lastConnectionMode;
    QHttpRequestHeader lastRequest;
    QIODevice *lastDevice;

    void tryAgain();
    void dumpState(States state);

  private slots:
    void updatePointers(int done, int total);
    void processResult(bool error);
    void saveCurrentRequest(const QHttpResponseHeader &resp);
    void initPointers();
    void evaluateNewState(int state);

  public:
    Retry(Interface *interface);
    virtual ~Retry() { }

    void setTries(int tries) { _tries = tries; }
    int tries() const { return _tries; }

    int get(const QString &path, QIODevice *to = 0);
    int post(const QString &path, QIODevice *data, QIODevice *to = 0);
    int post(const QString &path, const QByteArray &data, QIODevice *to = 0);
    int request(const QHttpRequestHeader &header, QIODevice *data = 0, QIODevice *to = 0);
    int request(const QHttpRequestHeader &header, const QByteArray &data, QIODevice *to = 0);
    int setHost(const QString &hostName, quint16 port = 80);
    int setHost(const QString &hostName, QHttp::ConnectionMode mode, quint16 port = 0);

  public slots:
    void abort();

  signals:
    void progress(int currentTry, int done, int total);
    void completed(bool error, const QByteArray &data);
  };
}

#endif
