/*-----------------------------------------------------------------------------
 *  FILE: Applet.cc
 *
 *      Copyright(c) 2006 Gareth Foster.
 *
 *      Applet class which controls the program.
 *
 *-----------------------------------------------------------------------------
 */

#include "Applet.hh"
#include <config.h>
#include <fstream>
#include <sstream>
#include <cmath>
#include <gtkmm/main.h>
#include <glib/gi18n.h>
#include <libgnomeuimm.h>
#include "config.h"

/** @defgroup WPTray WPTray
 * WPTray contains classes used in the Wallpaper Tray program.
 */

//------------------------------------------------------------------------
/**
<DisplayForward>
@brief Callback handler for the forward event.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void DisplayForward
(
	BonoboUIComponent *,
	void *									_pApplet,
	const gchar *
)
{
	static_cast<Applet *>(_pApplet)->OnForwardActivated();
}// end DisplayForward

//------------------------------------------------------------------------
/**
<DisplaySearch>
@brief Callback handler for the search event.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void DisplaySearch
(
	BonoboUIComponent *,
	void *									_pApplet,
	const gchar *
)
{
	static_cast<Applet *>(_pApplet)->OnSearchActivated();
}// end DisplaySearch

//------------------------------------------------------------------------
/**
<DisplayDelete>
@brief Callback handler for the delete event.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void DisplayDelete
(
	BonoboUIComponent *,
	void *									_pApplet,
	const gchar *
)
{
	static_cast<Applet *>(_pApplet)->OnDeleteActivated();
}// end DisplayDelete

//------------------------------------------------------------------------
/**
<DisplayPreferences>
@brief Callback handler for the preferences event.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void DisplayPreferences
(
	BonoboUIComponent *,
	void *									_pApplet,
	const gchar *
)
{
	static_cast<Applet *>(_pApplet)->OnPreferencesActivated();
}// end DisplayPreferences

//------------------------------------------------------------------------
/**
<DisplayHelp>
@brief Callback handler for the help event.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void DisplayHelp
(
	BonoboUIComponent *,
	void *									_pApplet,
	const gchar *
)
{
	static_cast<Applet *>(_pApplet)->OnHelpActivated();
}// end DisplayHelp

//------------------------------------------------------------------------
/**
<DisplayAbout>
@brief Callback handler for the about event.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void DisplayAbout
(
	BonoboUIComponent *,
	void *									_pApplet,
	const gchar *
)
{
	static_cast<Applet *>(_pApplet)->OnAboutActivated();
}// end DisplayAbout

//------------------------------------------------------------------------
/**
<Applet::Applet>
@brief Contruct a new Applet object.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
Applet::Applet
(
	PanelApplet *							_pPanelApplet
) :
	m_pPanelApplet(_pPanelApplet),
	m_pPreferencesDialog(NULL),
	m_pSearchDialog(NULL),
	m_Wallpapers(this)
{
	// set up drag and drop targets:
	std::list<Gtk::TargetEntry> ls_DropTarget;
	ls_DropTarget.push_back(Gtk::TargetEntry("STRING"));
	ls_DropTarget.push_back(Gtk::TargetEntry("text/plain"));

	// make the image a dnd drop destination
	m_GtkImage.drag_dest_set(Gtk::DEST_DEFAULT_ALL, Gdk::DragAction(GDK_ACTION_COPY));
	m_GtkImage.drag_dest_add_image_targets();
	m_GtkImage.drag_dest_add_text_targets();
	m_GtkImage.drag_dest_add_uri_targets();

	// connect dnd signals
	m_GtkImage.signal_drag_data_received().connect(sigc::mem_fun(*this, &Applet::OnDropDragDataReceived));

	// set up the gconf client
	m_pGconfClient = Gnome::Conf::Client::get_default_client();

	m_pGconfClient->add_dir("/apps/wp_tray");

	// add our directory to the list of directories the GConfClient will watch
	m_pGconfClient->add_dir("/desktop/gnome/background");

	// listen to changes to our key
	m_nCnxnUpdateTooltip = m_pGconfClient->notify_add	(
			"/desktop/gnome/background/picture_filename",
			sigc::mem_fun(*this, &Applet::CreateTooltip)
		);

	// set a random wallpaper if we have been told to at logon
	if(m_pGconfClient->get_bool("/apps/wp_tray/b_wp_logon") == true)
	{
		Applet::RenderThumbnail(m_pGconfClient->get_string("/desktop/gnome/background/picture_filename"));

		Applet::SetThumbnailOnPanel();

		m_Wallpapers.SetRandom(false);
	}// end if
	else
	{
		Applet::RenderThumbnail(m_pGconfClient->get_string("/desktop/gnome/background/picture_filename"));

		Applet::SetThumbnailOnPanel();
	}// end else

	// create app icon (for help dialog)
	try
	{
		m_pGdkIcon = Gdk::Pixbuf::create_from_file(GNOMEICONDIR"/wp_tray-applet.png");
	}// end try
	catch(...)
	{
		// it's a minor problem if we can't find the icon
		m_pGdkIcon = Glib::RefPtr<Gdk::Pixbuf>();
	}// end catch

	// enable and create tooltips
	m_GtkTooltips.enable();
	Applet::CreateTooltip(m_nCnxnUpdateTooltip, m_pGconfClient->get_entry("/desktop/gnome/background/picture_filename"));

	m_GtkEventBox.add_events(Gdk::BUTTON_PRESS_MASK);

	m_GtkEventBox.signal_button_press_event().connect(sigc::mem_fun(*this, &Applet::OnClicked));

	m_GtkEventBox.add(m_GtkImage);
	Applet::add(m_GtkEventBox);

	Applet::show_all();
	
	// setup menu
	static const char str_MenuXml[] =
		"<popup name=\"button3\">\n"
		"   <menuitem name=\"Forward Item\"\n"
		"             verb=\"WpTrayForward\" _label=\"_Forward\"\n"
		"             pixtype=\"stock\" pixname=\"gtk-go-forward\"/>\n"
		"   <menuitem name=\"Search Item\"\n"
		"             verb=\"WpTraySearch\" _label=\"_Search\"\n"
		"             pixtype=\"stock\" pixname=\"gtk-find\"/>\n"
		"   <menuitem name=\"Delete Item\"\n"
		"             verb=\"WpTrayDelete\" _label=\"_Delete\"\n"
		"             pixtype=\"stock\" pixname=\"gtk-delete\"/>\n"
		"   <menuitem name=\"Properties Item\"\n"
		"             verb=\"WpTrayPreferences\" _label=\"_Preferences\"\n"
		"             pixtype=\"stock\" pixname=\"gtk-properties\"/>\n"
		"   <menuitem name=\"Help Item\"\n"
		"             verb=\"WpTrayHelp\" _label=\"_Help\"\n"
		"             pixtype=\"stock\" pixname=\"gtk-help\"/>\n"
		"   <menuitem name=\"About Item\"\n"
		"             verb=\"WpTrayAbout\" _label=\"_About\"\n"
		"             pixtype=\"stock\" pixname=\"gnome-stock-about\"/>\n"
		"</popup>\n";

	static const BonoboUIVerb ar_MenuVerbs[] =
	{
		BONOBO_UI_VERB("WpTrayForward", &DisplayForward),
		BONOBO_UI_VERB("WpTraySearch", &DisplaySearch),
		BONOBO_UI_VERB("WpTrayDelete", &DisplayDelete),
		BONOBO_UI_VERB("WpTrayPreferences", &DisplayPreferences),
		BONOBO_UI_VERB("WpTrayHelp", &DisplayHelp),
		BONOBO_UI_VERB("WpTrayAbout", &DisplayAbout),    
		BONOBO_UI_VERB_END
  	};

	panel_applet_setup_menu(m_pPanelApplet, str_MenuXml, ar_MenuVerbs, this);
}// end Applet::Applet

//------------------------------------------------------------------------
/**
<Applet::~Applet>
@brief Destroy an Applet object.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
Applet::~Applet
(
)
{
	if(m_pPreferencesDialog != NULL)
	{
		delete m_pPreferencesDialog;
	}// end if
}// end Applet::~Applet

//------------------------------------------------------------------------
/**
<Applet::GetContainer>
@brief Gets a pointer to the applet as a Container.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
Gtk::Container & Applet::GetContainer
(
)
{
	return *this;
}// end Applet::GetContainer

//------------------------------------------------------------------------
/**
<Applet::GetSize>
@brief Gets the size of the Applet.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
int Applet::GetSize
(
) const
{
	// FIXME: the panel appears to lie about its true size
	// 2 pixels of frame decoration on both sides
	return panel_applet_get_size(m_pPanelApplet) - 2 * 2;
}// end Applet::GetSize

//------------------------------------------------------------------------
/**
<Applet::Horizontal>
@brief Gets the size of the Applet.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
bool Applet::Horizontal
(
) const
{
	PanelAppletOrient orient = panel_applet_get_orient(m_pPanelApplet);

	return orient == PANEL_APPLET_ORIENT_UP || orient == PANEL_APPLET_ORIENT_DOWN;
}// end Applet::Horizontal

//------------------------------------------------------------------------
/**
<Applet::RenderThumbnail>
@brief Renders a thumbnail of the given wallpaper to the m_pThumbnail member.

@param _szFile The file to thumbnail.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::RenderThumbnail
(
	Glib::ustring							_szFile
)
{
	Glib::RefPtr<Gdk::Pixbuf> p_GdkPixbuf = Glib::RefPtr<Gdk::Pixbuf>();

	int n_TargetW, n_TargetH;

	// get thumbnail size
	if(Applet::Horizontal() == true)
	{
	    n_TargetW = static_cast<int>(Applet::GetSize() * 1.3f);
	    n_TargetH = Applet::GetSize();
	}// end if
	else
	{
	    n_TargetW = Applet::GetSize();
	    n_TargetH = static_cast<int>(Applet::GetSize() * 0.77f);
	}

	// attempt to create a thumbnail
	try
	{
		p_GdkPixbuf = Gdk::Pixbuf::create_from_file(_szFile);
	}// end try
	catch(...)
	{
		// bugger, slap a black square in there for now
	    p_GdkPixbuf = Gdk::Pixbuf::create	(
				Gdk::COLORSPACE_RGB, 0, 8,
				n_TargetW, n_TargetH
			);

		p_GdkPixbuf->fill(0x000000ff);

		m_GtkImage.set(p_GdkPixbuf);

		return;
	}// end catch

	// only scale down if we're sure the starting image is completely larger than the destination size
	if(p_GdkPixbuf->get_width() < n_TargetW || p_GdkPixbuf->get_height() < n_TargetH)
	{
		m_pThumbnail = p_GdkPixbuf;
	}// end if
	else
	{
	    m_pThumbnail = Gnome::UI::thumbnail_scale_down_pixbuf	(
				p_GdkPixbuf,
				n_TargetW,
				n_TargetH
			);
	}// end else
}// end Applet::RenderThumbnail

//------------------------------------------------------------------------
/**
<Applet::SetThumbnailOnPanel>
@brief Sets a previously rendered thumbnail on the panel as the programs icon.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::SetThumbnailOnPanel
(
)
{
	if(m_pThumbnail)
	{
		m_GtkImage.set(m_pThumbnail);
	}// end if
}// end Applet::SetThumbnailOnPanel

//------------------------------------------------------------------------
/**
<Applet::RenderAndSetThumbnail>
@brief Sets and renders a thumbnail of the current wallpaper on the panel as the programs icon.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::RenderAndSetThumbnail
(
)
{
	Applet::RenderThumbnail(m_pGconfClient->get_string("/desktop/gnome/background/picture_filename"));

	Applet::SetThumbnailOnPanel();
}// end Applet::RenderAndSetThumbnail

//------------------------------------------------------------------------
/**
<Applet::GetThumbnail>
@brief Gets the thumbnail currently saved in the applet.

@return The thumbnail currently saved in the applet.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
Glib::RefPtr<Gdk::Pixbuf> Applet::GetThumbnail
(
)
{
	return m_pThumbnail;
}// end Applet::GetThumbnail

//------------------------------------------------------------------------
/**
<Applet::OnForwardActivated>
@brief Responds to the user choosing forward from the context menu.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnForwardActivated
(
)
{
	m_Wallpapers.SetRandom(true);
}// end Applet::OnForwardActivated

//------------------------------------------------------------------------
/**
<Applet::OnSearchActivated>
@brief Responds to the user choosing search from the context menu.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnSearchActivated
(
)
{
	if(m_pSearchDialog == NULL)
	{
		// load the glade file and instiate its widgets
		Glib::RefPtr<Gnome::Glade::Xml> ref_xml;

		try
		{
			ref_xml = Gnome::Glade::Xml::create(WP_TRAY_GLADEDIR"wp_tray_search.glade");
		}// end try
		catch(const Gnome::Glade::XmlError & ex)
		{
			std::cerr << ex.what() << std::endl;
		}// end catch

		// get the glade-instantiated dialog
		ref_xml->get_widget_derived("window1", m_pSearchDialog);
	}// end if

	m_pSearchDialog->set_icon(m_pGdkIcon);

	m_pSearchDialog->show();
}// end Applet::OnSearchActivated

//------------------------------------------------------------------------
/**
<Applet::OnDeleteActivated>
@brief Responds to the user choosing delete from the context menu.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnDeleteActivated
(
)
{
	Gtk::MessageDialog msgDlg(_("Are you sure you want to delete this wallpaper from disk?"), false, Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO, true);

	int n_Result = msgDlg.run();
	
	switch(n_Result)
	{
		case Gtk::RESPONSE_YES:
		{
			m_Wallpapers.DeleteCurrent();
		}// end case
		break;
		
		default:
		{
			// do i care? no.
		}// end default
		break;
	}// end switch

	Applet::SetThumbnailOnPanel();
}// end Applet::OnDeleteActivated

//------------------------------------------------------------------------
/**
<Applet::OnPreferencesActivated>
@brief Responds to the user choosing preferences from the context menu.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnPreferencesActivated
(
)
{
	if(m_pPreferencesDialog == NULL)
	{
		// load the Glade file and instiate its widgets:
		Glib::RefPtr<Gnome::Glade::Xml> refXml;

		try
		{
			refXml = Gnome::Glade::Xml::create(WP_TRAY_GLADEDIR"wp_tray_conf.glade");
		}// end try
		catch(const Gnome::Glade::XmlError & ex)
		{
			std::cerr << ex.what() << std::endl;
		}// end catch

		// get the glade-instantiated dialog
		refXml->get_widget_derived("window1", m_pPreferencesDialog);
	}// end if

	m_pPreferencesDialog->show();
}// end Applet::OnPreferencesActivated

//------------------------------------------------------------------------
/**
<Applet::OnHelpActivated>
@brief Responds to the user choosing help from the context menu.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnHelpActivated
(
)
{
}// end Applet::OnHelpActivated

//------------------------------------------------------------------------
/**
<Applet::OnAboutActivated>
@brief Responds to the user choosing about from the context menu.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnAboutActivated
(
)
{
	std::vector<Glib::ustring> ar_szAuthors;
	ar_szAuthors.push_back("Gareth Foster <earthworm@planetearthworm.com>");
  
	std::vector<Glib::ustring> ar_szDocumenters;
	// TODO: add documenters here

	// note to translators: please fill in your names and email addresses
	Glib::ustring sz_Translators = _("translator-credits");

	Glib::ustring sz_Description =
		_("Wallpaper Tray is a wallpaper utility that sits on your GNOME panel. "
		"It gives you a random wallpaper from a chosen directory at logon, on a timer, and allows you to "
		"select a new wallpaper at random from its menu.");
  
	if(m_pAboutBox.get() == 0)
	{
		m_pAboutBox.reset	(
			new Gnome::UI::About	(
					_("Wallpaper Tray"), VERSION,
					_("Copyright © 2006 Gareth Foster"),
					ar_szAuthors, ar_szDocumenters, sz_Description,
					sz_Translators == "translator-credits"
					? "" : sz_Translators,
					m_pGdkIcon
				)
			);

		m_pAboutBox->set_icon(m_pGdkIcon);
		m_pAboutBox->show();
	}
	else
	{
		m_pAboutBox->show();
		m_pAboutBox->raise();
	}// end else
}// end Applet::OnAboutActivated

//------------------------------------------------------------------------
/**
<Applet::OnDropDragDataReceived>
@brief Called when a user drops an object onto our applet.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::OnDropDragDataReceived
(
	const Glib::RefPtr<Gdk::DragContext> &	_context,
	int										_nX,
	int										_nY,
	const Gtk::SelectionData &				_selectionData,
	guint ,
	guint									_nTime
)
{
	if(_selectionData.get_text().find("http://") != Glib::ustring::npos || _selectionData.get_text().find("ftp://") != Glib::ustring::npos)
	{
		// TODO: copy the wallpaper to the collection
	}// end if
	else
	{
		// TODO: local file, move it to the collection
	}// end else

	_context->drag_finish(false, false, _nTime);
}// end Applet::OnDropDragDataReceived

//------------------------------------------------------------------------
/**
<Applet::OnClicked>
@brief Called when the user clicks the applet.

@param _refpGdkEventButton A C struct with details about the click.
@return Whether or not we handled the event. 
@date 15-11-2006 GAF Written
**/
//------------------------------------------------------------------------
bool Applet::OnClicked
(
	GdkEventButton * const &				_refpGdkEventButton
)
{
	if(_refpGdkEventButton->button == 1)
	{
		m_Wallpapers.SetRandom(false);

		return true;
	}// end if
	else
	{
		return false;
	}// end else
}// end Applet::OnClicked

//------------------------------------------------------------------------
/**
<Applet::CreateTooltip>
@brief Called when the wallpaper changes to update the applets tooltip with its location.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void Applet::CreateTooltip
(
	guint									_nCnxn,
	Gnome::Conf::Entry						_GconfEntry
)
{
	std::ostringstream ss;
	ss.imbue(std::locale("")); // use the user's locale for this stream

	// create tip text
	ss << _("Wallpaper Tray") << " ";
	ss << VERSION << " ";
	ss << m_pGconfClient->get_string("/desktop/gnome/background/picture_filename");

	// attach it
	m_GtkTooltips.set_tip(*this, Glib::locale_to_utf8(ss.str()));
}// end Applet::CreateTooltip
