#!/usr/bin/env python
#
# WAJIG - Debian Command Line System Administrator
#
# Copyright (c) Graham.Williams@togaware.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version. See the file LICENSE.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

########################################################################
# Standard python modules
#
import getopt,os,string,sys,re,tempfile

########################################################################
# Wajig modules
#
import documentation,commands,changes,perform

########################################################################
# Global Variables
#
progname = "wajig"
pause = False

#------------------------------------------------------------------------
#
# SUPPORT METHODS
#
#------------------------------------------------------------------------

def requires_no_args(command, args):
    if len(args) > 1: 
        print progname + ": Error: " + command +\
	      " requires no further arguments"
	finishup(1)

def requires_one_arg(command, args, message):
    if len(args) != 2:
        print progname + ": Error: " + command +\
	      " requires " + message
	finishup(1)

def requires_opt_arg(command, args, message):
    if len(args) > 2:
        print progname + ": Error: " + command +\
	      " requires " + message
	finishup(1)

def requires_args(command, args, type):
    if len(args) == 1: 
        print progname + ": Error: " + command +\
	      " requires " + type
	finishup(1)

def requires_package(package, path):
    if not os.path.exists(path):
        print 'The "' + package + '" package does not appear to be installed.'
        print 'Consider installing it with "wajig install ' + package + '".'
        finishup(1)

def finishup(code=0):
    global pause
    if pause:
        print "Press Enter to continue...",
        sys.stdin.readline()
    sys.exit(code)
#------------------------------------------------------------------------
#
# MAIN PROGRAM
#
#------------------------------------------------------------------------

def main():
    global pause
    #
    # Remove commas and insert the arguments appropriately.
    #
    #print sys.argv
    oldargv = sys.argv
    sys.argv = oldargv[0:2]
    for i in range(2,len(oldargv)):
        sys.argv += oldargv[i].split(",")
    #print sys.argv
    #
    # Process any command line options
    #
    try:
	#
	# Parse the command line arguments
	#
	sopts = "dhpqstv"
	lopts = ["debug", "help", "pause", "quiet","simulate","teaching",
                 "verbose=", "version"]
        opts, args = getopt.getopt(sys.argv[1:], sopts, lopts)
    except getopt.error, e:
	#
        # On error print help information and exit:
	#
	print e
        documentation.usage()
        finishup(2)
    #
    # Initialise variables
    #
    simulate = 0
    teaching = 0
    verbose  = 0
    debug = False
    #
    # Action the command line options
    #
    for o, a in opts:
        if o in ("-h", "--help"):
            documentation.usage()
            finishup()
        elif o in ("-d", "--debug"):
	    debug = True
        elif o in ("-p", "--pause"):
	    pause = True
            perform.pause = True
        elif o in ("-q", "--quiet"):
	    commands.set_quiet()
        elif o in ("-s", "--simulate"):
	    simulate = simulate + 1
            perform.set_simulate_level(simulate)
        elif o in ("-t", "--teaching"):
	    teaching = teaching + 1
            perform.set_teaching_level(teaching)
        elif o == "-v":
	    verbose = verbose + 1 
            commands.set_verbosity_level(verbose)
        elif o == "--verbose":
            verbose = string.atoi(a)
            commands.set_verbosity_level(verbose)
        elif o == "--version":
	    documentation.version()
 	    finishup()

    #
    # If no command then print usage
    #
    #print args
    #sys.exit(0)
    if len(args) == 0:
      documentation.usage()
      finishup()

    #
    # Process the command. Lowercase it so that we allow any case
    # for commands and allow hyphens and underscores and slash.
    #
    command = re.sub('-|_|/', '', string.lower(args[0]))
 
    #
    # Provide help up front - we don't need to initialise the system to give help
    #
    if   command == "doc" or command == "docs" or command == "documentation":
        requires_no_args(command, args)
	verbose=2
	documentation.help(verbose)
	finishup(0)
    elif command == "help":
        requires_no_args(command, args)
	documentation.help(verbose)
	finishup(0)

    #
    # Before we do any other command make sure the right files exist.
    #
    changes.ensure_initialised()

    if debug:
        select_command(command, args, verbose, teaching)
    else:
        try:
            select_command(command, args, verbose, teaching)
        except:
            print "Exiting...."
    finishup(0)

def select_command(command, args, verbose, teaching):
    """Select the appropriate command and execute it.

    This function was separated out of the main so that I could wrap
    it up with exception handling.
    """
    if   command == "addcdrom" or command == "cdromadd":
	requires_no_args(command, args)
        perform.execute("apt-cdrom add",root=1)

    elif command == "autodownload":
	requires_no_args(command, args)
        if verbose > 0:
            commands.do_update()
            filter_str = ""
        else:
            commands.do_update(quiet=1)
            filter_str = "| egrep -v '(http|ftp)'"
        perform.execute("apt-get --download-only --show-upgraded " +\
                        "--assume-yes dist-upgrade " + filter_str,
                        root=1)
	commands.do_describe_new()
	commands.do_newupgrades()

    elif command == "autoclean":
	requires_no_args(command, args)
        perform.execute("apt-get autoclean",root=1)

    elif command == "autoinstall":
	requires_args(command, args, "a list of package names")
        command = "apt-get install --assume-yes " + perform.concat(args[1:])
	perform.execute(command, root=1)

    elif command == "available" or command == "avail":
        requires_args(command, args, "a list of packages")
	perform.execute("apt-cache policy " + perform.concat(args[1:]))

    elif command == "bug" or command == "bugs":
        requires_one_arg(command, args, "a single named package")
	requires_package("reportbug", "/usr/bin/reportbug")
        perform.execute("reportbug " + args[1])

    elif command == "build":
        requires_args(command, args, "a list of package names")
        requires_package("fakeroot", "/usr/bin/fakeroot")
	perform.execute("fakeroot " +
                            "apt-get source -b " + perform.concat(args[1:]))

    elif command == "builddepend":
        requires_args(command, args, "a list of package names")
	perform.execute("apt-get build-dep " + perform.concat(args[1:]),
                            root=1)

    elif command == "changelog":
        requires_args(command, args, "a list of packages")
        commands.do_changelog(args[1:])

    elif command == "clean":
	requires_no_args(command, args)
        perform.execute("apt-get clean",root=1)

    elif command == "dailyupgrade":
        requires_no_args(command, args)
        commands.do_update()
        perform.execute("apt-get --show-upgraded dist-upgrade", root=1)

    elif command == "dependees":
        requires_one_arg(command, args, "package name")
        command = "apt-cache showpkg " + args[1] +\
                  " | awk \'BEGIN{found=0}/^Reverse Depends:/{found=1;next}/^[^ ]/{found=0}found==1{print}{next}\' " +\
                  " | sed \'s|Reverse Depends:||\' | tr \",\" \" \" " +\
                  " | awk \'{print $1}\' | sort -u | grep -v \'^$\'"
        perform.execute(command)

    elif command == "describe":
        requires_args(command, args, "a list of packages")
	commands.do_describe(args[1:])

    elif command == "describenew" or \
         command == "newdescribe":
        requires_no_args(command, args)
	commands.do_describe_new()

    elif command == "detail" or \
         command == "details" or \
         command == "show":
        requires_args(command, args, "a list of packages")
        verbose = 2
        commands.set_verbosity_level(verbose)
	commands.do_describe(args[1:])

    elif command == "detailnew" or \
         command == "newdetail":
        requires_no_args(command, args)
        verbose = 2
        commands.set_verbosity_level(verbose)
	commands.do_describe_new()

    elif command == "distupgrade":
        requires_no_args(command, args)
        perform.execute("apt-get -u dist-upgrade", root=1)

    elif command == "download":
        requires_args(command, args, "a list of packages")
        pkgs = args[1:]
        if len(pkgs) == 1 and pkgs[0] == "-":
            pkgs = string.split(
                string.join(map(string.strip,sys.stdin.readlines())))
        elif len(pkgs) == 2 and pkgs[0] == "-f":
            pkgs = string.split(string.join(map(string.strip,(open(pkgs[1])).readlines())))
        #
        # Since no messages are printed for the command, print message here.
        #
        print "Packages being downloaded to /var/cache/apt/archives..."
        #
        # Do the download, non-interactively (--quiet),
        # and force download for already installed packages (--reinstall)
        #
	perform.execute("apt-get --quiet=2 --reinstall " \
                        + "--download-only install " + perform.concat(pkgs),
                        root=1)

    elif command == "editsources" or \
         command == "setup":
	requires_no_args(command, args)
	requires_package("apt-setup", "/usr/sbin/apt-setup")
        perform.execute("apt-setup", root=1)

    elif command == "filedownload" or \
         command == "downloadfile":
        requires_one_arg(command, args, "a file name containing list of packages")
        pkgs = string.join(map(string.strip,(open(args[1])).readlines()))
	perform.execute("apt-get --download-only install " + pkgs, root=1)

    elif command == "fileinstall" or \
         command == "installfile":
        requires_one_arg(command, args, "a file name containing list of packages")
        pkgs = string.join(map(string.strip,(open(args[1])).readlines()))
        perform.execute("apt-get install " + pkgs, root=1)

    elif command == "fileremove" or \
         command == "removefile":
        requires_one_arg(command, args, "a file name containing list of packages")
        pkgs = string.join(map(string.strip,(open(args[1])).readlines()))
        perform.execute("apt-get remove " + pkgs, root=1)

    elif command == "findfile":
        requires_one_arg(command, args, "a file name")
        perform.execute("dpkg --search " + args[1])

    elif command in ["findpkg", "unofficial"]:
        requires_one_arg(command, args, "a package name")
        commands.do_findpkg(args[1])

    elif command == "fixconfigure":
	requires_no_args(command, args)
        perform.execute("dpkg --configure -a",root=1)

    elif command == "fixinstall":
	requires_no_args(command, args)
        perform.execute("apt-get --fix-broken install",root=1)

    elif command == "fixmissing":
	requires_no_args(command, args)
        perform.execute("apt-get --fix-missing upgrade",root=1)

    elif command == "force":
        requires_args(command, args, "a package name")
        commands.do_force(args[1:])

    elif command == "geturl":
        requires_one_arg(command, args, "a package name")
        # Not yet quite working
	perform.execute("/usr/lib/apt-move/fetch -t " + args[1], root=1)

    elif command == "hold":
        requires_args(command, args, "a list of packages to place on hold")
	commands.do_hold(args[1:])
	# TODO Perhaps I can use map to "execute" over each package

    elif command == "init":
        requires_no_args(command, args)
        changes.reset_files()

    elif command == "install" or \
         command == "isntall" :
        #
        # Okay, so I'm sometimes dyslexic :-)
        #
        requires_args(command, args, "a list of packages, .deb files, or url")
	commands.do_install(args[1:])

    elif command == "installr" or \
         command == "recommended":
        requires_args(command, args, "a list of packages")
	commands.do_install_suggest(args[1:], "Recommends")

    elif command == "installrs":
        requires_args(command, args, "a list of packages")
	commands.do_install_suggest(args[1:], "Both")

    elif command == "installs" or \
         command == "suggested":
        requires_args(command, args, "a list of packages")
	commands.do_install_suggest(args[1:], "Suggests")

    elif re.compile(r'install.*').match(command):
        # For example: install/unsable
        requires_args(command, args, "a list of packages, .deb files, or url")
        command = "apt-get --target-release %s install %s" %\
                  (re.compile(r'install').sub("", command), perform.concat(args[1:]))
        perform.execute(command, root=1)

    elif command == "integrity":
        requires_no_args(command, args)
	perform.execute("debsums -s -a")

    elif command == "large":
        commands.do_size(args[1:], 10000)

    elif command == "lastupdate":
        requires_no_args(command, args)
	perform.execute("/bin/ls -l --full-time " +\
			    changes.available_file +\
			    " 2>/dev/null |awk '{printf \"Last update was " +\
		            "%s %s %s\\n\"" +\
			    ", $6, $7, $8}' | sed 's|\.000000000||'")

    elif command == "list" or command == "listwide":
        requires_opt_arg(command, args, "at most one argument")
        cmd = ""
        if command == "listwide": cmd += "COLUMNS=200 " 
        cmd += "dpkg --list '*' | grep -v 'no description avail'"
        if len(args) > 1:
            cmd += " | egrep '" + args[1] + "' | sort"
        perform.execute(cmd)

    elif command == "listall":
        requires_no_args(command, args)
	perform.execute("apt-cache dumpavail |" +\
                            "egrep \"^(Package|Description): \" |" +\
                            "awk '/^Package: /{pkg=$2} /^Description: /" +\
                                 "{printf(\"%-24s %s\\n\", pkg," +\
                                 "substr($0,13))}' |" +\
                            "sort -u")

    elif command == "listalts" or command == "listalternatives":
        requires_no_args(command, args)
	perform.execute("ls /etc/alternatives/ | " +\
			    "egrep -v '(\.1|\.1\.gz)$'")

    elif command == "listcache":
        requires_opt_arg(command, args, "optional string to filter on")
	cmd = "printf 'Found %d files %s in the cache.\n\n'\
        $(ls /var/cache/apt/archives/ | wc -l) \
        $(ls -sh /var/cache/apt/archives/ | head -1 | awk '{print $2}')"
        perform.execute(cmd)
        cmd = "ls /var/cache/apt/archives/"
        if len(args) == 2:
            cmd = cmd + " | grep '" + args[1] + "'"
        cmd += "; echo"
	perform.execute(cmd)

    elif command == "listcommands" or command == "commands":
        requires_no_args(command, args)
        verbose=1
	documentation.help(verbose)

    elif command == "listdaemons":
        requires_no_args(command, args)
	perform.execute("printf 'Found %d daemons in /etc/init.d.\n\n'\
        $(ls /etc/init.d/ | \
        egrep -v '(~$|README|-(old|dist)|\.[0-9]*$)' | wc -l)")
	perform.execute("ls /etc/init.d/ | \
        egrep -v '(~$|README|-(old|dist)|\.[0-9]*$)' |\
        pr --columns=3 --omit-header")

    elif command == "listfiles":
        requires_one_arg(command, args, "the name of a single Debian package")
        perform.execute("dpkg --listfiles " + args[1])

    elif command == "listhold":
        requires_no_args(command, args)
	perform.execute("dpkg --get-selections | egrep 'hold$' | cut -f1")

    elif command == "listinstalled":
	requires_opt_arg(command, args, "at most one argument")
        commands.do_listinstalled(args[1:])
        
    elif command == "listnames":
	requires_opt_arg(command, args, "at most one argument")
	commands.do_listnames(args[1:])

    elif command == "liststatus":
        requires_opt_arg(command, args, "at most one argument")
        cmd = "COLUMNS=400 " 
        cmd += "dpkg --list '*' | grep -v 'no description avail'"
        cmd += " | awk '{print $1,$2}'"
        if len(args) > 1:
            cmd += " | egrep '" + args[1] + "' | sort"
        perform.execute(cmd)

    elif command == "localdistupgrade":
        requires_no_args(command, args)
        perform.execute("apt-get --no-download --ignore-missing " \
                            + "--show-upgraded dist-upgrade", root=1)

    elif command == "localupgrade":
        requires_no_args(command, args)
        perform.execute("apt-get --no-download --ignore-missing " \
                            + "--show-upgraded upgrade", root=1)

    elif command == "move":
        requires_no_args(command, args)
	perform.execute("apt-move update", root=1)
        # Then clean out the cached archive.
        perform.execute("apt-get clean",root=1)

    elif command == "new":
        requires_no_args(command, args)
	commands.do_describe_new()

    elif command == "news":
        requires_args(command, args, "a list of packages")
        commands.do_news(args[1:])

    elif command == "newupgrades" or command == "newupgrade":
        requires_no_args(command, args)
	commands.do_newupgrades()

    elif command == "nonfree":
        requires_no_args(command, args)
	requires_package("vrms", "/usr/bin/vrms")
        perform.execute("vrms")

    elif command == "orphans" or command == "listorphans":
        requires_no_args(command, args)
	requires_package("deborphan", "/usr/bin/deborphan")
        perform.execute("deborphan")

    elif command == "policy":
	perform.execute("apt-cache policy " + perform.concat(args[1:]))

    elif command == "purge":
        requires_args(command, args, "a list of packages")
	perform.execute("dpkg --purge " + perform.concat(args[1:]), root=1)

    elif command == "purgedepend":
        requires_one_arg(command, args, "a single package")
        commands.do_removedepend(args[1], purge = True)

    elif command == "purgeorphans":
        requires_no_args(command, args)
	requires_package("deborphan", "/usr/bin/deborphan")
	perform.execute("sh -c \"deborphan | xargs -r dpkg --purge\"", root=1)

    elif command == "recdownload":
        requires_args(command, args, "a list of packages")
        commands.do_recdownload(args[1:])

    elif command == "reconfigure":
        # With no args this will run gkdebconf
        if len(args) > 1:
            perform.execute("dpkg-reconfigure " +\
                            perform.concat(args[1:]),
                            root=1)
        else:
            perform.execute("gkdebconf", root=1)

    elif command == "reinstall":
        requires_args(command, args, "a list of packages")
	perform.execute("apt-get --reinstall install " +\
		   	     perform.concat(args[1:]), root=1)

    elif command == "reload":
        requires_one_arg(command, args, "name of service to " + command)
	perform.execute("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "remove":
        requires_args(command, args, "a list of packages")
	perform.execute("apt-get remove " + perform.concat(args[1:]), root=1)

    elif command == "removedepend":
        requires_one_arg(command, args, "a single package")
        #print changes.get_dependees("libclan2-mikmod")
        commands.do_removedepend(args[1])
            
    elif command == "removeorphans":
        requires_no_args(command, args)
	perform.execute("sh -c \"deborphan | xargs -r dpkg --remove\"", root=1)

    elif command == "repackage" or command == "package":
        requires_one_arg(command, args, "name of an installed package")
	requires_package("dpkg-repack", "/usr/bin/dpkg-repack")        
	perform.execute("dpkg-repack " + args[1], root=1)

    elif command == "reset":
        requires_no_args(command, args)
	changes.reset_files()

    elif command == "restart":
        requires_one_arg(command, args, "name of service to " + command)
	perform.execute("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "rpminstall":
        requires_one_arg(command, args, "a Red Hat package file name (.rpm)")
	perform.execute("alien --to-deb --install " + args[1], root=1)
        
    elif command == "rpmtodeb" or command == "rpm2deb":
        requires_one_arg(command, args, "a Red Hat package file name (.rpm)")
	perform.execute("alien -d " + args[1], root=1)

    elif command == "search":
        requires_args(command, args, "a list of words to search for")
	perform.execute("apt-cache search " + perform.concat(args[1:]))

    elif command == "searchapt":
        requires_one_arg(command, args, "one of stable|testing|unstable")
	requires_package("netselect-apt", "/usr/bin/netselect-apt")
        perform.execute("netselect-apt " + args[1], root=1)

    elif command == "showdistupgrade":
        requires_no_args(command, args)
        perform.execute("apt-get -u -s dist-upgrade", root=1)

    elif command == "showinstall":
        requires_args(command, args, "a list of packages")
        perform.execute("apt-get -u -s install " + perform.concat(args[1:]), root=1)

    elif command == "showremove":
        requires_args(command, args, "a list of packages")
        perform.execute("apt-get -u -s remove " + perform.concat(args[1:]), root=1)

    elif command == "showupgrade":
        requires_no_args(command, args)
        perform.execute("apt-get -u -s upgrade", root=1)

    elif command == "size" or command == "sizes":
        commands.do_size(args[1:], 0)

    elif command == "source":
        requires_args(command, args, "a list of package names")
	perform.execute("apt-get source " + perform.concat(args[1:]))

    elif command == "start":
        requires_one_arg(command, args, "name of service to " + command)
	perform.execute("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "status":
	commands.do_status(args[1:])

    elif command == "statusmatch" or command == "statussearch":
        requires_one_arg(command, args, "a search string for the package name")
        pkgs = map(lambda s: s.strip(),
                   commands.do_listnames(args[1:], pipe=True).readlines())
        if len(pkgs) > 0:
            commands.do_status(pkgs)
        else:
            print "No packages found matching '%s'" % args[1]
        #
        # Simplest thing to do is call wajig again.  Not the right way
        # but works for now.
        #
        # This was too slow and was not stopping when killed!
	#perform.execute("apt-cache search " \
        #                    + perform.concat(args[1:]) \
        #                    + " | awk '{print $1}' " \
        #                    + " | xargs wajig status ")

    elif command == "stop":
        requires_one_arg(command, args, "name of service to " + command)
	perform.execute("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "tasksel":
        requires_no_args(command, args)
	requires_package("gnome-tasksel", "/usr/bin/gnome-tasksel")
        perform.execute("gnome-tasksel", root = True)

    elif command == "toupgrade":
        requires_no_args(command, args)
        commands.do_toupgrade()

    # edd 03 Sep 2003  unhold patch based on hold semantics
    elif command == "unhold":
        requires_args(command, args, "a list of packages to remove from hold")
        commands.do_unhold(args[1:])
        # TODO Perhaps I can use map to "execute" over each package

    elif command == "update":
        requires_no_args(command, args)
        commands.do_update()

    # For testing only!
    elif command == "updateavailable":
        requires_no_args(command, args)
        changes.update_available()

    elif command == "updatealts" or command == "updatealternatives" or \
         command == "setalts" or command == "setalternatives":
        requires_one_arg(command, args, "name of alternative to update")
        perform.execute("update-alternatives --config " + args[1], root=1)

    elif command == "upgrade":
	if len(args) > 1:
	    perform.execute("apt-get install " + perform.concat(args[1:]),
				root=1)
        else:
            perform.execute("apt-get -u upgrade", root=1)

    elif command == "upgradesecurity":
        sources_list = tempfile.mkstemp(".security", "wajig.", "/tmp")[1]
        sources_file = open(sources_list, "w")
        # check dist
        sources_file.write("deb http://security.debian.org/ " +\
                           "testing/updates main contrib non-free\n")
        sources_file.close()
        command = "apt-get --no-list-cleanup -o Dir::Etc::SourceList=" +\
                  sources_list + " update"
        perform.execute(command, root=1)
        command = "apt-get -o Dir::Etc::SourceList=" +\
                  sources_list + " upgrade"
        perform.execute(command, root=1)
        if os.path.exists(sources_list): os.remove(sources_list)

    elif command == "version":
        documentation.version()

    elif command == "whatis":
        requires_args(command, args, "a list of package names")
	commands.do_describe(args[1:])

    elif command == "whichpkg" or command == "whichpackage":
        requires_one_arg(command, args, "a file name (possibly with a path)")
	commands.do_whichpkg(args[1])

    else:
        if command == args[0]:
            print "The command `" + command + "' was not recognised."
        else:
            print "The command `" + command + "' (entered as `" + args[0] + \
                  "') was not recognised."
        print "Perhaps it is not yet implemented or you misspelt the command."
        print "Try `wajig help' for further information."
        print "Try `wajig list-commands' for list of all commands."

#------------------------------------------------------------------------
#
# Start it all
#
#------------------------------------------------------------------------
if __name__ == '__main__':
    main()

