/*
 * Copyright (C) 1992 by Software Research Associates, Inc.
 *	Author:	Y. Kawabe <kawabe@sra.co.jp>
 *
 * Permission to use, copy, modify, and distribute, and sell this software
 * and its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  Software Research Associates
 * makes no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 */

#ifndef nls_wstring_h
#define nls_wstring_h

#include <NLS/wchar.h>
#include <hyperg/OS/enter-scope.h>

class Locale;
class String;

/*
 * class WString
 */
    
class WString {
  public:
    WString();
    WString(const WChar*);
    WString(const WChar*, int length);
    virtual ~WString();

  public:
    int length() const;
    const WChar* string() const;
    const WChar& item (int index) const;
    const WChar& operator [] (int index) const;

  public:
    virtual WString substr (int start, int length) const;
    WString left(int length) const;
    WString right(int start) const;

    virtual void set_to_substr(int start, int length);
    void set_to_left(int length);
    void set_to_right(int start);

    virtual int search(int start, const WChar&) const;
    int index(const WChar&) const;
    int rindex(const WChar&) const;

    virtual void reverse();

  public:
    virtual WString& operator = (const WString&);
    virtual boolean operator == (const WString&) const;
    virtual boolean operator != (const WString&) const;

    boolean null_terminated() const;

  public:
    /* Construct with CodeSet with Locale */
    WString(const Locale& locale, const char*);
    WString(const Locale& locale, const char*, int len);
    WString(const Locale& locale, const String&);

    /* Construct with CodeSet with Default Locale */
    WString(const char*);
    WString(const char*, int len);
    WString(const String&);

    /* return string with Locale */
    char* string(const Locale& locale) const;
    
  protected:
    const WChar* data_;
    int length_;
    boolean allocated_;
    boolean null_terminated_;
};

inline const WChar* WString::string() const { return data_; }
inline int WString::length() const { return length_; }
inline const WChar& WString::item(int index) const { return data_[index]; }
inline const WChar& WString::operator [] (int i) const { return data_[i]; }
inline boolean WString::null_terminated() const { return null_terminated_; }
inline WString WString::left(int length) const { return substr(0, length); }
inline WString WString::right(int start) const { return substr(start, -1); }
inline void WString::set_to_left(int length) { set_to_substr(0, length); }
inline void WString::set_to_right(int start) { set_to_substr(start, -1); }
inline int WString::index(const WChar &c) const { return search(0, c); }
inline int WString::rindex(const WChar &c) const { return search(-1, c); }

/*
 * class CopyWString
 */

class CopyWString : public WString {
  public:
    CopyWString();
    CopyWString(const WChar*);
    CopyWString(const WChar*, int length);
    CopyWString(const WString&);

  protected:
    void copy(const WChar*, int length);
};

/*
 * class UniqueWString
 */

class UniqueWStringPool;

class UniqueWString : public WString {
  public:
    UniqueWString();
    UniqueWString(const WString&);

  public:
    virtual boolean operator == (const WString&) const;
    virtual boolean operator != (const WString&) const;
    
  private:
    static UniqueWStringPool *pool_;
};

#endif
